<?php
// Simple non-session version of the message API for testing
// This bypasses the session checks for testing purposes

// Initialize response array
$response = [
    'status' => 'success',
    'message' => 'Test message sent successfully',
    'data' => null
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate required fields
    $requiredFields = ['recipient_id', 'message'];
    $missingFields = [];
    
    foreach ($requiredFields as $field) {
        if (!isset($_POST[$field]) || empty(trim($_POST[$field]))) {
            $missingFields[] = $field;
        }
    }
    
    if (!empty($missingFields)) {
        $response['status'] = 'error';
        $response['message'] = 'Missing required fields: ' . implode(', ', $missingFields);
    } else {
        // Get and sanitize inputs
        $senderId = 1; // Hardcoded for testing
        $recipientId = intval($_POST['recipient_id']);
        $message = trim($_POST['message']);
        $courseId = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        try {
            // Connect to database
            require_once '../../config/db_connect.php';
            
            if ($conn) {
                // Insert the message directly - skip teacher validation for testing
                $insertQuery = "INSERT INTO messages (sender_id, recipient_id, message_text, is_read, sent_at) 
                               VALUES (?, ?, ?, 0, NOW())";
                $stmt = $conn->prepare($insertQuery);
                $stmt->bind_param("iis", $senderId, $recipientId, $message);
                
                if ($stmt->execute()) {
                    $messageId = $conn->insert_id;
                    
                    // Get the inserted message with sender details
                    $messageQuery = "SELECT 
                                    m.message_id,
                                    m.sender_id,
                                    u.first_name as sender_first_name,
                                    u.last_name as sender_last_name,
                                    u.profile_picture as sender_profile_picture,
                                    m.recipient_id,
                                    m.message_text,
                                    m.is_read,
                                    m.sent_at
                                  FROM messages m
                                  JOIN users u ON m.sender_id = u.user_id
                                  WHERE m.message_id = ?";
                    $stmt = $conn->prepare($messageQuery);
                    $stmt->bind_param("i", $messageId);
                    $stmt->execute();
                    $msgResult = $stmt->get_result();
                    
                    if ($msgData = $msgResult->fetch_assoc()) {
                        // Success
                        $response['status'] = 'success';
                        $response['message'] = 'Message sent successfully';
                        $response['data'] = $msgData;
                    } else {
                        $response['status'] = 'success';
                        $response['message'] = 'Message sent but could not fetch details';
                    }
                } else {
                    $response['status'] = 'error';
                    $response['message'] = 'Could not send message: ' . $stmt->error;
                }
            }
        } catch (Exception $e) {
            $response['status'] = 'error';
            $response['message'] = 'Error: ' . $e->getMessage();
        }
    }
}

// Return JSON response
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>
