<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors, just log them

// Start output buffering to prevent any output before headers
ob_start();

session_start();

// Live server session configuration
ini_set('session.cookie_path', '/');
ini_set('session.cookie_domain', '.panaditeacademy.com');
ini_set('session.cookie_secure', 1);
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_samesite', 'Lax');

header('Content-Type: application/json');

// Function to ensure JSON response even on fatal errors
function sendJsonError($message, $code = 500) {
    ob_clean(); // Clear any output
    http_response_code($code);
    echo json_encode([
        'success' => false,
        'message' => $message
    ]);
    exit;
}

// Set error handler for fatal errors
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        sendJsonError('Server error: ' . $error['message']);
    }
});

// Include necessary files - EXACT same path as working submit_review.php
require_once '../../includes/db_connect.php';

// Try to include functions, but don't fail if not available
if (file_exists('../../includes/functions.php')) {
    require_once '../../includes/functions.php';
}

// Verify database connection (variable might be different in db_connect.php)
if (!isset($conn) && !isset($connection) && !isset($pdo)) {
    sendJsonError('Database connection not found after including db_connect.php');
}

// Use whichever connection variable exists
if (!isset($conn)) {
    if (isset($connection)) {
        $conn = $connection;
    } elseif (isset($pdo)) {
        $conn = $pdo;
    }
}

// Simple encryption/decryption functions if not available
if (!function_exists('encryptId')) {
    function encryptId($id) {
        return base64_encode($id);
    }
}

if (!function_exists('decryptId')) {
    function decryptId($encrypted) {
        $decoded = base64_decode($encrypted);
        return is_numeric($decoded) ? $decoded : false;
    }
}

// Enhanced decrypt function with multiple fallback strategies
function decryptReviewId($encrypted_id) {
    if (empty($encrypted_id)) {
        return false;
    }
    
    // Try direct decryption first
    if (function_exists('decryptId')) {
        $decrypted = decryptId($encrypted_id);
        if ($decrypted !== false && is_numeric($decrypted)) {
            return $decrypted;
        }
    }
    
    // Try base64 decode
    $base64_decoded = base64_decode($encrypted_id);
    if ($base64_decoded !== false && is_numeric($base64_decoded)) {
        return $base64_decoded;
    }
    
    // Try URL decode then decryption
    $url_decoded = urldecode($encrypted_id);
    if (function_exists('decryptId')) {
        $decrypted = decryptId($url_decoded);
        if ($decrypted !== false && is_numeric($decrypted)) {
            return $decrypted;
        }
    }
    
    // If all else fails and it's already numeric, use it
    if (is_numeric($encrypted_id)) {
        return $encrypted_id;
    }
    
    return false;
}

try {
    // Check if user is authenticated
    $user_id = null;
    $authenticated = false;

    // Primary authentication: session
    if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
        $user_id = $_SESSION['user_id'];
        $authenticated = true;
    }
    // Backup authentication: form data with token verification
    elseif (isset($_POST['user_id']) && isset($_POST['user_token'])) {
        $posted_user_id = $_POST['user_id'];
        $posted_token = $_POST['user_token'];
        $expected_token = md5($posted_user_id . 'panadite_review_token');
        
        if ($posted_token === $expected_token && is_numeric($posted_user_id)) {
            $user_id = $posted_user_id;
            $authenticated = true;
        }
    }

    if (!$authenticated || !$user_id) {
        throw new Exception('Authentication required');
    }

    // Get and validate input data
    $review_id = $_POST['review_id'] ?? '';
    $encrypted_course_id = $_POST['course_id'] ?? '';
    $rating = $_POST['rating'] ?? '';
    $review_title = trim($_POST['review_title'] ?? '');
    $review_text = trim($_POST['review_text'] ?? '');

    // Validate required fields
    if (!$review_id || !is_numeric($review_id)) {
        throw new Exception('Invalid review ID');
    }

    if (!$encrypted_course_id) {
        throw new Exception('Course ID is required');
    }

    if (!$rating || !in_array($rating, ['1', '2', '3', '4', '5'])) {
        throw new Exception('Please provide a valid rating (1-5 stars)');
    }

    if (strlen($review_title) < 3) {
        throw new Exception('Review title must be at least 3 characters long');
    }

    if (strlen($review_text) < 10) {
        throw new Exception('Review text must be at least 10 characters long');
    }

    if (strlen($review_title) > 255) {
        throw new Exception('Review title is too long (maximum 255 characters)');
    }

    if (strlen($review_text) > 2000) {
        throw new Exception('Review text is too long (maximum 2000 characters)');
    }

    // Find the review and verify ownership using EXACT submit_review.php structure
    $review_data = null;
    
    // Use the exact same query pattern as submit_review.php (line 292)
    $select_query = "SELECT * FROM course_reviews WHERE review_id = ? AND student_id = ?";
    
    $stmt = $conn->prepare($select_query);
    if ($stmt) {
        error_log("Using EXACT submit_review.php structure: $select_query with review_id=$review_id, user_id=$user_id");
        $stmt->bind_param("ii", $review_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $review_data = $result->fetch_assoc();
            error_log("SUCCESS! Found review with exact submit_review.php structure");
            error_log("Review data: " . print_r($review_data, true));
        } else {
            error_log("No results for exact structure query - review not found or user mismatch");
        }
        $stmt->close();
    } else {
        error_log("Failed to prepare exact structure query: " . $conn->error);
    }

    if (!$review_data) {
        throw new Exception('Review not found or you do not have permission to edit this review');
    }

    // DEBUG: Log the actual review data we found
    error_log("=== REVIEW DATA DEBUG ===");
    if ($review_data) {
        foreach ($review_data as $key => $value) {
            error_log("Review field - $key: $value");
        }
    }
    error_log("Update parameters - review_id: $review_id, user_id: $user_id, rating: $rating");
    error_log("Update parameters - title: '$review_title', text: '$review_text'");
    
    // EXACT MATCH TO WORKING SUBMIT_REVIEW.PHP STRUCTURE
    $updated = false;
    
    // Use the EXACT same table structure as submit_review.php
    $update_queries = [
        // This is the correct structure from submit_review.php:
        // Table: course_reviews, Columns: student_id (not user_id!), review_title, review_text
        "UPDATE course_reviews SET rating = ?, review_title = ?, review_text = ? WHERE review_id = ? AND student_id = ?",
    ];
    
    foreach ($update_queries as $update_query) {
        $update_stmt = $conn->prepare($update_query);
        if ($update_stmt) {
            error_log("Trying update query: $update_query");
            error_log("Parameters: rating=$rating, title='$review_title', text='$review_text', id=$review_id, user_id=$user_id");
            $update_stmt->bind_param("issii", $rating, $review_title, $review_text, $review_id, $user_id);
            
            if ($update_stmt->execute()) {
                if ($update_stmt->affected_rows > 0) {
                    $updated = true;
                    error_log("SUCCESS! Updated review with query: $update_query");
                    error_log("Affected rows: " . $update_stmt->affected_rows);
                    break; // Success!
                } else {
                    error_log("Query executed but no rows affected: $update_query");
                    error_log("This means no rows matched the WHERE condition");
                }
            } else {
                error_log("Execute failed for: $update_query. Error: " . $update_stmt->error);
            }
        } else {
            error_log("Prepare failed for: $update_query. Error: " . $conn->error);
        }
    }
    
    if (!$updated) {
        throw new Exception('Failed to update review in database - no rows affected by any query');
    }
    
    error_log("Review update completed successfully!");

    // Return success response
    ob_clean(); // Clear any output
    echo json_encode([
        'success' => true,
        'message' => 'Review updated successfully',
        'review_id' => $review_id,
        'rating' => (int)$rating,
        'title' => $review_title,
        'text' => $review_text
    ]);

} catch (Exception $e) {
    ob_clean(); // Clear any output
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} catch (Error $e) {
    ob_clean(); // Clear any output
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}

// Ensure output is sent and script ends cleanly
ob_end_flush();
exit;
?>
