/**
 * Panadite Academy - Courses Page JavaScript
 * Enhances the courses page with animations and interactive features
 */

document.addEventListener('DOMContentLoaded', function() {
    // Animate counter numbers in stat cards
    animateCounters();
    
    // Initialize course filters
    initCourseFilters();
    
    // Add animation classes to cards with delay
    animateCards();
});

/**
 * Animates the counter numbers in stat cards
 */
function animateCounters() {
    const counters = document.querySelectorAll('.stat-content h3[data-count]');
    
    counters.forEach(counter => {
        const target = parseInt(counter.getAttribute('data-count'));
        const duration = 1500; // ms
        const step = target / (duration / 30); // Update every 30ms
        let current = 0;
        
        setTimeout(() => {
            const timer = setInterval(() => {
                current += step;
                counter.textContent = Math.floor(current);
                
                if (current >= target) {
                    counter.textContent = target;
                    clearInterval(timer);
                }
            }, 30);
        }, 500); // Delay start
    });
}

/**
 * Initialize course filter buttons
 */
function initCourseFilters() {
    const filterButtons = document.querySelectorAll('.course-filters .btn');
    const courseCards = document.querySelectorAll('.enrolled-courses .course-card');
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            filterButtons.forEach(btn => btn.classList.remove('active'));
            
            // Add active class to clicked button
            this.classList.add('active');
            
            const filter = this.textContent.trim().toLowerCase();
            
            // Filter the courses
            courseCards.forEach(card => {
                if (filter === 'all') {
                    card.parentElement.style.display = 'block';
                    setTimeout(() => {
                        card.style.opacity = '1';
                        card.style.transform = '';
                    }, 50);
                } else {
                    const status = card.querySelector('.badge').textContent.trim().toLowerCase();
                    if (status === filter) {
                        card.parentElement.style.display = 'block';
                        setTimeout(() => {
                            card.style.opacity = '1';
                            card.style.transform = '';
                        }, 50);
                    } else {
                        card.style.opacity = '0';
                        card.style.transform = 'translateY(20px)';
                        setTimeout(() => {
                            card.parentElement.style.display = 'none';
                        }, 300);
                    }
                }
            });
        });
    });
}

/**
 * Apply hover effects and animations to cards
 */
function animateCards() {
    const cards = document.querySelectorAll('.course-card');
    
    cards.forEach(card => {
        // Apply custom hover effects
        card.addEventListener('mouseenter', function() {
            const thumbnail = this.querySelector('.course-thumbnail');
            if (thumbnail) {
                thumbnail.style.transform = 'scale(1.03)';
            }
        });
        
        card.addEventListener('mouseleave', function() {
            const thumbnail = this.querySelector('.course-thumbnail');
            if (thumbnail) {
                thumbnail.style.transform = '';
            }
        });
    });
    
    // Add a subtle parallax effect to thumbnails
    document.addEventListener('mousemove', function(e) {
        const thumbnails = document.querySelectorAll('.course-thumbnail');
        
        thumbnails.forEach(thumbnail => {
            const rect = thumbnail.getBoundingClientRect();
            const mouseX = (e.clientX - rect.left) / rect.width - 0.5;
            const mouseY = (e.clientY - rect.top) / rect.height - 0.5;
            
            if (isElementInViewport(thumbnail)) {
                thumbnail.style.transform = `perspective(1000px) rotateY(${mouseX * 3}deg) rotateX(${-mouseY * 3}deg) scale3d(1.02, 1.02, 1.02)`;
            }
        });
    });
}

/**
 * Check if element is in viewport
 */
function isElementInViewport(el) {
    const rect = el.getBoundingClientRect();
    return (
        rect.top >= 0 &&
        rect.left >= 0 &&
        rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
        rect.right <= (window.innerWidth || document.documentElement.clientWidth)
    );
}

// Add a scroll animation for elements
document.addEventListener('scroll', function() {
    const elements = document.querySelectorAll('.course-card, .section-header');
    
    elements.forEach(element => {
        if (isElementInViewport(element) && !element.classList.contains('animated')) {
            element.classList.add('animated');
            element.style.opacity = '1';
            element.style.transform = 'translateY(0)';
        }
    });
});
