<?php
// Use session_start.php for consistent session handling
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Add custom CSS for the videos page
$customCss = '<link rel="stylesheet" href="assets/css/pages/videos.css">';

// Initialize variables
$availableVideos = [];
$categories = [];
$error = null;
$success = null;

// Display messages from session if set
if (isset($_SESSION['video_success'])) {
    $success = $_SESSION['video_success'];
    unset($_SESSION['video_success']);
}

if (isset($_SESSION['video_error'])) {
    $error = $_SESSION['video_error'];
    unset($_SESSION['video_error']);
}

if (isset($_SESSION['video_info'])) {
    $info = $_SESSION['video_info'];
    unset($_SESSION['video_info']);
}

// Get user ID from session
$userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

if ($userId) {
    try {
        // Extract unique category names for filter
        $categoryQuery = "SELECT DISTINCT category_name FROM videos WHERE is_published = 1 ORDER BY category_name";
        $categoryResult = $conn->query($categoryQuery);
        
        if ($categoryResult) {
            while ($row = $categoryResult->fetch_assoc()) {
                if (!empty($row['category_name'])) {
                    $categories[] = ['name' => $row['category_name']];
                }
            }
        }
        
        // Get available videos (not purchased yet)
        $videoQuery = "
            SELECT 
                v.video_id,
                v.title,
                v.description,
                v.thumbnail,
                v.duration,
                v.price,
                v.is_free,
                v.category_name,
                v.views,
                u.first_name,
                u.last_name
            FROM videos v
            JOIN users u ON v.teacher_id = u.user_id
            LEFT JOIN video_purchases vp ON v.video_id = vp.video_id AND vp.user_id = ?
            WHERE vp.video_id IS NULL
            AND v.is_published = 1
            ORDER BY v.views DESC
        ";
        
        $stmt = $conn->prepare($videoQuery);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result) {
            $availableVideos = $result->fetch_all(MYSQLI_ASSOC);
        }
        
    } catch (Exception $e) {
        $error = "Error fetching available videos: " . $e->getMessage();
        error_log($error);
    }
}

// Include header
$pageTitle = "Available Videos";
include_once 'components/header.php';
?>

<div class="page-header">
    <div class="page-title">
        <h1>Available Videos</h1>
        <p>Discover videos available for purchase in the Panadite Academy</p>
    </div>
    <div class="page-actions">
        <a href="videos.php" class="btn btn-outline-primary">
            <i class="bx bx-video"></i> My Videos
        </a>
    </div>
</div>

<?php if ($error): ?>
<div class="alert alert-danger alert-dismissible fade show" role="alert">
    <?= $error ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
<?php endif; ?>

<?php if ($success): ?>
<div class="alert alert-success alert-dismissible fade show" role="alert">
    <?= $success ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
<?php endif; ?>

<?php if (isset($info)): ?>
<div class="alert alert-info alert-dismissible fade show" role="alert">
    <?= $info ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
<?php endif; ?>

<!-- Filter Controls -->
<div class="filter-controls mb-4">
    <div class="row g-3">
        <div class="col-md-4">
            <div class="input-group">
                <span class="input-group-text"><i class="bx bx-search"></i></span>
                <input type="text" id="videoSearchInput" class="form-control" placeholder="Search videos...">
            </div>
        </div>
        <div class="col-md-4">
            <select id="videoCategoryFilter" class="form-select">
                <option value="">All Categories</option>
                <?php foreach ($categories as $category): ?>
                <option value="<?= htmlspecialchars($category['name']) ?>"><?= htmlspecialchars($category['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
    </div>
</div>

<!-- Available Videos -->
<div class="row video-grid" id="videoGrid">
    <?php if (empty($availableVideos)): ?>
    <div class="col-12">
        <div class="card">
            <div class="card-body text-center p-5">
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="bx bx-video-off"></i>
                    </div>
                    <h4>No Available Videos</h4>
                    <p>You have already purchased all available videos.</p>
                </div>
            </div>
        </div>
    </div>
    <?php else: ?>
    <?php foreach ($availableVideos as $video): ?>
    <div class="col-md-6 col-lg-4 video-item" 
         data-category="<?= htmlspecialchars($video['category_name']) ?>"
         data-title="<?= htmlspecialchars($video['title']) ?>">
        <div class="card h-100">
            <div class="video-thumbnail">
                <img src="<?= !empty($video['thumbnail']) ? htmlspecialchars($video['thumbnail']) : 'assets/img/video-placeholder.jpg' ?>" 
                     class="card-img-top" alt="<?= htmlspecialchars($video['title']) ?>">
                <div class="video-duration">
                    <i class="bx bx-time"></i> <?= htmlspecialchars($video['duration'] ?? '00:00') ?>
                </div>
                <?php if ($video['is_free']): ?>
                <div class="video-badge">
                    <span class="badge bg-success">Free</span>
                </div>
                <?php endif; ?>
            </div>
            <div class="card-body">
                <div class="video-category"><?= htmlspecialchars($video['category_name']) ?></div>
                <h5 class="card-title"><?= htmlspecialchars($video['title']) ?></h5>
                <p class="card-text text-truncate-2"><?= htmlspecialchars($video['description']) ?></p>
                <div class="video-meta">
                    <div class="meta-item">
                        <i class="bx bx-user"></i> <?= htmlspecialchars($video['first_name'] . ' ' . $video['last_name']) ?>
                    </div>
                    <div class="meta-item">
                        <i class="bx bx-show"></i> <?= number_format($video['views']) ?>
                    </div>
                </div>
            </div>
            <div class="card-footer">
                <form action="video-purchase.php" method="post">
                    <input type="hidden" name="video_id" value="<?= $video['video_id'] ?>">
                    <input type="hidden" name="redirect" value="available-videos.php">
                    <button type="submit" class="btn btn-primary btn-sm">
                        <?php if ($video['is_free']): ?>
                        <i class="bx bx-plus"></i> Add to My Videos
                        <?php else: ?>
                        <i class="bx bx-cart-add"></i> Buy for $<?= number_format($video['price'], 2) ?>
                        <?php endif; ?>
                    </button>
                </form>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
// Filter videos by category and search
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('videoSearchInput');
    const categoryFilter = document.getElementById('videoCategoryFilter');
    const videoItems = document.querySelectorAll('.video-item');
    
    function filterVideos() {
        const searchTerm = searchInput.value.toLowerCase();
        const selectedCategory = categoryFilter.value.toLowerCase();
        
        videoItems.forEach(item => {
            const title = item.dataset.title.toLowerCase();
            const category = item.dataset.category.toLowerCase();
            
            const matchesSearch = title.includes(searchTerm);
            const matchesCategory = !selectedCategory || category === selectedCategory;
            
            if (matchesSearch && matchesCategory) {
                item.style.display = '';
            } else {
                item.style.display = 'none';
            }
        });
    }
    
    searchInput.addEventListener('input', filterVideos);
    categoryFilter.addEventListener('change', filterVideos);
});
</script>
<?php include_once 'components/footer.php'; ?>
