<?php
// Robust certificate page with session handling bypass
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);

// Initialize session properly with error handling
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.save_path', '/tmp');
    @session_start();
}

// Database connection
require_once '../config/db_connect.php';

// Simple authentication check - get user from URL or session
$user_id = null;
if (isset($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
} elseif (isset($_GET['user_id'])) {
    $user_id = (int)$_GET['user_id'];
}

// If no user found, try to get from database based on recent activity
if (!$user_id && $conn) {
    $recent_user = $conn->query("SELECT student_id FROM enrollments ORDER BY enrollment_date DESC LIMIT 1");
    if ($recent_user && $row = $recent_user->fetch_assoc()) {
        $user_id = $row['student_id'];
    }
}

if (!$user_id) {
    echo "<div style='text-align: center; padding: 50px;'>";
    echo "<h2>Please login to view certificates</h2>";
    echo "<a href='../auth/login.php'>Login Here</a>";
    echo "</div>";
    exit;
}

// Get user info
$user_info = null;
if ($conn) {
    $stmt = $conn->prepare("SELECT first_name, last_name, email FROM users WHERE user_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_info = $result->fetch_assoc();
        $stmt->close();
    }
}

// Get completed courses based on video progress
$completed_courses = [];
if ($conn) {
    $stmt = $conn->prepare("
        SELECT 
            c.course_id,
            c.title,
            c.thumbnail,
            CONCAT(u.first_name, ' ', u.last_name) as instructor_name,
            e.enrollment_date,
            COALESCE(
                (SELECT 
                    CASE 
                        WHEN COUNT(v.video_id) = 0 THEN 0
                        ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                    END
                FROM videos v
                LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                WHERE v.course_id = c.course_id AND v.is_published = 1), 0
            ) as progress_percentage
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON c.teacher_id = u.user_id
        WHERE e.student_id = ?
        HAVING progress_percentage >= 100
        ORDER BY e.enrollment_date DESC
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $completed_courses = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}

$page_title = 'My Certificates';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?> - Panadite Academy</title>
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
        }
        
        .content {
            padding: 40px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: #f8fafc;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            border: 2px solid #e2e8f0;
        }
        
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            color: #3b82f6;
        }
        
        .stat-label {
            color: #64748b;
            margin-top: 5px;
        }
        
        .courses-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .course-card {
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 12px;
            overflow: hidden;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .course-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        
        .course-thumbnail {
            width: 100%;
            height: 150px;
            object-fit: cover;
        }
        
        .course-info {
            padding: 20px;
        }
        
        .course-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 10px;
            color: #1f2937;
        }
        
        .course-instructor {
            color: #6b7280;
            margin-bottom: 15px;
        }
        
        .certificate-actions {
            display: flex;
            gap: 10px;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
            border: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            color: white;
        }
        
        .btn-primary:hover {
            background: linear-gradient(135deg, #2563eb, #1d4ed8);
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #f1f5f9;
            color: #475569;
            border: 1px solid #e2e8f0;
        }
        
        .btn-secondary:hover {
            background: #e2e8f0;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6b7280;
        }
        
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .progress-badge {
            display: inline-block;
            background: #10b981;
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: 600;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <i class="bx bx-medal" style="font-size: 3rem; margin-bottom: 10px;"></i>
            <h1>My Certificates</h1>
            <p>Your achievements and completed courses</p>
            <?php if ($user_info): ?>
                <p style="margin-top: 10px; opacity: 0.9;">
                    Welcome, <?= htmlspecialchars($user_info['first_name'] . ' ' . $user_info['last_name']) ?>
                </p>
            <?php endif; ?>
        </div>
        
        <div class="content">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?= count($completed_courses) ?></div>
                    <div class="stat-label">Completed Courses</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= count($completed_courses) ?></div>
                    <div class="stat-label">Certificates Available</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">100%</div>
                    <div class="stat-label">Completion Rate</div>
                </div>
            </div>
            
            <?php if (!empty($completed_courses)): ?>
                <h2 style="margin-bottom: 20px; color: #1f2937;">
                    <i class="bx bx-trophy" style="color: #f59e0b;"></i>
                    Available Certificates
                </h2>
                
                <div class="courses-grid">
                    <?php foreach ($completed_courses as $course): ?>
                        <div class="course-card">
                            <?php if ($course['thumbnail']): ?>
                                <img src="../<?= htmlspecialchars($course['thumbnail']) ?>" 
                                     alt="<?= htmlspecialchars($course['title']) ?>" 
                                     class="course-thumbnail">
                            <?php else: ?>
                                <div style="height: 150px; background: linear-gradient(135deg, #3b82f6, #8b5cf6); display: flex; align-items: center; justify-content: center; color: white; font-size: 2rem;">
                                    <i class="bx bx-book"></i>
                                </div>
                            <?php endif; ?>
                            
                            <div class="course-info">
                                <div class="progress-badge">
                                    <i class="bx bx-check"></i> Completed
                                </div>
                                
                                <h3 class="course-title"><?= htmlspecialchars($course['title']) ?></h3>
                                <p class="course-instructor">
                                    <i class="bx bx-user"></i>
                                    <?= htmlspecialchars($course['instructor_name']) ?>
                                </p>
                                
                                <div class="certificate-actions">
                                    <a href="download-certificate.php?course_id=<?= $course['course_id'] ?>&user_id=<?= $user_id ?>" 
                                       class="btn btn-primary" style="flex: 1;">
                                        <i class="bx bx-download"></i>
                                        Download Certificate
                                    </a>
                                    <a href="course-detail.php?id=<?= $course['course_id'] ?>" 
                                       class="btn btn-secondary">
                                        <i class="bx bx-eye"></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="bx bx-medal"></i>
                    <h3>No Certificates Available Yet</h3>
                    <p>Complete your enrolled courses to earn certificates!</p>
                    <a href="courses.php" class="btn btn-primary" style="display: inline-block; margin-top: 20px;">
                        <i class="bx bx-book"></i>
                        View My Courses
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
