<?php
/**
 * Student Course Catalog - Browse All Available Courses
 * Panadite Academy
 * 
 * Clean, modern course browsing interface with dynamic search
 */

// Authentication and session
require_once '../auth/functions.php';
requireRole('student');

// Initialize variables
$availableCourses = [];
$userId = $_SESSION['user_id'] ?? 0;

// Database operations with comprehensive error handling
try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $userId) {
        // Fetch all available courses with comprehensive data
        $coursesQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, c.price,
                   u.first_name, u.last_name, 'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   COALESCE(4.5, 4.5) as rating,
                   '10 hours' as duration, 
                   'Beginner' as level,
                   EXISTS(SELECT 1 FROM enrollments WHERE student_id = ? AND course_id = c.course_id) as is_enrolled
            FROM courses c
            JOIN users u ON c.teacher_id = u.user_id
            WHERE c.is_published = 1
            ORDER BY c.created_at DESC
        ";
        
        $stmt = $conn->prepare($coursesQuery);
        if (!$stmt) {
            error_log("Courses SQL prepare failed: " . $conn->error);
            throw new Exception("Courses database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("Courses SQL execute failed: " . $stmt->error);
            throw new Exception("Courses database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $availableCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                'price' => $row['price'],
                'rating' => $row['rating'],
                'duration' => $row['duration'],
                'level' => $row['level'],
                'category' => $row['category'],
                'students' => $row['total_students'],
                'is_enrolled' => $row['is_enrolled']
            ];
        }
        
    } else {
        error_log("Database connection failed or user not authenticated");
        throw new Exception("Database connection failed");
    }
    
} catch (Exception $e) {
    error_log("Course catalog error: " . $e->getMessage());
    $availableCourses = [];
}

// Set current page for navigation
$currentPage = 'course-catalog';
include_once 'components/header.php';

?>

<!-- Clean Course Catalog Page Content -->
<div class="page-content">
    
    <!-- Clean Page Header -->
    <div class="catalog-header">
        <div class="header-content">
            <h1><i class='bx bx-compass'></i> Browse Courses</h1>
            <p class="subtitle">Discover amazing courses and expand your knowledge</p>
        </div>
    </div>

    <!-- Dynamic Search Section -->
    <div class="search-section">
        <div class="search-container">
            <div class="search-box">
                <i class='bx bx-search'></i>
                <input type="text" id="courseSearch" placeholder="Search by course name, instructor, or topic..." autocomplete="off">
                <button type="button" id="clearSearch" class="clear-btn" style="display: none;">
                    <i class='bx bx-x'></i>
                </button>
            </div>
        </div>
        
        <!-- Search Results Info -->
        <div class="search-info">
            <span id="resultsCount"><?= count($availableCourses) ?> courses available</span>
            <div class="filter-options">
                <select id="levelFilter" class="filter-select">
                    <option value="">All Levels</option>
                    <option value="Beginner">Beginner</option>
                    <option value="Intermediate">Intermediate</option>
                    <option value="Advanced">Advanced</option>
                </select>
                <select id="priceFilter" class="filter-select">
                    <option value="">All Prices</option>
                    <option value="free">Free Courses</option>
                    <option value="paid">Paid Courses</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Courses Grid -->
    <div class="courses-container">
        <?php if (empty($availableCourses)): ?>
            <div class="empty-state">
                <div class="empty-illustration">
                    <i class='bx bx-book-open'></i>
                </div>
                <h3>No Courses Available</h3>
                <p>Check back later for new courses and learning opportunities</p>
            </div>
        <?php else: ?>
            <div class="courses-grid" id="coursesGrid">
                <?php foreach ($availableCourses as $index => $course): ?>
                    <div class="course-card" 
                         data-course-id="<?= $course['id'] ?>"
                         data-title="<?= strtolower($course['title']) ?>"
                         data-instructor="<?= strtolower($course['instructor']) ?>"
                         data-description="<?= strtolower($course['description']) ?>"
                         data-level="<?= $course['level'] ?>"
                         data-price="<?= $course['price'] <= 0 ? 'free' : 'paid' ?>"
                         style="animation-delay: <?= $index * 0.1 ?>s">
                        
                        <!-- Course Image -->
                        <div class="course-image-container">
                            <img src="<?= htmlspecialchars($course['thumbnail']) ?>" 
                                 alt="<?= htmlspecialchars($course['title']) ?>" 
                                 class="course-image"
                                 onerror="this.src='./assets/img/courses/default.jpg'">
                            
                            <!-- Price Badge -->
                            <?php if ($course['price'] <= 0): ?>
                                <div class="course-badge free-badge">Free</div>
                            <?php else: ?>
                                <div class="course-badge price-badge">R<?= number_format($course['price'], 2) ?></div>
                            <?php endif; ?>
                            
                            <!-- Enrollment Status -->
                            <?php if ($course['is_enrolled']): ?>
                                <div class="enrollment-badge">
                                    <i class='bx bx-check-circle'></i> Enrolled
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Course Content -->
                        <div class="course-content">
                            <!-- Course Title -->
                            <h3 class="course-title" title="<?= htmlspecialchars($course['title']) ?>">
                                <?= htmlspecialchars($course['title']) ?>
                            </h3>
                            
                            <!-- Course Description -->
                            <p class="course-description">
                                <?= htmlspecialchars(substr($course['description'], 0, 120)) ?>...
                            </p>
                            
                            <!-- Instructor Info -->
                            <div class="instructor-info">
                                <i class='bx bx-user'></i>
                                <span><?= htmlspecialchars($course['instructor']) ?></span>
                            </div>
                            
                            <!-- Course Meta -->
                            <div class="course-meta">
                                <div class="meta-item">
                                    <i class='bx bx-time'></i>
                                    <span><?= $course['duration'] ?></span>
                                </div>
                                <div class="meta-item">
                                    <i class='bx bx-signal-2'></i>
                                    <span><?= $course['level'] ?></span>
                                </div>
                                <div class="meta-item">
                                    <i class='bx bx-group'></i>
                                    <span><?= $course['students'] ?> students</span>
                                </div>
                            </div>
                            
                            <!-- Rating -->
                            <div class="course-rating">
                                <div class="stars">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <i class='bx <?= $i <= $course['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                    <?php endfor; ?>
                                </div>
                                <span class="rating-text"><?= $course['rating'] ?> (<?= rand(10, 100) ?> reviews)</span>
                            </div>
                        </div>
                        
                        <!-- Course Actions -->
                        <div class="course-actions">
                            <?php if ($course['is_enrolled']): ?>
                                <button class="action-btn continue-btn" onclick="continueCourse(<?= $course['id'] ?>)" title="Continue Learning">
                                    <i class='bx bx-play-circle'></i>
                                </button>
                            <?php else: ?>
                                <button class="action-btn cart-btn" onclick="addToCart(<?= $course['id'] ?>)" title="Add to Cart">
                                    <i class='bx bx-cart-add'></i>
                                </button>
                                <?php if ($course['price'] <= 0): ?>
                                    <button class="action-btn enroll-btn" onclick="enrollInCourse(<?= $course['id'] ?>)" title="Enroll Now">
                                        <i class='bx bx-plus'></i>
                                    </button>
                                <?php else: ?>
                                    <button class="action-btn enroll-btn" onclick="enrollInCourse(<?= $course['id'] ?>, <?= $course['price'] ?>)" title="Enroll - R<?= number_format($course['price'], 0) ?>">
                                        <i class='bx bx-credit-card'></i>
                                    </button>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <button class="action-btn view-btn" onclick="previewCourse(<?= $course['id'] ?>)" title="Preview Course">
                                <i class='bx bx-show'></i>
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modern Course Catalog CSS -->
<style>
/* Clean Page Header */
.catalog-header {
    text-align: center;
    padding: 2rem 0 1rem 0;
    margin-bottom: 2rem;
}

.header-content h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
}

.header-content h1 i {
    font-size: 2.2rem;
    color: #3b82f6;
}

.subtitle {
    font-size: 1.1rem;
    color: #6b7280;
    font-weight: 400;
    margin: 0;
}

/* Dynamic Search Section */
.search-section {
    background: #ffffff;
    padding: 2rem;
    margin-bottom: 2rem;
    border-radius: 15px;
    border: 1px solid #e5e7eb;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
}

.search-container {
    max-width: 600px;
    margin: 0 auto 1.5rem auto;
}

.search-box {
    position: relative;
    background: #f8fafc;
    border-radius: 50px;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
    overflow: hidden;
}

.search-box:focus-within {
    border-color: #3b82f6;
    background: #ffffff;
    box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.1);
}

.search-box i {
    position: absolute;
    left: 20px;
    top: 50%;
    transform: translateY(-50%);
    color: #6b7280;
    font-size: 1.2rem;
    z-index: 2;
}

.search-box input {
    width: 100%;
    padding: 18px 60px 18px 50px;
    border: none;
    background: transparent;
    font-size: 1.1rem;
    color: #1f2937;
    outline: none;
}

.search-box input::placeholder {
    color: #9ca3af;
    font-weight: 400;
}

.clear-btn {
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
    background: #ef4444;
    color: white;
    border: none;
    border-radius: 50%;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
    z-index: 2;
}

.clear-btn:hover {
    background: #dc2626;
    transform: translateY(-50%) scale(1.1);
}

/* Search Info */
.search-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

#resultsCount {
    color: #6b7280;
    font-weight: 500;
    font-size: 0.95rem;
}

.filter-options {
    display: flex;
    gap: 1rem;
}

.filter-select {
    padding: 8px 16px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    background: white;
    color: #374151;
    font-size: 0.9rem;
    cursor: pointer;
    transition: all 0.2s ease;
}

.filter-select:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

/* Courses Grid */
.courses-container {
    margin-top: 1rem;
}

.courses-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1.2rem;
    padding: 0;
}

.course-card {
    background: #ffffff;
    border-radius: 16px;
    border: 1px solid #e5e7eb;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: slideUp 0.6s ease forwards;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    display: flex;
    flex-direction: column;
}

.course-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
    border-color: #cbd5e1;
}

/* Course Image */
.course-image-container {
    position: relative;
    width: 100%;
    height: 140px;
    overflow: hidden;
}

.course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.course-card:hover .course-image {
    transform: scale(1.05);
}

.course-badge {
    position: absolute;
    top: 15px;
    right: 15px;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.free-badge {
    background: rgba(16, 185, 129, 0.9);
    color: white;
}

.price-badge {
    background: rgba(59, 130, 246, 0.9);
    color: white;
}

.enrollment-badge {
    position: absolute;
    top: 15px;
    left: 15px;
    background: rgba(34, 197, 94, 0.95);
    color: white;
    padding: 6px 10px;
    border-radius: 15px;
    font-size: 11px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
}

/* Course Content */
.course-content {
    padding: 12px;
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.course-title {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1f2937;
    line-height: 1.3;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
    height: 1.4em;
}

.course-title:hover {
    color: #3b82f6;
}

.course-description {
    font-size: 0.85rem;
    color: #6b7280;
    line-height: 1.4;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    height: 1.2em;
}

.instructor-info {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 0.9rem;
    color: #374151;
    font-weight: 500;
}

.instructor-info i {
    font-size: 1rem;
    color: #6b7280;
}

.course-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 16px;
    padding: 12px 0;
    border-top: 1px solid #f3f4f6;
    border-bottom: 1px solid #f3f4f6;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 0.85rem;
    color: #6b7280;
}

.meta-item i {
    font-size: 1rem;
}

.course-rating {
    display: flex;
    align-items: center;
    gap: 8px;
}

.stars {
    display: flex;
    gap: 2px;
}

.stars i {
    font-size: 1rem;
    color: #fbbf24;
}

.rating-text {
    font-size: 0.85rem;
    color: #6b7280;
    font-weight: 500;
}

/* Course Actions */
.course-actions {
    padding: 0 12px 12px 12px;
    display: flex;
    justify-content: center;
    gap: 6px;
}

.action-btn {
    width: 36px;
    height: 36px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    color: #64748b;
}

.action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.view-btn {
    background: #64748b;
    color: white;
    border-color: #64748b;
}

.view-btn:hover {
    background: #475569;
    border-color: #475569;
}

.cart-btn {
    background: #fbbf24;
    color: white;
    border-color: #fbbf24;
}

.cart-btn:hover {
    background: #f59e0b;
    border-color: #f59e0b;
}

.enroll-btn {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.enroll-btn:hover {
    background: #2563eb;
    border-color: #2563eb;
}

.continue-btn {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.continue-btn:hover {
    background: #2563eb;
    border-color: #2563eb;
}



/* Empty State */
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-illustration i {
    font-size: 5rem;
    color: #e5e7eb;
    margin-bottom: 1.5rem;
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #6b7280;
    font-size: 1rem;
}

/* Animations */
@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive Design for 5-Card Layout */
@media (max-width: 1400px) {
    .courses-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 1rem;
    }
}

@media (max-width: 1200px) {
    .courses-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 1rem;
    }
}

@media (max-width: 900px) {
    .courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
    }
    
    .course-image-container {
        height: 140px;
    }
}

@media (max-width: 768px) {
    .catalog-header {
        padding: 1rem 0;
        margin-bottom: 1.5rem;
    }
    
    .header-content h1 {
        font-size: 2rem;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .search-section {
        padding: 1.5rem;
        margin: 0 1rem 1.5rem 1rem;
    }
    
    .search-box input {
        padding: 16px 50px 16px 45px;
        font-size: 1rem;
    }
    
    .search-info {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
    
    .filter-options {
        justify-content: space-between;
    }
    
    .courses-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
        padding: 0 1rem;
    }
    
    .course-actions {
        flex-direction: column;
        gap: 8px;
    }
    
    .course-meta {
        gap: 12px;
    }
    
    .course-image-container {
        height: 180px;
    }
}

@media (max-width: 480px) {
    .search-box input {
        padding: 14px 45px 14px 40px;
        font-size: 0.95rem;
    }
    
    .courses-grid {
        padding: 0 0.5rem;
    }
    
    .course-content {
        padding: 16px;
    }
    
    .course-actions {
        padding: 0 16px 16px 16px;
    }
}

/* Desktop 5-Card Layout Optimization */
@media (min-width: 1600px) {
    .courses-grid {
        grid-template-columns: repeat(5, 1fr);
        gap: 1.5rem;
    }
    
    .course-content {
        padding: 18px;
    }
    
    .course-image-container {
        height: 180px;
    }
}
</style>

<!-- Dynamic Search and Filter JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('courseSearch');
    const clearBtn = document.getElementById('clearSearch');
    const levelFilter = document.getElementById('levelFilter');
    const priceFilter = document.getElementById('priceFilter');
    const coursesGrid = document.getElementById('coursesGrid');
    const resultsCount = document.getElementById('resultsCount');
    const allCourses = document.querySelectorAll('.course-card');
    
    let searchTimeout;
    
    // Search functionality
    function performSearch() {
        const searchTerm = searchInput.value.toLowerCase().trim();
        const levelValue = levelFilter.value;
        const priceValue = priceFilter.value;
        
        let visibleCount = 0;
        
        allCourses.forEach(course => {
            const title = course.dataset.title || '';
            const instructor = course.dataset.instructor || '';
            const description = course.dataset.description || '';
            const level = course.dataset.level || '';
            const price = course.dataset.price || '';
            
            // Text search match
            const searchMatch = !searchTerm || 
                title.includes(searchTerm) || 
                instructor.includes(searchTerm) || 
                description.includes(searchTerm);
            
            // Level filter match
            const levelMatch = !levelValue || level === levelValue;
            
            // Price filter match
            const priceMatch = !priceValue || price === priceValue;
            
            // Show/hide course
            if (searchMatch && levelMatch && priceMatch) {
                course.style.display = 'flex';
                visibleCount++;
            } else {
                course.style.display = 'none';
            }
        });
        
        // Update results count
        resultsCount.textContent = `${visibleCount} course${visibleCount !== 1 ? 's' : ''} available`;
        
        // Show/hide clear button
        if (searchTerm) {
            clearBtn.style.display = 'flex';
        } else {
            clearBtn.style.display = 'none';
        }
        
        // Show no results message if needed
        showNoResultsMessage(visibleCount === 0);
    }
    
    // Show/hide no results message
    function showNoResultsMessage(show) {
        let noResultsMsg = document.getElementById('noResultsMessage');
        
        if (show && !noResultsMsg) {
            noResultsMsg = document.createElement('div');
            noResultsMsg.id = 'noResultsMessage';
            noResultsMsg.className = 'empty-state';
            noResultsMsg.innerHTML = `
                <div class="empty-illustration">
                    <i class='bx bx-search-alt-2'></i>
                </div>
                <h3>No Courses Found</h3>
                <p>Try adjusting your search criteria or browse all available courses</p>
            `;
            coursesGrid.appendChild(noResultsMsg);
        } else if (!show && noResultsMsg) {
            noResultsMsg.remove();
        }
    }
    
    // Real-time search with debounce
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(performSearch, 300);
    });
    
    // Enter key search
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            clearTimeout(searchTimeout);
            performSearch();
        }
    });
    
    // Clear search
    clearBtn.addEventListener('click', function() {
        searchInput.value = '';
        clearBtn.style.display = 'none';
        performSearch();
        searchInput.focus();
    });
    
    // Filter changes
    levelFilter.addEventListener('change', performSearch);
    priceFilter.addEventListener('change', performSearch);
    
    // Focus search on page load
    searchInput.focus();
});

// Course action functions
function enrollInCourse(courseId, price = 0) {
    const button = event.target;
    const originalHTML = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Processing...';
    button.disabled = true;
    
    if (price > 0) {
        // Redirect to payment for paid courses
        window.location.href = `process-payment.php?course_id=${courseId}&amount=${price}`;
    } else {
        // Direct enrollment for free courses
        fetch('enroll-course.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `course_id=${courseId}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.innerHTML = '<i class="bx bx-check-circle"></i> Enrolled!';
                button.className = 'btn-primary enrolled-btn';
                button.onclick = () => continueCourse(courseId);
                
                // Show success notification
                showNotification('Successfully enrolled in course!', 'success');
                
                setTimeout(() => {
                    button.innerHTML = '<i class="bx bx-play-circle"></i> Continue Learning';
                }, 2000);
            } else {
                throw new Error(data.message || 'Enrollment failed');
            }
        })
        .catch(error => {
            console.error('Enrollment error:', error);
            button.innerHTML = originalHTML;
            button.disabled = false;
            showNotification('Failed to enroll in course', 'error');
        });
    }
}

function continueCourse(courseId) {
    window.location.href = `course-detail.php?id=${courseId}`;
}

function previewCourse(courseId) {
    window.location.href = `course-detail.php?id=${courseId}&preview=1`;
}

function addToCart(courseId) {
    const button = event.target.closest('.cart-btn');
    const originalHTML = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    button.disabled = true;
    
    // Simulate adding to cart (you can implement actual cart functionality)
    setTimeout(() => {
        showNotification('Course added to cart successfully!', 'success');
        button.innerHTML = '<i class="bx bx-check"></i>';
        
        // Reset button after 2 seconds
        setTimeout(() => {
            button.innerHTML = originalHTML;
            button.disabled = false;
        }, 2000);
    }, 500);
}

// Notification system
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 10000;
        animation: slideInRight 0.3s ease;
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Add notification animations
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    .notification-content {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 500;
    }
`;
document.head.appendChild(notificationStyles);
</script>


