<?php
/**
 * Student Navbar Component - Airbnb Style Design
 * Clean, modern navbar matching Airbnb's design exactly
 */

// Include comprehensive SEO optimization
require_once __DIR__ . '/../../includes/seo_header.php';

// Get user information
$userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
$userInitials = 'ST';
$currentUserName = 'Student';
$currentUserRole = 'Student';
$user = null; // Initialize user variable

if ($userId && isset($conn) && $conn) {
    try {
        $stmt = $conn->prepare("SELECT first_name, last_name, email, profile_picture FROM users WHERE user_id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($userData = $result->fetch_assoc()) {
            $user = $userData; // Store complete user data
            $currentUserName = $user['first_name'] . ' ' . $user['last_name'];
            $userInitials = strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1));
        }
    } catch (Exception $e) {
        // Use defaults
    }
}
?>

<!-- Airbnb-Style Navbar -->
<nav class="airbnb-navbar">
    <div class="navbar-container">
        <!-- Left: Menu Toggle -->
        <div class="navbar-left">
            <button class="menu-toggle-btn" onclick="toggleSidebar()">
                <i class="bx bx-menu"></i>
            </button>
        </div>

        <!-- Center: Search -->
        <div class="navbar-center">
            <!-- Boxicons -->
            <link href='https://unpkg.com/boxicons@2.0.9/css/boxicons.min.css' rel='stylesheet'>
            
            <!-- Google Fonts -->
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
            
            <style>
            /* Complete Student Navbar CSS - Desktop + Mobile Responsive */

            /* Fix body padding for fixed navbar */
            body {
                padding-top: 60px !important;
            }

            /* Base Navbar Styles */
            .airbnb-navbar {
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                height: 80px;
                background: #fff;
                border-bottom: 1px solid #e0e0e0;
                z-index: 1000;
                box-shadow: 0 1px 8px rgba(0, 0, 0, 0.08);
                font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            }

            .navbar-container {
                max-width: 1200px;
                margin: 0 auto;
                height: 100%;
                display: flex;
                align-items: center;
                justify-content: space-between;
                padding: 0 24px;
            }

            /* Left Section - Menu Toggle */
            .navbar-left {
                flex: 0 0 auto;
            }

            .menu-toggle-btn {
                background: none;
                border: none;
                font-size: 24px;
                color: #222;
                cursor: pointer;
                padding: 8px;
                border-radius: 8px;
                display: flex;
                align-items: center;
                justify-content: center;
                transition: background-color 0.2s ease;
            }

            .menu-toggle-btn:hover {
                background-color: #f7f7f7;
            }

            .menu-toggle-btn i {
                font-size: 24px;
            }

            /* Center Section - Search */
            .navbar-center {
                flex: 1;
                display: flex;
                justify-content: center;
                margin: 0 40px;
                position: relative;
            }

            .search-container {
                display: flex;
                align-items: center;
                background: #f7f7f7;
                border: 1px solid #ddd;
                border-radius: 40px;
                height: 48px;
                max-width: 400px;
                width: 100%;
                overflow: visible !important;
                transition: box-shadow 0.2s ease;
                position: relative;
            }

            /* Search Segments */
            .search-segment {
                flex: 1;
                padding: 0 16px;
                display: flex;
                flex-direction: column;
                justify-content: center;
                height: 100%;
            }

            .search-segment label {
                font-size: 11px;
                font-weight: 600;
                color: #666;
                margin: 0 0 4px 0;
                text-transform: uppercase;
                letter-spacing: 0.3px;
                line-height: 1;
                height: 13px;
                display: block;
            }

            .search-segment input {
                border: none;
                background: none;
                font-size: 14px;
                color: #222;
                outline: none;
                width: 100%;
                padding: 0;
                margin: 0;
                line-height: 1.2;
            }

            .search-divider {
                width: 1px;
                height: 24px;
                background: #ddd;
                flex-shrink: 0;
            }

            /* Ensure dropdown displays properly */
            .navbar-search-dropdown {
                z-index: 99999 !important;
                position: absolute !important;
                top: 100% !important;
                left: 50% !important;
                transform: translateX(-50%) !important;
                margin-top: 8px !important;
            }

            .navbar-search-dropdown.show {
                display: block !important;
            }

            /* Category Selector Button */
            .category-selector-btn {
                width: 100%;
                height: auto;
                background: none;
                border: none;
                display: flex;
                align-items: center;
                justify-content: space-between;
                padding: 0;
                font-size: 14px;
                color: #222;
                cursor: pointer;
                transition: all 0.2s ease;
                line-height: 1.2;
                text-align: left;
            }

            .category-selector-btn span {
                flex: 1;
                text-align: left;
                line-height: 1.2;
                margin-right: 8px;
            }

            .category-selector-btn:hover {
                color: #222;
            }

            .category-selector-btn i {
                font-size: 16px;
                transition: transform 0.2s ease;
                flex-shrink: 0;
            }

            .category-selector-btn:hover i {
                transform: rotate(180deg);
            }

            /* Category Modal Styles */
            .category-modal-overlay {
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                backdrop-filter: blur(5px);
                z-index: 999999;
                display: none;
                align-items: center;
                justify-content: center;
                animation: fadeIn 0.3s ease-out;
            }

            .category-modal-overlay.show {
                display: flex;
            }

            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }

            @keyframes slideUp {
                from { 
                    opacity: 0; 
                    transform: translateY(50px) scale(0.9); 
                }
                to { 
                    opacity: 1; 
                    transform: translateY(0) scale(1); 
                }
            }

            .category-modal {
                background: white;
                border-radius: 16px;
                box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
                max-width: 600px;
                width: 90vw;
                max-height: 80vh;
                overflow: hidden;
                animation: slideUp 0.3s ease-out;
            }

            .category-modal-header {
                padding: 24px;
                border-bottom: 1px solid #f0f0f0;
                display: flex;
                align-items: center;
                justify-content: space-between;
                background: linear-gradient(135deg, #f8f9ff 0%, #f0f4ff 100%);
            }

            .category-modal-header h3 {
                margin: 0;
                font-size: 20px;
                font-weight: 600;
                color: #1a1a1a;
            }

            .close-modal-btn {
                background: none;
                border: none;
                font-size: 24px;
                color: #666;
                cursor: pointer;
                padding: 8px;
                border-radius: 50%;
                transition: all 0.2s ease;
            }

            .close-modal-btn:hover {
                background: rgba(0, 0, 0, 0.1);
                color: #333;
            }

            .category-modal-content {
                padding: 24px;
                max-height: 400px;
                overflow-y: auto;
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                gap: 16px;
            }

            /* Category Cards */
            .category-card {
                background: white;
                border: 2px solid #f0f0f0;
                border-radius: 12px;
                padding: 20px;
                cursor: pointer;
                transition: all 0.3s ease;
                display: flex;
                align-items: center;
                gap: 16px;
                position: relative;
                overflow: hidden;
            }

            .category-card::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, transparent 0%, rgba(255, 255, 255, 0.1) 100%);
                opacity: 0;
                transition: opacity 0.3s ease;
            }

            .category-card:hover {
                transform: translateY(-4px);
                box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
                border-color: #2563eb;
            }

            .category-card:hover::before {
                opacity: 1;
            }

            .category-icon {
                width: 50px;
                height: 50px;
                border-radius: 12px;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 24px;
                color: white;
                position: relative;
                z-index: 1;
            }

            .category-info h4 {
                margin: 0 0 4px 0;
                font-size: 16px;
                font-weight: 600;
                color: #1a1a1a;
            }

            .course-count {
                font-size: 13px;
                color: #666;
                font-weight: 500;
            }

            /* Individual Category Colors */
            .all-categories .category-icon {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .cyber-security .category-icon {
                background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            }

            .learning-development .category-icon {
                background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            }

            .new-venture .category-icon {
                background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
            }

            .programming .category-icon {
                background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
            }

            .technopreneur .category-icon {
                background: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%);
            }

            /* Mobile Responsive */
            @media (max-width: 768px) {
                .category-modal {
                    width: 95vw;
                    max-height: 85vh;
                }

                .category-modal-content {
                    grid-template-columns: 1fr;
                    gap: 12px;
                    padding: 16px;
                }

                .category-card {
                    padding: 16px;
                }

                .category-icon {
                    width: 45px;
                    height: 45px;
                    font-size: 20px;
                }

                .category-info h4 {
                    font-size: 15px;
                }
            }

            .search-container:hover {
                box-shadow: 0 2px 16px rgba(0, 0, 0, 0.12);
            }

            .search-container:focus-within {
                box-shadow: 0 2px 16px rgba(0, 0, 0, 0.15);
                border-color: #222;
            }

            .search-segment {
                flex: 1;
                padding: 0 16px;
                display: flex;
                flex-direction: column;
                justify-content: center;
                cursor: pointer;
                transition: background-color 0.2s ease;
            }

            .search-segment:hover {
                background-color: #ebebeb;
            }

            .search-segment label {
                font-size: 12px;
                font-weight: 600;
                color: #222;
                margin-bottom: 2px;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }

            .search-segment input,
            .search-segment select {
                border: none;
                background: transparent;
                font-size: 14px;
                color: #222;
                outline: none;
                width: 100%;
                font-family: inherit;
            }

            .search-segment input::placeholder {
                color: #717171;
            }

            .search-segment select {
                cursor: pointer;
                appearance: none;
                color: #717171;
            }

            .search-segment select:focus {
                color: #222;
            }

            .search-divider {
                width: 1px;
                height: 24px;
                background: #ddd;
                flex-shrink: 0;
            }

            .search-button-container {
                padding: 0 8px;
                flex-shrink: 0;
            }

            .search-button {
                width: 32px;
                height: 32px;
                border-radius: 50%;
                background: #f59e0b;
                border: none;
                display: flex;
                align-items: center;
                justify-content: center;
                cursor: pointer;
                transition: background-color 0.2s ease;
            }

            .search-button:hover {
                background: #d97706;
            }

            .search-button i {
                color: white;
                font-size: 16px;
            }

            /* Right Section - User Menu */
            .navbar-right {
                flex: 0 0 auto;
                display: flex;
                align-items: center;
                gap: 16px;
            }

            .become-instructor {
                text-decoration: none;
                color: #222;
                font-weight: 600;
                font-size: 14px;
                padding: 12px 16px;
                border-radius: 22px;
                transition: background-color 0.2s ease;
                white-space: nowrap;
            }

            .become-instructor:hover {
                background-color: #f7f7f7;
                color: #222;
                text-decoration: none;
            }

            /* Cart Button */
            .cart-container {
                position: relative;
            }

            .cart-btn {
                background: none;
                border: none;
                padding: 8px;
                border-radius: 8px;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                position: relative;
                transition: background-color 0.2s ease;
            }

            .cart-btn:hover {
                background-color: #f7f7f7;
            }

            .cart-btn i {
                font-size: 20px;
                color: #222;
            }

            .cart-count {
                position: absolute;
                top: -2px;
                right: -2px;
                background: #f59e0b;
                color: white;
                font-size: 11px;
                font-weight: 600;
                padding: 2px 6px;
                border-radius: 10px;
                min-width: 16px;
                height: 16px;
                display: flex;
                align-items: center;
                justify-content: center;
                line-height: 1;
            }

            .cart-count:empty,
            .cart-count[data-count="0"] {
                display: none;
            }

            /* User Menu */
            .user-menu-container {
                position: relative;
            }

            .user-menu-toggle {
                display: flex;
                align-items: center;
                gap: 8px;
                padding: 6px 6px 6px 12px;
                border: 1px solid #ddd;
                border-radius: 21px;
                background: white;
                cursor: pointer;
                transition: box-shadow 0.2s ease;
            }

            .user-menu-toggle:hover {
                box-shadow: 0 2px 16px rgba(0, 0, 0, 0.12);
            }

            .user-menu-toggle i {
                font-size: 16px;
                color: #717171;
            }

            .user-avatar {
                width: 32px;
                height: 32px;
                border-radius: 50%;
                background: #6c757d;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-weight: 500;
                font-size: 14px;
                overflow: hidden;
            }
            
            .avatar-image {
                width: 100%;
                height: 100%;
                object-fit: cover;
                border-radius: 50%;
            }

            /* Dropdown Menu */
            .user-dropdown {
                position: absolute;
                top: 100%;
                right: 0;
                margin-top: 8px;
                width: 240px;
                background: white;
                border-radius: 12px;
                box-shadow: 0 8px 28px rgba(0, 0, 0, 0.28);
                display: none;
                z-index: 1001;
            }

            .user-dropdown.show {
                display: block;
            }

            .dropdown-header {
                padding: 16px;
            }

            .user-info {
                display: flex;
                align-items: center;
                gap: 12px;
            }

            .user-avatar-large {
                width: 40px;
                height: 40px;
                border-radius: 50%;
                background: #3b82f6;
                display: flex;
                align-items: center;
                justify-content: center;
                overflow: hidden;
            }
            
            .avatar-image-large {
                width: 100%;
                height: 100%;
                object-fit: cover;
                border-radius: 50%;
            }

            .user-avatar-large span {
                color: white;
                font-size: 16px;
                font-weight: 600;
            }

            .user-name {
                font-weight: 600;
                color: #222;
                font-size: 14px;
            }

            .user-role {
                color: #717171;
                font-size: 12px;
            }

            .dropdown-divider {
                height: 1px;
                background: #ebebeb;
                margin: 0 16px;
            }

            .dropdown-menu {
                padding: 8px 0;
            }

            .dropdown-item {
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 12px 16px;
                text-decoration: none;
                color: #222;
                font-size: 14px;
                transition: background-color 0.2s ease;
            }

            .dropdown-item:hover {
                background-color: #f7f7f7;
                color: #222;
                text-decoration: none;
            }

            .dropdown-item i {
                font-size: 16px;
                color: #717171;
                width: 16px;
            }

            .dropdown-item.logout {
                color: #ff385c;
            }

            .dropdown-item.logout i {
                color: #ff385c;
            }

            /* Search Dropdown Styles */
            .navbar-search-dropdown {
                position: absolute;
                top: 100%;
                left: 0;
                right: 0;
                background: white;
                border-radius: 12px;
                box-shadow: 0 8px 28px rgba(0, 0, 0, 0.28);
                margin-top: 8px;
                max-height: 400px;
                overflow-y: auto;
                display: none;
                z-index: 1002;
            }

            .navbar-search-dropdown.show {
                display: block;
            }

            .navbar-search-results {
                padding: 8px 0;
            }

            /* Search Result Item Styles */
            .navbar-search-item {
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 12px 16px;
                text-decoration: none;
                color: inherit;
                transition: background-color 0.2s ease;
            }

            .navbar-search-item:hover {
                background-color: #f7f7f7;
                text-decoration: none;
                color: inherit;
            }

            .navbar-course-info {
                flex: 1;
                min-width: 0; /* Allow shrinking */
                overflow: hidden;
            }

            .navbar-course-title {
                font-size: 14px;
                font-weight: 600;
                color: #222;
                margin: 0 0 4px 0;
                white-space: nowrap;
                overflow: hidden;
                text-overflow: ellipsis;
                line-height: 1.3;
            }

            .navbar-course-meta {
                font-size: 12px;
                color: #717171;
                white-space: nowrap;
                overflow: hidden;
                text-overflow: ellipsis;
            }

            .navbar-course-price {
                font-weight: 600;
                color: #f59e0b;
            }

            .navbar-meta-separator {
                margin: 0 6px;
                opacity: 0.6;
            }

            .navbar-search-footer {
                padding: 8px 16px;
                border-top: 1px solid #ebebeb;
                margin-top: 4px;
            }

            .navbar-view-all-btn {
                width: 100%;
                padding: 8px 16px;
                background: #f7f7f7;
                border: none;
                border-radius: 6px;
                color: #222;
                font-weight: 500;
                font-size: 13px;
                cursor: pointer;
                transition: background-color 0.2s ease;
            }

            .navbar-view-all-btn:hover {
                background: #ebebeb;
            }

            /* MOBILE RESPONSIVE STYLES */
            @media (max-width: 768px) {
                /* Adjust body padding for mobile navbar */
                body {
                    padding-top: 100px !important; /* Extra padding for 2-row mobile navbar */
                }
                
                .airbnb-navbar {
                    height: auto;
                    min-height: 80px;
                }
                
                /* Make navbar container wrap to two rows */
                .navbar-container {
                    flex-wrap: wrap;
                    padding: 12px 16px 8px;
                }
                
                /* Row 1: Hamburger (left) + User Menu (right) */
                .navbar-left {
                    order: 1;
                    flex: 0 0 auto;
                }
                
                .navbar-right {
                    order: 2;
                    flex: 0 0 auto;
                    margin-left: auto;
                    gap: 12px;
                }
                
                /* Row 2: Search goes full width below */
                .navbar-center {
                    order: 3;
                    flex: 0 0 100%;
                    margin: 12px 0 8px 0;
                    justify-content: stretch;
                }
                
                /* Hide become instructor on mobile */
                .become-instructor {
                    display: none;
                }
                
                /* Mobile search container */
                .search-container {
                    max-width: none;
                    width: 100%;
                    height: 52px;
                    border-radius: 26px;
                    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
                }
                
                /* Hide Level dropdown and divider on mobile */
                .search-segment:nth-child(3), /* Level segment */
                .search-divider {
                    display: none !important;
                }
                
                /* Make What input full width */
                .search-segment:first-child {
                    flex: 1;
                    padding: 0 20px;
                }
                
                .search-segment:first-child input {
                    font-size: 16px; /* Prevent zoom on iOS */
                    padding: 0;
                }
                
                .search-segment:first-child label {
                    display: none; /* Hide "What" label on mobile */
                }
                
                /* Search button on mobile */
                .search-button-container {
                    padding: 0 10px;
                }
                
                .search-button {
                    width: 36px;
                    height: 36px;
                }
                
                /* Mobile user menu adjustments */
                .user-menu-toggle {
                    padding: 4px 4px 4px 8px;
                }
                
                .user-dropdown {
                    right: -8px; /* Account for container padding */
                    width: 220px;
                }
                
                /* Mobile cart button */
                .cart-btn {
                    padding: 6px;
                }
            }

            /* Extra small mobile devices */
            @media (max-width: 480px) {
                .navbar-container {
                    padding: 10px 12px 6px;
                }
                
                .search-container {
                    height: 48px;
                    border-radius: 24px;
                }
                
                .search-segment:first-child {
                    padding: 0 16px;
                }
                
                .search-button {
                    width: 32px;
                    height: 32px;
                }
                
                .search-button i {
                    font-size: 14px;
                }
            }
            </style>
            <link rel="stylesheet" href="/assets/css/search-dropdown.css">
            <link rel="stylesheet" href="/assets/css/mobile-navbar.css">
            <div class="search-container" id="searchContainer">
                <div class="search-segment">
                    <label>What</label>
                    <input type="text" placeholder="Search courses, topics" id="search-what">
                </div>
                <div class="search-divider"></div>
                <div class="search-segment">
                    <label>Category</label>
                    <button type="button" id="category-selector" class="category-selector-btn" onclick="openCategoryModal()">
                        <span id="selected-category-text">All categories</span>
                        <i class="bx bx-chevron-down"></i>
                    </button>
                    <input type="hidden" id="search-category" value="">
                </div>
                <div class="search-button-container">
                    <button class="search-button" type="button" onclick="performSearch()">
                        <i class="bx bx-search"></i>
                    </button>
                </div>
                
                <!-- Dynamic Search Dropdown -->
                <div class="navbar-search-dropdown" id="searchDropdown">
                    <div class="navbar-search-results" id="searchResults">
                        <!-- Dynamic results will be inserted here -->
                    </div>
                </div>
            </div>
        </div>

        <!-- Category Selection Modal -->
        <div class="category-modal-overlay" id="categoryModalOverlay" onclick="closeCategoryModal()">
            <div class="category-modal" onclick="event.stopPropagation()">
                <div class="category-modal-header">
                    <h3>Select a Category</h3>
                    <button class="close-modal-btn" onclick="closeCategoryModal()">
                        <i class="bx bx-x"></i>
                    </button>
                </div>
                
                <div class="category-modal-content">
                    <!-- All Categories Option -->
                    <div class="category-card all-categories" onclick="selectCategory('', 'All categories')">
                        <div class="category-icon">
                            <i class="bx bx-grid-alt"></i>
                        </div>
                        <div class="category-info">
                            <h4>All Categories</h4>
                            <span class="course-count">10 courses</span>
                        </div>
                    </div>
                    
                    <!-- Individual Categories -->
                    <div class="category-card cyber-security" onclick="selectCategory('Cyber Security', 'Cyber Security')">
                        <div class="category-icon">
                            <i class="bx bx-shield"></i>
                        </div>
                        <div class="category-info">
                            <h4>Cyber Security</h4>
                            <span class="course-count">1 course</span>
                        </div>
                    </div>
                    
                    <div class="category-card learning-development" onclick="selectCategory('Learning and Development', 'Learning and Development')">
                        <div class="category-icon">
                            <i class="bx bx-graduation"></i>
                        </div>
                        <div class="category-info">
                            <h4>Learning and Development</h4>
                            <span class="course-count">3 courses</span>
                        </div>
                    </div>
                    
                    <div class="category-card new-venture" onclick="selectCategory('New Venture Creation', 'New Venture Creation')">
                        <div class="category-icon">
                            <i class="bx bx-rocket"></i>
                        </div>
                        <div class="category-info">
                            <h4>New Venture Creation</h4>
                            <span class="course-count">1 course</span>
                        </div>
                    </div>
                    
                    <div class="category-card programming" onclick="selectCategory('Programming', 'Programming')">
                        <div class="category-icon">
                            <i class="bx bx-code-alt"></i>
                        </div>
                        <div class="category-info">
                            <h4>Programming</h4>
                            <span class="course-count">3 courses</span>
                        </div>
                    </div>
                    
                    <div class="category-card technopreneur" onclick="selectCategory('Technoprenuer', 'Technoprenuer')">
                        <div class="category-icon">
                            <i class="bx bx-briefcase"></i>
                        </div>
                        <div class="category-info">
                            <h4>Technoprenuer</h4>
                            <span class="course-count">2 courses</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right: User Menu -->
        <div class="navbar-right">
            <a href="#" class="become-instructor">Become an Instructor</a>
            
            <div class="cart-container">
                <button class="cart-btn" onclick="toggleCart()">
                    <i class="bx bx-shopping-bag"></i>
                    <span class="cart-count" id="cartCount" style="display: none;"></span>
                </button>
            </div>
            
            <div class="user-menu-container">
                <button class="user-menu-toggle" onclick="toggleUserMenu()">
                    <i class="bx bx-menu"></i>
                    <div class="user-avatar">
                        <?php if (!empty($user['profile_picture'])): ?>
                            <img src="../uploads/profile_pictures/<?php echo htmlspecialchars($user['profile_picture']); ?>" 
                                 alt="Profile Picture" class="avatar-image">
                        <?php else: ?>
                            <span><?php echo $userInitials; ?></span>
                        <?php endif; ?>
                    </div>
                </button>
                
                <div class="user-dropdown" id="userDropdown">
                    <div class="dropdown-header">
                        <div class="user-info">
                            <div class="user-avatar-large">
                                <?php if (!empty($user['profile_picture'])): ?>
                                    <img src="../uploads/profile_pictures/<?php echo htmlspecialchars($user['profile_picture']); ?>" 
                                         alt="Profile Picture" class="avatar-image-large">
                                <?php else: ?>
                                    <span><?php echo $userInitials; ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="user-details">
                                <div class="user-name"><?php echo htmlspecialchars($currentUserName); ?></div>
                                <div class="user-role"><?php echo $currentUserRole; ?></div>
                            </div>
                        </div>
                    </div>
                    <div class="dropdown-divider"></div>
                    <div class="dropdown-menu">
                        <a href="dashboard.php" class="dropdown-item">
                            <i class="bx bx-home"></i>
                            <span>Dashboard</span>
                        </a>
                        <a href="courses.php" class="dropdown-item">
                            <i class="bx bx-book"></i>
                            <span>My Courses</span>
                        </a>
                        <a href="assignments.php" class="dropdown-item">
                            <i class="bx bx-task"></i>
                            <span>Assignments</span>
                        </a>
                        <a href="#" class="dropdown-item">
                            <i class="bx bx-user"></i>
                            <span>Profile</span>
                        </a>
                        <a href="#" class="dropdown-item">
                            <i class="bx bx-cog"></i>
                            <span>Settings</span>
                        </a>
                        <div class="dropdown-divider"></div>
                        <a href="../logout.php" class="dropdown-item logout">
                            <i class="bx bx-log-out"></i>
                            <span>Sign out</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</nav>

<style>



</style>

<script>
// Sidebar toggle functionality
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        if (window.innerWidth <= 768) {
            // Mobile: toggle between hidden and fully open
            sidebar.classList.toggle('open');
        } else {
            // Desktop: toggle between full and collapsed (icons only)
            sidebar.classList.toggle('close');
        }
    }
}

// Cart toggle functionality
function toggleCart() {
    // Redirect to cart page or show cart dropdown
    window.location.href = 'cart.php';
}

// Update cart count
function updateCartCount() {
    // Get cart count from localStorage or make AJAX call
    const cartItems = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cartItems.length;
    const cartCountElement = document.getElementById('cartCount');
    
    if (cartCountElement) {
        cartCountElement.textContent = cartCount;
        cartCountElement.setAttribute('data-count', cartCount);
        
        // Show/hide count badge
        if (cartCount > 0) {
            cartCountElement.style.display = 'flex';
        } else {
            cartCountElement.style.display = 'none';
        }
    }
}

// User menu toggle functionality
function toggleUserMenu() {
    const dropdown = document.getElementById('userDropdown');
    dropdown.classList.toggle('show');
}

// Close dropdown when clicking outside
document.addEventListener('click', function(event) {
    const userMenu = document.querySelector('.user-menu-container');
    const dropdown = document.getElementById('userDropdown');
    
    if (!userMenu.contains(event.target)) {
        dropdown.classList.remove('show');
    }
});

// Category Modal Functions
function openCategoryModal() {
    const modal = document.getElementById('categoryModalOverlay');
    modal.classList.add('show');
    document.body.style.overflow = 'hidden'; // Prevent background scrolling
}

function closeCategoryModal() {
    const modal = document.getElementById('categoryModalOverlay');
    modal.classList.remove('show');
    document.body.style.overflow = ''; // Restore scrolling
}

function selectCategory(categoryValue, categoryText) {
    // Update the hidden input and display text
    document.getElementById('search-category').value = categoryValue;
    document.getElementById('selected-category-text').textContent = categoryText;
    
    // Close the modal
    closeCategoryModal();
    
    // Trigger search with new category
    performDynamicSearch();
}

// Close modal with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeCategoryModal();
    }
});

// Dynamic Search functionality
let searchTimeout;
let currentSearchRequest;

function performDynamicSearch() {
    const what = document.getElementById('search-what').value.trim();
    const category = document.getElementById('search-category').value;
    const dropdown = document.getElementById('searchDropdown');
    const results = document.getElementById('searchResults');
    
    // Clear previous timeout
    clearTimeout(searchTimeout);
    
    // Cancel previous request
    if (currentSearchRequest) {
        currentSearchRequest.abort();
    }
    
    // If no search query, hide dropdown
    if (!what && !category) {
        dropdown.classList.remove('show');
        return;
    }
    
    // Show loading state
    dropdown.classList.add('show');
    
    // Add spinner keyframes directly to page
    if (!document.querySelector('#navbar-spinner-styles')) {
        const spinnerStyle = document.createElement('style');
        spinnerStyle.id = 'navbar-spinner-styles';
        spinnerStyle.innerHTML = `
            @keyframes navbarSpinner {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            .navbar-spinner {
                animation: navbarSpinner 1s linear infinite !important;
            }
        `;
        document.head.appendChild(spinnerStyle);
    }
    
    results.innerHTML = `
        <div style="
            text-align: center;
            padding: 25px 20px;
            background: #ffffff;
            border-top: 3px solid #007bff;
            color: #495057;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            border-radius: 0 0 12px 12px;
        ">
            <div style="
                width: 32px;
                height: 32px;
                border: 3px solid #e3f2fd;
                border-top: 3px solid #007bff;
                border-radius: 50%;
                margin: 0 auto 15px auto;
                animation: navbarSpinner 1s linear infinite;
            "></div>
            <div style="
                font-size: 15px;
                font-weight: 500;
                color: #495057;
                margin: 0;
                letter-spacing: 0.3px;
            ">Searching courses...</div>
        </div>
    `;
    
    // Debounce search
    searchTimeout = setTimeout(() => {
        const params = new URLSearchParams();
        if (what) params.append('q', what);
        if (category) params.append('category', category);
        params.append('limit', '6');
        
        currentSearchRequest = new AbortController();
        
        fetch(`api/navbar_search.php?${params.toString()}`, {
            signal: currentSearchRequest.signal
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data.courses.length > 0) {
                displaySearchResults(data.data.courses, what, category);
            } else {
                showNoResults();
            }
        })
        .catch(error => {
            if (error.name !== 'AbortError') {
                console.error('Search error:', error);
                showSearchError();
            }
        });
    }, 300);
}

function displaySearchResults(courses, query, category) {
    const results = document.getElementById('searchResults');
    
    let html = '';
    
    courses.forEach(course => {
        const courseInitials = course.title ? course.title.substring(0, 2).toUpperCase() : 'CO';
        
        html += `
            <a href="${course.url}" class="navbar-search-item">
                <div class="navbar-course-thumb" style="
                    width: 45px; 
                    height: 32px; 
                    border-radius: 6px; 
                    overflow: hidden;
                    flex-shrink: 0;
                    border: 1px solid #e1e5e9;
                    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
                    margin: 0;
                    padding: 0;
                ">
                    <img src="${course.thumbnail}" 
                         alt="${course.title}" 
                         style="width: 100%; height: 100%; object-fit: cover;"
                         onerror="this.style.display='none'; this.parentElement.innerHTML='${courseInitials}'; this.parentElement.style.background='linear-gradient(135deg, #667eea 0%, #764ba2 100%)'; this.parentElement.style.color='white'; this.parentElement.style.display='flex'; this.parentElement.style.alignItems='center'; this.parentElement.style.justifyContent='center'; this.parentElement.style.fontWeight='600'; this.parentElement.style.fontSize='11px';">
                </div>
                <div class="navbar-course-info">
                    <div class="navbar-course-title">${course.title}</div>
                    <div class="navbar-course-meta">
                        <span class="navbar-course-price">${course.price}</span>
                        <span class="navbar-meta-separator">•</span>
                        <span class="navbar-course-category">${course.category}</span>
                        <span class="navbar-meta-separator">•</span>
                        <span class="navbar-course-teacher">${course.teacher_name}</span>
                    </div>
                </div>
            </a>
        `;
    });
    
    // Add "View All Results" button if there are results
    if (courses.length > 0) {
        html += `
            <div class="navbar-search-footer">
                <button class="navbar-view-all-btn" onclick="viewAllResults('${query}', '')">
                    View All Results
                </button>
            </div>
        `;
    }
    
    results.innerHTML = html;
}

function showSearchLoading() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="navbar-search-loading">
            <i class="bx bx-loader-alt"></i>
            <div>Searching courses...</div>
        </div>
    `;
}

function showNoResults() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="navbar-search-empty">
            <i class="bx bx-search-alt"></i>
            <div>No courses found</div>
            <div>Try adjusting your search terms</div>
        </div>
    `;
}

function showSearchError() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="search-no-results">
            <i class="bx bx-error"></i>
            <div>Search temporarily unavailable</div>
            <div style="font-size: 12px; margin-top: 4px;">Please try again later</div>
        </div>
    `;
}

function viewAllResults(query, level) {
    let url = 'courses.php?';
    if (query) url += 'search=' + encodeURIComponent(query) + '&';
    if (level) url += 'level=' + encodeURIComponent(level) + '&';
    window.location.href = url.slice(0, -1);
}

function performSearch() {
    // Keep the old function for the search button - redirect to full results
    const what = document.getElementById('search-what').value.trim();
    const level = document.getElementById('search-level').value;
    viewAllResults(what, level);
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('search-what');
    const searchCategory = document.getElementById('search-category');
    const searchContainer = document.getElementById('searchContainer');
    const dropdown = document.getElementById('searchDropdown');
    
    if (searchInput) {
        // Real-time search as user types
        searchInput.addEventListener('input', performDynamicSearch);
        
        // Show dropdown when input is focused
        searchInput.addEventListener('focus', function() {
            if (this.value.trim()) {
                performDynamicSearch();
            }
        });
        
        // Enter key search
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });
    }
    
    if (searchCategory) {
        searchCategory.addEventListener('change', performDynamicSearch);
    }
    
    // Hide dropdown when clicking outside
    document.addEventListener('click', function(event) {
        if (!searchContainer.contains(event.target)) {
            dropdown.classList.remove('show');
        }
    });
    
    // Initialize cart count on page load
    updateCartCount();
});
</script>
