<?php
/**
 * Course Details Page
 * Panadite Academy
 */

require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');

// Decrypt the course ID from URL
$courseId = getSecureId('id');
$userId = $_SESSION['user_id'] ?? 0;

// Redirect if invalid or missing encrypted ID
if (!$courseId) {
    header('Location: courses.php?error=invalid_course');
    exit();
}

// Initialize variables
$course = null;
$isEnrolled = false;
$enrollmentData = null;

try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $courseId) {
        // Fetch course details with instructor social links and profile picture
        $courseQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, c.price,
                   u.first_name, u.last_name, u.bio as instructor_bio, u.profile_picture, c.created_at,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube
            FROM courses c
            JOIN users u ON c.teacher_id = u.user_id
            LEFT JOIN teachers t ON u.user_id = t.user_id
            WHERE c.course_id = ? AND c.is_published = 1
        ";
        
        $stmt = $conn->prepare($courseQuery);
        if ($stmt) {
            $stmt->bind_param("i", $courseId);
            $stmt->execute();
            $result = $stmt->get_result();
            $course = $result->fetch_assoc();
        }
        
        // Check if user is enrolled
        if ($course && $userId) {
            $enrollmentQuery = "
                SELECT enrolled_date, progress 
                FROM enrollments 
                WHERE student_id = ? AND course_id = ?
            ";
            $stmt = $conn->prepare($enrollmentQuery);
            if ($stmt) {
                $stmt->bind_param("ii", $userId, $courseId);
                $stmt->execute();
                $result = $stmt->get_result();
                $enrollmentData = $result->fetch_assoc();
                $isEnrolled = !empty($enrollmentData);
            }
        }
        
        // Get curriculum preview data using EXACT same implementation as course-detail.php
        $curriculumLessons = [];
        
        // EXACT COPY from course-detail.php lines 221-340
        if (isset($conn)) {
            $video_query = "
                SELECT v.*, 
                       COALESCE(vp.completion_percentage, 0) as watch_progress,
                       CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
                       CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
                FROM videos v
                LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
                WHERE v.course_id = ? AND v.is_published = 1
                ORDER BY v.video_order ASC, v.created_at ASC
            ";
            
            if ($stmt = $conn->prepare($video_query)) {
                $stmt->bind_param("ii", $userId, $courseId);
                $stmt->execute();
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()) {
                    // Handle both file uploads and URL videos
                    if (!empty($row['external_url'])) {
                        // URL video - convert to proper embed format
                        $external_url = $row['external_url'];
                        
                        // Convert YouTube URLs to embed format
                        if (strpos($external_url, 'youtube.com/watch') !== false) {
                            parse_str(parse_url($external_url, PHP_URL_QUERY), $query);
                            if (isset($query['v'])) {
                                $row['video_url'] = 'https://www.youtube.com/embed/' . $query['v'];
                            } else {
                                $row['video_url'] = $external_url;
                            }
                        } elseif (strpos($external_url, 'youtu.be') !== false) {
                            $video_id = basename(parse_url($external_url, PHP_URL_PATH));
                            $row['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
                        } elseif (strpos($external_url, 'youtube.com/embed') !== false) {
                            $row['video_url'] = $external_url;
                        } else {
                            $row['video_url'] = $external_url;
                        }
                        
                        $row['is_external'] = true;
                    } elseif (!empty($row['video_url'])) {
                        // File upload - fix video URL path for web access
                        if (strpos($row['video_url'], 'http') === 0) {
                            $row['video_url'] = $row['video_url'];
                        } else {
                            $row['video_url'] = '/' . ltrim($row['video_url'], './');
                        }
                        $row['is_external'] = false;
                    } else {
                        $row['video_url'] = '';
                        $row['is_external'] = false;
                    }
                    
                    // Fix thumbnail path for web access - EXACT COPY
                    if (!empty($row['thumbnail'])) {
                        if (strpos($row['thumbnail'], 'http') === 0) {
                            $row['thumbnail'] = $row['thumbnail'];
                        } else {
                            $row['thumbnail'] = '/' . ltrim($row['thumbnail'], './');
                        }
                    } else {
                        $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
                    }
                    
                    // Check for related content - EXACT COPY from course-detail.php
                    $row['has_assignment'] = false;
                    $row['has_quiz'] = false;
                    $row['has_materials'] = false;
                    
                    // Check assignments for this course
                    $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE course_id = ?");
                    if ($assignment_check) {
                        $assignment_check->bind_param("i", $courseId);
                        $assignment_check->execute();
                        $assignment_result = $assignment_check->get_result();
                        $assignment_data = $assignment_result->fetch_assoc();
                        $row['has_assignment'] = $assignment_data['count'] > 0;
                        $assignment_check->close();
                    }
                    
                    // Check quizzes for this course
                    $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
                    if ($quiz_check) {
                        $quiz_check->bind_param("i", $courseId);
                        $quiz_check->execute();
                        $quiz_result = $quiz_check->get_result();
                        $quiz_data = $quiz_result->fetch_assoc();
                        $row['has_quiz'] = $quiz_data['count'] > 0;
                        $quiz_check->close();
                    }
                    
                    // Check materials for this course
                    $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE course_id = ?");
                    if ($material_check) {
                        $material_check->bind_param("i", $courseId);
                        $material_check->execute();
                        $material_result = $material_check->get_result();
                        $material_data = $material_result->fetch_assoc();
                        $row['has_materials'] = $material_data['count'] > 0;
                        $material_check->close();
                    }
                    
                    $row['lesson_id'] = $row['video_id']; // For compatibility
                    $curriculumLessons[] = $row;
                }
                $stmt->close();
            }
        }
    }
    
    // EXACT REFERRAL DISCOUNT LOGIC FROM COURSE-CATALOG.PHP
    if ($course && isset($conn)) {
        // FETCH USER'S AVAILABLE DISCOUNTS - EXACT SAME AS COURSE-CATALOG.PHP
        $available_discounts = [];
        $discountsQuery = "SELECT * FROM student_referral_discounts 
                          WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                          ORDER BY discount_percentage DESC LIMIT 1";
        $discountStmt = $conn->prepare($discountsQuery);
        if ($discountStmt) {
            $discountStmt->bind_param("i", $userId);
            $discountStmt->execute();
            $discountResult = $discountStmt->get_result();
            if ($discountResult->num_rows > 0) {
                $available_discounts = $discountResult->fetch_assoc();
            }
        }
        
        // APPLY DISCOUNT LOGIC - EXACT SAME AS COURSE-CATALOG.PHP
        $original_price = $course['price'];
        $discounted_price = $original_price;
        $has_discount = false;
        
        // APPLY DISCOUNT IF AVAILABLE
        if (!empty($available_discounts) && $original_price > 0) {
            $discount_amount = min(
                ($original_price * $available_discounts['discount_percentage'] / 100),
                $available_discounts['discount_amount']
            );
            $discounted_price = $original_price - $discount_amount;
            $has_discount = true;
        }
        
        // UPDATE COURSE DATA - EXACT SAME STRUCTURE AS COURSE-CATALOG.PHP
        $course['price'] = $original_price;
        $course['discounted_price'] = $discounted_price;
        $course['has_discount'] = $has_discount;
        $course['discount_percentage'] = $has_discount ? $available_discounts['discount_percentage'] : 0;
        $course['savings'] = $has_discount ? ($original_price - $discounted_price) : 0;
        $course['thumbnail'] = $course['thumbnail'] ?: 'default-course.jpg';
    }
} catch (Exception $e) {
    error_log("Course details error: " . $e->getMessage());
}

if (!$course) {
    header("Location: courses.php");
    exit;
}

include_once 'components/header.php';
?>

<!-- Course Details Content -->
<div class="course-details-container">
    <!-- Course Header -->
    <div class="course-header">
        <div class="course-hero">
            <div class="course-hero-content">
                <div class="course-breadcrumb">
                    <a href="courses.php"><i class='bx bx-arrow-back'></i> Back to Courses</a>
                </div>
                
                <div class="course-main-info">
                    <div class="course-content">
                        <h1 class="course-title"><?= htmlspecialchars($course['title']) ?></h1>
                        <p class="course-description"><?= htmlspecialchars($course['description']) ?></p>
                        
                        <div class="course-meta">
                            <div class="instructor-info">
                                <i class='bx bx-user'></i>
                                <span>by <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?></span>
                            </div>
                            <div class="students-count">
                                <i class='bx bx-group'></i>
                                <span><?= $course['total_students'] ?> students enrolled</span>
                            </div>
                            <div class="created-date">
                                <i class='bx bx-calendar'></i>
                                <span>Created <?= date('M Y', strtotime($course['created_at'])) ?></span>
                            </div>
                        </div>
                        
                        <div class="course-actions">
                            <?php if ($isEnrolled): ?>
                                <a href="course-detail.php?id=<?= encryptId($courseId) ?>" class="btn-primary">
                                    <i class='bx bx-play-circle'></i>
                                    Continue Learning
                                </a>
                                <div class="enrollment-status">
                                    <i class='bx bx-check-circle'></i>
                                    <span>Enrolled on <?= date('M j, Y', strtotime($enrollmentData['enrolled_date'])) ?></span>
                                    
                                    <?php
                                    // Calculate REAL progress using SAME logic as certificates/dashboard
                                    
                                    // 1. Video completion (FIXED - uses user_video_completions)
                                    $videoQuery = $conn->prepare("
                                        SELECT 
                                            COUNT(DISTINCT v.video_id) as total_videos,
                                            COUNT(DISTINCT uvc.video_id) as completed_videos
                                        FROM videos v
                                        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                                            AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                                        WHERE v.course_id = ? AND v.is_published = 1
                                    ");
                                    
                                    $total_videos = 0;
                                    $completed_videos = 0;
                                    
                                    if ($videoQuery) {
                                        $videoQuery->bind_param("ii", $userId, $courseId);
                                        $videoQuery->execute();
                                        $videoResult = $videoQuery->get_result();
                                        if ($videoRow = $videoResult->fetch_assoc()) {
                                            $total_videos = (int)$videoRow['total_videos'];
                                            $completed_videos = (int)$videoRow['completed_videos'];
                                        }
                                        $videoQuery->close();
                                    }
                                    
                                    // 2. Quiz performance 
                                    $quizQuery = $conn->prepare("
                                        SELECT 
                                            COUNT(DISTINCT q.quiz_id) as total_quizzes,
                                            COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes
                                        FROM quizzes q
                                        LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                                        WHERE q.course_id = ? AND q.is_published = 1
                                    ");
                                    
                                    $total_quizzes = 0;
                                    $passed_quizzes = 0;
                                    
                                    if ($quizQuery) {
                                        $quizQuery->bind_param("ii", $userId, $courseId);
                                        $quizQuery->execute();
                                        $quizResult = $quizQuery->get_result();
                                        if ($quizRow = $quizResult->fetch_assoc()) {
                                            $total_quizzes = (int)$quizRow['total_quizzes'];
                                            $passed_quizzes = (int)$quizRow['passed_quizzes'];
                                        }
                                        $quizQuery->close();
                                    }
                                    
                                    // 🚨 CRITICAL PROFESSIONAL FIX: Use SAME logic as other fixed files
                                    // PREVENT courses with NO CONTENT from showing 100% completion!
                                    
                                    $has_any_content = ($total_videos > 0 || $total_quizzes > 0);
                                    
                                    if (!$has_any_content) {
                                        // 🚫 NO CONTENT = 0% progress (NOT 100%!)
                                        $real_course_progress = 0;
                                        error_log("COURSE-DETAILS.PHP PROGRESS: Course ID $courseId has NO CONTENT - Progress set to 0%");
                                    } else {
                                        // Course has content - calculate normally
                                        $video_progress = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                                        $quiz_progress = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 0;
                                        
                                        // Only weight if both exist, otherwise use the available one
                                        if ($total_videos > 0 && $total_quizzes > 0) {
                                            $real_course_progress = round(($video_progress * 0.6) + ($quiz_progress * 0.4));
                                        } elseif ($total_videos > 0) {
                                            $real_course_progress = round($video_progress);
                                        } elseif ($total_quizzes > 0) {
                                            $real_course_progress = round($quiz_progress);
                                        } else {
                                            $real_course_progress = 0;
                                        }
                                    }
                                    ?>
                                    
                                    <!-- Clean Circular Progress Indicator -->
                                    <div class="course-progress-container">
                                        <div class="progress-circle-wrapper">
                                            <div class="progress-circle" data-progress="<?= $real_course_progress ?>">
                                                <div class="progress-circle-inner">
                                                    <span class="progress-text"><?= $real_course_progress ?>%</span>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="progress-details">
                                            <div class="progress-label">Course Progress</div>
                                            <div class="progress-breakdown">
                                                <small class="text-muted">
                                                    Videos: <?= $completed_videos ?>/<?= $total_videos ?> • 
                                                    Quizzes: <?= $passed_quizzes ?>/<?= $total_quizzes ?>
                                                </small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php else: ?>
                                <?php if ($course['price'] > 0): ?>
                                    <div class="course-pricing">
                                        <?php if ($course['has_discount']): ?>
                                            <div class="price-with-discount">
                                                <span class="original-price">R<?= number_format($course['price'], 2) ?></span>
                                                <span class="discounted-price">R<?= number_format($course['discounted_price'], 2) ?></span>
                                                <div class="discount-badge">
                                                    <i class="bx bx-gift"></i>
                                                    <?= number_format($course['discount_percentage'], 1) ?>% OFF
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <span class="price-tag">R<?= number_format($course['price'], 2) ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <button class="btn-primary cart-button" onclick="addToCart('<?= htmlspecialchars($_GET['id']) ?>')">
                                        <i class='bx bx-cart-add'></i>
                                        <span class="btn-text">Add to Cart</span>
                                        <span class="btn-loading" style="display: none;">
                                            <i class='bx bx-loader-alt bx-spin'></i>
                                            Adding...
                                        </span>
                                    </button>
                                    <a href="course-payment.php?id=<?= $_GET['id'] ?>" class="btn-outline buy-now-button">
                                        <i class='bx bx-credit-card'></i>
                                        Buy Now
                                    </a>
                                <?php else: ?>
                                    <button class="btn-primary" onclick="enrollCourse(<?= $courseId ?>)">
                                        <i class='bx bx-plus'></i>
                                        Enroll for Free
                                    </button>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="course-thumbnail-container">
                        <img src="<?= htmlspecialchars($course['thumbnail'] ?? '../assets/images/default-course.jpg') ?>" 
                             alt="<?= htmlspecialchars($course['title']) ?>" 
                             class="course-thumbnail"
                             onerror="this.onerror=null; this.src='../assets/images/default-course.jpg';">
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Course Content Sections -->
    <div class="course-content-sections">

        <div class="content-section">
            <h3><i class='bx bx-list-ul'></i> Course Curriculum</h3>
            <div class="section-content">
                <?php if (!empty($curriculumLessons)): ?>
                    <div class="curriculum-preview">
                        <?php if (!$isEnrolled): ?>
                            <div class="locked-notice">
                                <i class='bx bx-lock-alt'></i>
                                <span>Enroll in this course to access all lessons and materials</span>
                            </div>
                        <?php endif; ?>
                        
                        <div class="accordion curriculum-accordion" id="curriculumAccordion">
                            <?php foreach ($curriculumLessons as $index => $lesson): ?>
                                <div class="accordion-item lesson-preview-item <?= !$isEnrolled ? 'locked' : '' ?>">
                                    <h2 class="accordion-header" id="headingLesson<?= $lesson['lesson_id'] ?>">
                                        <button class="accordion-button collapsed <?= !$isEnrolled ? 'disabled' : '' ?>" 
                                                type="button" 
                                                <?= $isEnrolled ? 'data-bs-toggle="collapse" data-bs-target="#collapseLesson' . $lesson['lesson_id'] . '"' : 'disabled' ?>
                                                aria-expanded="false" 
                                                aria-controls="collapseLesson<?= $lesson['lesson_id'] ?>">
                                            
                                            <div class="d-flex align-items-center w-100">
                                                <!-- Lesson Thumbnail -->
                                                <div class="lesson-thumbnail-preview me-3">
                                                    <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" 
                                                         alt="Lesson <?= $index + 1 ?>" 
                                                         class="rounded"
                                                         onerror="this.onerror=null; this.src='/assets/images/default-video-thumbnail.jpg';">
                                                    <?php if (!$isEnrolled): ?>
                                                        <div class="lock-overlay">
                                                            <i class='bx bx-lock-alt'></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <!-- Lesson Info -->
                                                <div class="flex-grow-1">
                                                    <div class="lesson-title-preview">
                                                        <span class="lesson-number">Lesson <?= $index + 1 ?>:</span>
                                                        <span class="lesson-name"><?= htmlspecialchars($lesson['title']) ?></span>
                                                    </div>
                                                    
                                                    <div class="lesson-meta-preview">
                                                        <?php if (!empty($lesson['duration'])): ?>
                                                            <span class="meta-item">
                                                                <i class='bx bx-time'></i> <?= htmlspecialchars($lesson['duration']) ?>
                                                            </span>
                                                        <?php endif; ?>
                                                        
                                                        <?php 
                                                        $contentCount = ($lesson['has_quiz'] ? 1 : 0) + ($lesson['has_materials'] ? 1 : 0) + ($lesson['has_assignment'] ? 1 : 0);
                                                        if ($contentCount > 0): 
                                                        ?>
                                                            <span class="meta-item">
                                                                <i class='bx bx-folder'></i> <?= $contentCount ?> resource<?= $contentCount > 1 ? 's' : '' ?>
                                                            </span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                
                                                <!-- Lock Icon for Non-enrolled -->
                                                <?php if (!$isEnrolled): ?>
                                                    <div class="lock-indicator">
                                                        <i class='bx bx-lock-alt'></i>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </button>
                                    </h2>
                                    
                                    <?php if ($isEnrolled): ?>
                                        <div id="collapseLesson<?= $lesson['lesson_id'] ?>" 
                                             class="accordion-collapse collapse" 
                                             aria-labelledby="headingLesson<?= $lesson['lesson_id'] ?>" 
                                             data-bs-parent="#curriculumAccordion">
                                            <div class="accordion-body">
                                                <div class="lesson-content-preview">
                                                    <?php if (!empty($lesson['description'])): ?>
                                                        <p class="lesson-description"><?= htmlspecialchars($lesson['description']) ?></p>
                                                    <?php endif; ?>
                                                    
                                                    <!-- Content breakdown for enrolled users -->
                                                    <div class="content-breakdown">
                                                        <?php if ($lesson['has_quiz']): ?>
                                                            <div class="content-group">
                                                                <h6><i class='bx bx-help-circle'></i> Quizzes Available</h6>
                                                                <small>Interactive quizzes to test your knowledge</small>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <?php if ($lesson['has_materials']): ?>
                                                            <div class="content-group">
                                                                <h6><i class='bx bx-folder'></i> Learning Materials</h6>
                                                                <small>Downloadable resources and documents</small>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <?php if ($lesson['has_assignment']): ?>
                                                            <div class="content-group">
                                                                <h6><i class='bx bx-task'></i> Assignments</h6>
                                                                <small>Practice exercises and projects</small>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <!-- Locked content preview for non-enrolled users -->
                                        <div class="locked-content-preview">
                                            <div class="locked-content-info">
                                                <p><i class='bx bx-info-circle'></i> This lesson contains:</p>
                                                <div class="locked-content-summary">
                                                    <?php if ($lesson['has_quiz']): ?>
                                                        <span class="content-badge quiz-badge">
                                                            <i class='bx bx-help-circle'></i> Quizzes Available
                                                        </span>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($lesson['has_materials']): ?>
                                                        <span class="content-badge material-badge">
                                                            <i class='bx bx-folder'></i> Learning Materials
                                                        </span>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($lesson['has_assignment']): ?>
                                                        <span class="content-badge assignment-badge">
                                                            <i class='bx bx-task'></i> Assignments Available
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <?php if (!$isEnrolled): ?>
                            <div class="enroll-cta-bottom">
                                <p><strong>Ready to start learning?</strong></p>
                                <p>Enroll now to access all <?= count($curriculumLessons) ?> lessons and their resources!</p>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <div class="curriculum-placeholder">
                        <i class='bx bx-video-off'></i>
                        <p>No curriculum available for this course yet.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="content-section">
            <h3><i class='bx bx-user-check'></i> Instructor</h3>
            <div class="section-content">
                <div class="instructor-profile">
                    <div class="instructor-header">
                        <div class="instructor-avatar">
                            <?php if(!empty($course['profile_picture'])): ?>
                                <img src="../<?= htmlspecialchars($course['profile_picture']) ?>" class="avatar-image" alt="<?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="avatar-circle" style="display: none;">
                                    <?= strtoupper(substr($course['first_name'], 0, 1) . substr($course['last_name'], 0, 1)) ?>
                                </div>
                            <?php else: ?>
                                <div class="avatar-circle">
                                    <?= strtoupper(substr($course['first_name'], 0, 1) . substr($course['last_name'], 0, 1)) ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="instructor-info">
                            <div class="instructor-name">
                                <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>
                            </div>
                            <div class="instructor-title">
                                <span class="badge bg-primary-light text-primary">Verified Instructor</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="instructor-bio">
                        <?= htmlspecialchars($course['instructor_bio'] ?? 'Experienced instructor passionate about education and helping students succeed.') ?>
                    </div>
                    
                    <?php if (!empty($course['website']) || !empty($course['social_linkedin']) || !empty($course['social_twitter']) || !empty($course['social_facebook']) || !empty($course['social_youtube'])): ?>
                    <div class="instructor-social-links">
                        <h6 class="social-title">Connect with me:</h6>
                        <div class="social-links-container">
                            <?php if (!empty($course['website'])): ?>
                                <a href="<?= htmlspecialchars($course['website']) ?>" target="_blank" class="social-link website" title="Website">
                                    <i class='bx bx-globe'></i>
                                    <span>Website</span>
                                </a>
                            <?php endif; ?>
                            
                            <?php if (!empty($course['social_linkedin'])): ?>
                                <a href="<?= htmlspecialchars($course['social_linkedin']) ?>" target="_blank" class="social-link linkedin" title="LinkedIn">
                                    <i class='bx bxl-linkedin'></i>
                                    <span>LinkedIn</span>
                                </a>
                            <?php endif; ?>
                            
                            <?php if (!empty($course['social_twitter'])): ?>
                                <a href="<?= htmlspecialchars($course['social_twitter']) ?>" target="_blank" class="social-link twitter" title="Twitter">
                                    <i class='bx bxl-twitter'></i>
                                    <span>Twitter</span>
                                </a>
                            <?php endif; ?>
                            
                            <?php if (!empty($course['social_facebook'])): ?>
                                <a href="<?= htmlspecialchars($course['social_facebook']) ?>" target="_blank" class="social-link facebook" title="Facebook">
                                    <i class='bx bxl-facebook'></i>
                                    <span>Facebook</span>
                                </a>
                            <?php endif; ?>
                            
                            <?php if (!empty($course['social_youtube'])): ?>
                                <a href="<?= htmlspecialchars($course['social_youtube']) ?>" target="_blank" class="social-link youtube" title="YouTube">
                                    <i class='bx bxl-youtube'></i>
                                    <span>YouTube</span>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Clean Circular Progress Indicator */
.course-progress-container {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-top: 0.75rem;
}

.progress-circle-wrapper {
    position: relative;
    flex-shrink: 0;
}

.progress-circle {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    background: conic-gradient(
        #667eea 0deg,
        #667eea calc(var(--progress, 0) * 3.6deg),
        #e5e7eb calc(var(--progress, 0) * 3.6deg),
        #e5e7eb 360deg
    );
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    transition: all 0.3s ease;
}

.progress-circle-inner {
    width: 46px;
    height: 46px;
    background: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.progress-text {
    font-size: 0.875rem;
    font-weight: 600;
    color: #1f2937;
}

.progress-details {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.progress-label {
    font-size: 0.875rem;
    font-weight: 600;
    color: #374151;
}

.progress-breakdown {
    font-size: 0.75rem;
    color: #6b7280;
}

/* Responsive adjustments for progress indicator */
@media (max-width: 768px) {
    .course-progress-container {
        gap: 0.75rem;
    }
    
    .progress-circle {
        width: 50px;
        height: 50px;
    }
    
    .progress-circle-inner {
        width: 38px;
        height: 38px;
    }
    
    .progress-text {
        font-size: 0.75rem;
    }
    
    .progress-label {
        font-size: 0.8rem;
    }
    
    .progress-breakdown {
        font-size: 0.7rem;
    }
}

/* Course Details Styling */
.course-details-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem 1rem;
}

.course-header {
    margin-bottom: 3rem;
}

.course-hero {
    background: linear-gradient(135deg, #fff5e6 0%, #ffecd1 100%);
    border-radius: 20px;
    padding: 3rem;
    border: 1px solid rgba(255, 149, 0, 0.1);
}

.course-breadcrumb {
    margin-bottom: 1.5rem;
}

.course-breadcrumb a {
    color: #4A90E2;
    text-decoration: none;
    font-weight: 500;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: color 0.3s ease;
}

.course-breadcrumb a:hover {
    color: #357ABD;
}

.course-main-info {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 3rem;
    align-items: start;
}

.course-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 1rem;
    line-height: 1.2;
}

.course-description {
    font-size: 1.1rem;
    color: #7f8c8d;
    margin-bottom: 2rem;
    line-height: 1.6;
}

.course-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 2rem;
    margin-bottom: 2rem;
}

.course-meta > div {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #7f8c8d;
    font-size: 0.95rem;
}

.course-meta i {
    color: #4A90E2;
}

.course-actions {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    align-items: center;
}

.btn-primary, .btn-outline {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.875rem 1.5rem;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    cursor: pointer;
    border: none;
}

.btn-primary {
    background: linear-gradient(135deg, #4A90E2, #357ABD);
    color: white;
    box-shadow: 0 4px 15px rgba(74, 144, 226, 0.3);
}

.btn-primary:hover {
    background: linear-gradient(135deg, #357ABD, #2E6DA4);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(74, 144, 226, 0.4);
    color: white;
    text-decoration: none;
}

.btn-outline {
    background: white;
    color: #4A90E2;
    border: 2px solid #4A90E2;
}

.btn-outline:hover {
    background: #4A90E2;
    color: white;
    text-decoration: none;
}

.course-thumbnail-container {
    position: relative;
}

.course-thumbnail {
    width: 100%;
    height: 250px;
    object-fit: cover;
    border-radius: 16px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

.course-pricing {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.price-tag {
    font-size: 2rem;
    font-weight: 700;
    color: #FF9500;
}

.enrollment-status {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #10b981;
    font-weight: 500;
}

.content-section {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
    border: 1px solid #e1e8ed;
}

.content-section h3 {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    color: #2c3e50;
    margin-bottom: 1.5rem;
    font-size: 1.5rem;
}

.content-section h3 i {
    color: #4A90E2;
}

.instructor-profile {
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 12px;
}

.instructor-name {
    font-size: 1.25rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.instructor-bio {
    color: #7f8c8d;
}

/* Cart Animation Effects */
.cart-updated {
    animation: cartBounce 0.6s ease-out;
    background: linear-gradient(135deg, #10b981, #059669) !important;
    color: white !important;
    border-radius: 50%;
    transform: scale(1.1);
}

@keyframes cartBounce {
    0% { transform: scale(1); }
    50% { transform: scale(1.3); }
    100% { transform: scale(1.1); }
}

.cart-icon.has-items {
    animation: cartShake 0.5s ease-in-out;
}

@keyframes cartShake {
    0%, 100% { transform: translateX(0); }
    25% { transform: translateX(-3px); }
    75% { transform: translateX(3px); }
}

/* Button Loading States */
.btn-loading {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-loading .bx-spin {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Cart Options Modal */
.cart-options-modal {
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    padding: 2rem;
    border-radius: 16px;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
    z-index: 1000;
    max-width: 400px;
    width: 90%;
    text-align: center;
}

.cart-options-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    z-index: 999;
}

@media (max-width: 768px) {
    .course-main-info {
        grid-template-columns: 1fr;
        gap: 2rem;
    }
    
    .course-title {
        font-size: 2rem;
    }
    
    .course-hero {
        padding: 2rem;
    }
    
    .course-actions {
        flex-direction: column;
        align-items: stretch;
    }
    
    .btn-primary, .btn-outline {
        justify-content: center;
    }
}

/* Curriculum Preview Styling */
.curriculum-preview {
    margin-top: 1rem;
}

.locked-notice {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border: 1px solid #f59e0b;
    border-radius: 12px;
    padding: 1rem 1.5rem;
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    color: #92400e;
    font-weight: 500;
}

.locked-notice i {
    font-size: 1.2rem;
    color: #f59e0b;
}

.curriculum-accordion .accordion-item {
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    margin-bottom: 1rem;
    overflow: hidden;
    transition: all 0.3s ease;
}

.curriculum-accordion .accordion-item:hover {
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    transform: translateY(-2px);
}

.curriculum-accordion .accordion-item.locked {
    background: linear-gradient(135deg, #f9fafb 0%, #f3f4f6 100%);
    border-color: #d1d5db;
}

.curriculum-accordion .accordion-button {
    background: white;
    border: none;
    padding: 1.5rem;
    font-weight: 600;
    color: #374151;
    text-align: left;
}

.curriculum-accordion .accordion-button.disabled {
    cursor: not-allowed;
    background: #f9fafb;
    color: #9ca3af;
}

.curriculum-accordion .accordion-button:not(.collapsed) {
    background: #fef3c7;
    color: #92400e;
    border-color: #f59e0b;
}

.curriculum-accordion .accordion-button:focus {
    box-shadow: 0 0 0 0.25rem rgba(245, 158, 11, 0.25);
    border-color: #f59e0b;
}

.lesson-thumbnail-preview {
    position: relative;
    width: 80px;
    height: 60px;
    border-radius: 8px;
    overflow: hidden;
    flex-shrink: 0;
}

.lesson-thumbnail-preview img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.lock-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.lesson-title-preview {
    font-size: 1rem;
    margin-bottom: 0.5rem;
    line-height: 1.4;
}

.lesson-number {
    color: #f59e0b;
    font-weight: 700;
    margin-right: 0.5rem;
}

.lesson-name {
    color: #374151;
}

.lesson-meta-preview {
    display: flex;
    gap: 1rem;
    font-size: 0.875rem;
    color: #6b7280;
}

.lesson-meta-preview .meta-item {
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.lesson-meta-preview i {
    font-size: 1rem;
}

.lock-indicator {
    color: #9ca3af;
    font-size: 1.5rem;
    margin-left: 1rem;
}

.locked.lesson-preview-item .lock-indicator {
    color: #f59e0b;
}

.lesson-content-preview {
    padding: 1rem 0;
}

.lesson-description {
    color: #6b7280;
    font-style: italic;
    margin-bottom: 1rem;
    line-height: 1.6;
}

.content-breakdown {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.content-group h6 {
    color: #374151;
    font-weight: 600;
    margin-bottom: 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.content-group i {
    font-size: 1.1rem;
}

.content-list {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
    padding-left: 1.5rem;
}

.content-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    padding: 0.5rem;
    background: #f9fafb;
    border-radius: 6px;
    border-left: 3px solid #e5e7eb;
}

.content-item span {
    font-weight: 500;
    color: #374151;
}

.content-item small {
    color: #6b7280;
    font-size: 0.8rem;
}

.locked-content-preview {
    background: #f9fafb;
    border-radius: 12px;
    padding: 1.5rem;
    margin-top: 1rem;
    border: 2px dashed #d1d5db;
}

.locked-content-info p {
    color: #6b7280;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.locked-content-summary {
    display: flex;
    flex-wrap: wrap;
    gap: 0.75rem;
}

.content-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 0.75rem;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 500;
}

.quiz-badge {
    background: #dbeafe;
    color: #1d4ed8;
    border: 1px solid #3b82f6;
}

.material-badge {
    background: #d1fae5;
    color: #047857;
    border: 1px solid #10b981;
}

.assignment-badge {
    background: #fef3c7;
    color: #92400e;
    border: 1px solid #f59e0b;
}

.notes-badge {
    background: #f3e8ff;
    color: #7c3aed;
    border: 1px solid #8b5cf6;
}

.enroll-cta-bottom {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 2rem;
    border-radius: 16px;
    text-align: center;
    margin-top: 2rem;
}

.enroll-cta-bottom p {
    margin: 0;
    color: white;
}

.enroll-cta-bottom p:first-child {
    font-size: 1.2rem;
    margin-bottom: 0.5rem;
}

.curriculum-placeholder {
    text-align: center;
    padding: 3rem 2rem;
    color: #9ca3af;
}

.curriculum-placeholder i {
    font-size: 3rem;
    margin-bottom: 1rem;
    display: block;
}

/* Responsive Design for Curriculum */
@media (max-width: 768px) {
    .lesson-thumbnail-preview {
        width: 60px;
        height: 45px;
    }
    
    .lesson-meta-preview {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .locked-content-summary {
        flex-direction: column;
    }
    
    .content-badge {
        justify-content: center;
    }
    
    .curriculum-accordion .accordion-button {
        padding: 1rem;
    }
}

/* Enhanced Instructor Profile Styling */
.instructor-profile {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 16px;
    padding: 2rem;
    border: 1px solid #e2e8f0;
}

.instructor-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.instructor-avatar {
    flex-shrink: 0;
}

.avatar-circle {
    width: 64px;
    height: 64px;
    background: linear-gradient(135deg, #4A90E2, #357ABD);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 700;
    font-size: 1.5rem;
    box-shadow: 0 4px 15px rgba(74, 144, 226, 0.3);
}

.avatar-image {
    width: 64px;
    height: 64px;
    border-radius: 50%;
    object-fit: cover;
    box-shadow: 0 4px 15px rgba(74, 144, 226, 0.3);
    border: 3px solid white;
}

.instructor-info {
    flex: 1;
}

.instructor-name {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 0.5rem;
}

.instructor-title {
    margin-top: 0.5rem;
}

.bg-primary-light {
    background-color: #dbeafe !important;
}

.text-primary {
    color: #1d4ed8 !important;
}

.instructor-bio {
    color: #64748b;
    line-height: 1.6;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: white;
    border-radius: 12px;
    border-left: 4px solid #4A90E2;
    font-style: italic;
}

.instructor-social-links {
    margin-top: 2rem;
    padding-top: 1.5rem;
    border-top: 1px solid #e2e8f0;
}

.social-title {
    font-size: 1rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.social-title::before {
    content: '';
    width: 20px;
    height: 2px;
    background: #4A90E2;
    border-radius: 1px;
}

.social-links-container {
    display: flex;
    flex-wrap: wrap;
    gap: 0.75rem;
}

.social-link {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1rem;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    color: #64748b;
    text-decoration: none;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.3s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.social-link:hover {
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.social-link.website:hover {
    background: #0ea5e9;
    color: white;
    border-color: #0ea5e9;
}

.social-link.linkedin:hover {
    background: #0077b5;
    color: white;
    border-color: #0077b5;
}

.social-link.twitter:hover {
    background: #1da1f2;
    color: white;
    border-color: #1da1f2;
}

.social-link.facebook:hover {
    background: #1877f2;
    color: white;
    border-color: #1877f2;
}

.social-link.youtube:hover {
    background: #ff0000;
    color: white;
    border-color: #ff0000;
}

.social-link i {
    font-size: 1.25rem;
}

/* Responsive Design for Social Links */
@media (max-width: 768px) {
    .instructor-header {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .instructor-profile {
        padding: 1.5rem;
    }
    
    .social-links-container {
        justify-content: center;
    }
    
    .social-link {
        flex: 1;
        justify-content: center;
        min-width: 120px;
    }
    
    .avatar-circle {
        width: 56px;
        height: 56px;
        font-size: 1.25rem;
    }
    
    .locked-notice {
        padding: 1rem;
        font-size: 0.9rem;
    }
}
</style>

<script>
// Working Add to Cart Function from courses.php
function addToCart(courseId) {
    // Get the cart button for this specific course
    const cartButton = document.querySelector(`[data-course-id="${courseId}"] .cart-btn`) || document.querySelector('.cart-button');
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        return;
    }
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Adding...';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i> Added!';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-cart-add"></i> In Cart';
                cartButton.style.background = '#f59e0b';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                
                // Add "in-cart" class for styling
                cartButton.classList.add('in-cart');
            }, 1500);
            
        } else if (data.duplicate) {
            // Handle duplicate course (already in cart)
            cartButton.innerHTML = '<i class="bx bx-info-circle"></i> Already in Cart';
            cartButton.style.background = '#f59e0b';
            cartButton.style.color = 'white';
            cartButton.disabled = true;
            cartButton.classList.add('in-cart');
            
            // Show duplicate warning
            showNotification(`📚 ${data.message}`, 'warning');
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i> Error';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

// EXACT ADD TO CART FUNCTION FROM COURSE-CATALOG.PHP
function addToCart(courseId, buttonElement) {
    console.log('addToCart called with courseId:', courseId, 'buttonElement:', buttonElement);
    
    // Get the cart button - either passed as parameter or find by course ID
    let cartButton = buttonElement || document.querySelector('.cart-button');
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        showNotification('❌ Unable to add course to cart - button not found', 'error');
        return;
    }
    
    console.log('Cart button found:', cartButton);
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data - EXACT SAME AS COURSE-CATALOG.PHP
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-check"></i>';
                cartButton.style.background = '#10b981';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                cartButton.classList.add('in-cart');
                cartButton.title = 'Added to Cart';
            }, 1500);
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i>';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

// EXACT UPDATE NAVBAR CART COUNT FUNCTION FROM COURSE-CATALOG.PHP
function updateNavbarCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cart.length;
    
    // Update cart badge in navbar
    const cartBadge = document.querySelector('.cart-count');
    if (cartBadge) {
        cartBadge.textContent = cartCount;
        cartBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    // Update cart icon badge (alternative selector)
    const cartIconBadge = document.querySelector('#cart-count');
    if (cartIconBadge) {
        cartIconBadge.textContent = cartCount;
        cartIconBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    console.log('Cart count updated:', cartCount);
}

// EXACT NOTIFICATION SYSTEM FROM COURSE-CATALOG.PHP
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : type === 'warning' ? '#f59e0b' : '#3b82f6'};
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 10000;
        animation: slideInRight 0.3s ease;
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Add notification animations - EXACT SAME AS COURSE-CATALOG.PHP
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    .notification-content {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 500;
    }
`;
document.head.appendChild(notificationStyles);

// Enroll in free course
function enrollCourse(courseId) {
    fetch('api/enroll-course.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({ course_id: courseId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Successfully enrolled in course!', 'success');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showNotification(data.message || 'Failed to enroll in course', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('An error occurred. Please try again.', 'error');
    });
}

// Notification system
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `course-notification ${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : type === 'warning' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles if not already present
    if (!document.querySelector('#notification-styles')) {
        const style = document.createElement('style');
        style.id = 'notification-styles';
        style.textContent = `
            .course-notification {
                position: fixed;
                top: 100px;
                right: 20px;
                background: white;
                padding: 16px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
                border-left: 4px solid;
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 400px;
            }
            .course-notification.success { border-left-color: #10b981; }
            .course-notification.error { border-left-color: #ef4444; }
            .course-notification.warning { border-left-color: #f59e0b; }
            .course-notification.info { border-left-color: #3b82f6; }
            .notification-content {
                display: flex;
                align-items: center;
                gap: 12px;
                font-size: 14px;
                font-weight: 500;
            }
            .notification-content i {
                font-size: 18px;
                color: inherit;
            }
            .course-notification.success .notification-content i { color: #10b981; }
            .course-notification.error .notification-content i { color: #ef4444; }
            .course-notification.warning .notification-content i { color: #f59e0b; }
            .course-notification.info .notification-content i { color: #3b82f6; }
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOutRight {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 4 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 4000);
}

// Keep the updateCartCount as alias to updateNavbarCartCount for compatibility
function updateCartCount() {
    updateNavbarCartCount();
}

// Show cart options after adding item
function showCartOptions() {
    const modal = document.createElement('div');
    modal.className = 'cart-options-modal';
    modal.innerHTML = `
        <h3>🎉 Added to Cart!</h3>
        <p>Course has been added to your cart successfully.</p>
        <div style="display: flex; gap: 1rem; justify-content: center; margin-top: 1.5rem;">
            <a href="checkout.php" class="btn-primary">
                <i class='bx bx-credit-card'></i>
                Go to Checkout
            </a>
            <button onclick="closeCartModal()" class="btn-outline">
                Continue Shopping
            </button>
        </div>
    `;
    
    const backdrop = document.createElement('div');
    backdrop.className = 'cart-options-backdrop';
    backdrop.onclick = closeCartModal;
    
    document.body.appendChild(backdrop);
    document.body.appendChild(modal);
    
    // Auto close after 5 seconds
    setTimeout(closeCartModal, 5000);
}

// Close cart modal
function closeCartModal() {
    const modal = document.querySelector('.cart-options-modal');
    const backdrop = document.querySelector('.cart-options-backdrop');
    
    if (modal) document.body.removeChild(modal);
    if (backdrop) document.body.removeChild(backdrop);
}

// Initialize cart on page load (working version from courses.php)
function initializeCart() {
    // Create notification container if it doesn't exist
    if (!document.querySelector('.cart-notification')) {
        const notificationContainer = document.createElement('div');
        notificationContainer.className = 'cart-notification';
        document.body.appendChild(notificationContainer);
    }
    
    // Update cart count on page load
    updateNavbarCartCount();
    
    // Check if current course is already in cart
    const currentCourseId = <?= $courseId ?>;
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const isInCart = cart.some(item => item.id == currentCourseId);
    
    if (isInCart) {
        const cartButton = document.querySelector('.cart-button');
        if (cartButton) {
            cartButton.innerHTML = '<i class="bx bx-check"></i> In Cart';
            cartButton.style.background = '#f59e0b';
            cartButton.style.color = 'white';
            cartButton.disabled = true;
            cartButton.classList.add('in-cart');
        }
    }
    
    console.log('Cart initialized. Current cart items:', cart.length);
}

// Initialize page functionality
document.addEventListener('DOMContentLoaded', function() {
    // Initialize cart functionality
    initializeCart();
    
    // Animate circular progress indicator
    const progressCircle = document.querySelector('.progress-circle');
    if (progressCircle) {
        const progressValue = parseInt(progressCircle.getAttribute('data-progress')) || 0;
        
        // Set CSS custom property for the conic gradient
        progressCircle.style.setProperty('--progress', progressValue);
        
        // Add smooth animation effect
        let currentProgress = 0;
        const animationDuration = 1500; // 1.5 seconds
        const startTime = performance.now();
        
        function animateProgress(timestamp) {
            const elapsed = timestamp - startTime;
            const progress = Math.min(elapsed / animationDuration, 1);
            
            // Use easing function for smooth animation
            const easeOut = 1 - Math.pow(1 - progress, 3);
            currentProgress = Math.round(progressValue * easeOut);
            
            progressCircle.style.setProperty('--progress', currentProgress);
            
            if (progress < 1) {
                requestAnimationFrame(animateProgress);
            }
        }
        
        // Start animation after a small delay
        setTimeout(() => {
            requestAnimationFrame(animateProgress);
        }, 300);
    }
    
    // Add notification styles if not present
    if (!document.querySelector('#notification-styles')) {
        const styles = document.createElement('style');
        styles.id = 'notification-styles';
        styles.textContent = `
            .course-actions .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            padding: 1rem 2rem;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        /* Referral Discount Styles */
        .course-pricing {
            margin-bottom: 1rem;
        }
        
        .price-with-discount {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
            gap: 0.5rem;
        }
        
        .original-price {
            color: #6b7280;
            text-decoration: line-through;
            font-size: 1rem;
            font-weight: 500;
        }
        
        .discounted-price {
            color: #059669;
            font-size: 1.75rem;
            font-weight: 700;
            margin: 0;
        }
        
        .price-tag {
            color: #1f2937;
            font-size: 1.75rem;
            font-weight: 700;
            margin: 0;
        }
        
        .discount-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.875rem;
            font-weight: 600;
            box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
        }
        
        .discount-badge i {
            font-size: 1rem;
        }
            
            .course-notification.show {
                transform: translateX(0);
            }
            
            .course-notification {
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 1rem 1.5rem;
                border-radius: 12px;
                color: white;
                font-weight: 500;
                z-index: 10000;
                transform: translateX(400px);
                transition: all 0.3s ease;
                max-width: 400px;
                box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
            }
            
            .course-notification.success {
                background: linear-gradient(135deg, #10b981, #059669);
            }
            
            .course-notification.error {
                background: linear-gradient(135deg, #ef4444, #dc2626);
            }
            
            .course-notification.warning {
                background: linear-gradient(135deg, #f59e0b, #d97706);
            }
            
            .notification-content {
                display: flex;
                align-items: center;
                gap: 0.75rem;
            }
            
            .notification-content i {
                font-size: 1.25rem;
            }
        `;
        document.head.appendChild(styles);
    }
});
</script>

<?php include_once 'components/footer.php'; ?>
