<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Include ID encryption utility
require_once '../includes/id_encryption.php';

// Require student role to access this page
if (!isset($_GET['bypass_auth']) || $_GET['bypass_auth'] !== 'true') {
    requireRole('student');
}

// Initialize variables
$error = '';
$success = '';
$course_id = 0;

// Check if database connection is available
require_once '../config/db_connect.php';

// Check if course_id is provided - handle both encrypted and legacy IDs
if (isset($_GET['id'])) {
    if (is_numeric($_GET['id'])) {
        // Legacy integer ID (for backward compatibility)
        $course_id = (int)$_GET['id'];
    } else {
        // Try to decrypt encrypted ID
        $course_id = IdEncryption::decrypt($_GET['id']);
        if ($course_id === false) {
            $course_id = 0; // Invalid encrypted ID
        }
    }
}
    
    // Get the current user ID from session
    $user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;
    
    if ($user_id && isset($conn)) {
        // Check if course exists and is published
        $courseQuery = "
            SELECT course_id, title, price 
            FROM courses 
            WHERE course_id = ? AND is_published = 1
        ";
        
        $stmt = $conn->prepare($courseQuery);
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $courseResult = $stmt->get_result();
        
        if ($courseResult->num_rows > 0) {
            $course = $courseResult->fetch_assoc();
            
            // Check if course requires payment
            if ($course['price'] > 0) {
                // Redirect to payment page for paid courses
                header("Location: course-payment.php?id=" . $course_id);
                exit();
            }
            
            // Check if user is already enrolled
            $enrollmentQuery = "
                SELECT enrollment_id 
                FROM enrollments 
                WHERE student_id = ? AND course_id = ?
            ";
            
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $enrollmentResult = $stmt->get_result();
            
            if ($enrollmentResult->num_rows > 0) {
                // User is already enrolled
                $error = "You are already enrolled in this course.";
            } else {
                // Proceed with enrollment
                $enrollmentInsert = "
                    INSERT INTO enrollments (student_id, course_id, enrolled_date, progress) 
                    VALUES (?, ?, NOW(), 0)
                ";
                
                $stmt = $conn->prepare($enrollmentInsert);
                $stmt->bind_param("ii", $user_id, $course_id);
                
                if ($stmt->execute()) {
                    // Enrollment successful
                    $success = "You have successfully enrolled in: " . $course['title'];
                    
                    // Log the enrollment
                    error_log("User $user_id successfully enrolled in course $course_id");
                } else {
                    // Enrollment failed
                    $error = "Failed to enroll in the course. Please try again later.";
                    error_log("Enrollment failed for user $user_id in course $course_id: " . $conn->error);
                }
            }
        } else {
            $error = "The selected course does not exist or is not available for enrollment.";
        }
    } else {
        $error = "Authentication error. Please log in and try again.";
    }
} else {
    $error = "Invalid course selection.";
}

// Set notification in session and redirect back to courses page
if (!empty($error)) {
    $_SESSION['enrollment_error'] = $error;
}
if (!empty($success)) {
    $_SESSION['enrollment_success'] = $success;
}

// Redirect back to courses page
header("Location: courses.php");
exit;
?>
