<?php
/**
 * Course Grade Details - Student's comprehensive grade view for a specific course
 * Shows detailed breakdown of assignments, quizzes, and overall grade calculation
 */

// Set current page for navigation
$currentPage = 'courses';

// Authentication - Require student role
require_once '../auth/functions.php';
requireRole('student');

require_once '../config/db_connect.php';
require_once '../includes/course_grade_calculator.php';

$student_id = $_SESSION['user_id'];
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;

if (!$course_id) {
    header("Location: dashboard.php");
    exit();
}

// Verify student is enrolled in this course
$enrollment_check = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?");
if (!$enrollment_check) {
    error_log("Database prepare failed for enrollment check: " . $conn->error);
    header("Location: dashboard.php");
    exit();
}
$enrollment_check->bind_param("ii", $student_id, $course_id);
$enrollment_check->execute();
if (!$enrollment_check->get_result()->fetch_assoc()) {
    header("Location: dashboard.php");
    exit();
}

// Get course information
$course_query = "
    SELECT 
        c.*,
        u.first_name as instructor_first,
        u.last_name as instructor_last
    FROM courses c
    JOIN users u ON c.teacher_id = u.user_id
    WHERE c.course_id = ?
";
$stmt = $conn->prepare($course_query);
if (!$stmt) {
    error_log("Database prepare failed for course query: " . $conn->error);
    header("Location: dashboard.php");
    exit();
}
$stmt->bind_param("i", $course_id);
$stmt->execute();
$course = $stmt->get_result()->fetch_assoc();

if (!$course) {
    header("Location: dashboard.php");
    exit();
}

// Calculate comprehensive grade data
$calculator = new CourseGradeCalculator($conn);
$grade_data = $calculator->calculateFinalGrade($student_id, $course_id);

// Get detailed assignment data
$assignments_query = "
    SELECT 
        a.*,
        s.grade,
        s.submitted_at,
        s.graded_at,
        s.feedback
    FROM assignments a
    LEFT JOIN submissions s ON a.assignment_id = s.assignment_id AND s.student_id = ?
    WHERE a.course_id = ?
    ORDER BY a.due_date DESC
";
$stmt = $conn->prepare($assignments_query);
if (!$stmt) {
    error_log("Database prepare failed for assignments query: " . $conn->error);
    $assignments = [];
} else {
    $stmt->bind_param("ii", $student_id, $course_id);
    $stmt->execute();
    $assignments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
}

// Get detailed quiz data
$quizzes_query = "
    SELECT 
        q.*,
        qa.score,
        qa.total_questions,
        qa.start_time,
        qa.end_time,
        qa.time_taken
    FROM quizzes q
    LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
    WHERE q.course_id = ? AND q.is_published = 1
    ORDER BY q.created_at DESC
";
$stmt = $conn->prepare($quizzes_query);
if (!$stmt) {
    error_log("Database prepare failed for quizzes query: " . $conn->error);
    $quizzes = [];
} else {
    $stmt->bind_param("ii", $student_id, $course_id);
    $stmt->execute();
    $quizzes = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
}

$instructor_name = trim($course['instructor_first'] . ' ' . $course['instructor_last']);
$final_grade = $grade_data['final_grade'];
$certificate_eligible = $final_grade >= 50;

// Include header
include_once __DIR__ . '/components/header.php';
?>

<style>
    /* Clean Panadite Academy Course Grade Details Styling */
    body {
        background: #f8fafc;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .dashboard-content {
        padding: 1.5rem;
        max-width: 1200px;
        margin: 0 auto;
    }

    /* Clean Course Header - Panadite Branding */
    .course-grade-header {
        background: linear-gradient(135deg, #3366ff 0%, #4facfe 100%);
        color: white;
        padding: 2rem;
        border-radius: 1rem;
        margin-bottom: 2rem;
        border: 1px solid #e0ebff;
        position: relative;
        overflow: hidden;
    }

    .course-grade-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 200px;
        height: 200px;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 50%;
        transform: translate(25%, -25%);
    }

    .course-info {
        position: relative;
        z-index: 2;
    }

    .course-title {
        font-size: 2rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        color: white;
    }

    .course-instructor {
        font-size: 1.1rem;
        opacity: 0.9;
        margin-bottom: 1rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .course-instructor i {
        font-size: 1.2rem;
    }

    .course-grading-info {
        background: rgba(255, 255, 255, 0.15);
        padding: 1rem;
        border-radius: 0.5rem;
        font-size: 0.9rem;
        margin-top: 1rem;
        backdrop-filter: blur(10px);
    }

    /* Clean Grade Summary Cards */
    .grade-summary-cards {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .grade-summary-card {
        background: white;
        padding: 1.75rem;
        border-radius: 0.75rem;
        border: 1px solid #e8ecf4;
        text-align: center;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .grade-summary-card:hover {
        transform: translateY(-2px);
        border-color: #3366ff;
        box-shadow: 0 8px 25px rgba(51, 102, 255, 0.12);
    }

    .grade-summary-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #3366ff, #4facfe);
    }

    .grade-number {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        line-height: 1;
    }

    /* Panadite Grade Colors */
    .grade-excellent { color: #22c55e; }
    .grade-good { color: #3366ff; }
    .grade-average { color: #f59e0b; }
    .grade-poor { color: #ef4444; }
    .grade-none { color: #6b7280; }

    .grade-label {
        color: #6b7280;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    /* Clean Detailed Breakdown Sections */
    .detailed-breakdown {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
        margin-bottom: 2rem;
    }

    @media (max-width: 768px) {
        .detailed-breakdown {
            grid-template-columns: 1fr;
        }
    }

    .breakdown-section {
        background: white;
        padding: 1.75rem;
        border-radius: 0.75rem;
        border: 1px solid #e8ecf4;
        transition: all 0.3s ease;
    }

    .breakdown-section:hover {
        border-color: #3366ff;
        box-shadow: 0 4px 15px rgba(51, 102, 255, 0.08);
    }

    .breakdown-header {
        display: flex;
        align-items: center;
        margin-bottom: 1.5rem;
        padding-bottom: 1rem;
        border-bottom: 2px solid #f1f5f9;
    }

    .breakdown-icon {
        width: 40px;
        height: 40px;
        background: linear-gradient(135deg, #3366ff, #4facfe);
        border-radius: 0.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.2rem;
        color: white;
        margin-right: 1rem;
    }

    .breakdown-title {
        font-size: 1.3rem;
        font-weight: 700;
        margin: 0;
        color: #1a2332;
    }

    .item-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }

    .grade-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 1.25rem;
        border: 1px solid #e8ecf4;
        border-radius: 0.5rem;
        transition: all 0.3s ease;
        background: #fafbff;
    }

    .grade-item:hover {
        border-color: #3366ff;
        background: #f0f4ff;
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(51, 102, 255, 0.08);
    }

    .item-info h4 {
        margin: 0 0 0.4rem 0;
        font-size: 1rem;
        font-weight: 600;
        color: #1a2332;
    }

    .item-meta {
        font-size: 0.8rem;
        color: #6b7280;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .item-grade {
        text-align: right;
        display: flex;
        flex-direction: column;
        align-items: flex-end;
        gap: 0.5rem;
    }

    .item-score {
        font-size: 1.2rem;
        font-weight: 700;
        color: #1a2332;
    }

    .item-status {
        font-size: 0.7rem;
        padding: 0.3rem 0.8rem;
        border-radius: 1rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-weight: 600;
    }

    /* Clean Panadite Status Colors */
    .status-submitted { background: #dbeafe; color: #1e40af; }
    .status-graded { background: #dcfce7; color: #166534; }
    .status-pending { background: #fef3c7; color: #92400e; }
    .status-not-submitted { background: #fee2e2; color: #dc2626; }
    .status-not-attempted { background: #f1f5f9; color: #64748b; }
    .status-completed { background: #dcfce7; color: #166534; }

    .contributes-badge {
        font-size: 0.65rem;
        padding: 0.25rem 0.6rem;
        border-radius: 1rem;
        margin-left: 0.5rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-weight: 600;
    }

    .contributes-yes { 
        background: #22c55e; 
        color: white; 
    }
    .contributes-no { 
        background: #6b7280; 
        color: white; 
    }

    /* Clean Back Button */
    .back-button {
        background: white;
        border: 1px solid #e8ecf4;
        color: #3366ff;
        padding: 0.75rem 1.5rem;
        border-radius: 0.5rem;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        font-weight: 600;
        transition: all 0.3s ease;
        margin-bottom: 1.5rem;
    }

    .back-button:hover {
        background: #3366ff;
        color: white;
        border-color: #3366ff;
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(51, 102, 255, 0.15);
    }

    .back-button i {
        font-size: 1.1rem;
    }

    /* Empty State Styling */
    .empty-state {
        text-align: center;
        padding: 3rem 1rem;
        color: #6b7280;
    }

    .empty-state i {
        font-size: 3rem;
        color: #d1d5db;
        margin-bottom: 1rem;
    }

    .empty-state h4 {
        font-size: 1.2rem;
        margin-bottom: 0.5rem;
        color: #374151;
    }

    .empty-state p {
        font-size: 0.9rem;
        margin: 0;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .dashboard-content {
            padding: 1rem;
        }
        
        .course-grade-header {
            padding: 1.5rem;
        }
        
        .course-title {
            font-size: 1.5rem;
        }
        
        .grade-summary-cards {
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 1rem;
        }
        
        .grade-summary-card {
            padding: 1.25rem;
        }
        
        .grade-number {
            font-size: 2rem;
        }
        
        .breakdown-section {
            padding: 1.25rem;
        }
        
        .grade-item {
            padding: 1rem;
            flex-direction: column;
            align-items: flex-start;
            gap: 0.75rem;
        }
        
        .item-grade {
            align-items: flex-start;
            width: 100%;
        }
    }

    .certificate-section {
        background: white;
        padding: 2rem;
        border-radius: 12px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        text-align: center;
        border: 1px solid #e9ecef;
    }

    .certificate-icon {
        font-size: 3rem;
        margin-bottom: 1rem;
    }

    .certificate-eligible .certificate-icon { color: #28a745; }
    .certificate-not-eligible .certificate-icon { color: #6c757d; }

    .certificate-btn {
        background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        color: white;
        border: none;
        padding: 1rem 2rem;
        border-radius: 25px;
        font-size: 1.1rem;
        font-weight: 600;
        text-decoration: none;
        transition: all 0.3s ease;
        display: inline-block;
        margin-top: 1rem;
    }

    .certificate-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.2);
        color: white;
        text-decoration: none;
    }

    .certificate-btn:disabled {
        background: #6c757d;
        cursor: not-allowed;
        transform: none;
    }

    .weight-info {
        background: #f8f9fa;
        padding: 1rem;
        border-radius: 8px;
        margin-bottom: 1rem;
        font-size: 0.9rem;
        color: #495057;
    }

    .back-button {
        background: #6c757d;
        color: white;
        border: none;
        padding: 0.8rem 1.5rem;
        border-radius: 20px;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        margin-bottom: 2rem;
        transition: all 0.3s ease;
    }

    .back-button:hover {
        background: #5a6268;
        color: white;
        text-decoration: none;
        transform: translateX(-2px);
    }

    .back-button i {
        margin-right: 0.5rem;
    }
</style>

<div class="dashboard-content">
    <!-- Back Button -->
    <a href="dashboard.php" class="back-button">
        <i class='bx bx-arrow-back'></i>
        Back to Dashboard
    </a>

    <!-- Course Header -->
    <div class="course-grade-header">
        <div class="course-info">
            <h1 class="course-title"><?= htmlspecialchars($course['title']) ?></h1>
            <div class="course-instructor">
                <i class='bx bx-user-circle'></i> Instructor: <?= htmlspecialchars($instructor_name) ?>
            </div>
            <div class="course-grading-info">
                <i class='bx bx-info-circle'></i>
                Course Grading: <?= $course['assignment_weight'] ?>% Assignments, <?= $course['quiz_weight'] ?>% Quizzes
            </div>
        </div>
    </div>

    <!-- Grade Summary Cards -->
    <div class="grade-summary-cards">
        <div class="grade-summary-card">
            <div class="grade-number <?= $final_grade >= 90 ? 'grade-excellent' : ($final_grade >= 80 ? 'grade-good' : ($final_grade >= 70 ? 'grade-average' : ($final_grade >= 50 ? 'grade-poor' : 'grade-none'))) ?>">
                <?= $final_grade ?>%
            </div>
            <div class="grade-label">Final Grade</div>
        </div>
        <div class="grade-summary-card">
            <div class="grade-number <?= $final_grade >= 90 ? 'grade-excellent' : ($final_grade >= 80 ? 'grade-good' : ($final_grade >= 70 ? 'grade-average' : ($final_grade >= 50 ? 'grade-poor' : 'grade-none'))) ?>">
                <?= $grade_data['letter_grade'] ?>
            </div>
            <div class="grade-label">Letter Grade</div>
        </div>
        <div class="grade-summary-card">
            <div class="grade-number <?= $grade_data['assignment_component']['average_grade'] >= 80 ? 'grade-good' : ($grade_data['assignment_component']['average_grade'] >= 70 ? 'grade-average' : 'grade-poor') ?>">
                <?= number_format($grade_data['assignment_component']['average_grade'], 1) ?>%
            </div>
            <div class="grade-label">Assignment Average</div>
        </div>
        <div class="grade-summary-card">
            <div class="grade-number <?= $grade_data['quiz_component']['average_grade'] >= 80 ? 'grade-good' : ($grade_data['quiz_component']['average_grade'] >= 70 ? 'grade-average' : 'grade-poor') ?>">
                <?= number_format($grade_data['quiz_component']['average_grade'], 1) ?>%
            </div>
            <div class="grade-label">Quiz Average</div>
        </div>
    </div>

    <!-- Detailed Breakdown -->
    <div class="detailed-breakdown">
        <!-- Assignments Section -->
        <div class="breakdown-section">
            <div class="breakdown-header">
                <i class='bx bx-task breakdown-icon'></i>
                <h3 class="breakdown-title">Assignments (<?= $course['assignment_weight'] ?>%)</h3>
            </div>
            <div class="item-list">
                <?php if (empty($assignments)): ?>
                    <div class="grade-item">
                        <div class="item-info">
                            <h4>No assignments available</h4>
                            <div class="item-meta">Check back later for assignments</div>
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($assignments as $assignment): ?>
                        <div class="grade-item">
                            <div class="item-info">
                                <h4>
                                    <?= htmlspecialchars($assignment['title']) ?>
                                    <?php if ($assignment['contributes_to_final_grade']): ?>
                                        <span class="contributes-badge contributes-yes">Counts</span>
                                    <?php else: ?>
                                        <span class="contributes-badge contributes-no">Practice</span>
                                    <?php endif; ?>
                                </h4>
                                <div class="item-meta">
                                    Due: <?= date('M j, Y', strtotime($assignment['due_date'])) ?>
                                    <?php if ($assignment['submitted_at']): ?>
                                        | Submitted: <?= date('M j, Y', strtotime($assignment['submitted_at'])) ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="item-grade">
                                <?php if ($assignment['grade'] !== null): ?>
                                    <div class="item-score <?= $assignment['grade'] >= 80 ? 'grade-good' : ($assignment['grade'] >= 70 ? 'grade-average' : 'grade-poor') ?>">
                                        <?= $assignment['grade'] ?>%
                                    </div>
                                    <div class="item-status status-graded">Graded</div>
                                <?php elseif ($assignment['submitted_at']): ?>
                                    <div class="item-score grade-none">-</div>
                                    <div class="item-status status-submitted">Submitted</div>
                                <?php else: ?>
                                    <div class="item-score grade-none">-</div>
                                    <div class="item-status status-not-submitted">Not Submitted</div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Quizzes Section -->
        <div class="breakdown-section">
            <div class="breakdown-header">
                <i class='bx bx-brain breakdown-icon'></i>
                <h3 class="breakdown-title">Quizzes (<?= $course['quiz_weight'] ?>%)</h3>
            </div>
            <div class="item-list">
                <?php if (empty($quizzes)): ?>
                    <div class="grade-item">
                        <div class="item-info">
                            <h4>No quizzes available</h4>
                            <div class="item-meta">Check back later for quizzes</div>
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($quizzes as $quiz): ?>
                        <div class="grade-item">
                            <div class="item-info">
                                <h4>
                                    <?= htmlspecialchars($quiz['title']) ?>
                                    <?php if ($quiz['contributes_to_final_grade']): ?>
                                        <span class="contributes-badge contributes-yes">Counts</span>
                                    <?php else: ?>
                                        <span class="contributes-badge contributes-no">Practice</span>
                                    <?php endif; ?>
                                </h4>
                                <div class="item-meta">
                                    <?php if ($quiz['end_time']): ?>
                                        Completed: <?= date('M j, Y', strtotime($quiz['end_time'])) ?>
                                        | Time: <?= gmdate('H:i:s', $quiz['time_taken']) ?>
                                    <?php else: ?>
                                        Available for attempt
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="item-grade">
                                <?php if ($quiz['score'] !== null): ?>
                                    <?php 
                                    $percentage = $quiz['total_questions'] > 0 ? round(($quiz['score'] / $quiz['total_questions']) * 100) : 0;
                                    ?>
                                    <div class="item-score <?= $percentage >= 80 ? 'grade-good' : ($percentage >= 70 ? 'grade-average' : 'grade-poor') ?>">
                                        <?= $percentage ?>%
                                    </div>
                                    <div class="item-status status-completed">Completed</div>
                                <?php else: ?>
                                    <div class="item-score grade-none">-</div>
                                    <div class="item-status status-not-attempted">Not Attempted</div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Certificate Section -->
    <div class="certificate-section <?= $certificate_eligible ? 'certificate-eligible' : 'certificate-not-eligible' ?>">
        <div class="certificate-icon">
            <i class='bx <?= $certificate_eligible ? 'bx-award' : 'bx-lock' ?>'></i>
        </div>
        
        <?php if ($certificate_eligible): ?>
            <h3>🎉 Congratulations!</h3>
            <p>You have achieved a passing grade of <?= $final_grade ?>% in this course. You are eligible to download your certificate!</p>
            <a href="certificates.php?course_id=<?= $course_id ?>&action=preview" class="certificate-btn">
                <i class='bx bx-award'></i> View Certificate
            </a>
        <?php else: ?>
            <h3>Certificate Not Available</h3>
            <p>You need a minimum grade of 50% to earn a certificate. Your current grade is <?= $final_grade ?>%. Keep working on your assignments and quizzes!</p>
            <button class="certificate-btn" disabled>
                <i class='bx bx-lock'></i> Certificate Locked
            </button>
        <?php endif; ?>
    </div>
</div>

<?php include_once __DIR__ . '/components/footer.php'; ?>
