<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Decrypt and validate course ID
$course_id = getSecureId('id');
if (!$course_id) {
    $_SESSION['error'] = "Invalid or corrupted course ID";
    header('Location: courses.php');
    exit();
}

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

// Get course details
$courseQuery = "
    SELECT c.*, cat.category_name, 
           u.first_name, u.last_name,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as enrolled_students
    FROM courses c
    JOIN categories cat ON c.category_id = cat.category_id
    JOIN users u ON c.teacher_id = u.user_id
    WHERE c.course_id = ?
";

$courseStmt = $conn->prepare($courseQuery);
$courseStmt->bind_param("i", $course_id);
$courseStmt->execute();
$courseResult = $courseStmt->get_result();

if ($courseResult->num_rows == 0) {
    $_SESSION['error'] = "Course not found";
    header('Location: courses.php');
    exit();
}

$course = $courseResult->fetch_assoc();
$page_title = $course['title'] . " - Materials";

// Include header
include_once 'components/header.php';
?>

<!-- Course Header -->
<div class="container-fluid pt-4">
    <div class="row">
        <div class="col-12">
            <div class="course-banner">
                <div class="course-banner-content">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="course-title"><?= htmlspecialchars($course['title']) ?></h2>
                            <p class="course-instructor">
                                <i class='bx bx-user'></i> Instructor: <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>
                            </p>
                            <div class="mt-2">
                                <span class="course-category"><?= htmlspecialchars($course['category_name'] ?? 'General') ?></span>
                                <?php if (!empty($course['level'])): ?>
                                <span class="course-category"><?= htmlspecialchars(ucfirst($course['level'])) ?> Level</span>
                                <?php endif; ?>
                                <?php if (!empty($course['enrolled_students'])): ?>
                                <span class="course-category"><i class='bx bx-group'></i> <?= $course['enrolled_students'] ?> students</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if ($course && $is_enrolled): ?>
<!-- Course Navigation Menu -->
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="course-nav">
                <ul class="nav nav-pills nav-fill">
                    <li class="nav-item">
                        <a href="course-detail.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-book-open'></i> Course Content
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-materials.php?id=<?php echo $course_id; ?>" class="nav-link active">
                            <i class='bx bx-file'></i> Materials
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-assignments.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-task'></i> Assignments
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-quizzes.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-question-mark'></i> Quizzes
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-discussions.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-chat'></i> Discussions
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Materials Content -->
<div class="container-fluid mb-5">
    <div class="row">
        <div class="col-12">
            <div class="content-card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Course Materials</h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="filter-buttons mb-4">
                        <button type="button" class="filter-button active material-filter" data-filter="all">All Materials</button>
                        <button type="button" class="filter-button material-filter" data-filter="document">Documents</button>
                        <button type="button" class="filter-button material-filter" data-filter="video">Videos</button>
                        <button type="button" class="filter-button material-filter" data-filter="audio">Audio</button>
                        <button type="button" class="filter-button material-filter" data-filter="image">Images</button>
                        <button type="button" class="filter-button material-filter" data-filter="other">Other</button>
                    </div>
                    
                    <?php
                    // Get course materials
                    $materialsQuery = "
                        SELECT m.*, u.first_name, u.last_name,
                               (SELECT COUNT(*) FROM material_views WHERE material_id = m.material_id AND student_id = ?) as viewed
                        FROM course_materials m
                        JOIN users u ON m.teacher_id = u.user_id
                        WHERE m.course_id = ? AND m.status = 'active'
                        ORDER BY m.date_added DESC
                    ";
                    
                    $materialStmt = $conn->prepare($materialsQuery);
                    if ($materialStmt) {
                        $materialStmt->bind_param("ii", $student_id, $course_id);
                        $materialStmt->execute();
                        $materialsResult = $materialStmt->get_result();
                        
                        if ($materialsResult->num_rows > 0) {
                            echo '<div class="table-responsive">';
                            echo '<table class="table table-hover">';
                            echo '<thead class="table-light">';
                            echo '<tr>';
                            echo '<th>Title</th>';
                            echo '<th>Added By</th>';
                            echo '<th>Category</th>';
                            echo '<th>Type</th>';
                            echo '<th>Size</th>';
                            echo '<th>Date</th>';
                            echo '<th>Actions</th>';
                            echo '</tr>';
                            echo '</thead>';
                            echo '<tbody>';
                            
                            while ($material = $materialsResult->fetch_assoc()) {
                                // Determine file type and icon
                                $fileType = $material['file_type'] ?? 'unknown';
                                $fileIcon = 'bx-file';
                                $fileCategory = 'document';
                                
                                if (strpos($fileType, 'pdf') !== false) {
                                    $fileIcon = 'bxs-file-pdf';
                                    $fileCategory = 'document';
                                } else if (strpos($fileType, 'doc') !== false || strpos($fileType, 'word') !== false) {
                                    $fileIcon = 'bxs-file-doc';
                                    $fileCategory = 'document';
                                } else if (strpos($fileType, 'xls') !== false || strpos($fileType, 'sheet') !== false) {
                                    $fileIcon = 'bxs-file-xls';
                                    $fileCategory = 'document';
                                } else if (strpos($fileType, 'ppt') !== false || strpos($fileType, 'presentation') !== false) {
                                    $fileIcon = 'bxs-file-ppt';
                                    $fileCategory = 'presentation';
                                } else if (strpos($fileType, 'image') !== false) {
                                    $fileIcon = 'bxs-image';
                                    $fileCategory = 'image';
                                } else if (strpos($fileType, 'video') !== false) {
                                    $fileIcon = 'bxs-video';
                                    $fileCategory = 'video';
                                } else if (strpos($fileType, 'audio') !== false) {
                                    $fileIcon = 'bxs-music';
                                    $fileCategory = 'audio';
                                } else if (strpos($fileType, 'zip') !== false || strpos($fileType, 'rar') !== false) {
                                    $fileIcon = 'bxs-file-archive';
                                    $fileCategory = 'archive';
                                }
                                
                                // Format date
                                $dateAdded = new DateTime($material['date_added']);
                                $formattedDate = $dateAdded->format('M d, Y');
                                
                                // Format size
                                $size = $material['file_size'] ?? 0;
                                $formattedSize = '0 KB';
                                
                                if ($size > 0) {
                                    if ($size < 1024) {
                                        $formattedSize = $size . ' B';
                                    } else if ($size < 1024 * 1024) {
                                        $formattedSize = round($size / 1024, 1) . ' KB';
                                    } else if ($size < 1024 * 1024 * 1024) {
                                        $formattedSize = round($size / (1024 * 1024), 1) . ' MB';
                                    } else {
                                        $formattedSize = round($size / (1024 * 1024 * 1024), 1) . ' GB';
                                    }
                                }
                                
                                // Display material row
                                echo "<tr class='material-item' data-category='{$fileCategory}'>";
                                echo "<td>";
                                
                                // Viewed status
                                if ($material['viewed'] > 0) {
                                    echo "<span class='badge bg-success me-2'><i class='bx bx-check'></i></span>";
                                }
                                
                                echo "<i class='bx {$fileIcon} me-2'></i> " . htmlspecialchars($material['title']);
                                echo "</td>";
                                
                                echo "<td>" . htmlspecialchars($material['first_name'] . ' ' . $material['last_name']) . "</td>";
                                echo "<td><span class='badge bg-light text-dark'>" . htmlspecialchars($material['category']) . "</span></td>";
                                echo "<td>" . htmlspecialchars($fileType) . "</td>";
                                echo "<td>{$formattedSize}</td>";
                                echo "<td>{$formattedDate}</td>";
                                echo "<td>";
                                echo "<div class='btn-group'>";
                                echo "<a href='view_material.php?id={$material['material_id']}' class='btn btn-sm btn-primary me-1'><i class='bx bx-show'></i> View</a>";
                                echo "<a href='download_resource.php?id={$material['material_id']}' class='btn btn-sm btn-outline-secondary'><i class='bx bx-download'></i> Download</a>";
                                echo "</div>";
                                echo "</td>";
                                echo "</tr>";
                                
                                // Optional: Add description row if it exists
                                if (!empty($material['description'])) {
                                    echo "<tr class='material-description-row' data-category='{$fileCategory}'>";
                                    echo "<td colspan='6' class='text-muted small'>" . htmlspecialchars($material['description']) . "</td>";
                                    echo "</tr>";
                                }
                            }
                            
                            echo '</tbody>';
                            echo '</table>';
                            echo '</div>';
                        } else {
                            echo '<div class="empty-state">';
                            echo '<i class="bx bx-file"></i>';
                            echo '<h5 class="mt-3">No materials yet</h5>';
                            echo '<p class="text-muted">No materials have been uploaded to this course yet.</p>';
                            echo '</div>';
                        }
                    }
                    ?>
                    
                    <script>
                    // Filter materials by category
                    document.addEventListener('DOMContentLoaded', function() {
                        const filterButtons = document.querySelectorAll('.material-filter');
                        const materialItems = document.querySelectorAll('.material-item, .material-description-row');
                        
                        filterButtons.forEach(button => {
                            button.addEventListener('click', function() {
                                const filterValue = this.getAttribute('data-filter');
                                
                                // Toggle active class
                                filterButtons.forEach(btn => btn.classList.remove('active'));
                                this.classList.add('active');
                                
                                // Filter items
                                materialItems.forEach(item => {
                                    if (filterValue === 'all' || item.getAttribute('data-category') === filterValue) {
                                        item.style.display = '';
                                    } else {
                                        item.style.display = 'none';
                                    }
                                });
                            });
                        });
                    });
                    </script>
                </div>
            </div>
        </div>
    </div>
</div>
<?php else: ?>
    <!-- Content for non-enrolled users -->
    <div class="container-fluid mb-5">
        <div class="row">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-body p-4">
                        <div class="alert alert-warning">
                            <h5><i class="bx bx-lock me-2"></i> Course materials are only available to enrolled students</h5>
                            <p>To access this content, please enroll in this course.</p>
                            <a href="course-enrollment.php?id=<?= $course_id ?>" class="btn btn-primary mt-2">Enroll Now</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
<?php endif; ?>

<?php

?>
