<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';

// Require student role to access this page
requireRole('student');

// Include PayFast functions
require_once '../payments/payfast.php';

// Initialize variables
$error = '';
$success = '';
$course_id = 0;
$course = null;
$user = null;

// Check if database connection is available
require_once '../config/db_connect.php';

// Decrypt and validate course ID
$course_id = getSecureId('id');
if (!$course_id) {
    header('Location: courses.php?error=invalid_course');
    exit();
}

// Get the current user ID from session
$user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

if ($user_id && isset($conn)) {
    // Get user details
    $userQuery = "SELECT user_id, first_name, last_name, email FROM users WHERE user_id = ?";
    $stmt = $conn->prepare($userQuery);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $userResult = $stmt->get_result();

    if ($userResult->num_rows > 0) {
        $user = $userResult->fetch_assoc();
    }

    // Check if course exists and is published
    $courseQuery = "
        SELECT c.course_id, c.title, c.description, c.price, c.thumbnail,
               u.first_name as teacher_first_name, u.last_name as teacher_last_name,
               cat.category_name
        FROM courses c 
        JOIN users u ON c.teacher_id = u.user_id
        JOIN categories cat ON c.category_id = cat.category_id
        WHERE c.course_id = ? AND c.is_published = 1
    ";

    $stmt = $conn->prepare($courseQuery);
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $courseResult = $stmt->get_result();

    if ($courseResult->num_rows > 0) {
        $course = $courseResult->fetch_assoc();

        // Check if user is already enrolled
        $enrollmentQuery = "
            SELECT enrollment_id 
            FROM enrollments 
            WHERE student_id = ? AND course_id = ?
        ";
        
        $enrollmentStmt = $conn->prepare($enrollmentQuery);
        $enrollmentStmt->bind_param("ii", $user_id, $course_id);
        $enrollmentStmt->execute();
        $enrollmentResult = $enrollmentStmt->get_result();
        
        if ($enrollmentResult->num_rows > 0) {
            // User is already enrolled, redirect to course
            header("Location: course-detail.php?id=" . base64_encode($course_id));
            exit();
        }
        
        $stmt = $conn->prepare($courseQuery);
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $courseResult = $stmt->get_result();
        
        if ($courseResult->num_rows > 0) {
            $course = $courseResult->fetch_assoc();
            
            // Initialize error variable
            $error = '';
            
            // Check if user has already paid for this course
            if (hasUserPurchasedCourse($user_id, $course_id)) {
                // User has paid but not enrolled - auto enroll them
                $enrollQuery = "
                    INSERT INTO enrollments (student_id, course_id, enrolled_date, progress) 
                    VALUES (?, ?, NOW(), 0)
                ";
                
                $stmt = $conn->prepare($enrollQuery);
                $stmt->bind_param("ii", $user_id, $course_id);
                
                if ($stmt->execute()) {
                    // Create notification for successful enrollment
                    createNotification($user_id, 'enrollment', "You have been successfully enrolled in course: " . $course['title']);
                    
                    header("Location: courses.php?enrolled=success&course=" . urlencode($course['title']));
                    exit();
                } else {
                    $error = "Failed to enroll you in the course. Please contact support.";
                }
            }
            
            // EXACT REFERRAL DISCOUNT LOGIC FROM COURSE-CATALOG.PHP
            if ($course) {
                // FETCH USER'S AVAILABLE DISCOUNTS - EXACT SAME AS COURSE-CATALOG.PHP
                $available_discounts = [];
                $discountsQuery = "SELECT * FROM student_referral_discounts 
                                  WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                                  ORDER BY discount_percentage DESC LIMIT 1";
                $discountStmt = $conn->prepare($discountsQuery);
                if ($discountStmt) {
                    $discountStmt->bind_param("i", $user_id);
                    $discountStmt->execute();
                    $discountResult = $discountStmt->get_result();
                    if ($discountResult->num_rows > 0) {
                        $available_discounts = $discountResult->fetch_assoc();
                    }
                }
                
                // APPLY DISCOUNT LOGIC - EXACT SAME AS COURSE-CATALOG.PHP
                $original_price = $course['price'];
                $discounted_price = $original_price;
                $has_discount = false;
                
                // APPLY DISCOUNT IF AVAILABLE
                if (!empty($available_discounts) && $original_price > 0) {
                    $discount_amount = min(
                        ($original_price * $available_discounts['discount_percentage'] / 100),
                        $available_discounts['discount_amount']
                    );
                    $discounted_price = $original_price - $discount_amount;
                    $has_discount = true;
                }
                
                // UPDATE COURSE DATA - EXACT SAME STRUCTURE AS COURSE-CATALOG.PHP
                $course['price'] = $original_price;
                $course['discounted_price'] = $discounted_price;
                $course['has_discount'] = $has_discount;
                $course['discount_percentage'] = $has_discount ? $available_discounts['discount_percentage'] : 0;
                $course['savings'] = $has_discount ? ($original_price - $discounted_price) : 0;
                $course['thumbnail'] = $course['thumbnail'] ?: 'default-course.jpg';
            }
            
            // If course is free, redirect to regular enrollment
            if ($course['discounted_price'] == 0) {
                header("Location: course-enrollment.php?id=" . $course_id);
                exit();
            }
            
        } else {
            $error = "Course not found or not available.";
        }
    } else {
        $error = "Unable to process your request. Please try again.";
    }
} else {
    $error = "Invalid course ID provided.";
}

// If we have the course and user data, prepare PayFast payment
$paymentFormData = [];
if ($course && $user && !$error) {
    // If everything is good, prepare PayFast payment form data
    if (!$error && $course && $user) {
        // Generate PayFast form data using DISCOUNTED price
        $final_price = isset($course['discounted_price']) ? $course['discounted_price'] : $course['price'];
        $item = [
            'title' => $course['title'],
            'amount' => $final_price,
            'item_type' => 'course',
            'item_id' => $course['course_id']
        ];
        
        // Store payment info in session for return page
        $_SESSION['payment_amount'] = $final_price;
        $_SESSION['payment_course_title'] = $course['title'];
        // Store course ID in session for payment return page to use
        $_SESSION['payment_course_id'] = $course_id;
        $_SESSION['last_viewed_course'] = $course_id;
        
        // Generate return and cancel URLs with course ID using site config
        global $site_config;
        $returnUrl = $site_config['site_url'] . '/student/payment-return.php?course_id=' . $course_id;
        $cancelUrl = $site_config['site_url'] . '/student/course-payment.php?id=' . $course_id;
        $notifyUrl = $site_config['site_url'] . '/payments/notify.php';
        
        $paymentFormData = generatePayfastFormData($item, $user, $returnUrl, $cancelUrl, $notifyUrl);
    }
}

// Include header
include_once 'components/header.php';
?>

<style>
/* Clean Amber Course Payment Page Styling */
body {
    background: linear-gradient(135deg, #fffbeb 0%, #fef3c7 50%, #fde68a 100%);
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
    margin: 0;
    padding: 0;
}

.main-wrapper {
    padding: 40px 20px;
    max-width: 1400px;
    margin: 0 auto;
}

.payment-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0;
}

/* Beautiful Header with Floating Icons */
.payment-header {
    text-align: center;
    margin-bottom: 50px;
    position: relative;
    background: linear-gradient(135deg, #fef9e7 0%, #fef3c7 100%);
    border-radius: 25px;
    padding: 60px 40px;
    box-shadow: 0 10px 40px rgba(245, 158, 11, 0.1);
    border: 2px solid rgba(245, 158, 11, 0.1);
    overflow: hidden;
}

/* Floating Background Icons */
.payment-header::before,
.payment-header::after {
    content: '';
    position: absolute;
    width: 60px;
    height: 60px;
    background: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' fill='%23f59e0b' viewBox='0 0 24 24'%3E%3Cpath d='M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z'/%3E%3C/svg%3E") no-repeat center;
    background-size: contain;
    opacity: 0.05;
    animation: float 6s ease-in-out infinite;
}

.payment-header::before {
    top: 20px;
    left: 30px;
    animation-delay: 0s;
}

.payment-header::after {
    top: 30px;
    right: 40px;
    animation-delay: 3s;
    background: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' fill='%23f59e0b' viewBox='0 0 24 24'%3E%3Cpath d='M2 17h20v2H2zm1.15-4.05L4 11.47l.85 1.48c.2.34.57.55.98.55s.77-.21.98-.55L8.65 10l1.34 2.32c.2.34.57.55.98.55s.77-.21.98-.55L14.83 8l1.85 3.22c.2.34.57.55.98.55s.77-.21.98-.55L20.49 8l1.46 2.53c.2.34.57.55.98.55H24v2h-1.07l-2.35-4.08c-.2-.34-.57-.55-.98-.55s-.77.21-.98.55L16.27 11l-1.85-3.22c-.2-.34-.57-.55-.98-.55s-.77.21-.98.55L9.62 11l-1.34-2.32c-.2-.34-.57-.55-.98-.55s-.77.21-.98.55L3.17 11l-1.02-1.77z'/%3E%3C/svg%3E") no-repeat center;
}

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    50% { transform: translateY(-20px) rotate(180deg); }
}

.payment-header h1 {
    font-size: 2.5rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 15px;
    text-shadow: none;
    position: relative;
    z-index: 2;
}

.payment-header p {
    font-size: 1.1rem;
    color: #6b7280;
    font-weight: 400;
    position: relative;
    z-index: 2;
    max-width: 600px;
    margin: 0 auto;
}

/* Clean Course Preview Card */
.course-preview {
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 40px rgba(245, 158, 11, 0.08);
    overflow: hidden;
    margin-bottom: 30px;
    transition: all 0.3s ease;
    border: 1px solid rgba(245, 158, 11, 0.1);
    padding: 25px;
    margin: 0 15px 30px 15px;
}

.course-preview:hover {
    transform: translateY(-8px);
    box-shadow: 0 20px 60px rgba(245, 158, 11, 0.15);
    border-color: rgba(245, 158, 11, 0.2);
}

.course-thumbnail {
    width: 100%;
    height: 280px;
    background-size: cover;
    background-position: center;
    position: relative;
    overflow: hidden;
    border-radius: 15px;
    margin-bottom: 20px;
}

.course-thumbnail::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, rgba(245, 158, 11, 0.9) 0%, rgba(251, 191, 36, 0.9) 100%);
    opacity: 0;
    transition: all 0.3s ease;
}

.course-preview:hover .course-thumbnail::before {
    opacity: 1;
}

.course-content-wrapper {
    padding: 0;
}

.course-content-wrapper h4 {
    color: #1f2937;
    font-size: 1.75rem;
    font-weight: 700;
    margin-bottom: 25px;
    line-height: 1.4;
}

.course-meta {
    display: flex;
    align-items: center;
    gap: 1.5rem;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.95rem;
    font-weight: 500;
}

.meta-item i {
    color: #4e73df;
    font-size: 1.1rem;
}

.instructor-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 12px;
    margin-bottom: 1.5rem;
    border-left: 4px solid #4e73df;
}

.instructor-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 700;
    font-size: 1.2rem;
}

.instructor-details h6 {
    margin: 0;
    color: #2c3e50;
    font-weight: 600;
    font-size: 1rem;
}

.instructor-details .instructor-title {
    color: #64748b;
    font-size: 0.85rem;
    margin-top: 0.25rem;
}

.course-description {
    color: #4a5568;
    line-height: 1.8;
    margin-bottom: 1.5rem;
    font-size: 1rem;
}

/* Enhanced Payment Summary Card */
.price-breakdown {
    background: white;
    border-radius: 20px;
    padding: 35px;
    margin: 0 15px 30px 15px;
    box-shadow: 0 10px 40px rgba(245, 158, 11, 0.08);
    border: 1px solid rgba(245, 158, 11, 0.1);
    transition: all 0.3s ease;
}

.price-breakdown:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 60px rgba(245, 158, 11, 0.15);
    border-color: rgba(245, 158, 11, 0.2);
}

.price-breakdown h5 {
    color: #1f2937;
    font-size: 1.4rem;
    font-weight: 700;
    margin-bottom: 30px;
    display: flex;
    align-items: center;
}

.price-breakdown h5 i {
    margin-right: 12px;
    color: #f59e0b;
    font-size: 1.3rem;
}

.price-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px 0;
    border-bottom: 1px solid rgba(245, 158, 11, 0.1);
    font-size: 1rem;
}

.price-item:last-of-type:not(.price-total) {
    border-bottom: none;
    margin-bottom: 20px;
}

.price-total {
    display: flex;
    justify-content: space-between;
    font-weight: 800;
    font-size: 1.3rem;
    padding: 20px 0 0 0;
    border-top: 2px solid rgba(245, 158, 11, 0.2);
    margin-top: 20px;
    color: #1f2937;
}

.price-value {
    color: #4e73df;
}

/* Enhanced Security Info Card */
.security-info {
    background: linear-gradient(135deg, #fef9e7 0%, #fef3c7 100%);
    border: 1px solid rgba(245, 158, 11, 0.2);
    border-radius: 18px;
    padding: 30px;
    margin: 0 15px 30px 15px;
    position: relative;
    box-shadow: 0 5px 20px rgba(245, 158, 11, 0.05);
}

.security-info::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, #f59e0b 0%, #d97706 100%);
    border-radius: 4px 4px 0 0;
}

.security-info h6 {
    color: #1f2937;
    font-weight: 700;
    font-size: 1.2rem;
    margin-bottom: 15px;
    display: flex;
    align-items: center;
}

.security-info h6 i {
    margin-right: 10px;
    color: #f59e0b;
    font-size: 1.3rem;
}

.security-info p {
    color: #4b5563;
    font-size: 1rem;
    margin: 0;
    line-height: 1.7;
    line-height: 1.6;
}

/* Enhanced PayFast Form Card */
.payfast-form {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
}

.payfast-form:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 60px rgba(245, 158, 11, 0.15);
    border-color: rgba(245, 158, 11, 0.2);
}

.payfast-form h5 {
    color: #1f2937;
    font-size: 1.4rem;
    font-weight: 700;
    margin-bottom: 30px;
    display: flex;
    align-items: center;
}

.payfast-form h5 i {
    margin-right: 12px;
    color: #f59e0b;
    font-size: 1.3rem;
}

.payment-button {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    border: none;
    color: white;
    padding: 20px 40px;
    border-radius: 15px;
    font-size: 1.2rem;
    font-weight: 700;
    width: 100%;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 8px 30px rgba(245, 158, 11, 0.3);
    position: relative;
    overflow: hidden;
}

.payment-button:hover {
    transform: translateY(-3px);
    box-shadow: 0 15px 40px rgba(245, 158, 11, 0.4);
    background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
}

.payment-button:active {
    transform: translateY(-1px);
}

.payment-button::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.5s;
}

.payment-button:hover::before {
    left: 100%;
}

/* Course info tags */
.course-info-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 10px;
    margin-top: 20px;
}

.course-tag {
    display: inline-flex;
    align-items: center;
    background: #f0f4ff;
    border-radius: 50px;
    padding: 6px 12px;
    font-size: 0.85rem;
    color: #4e73df;
    font-weight: 500;
}

.course-tag i {
    margin-right: 5px;
    font-size: 1.1rem;
}

/* Payment form container */
.payment-form-container {
    position: relative;
}

/* Add payment logo */
.payment-security-badge {
    position: absolute;
    top: -15px;
    right: 0;
    background: white;
    border-radius: 50%;
    width: 60px;
    height: 60px;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 5px 15px rgba(0,0,0,0.08);
}

.alert {
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.05);
    border: none;
    padding: 15px 20px;
}

/* Responsive adjustments */
@media (max-width: 767px) {
    .main-wrapper {
        padding: 20px 0;
    }
    
    .course-preview,
    .price-breakdown,
    .security-info,
    .payfast-form {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .course-thumbnail {
        height: 200px;
    }
}
</style>

<!-- Modern Payment Page Container -->
<div class="main-wrapper">
    <div class="payment-container">
        <!-- Modern Header -->
        <div class="payment-header">
            <h1><i class="bx bx-credit-card me-3"></i>Course Payment</h1>
            <p>Secure and easy payment for your learning journey</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error-circle me-2"></i>
                <?= htmlspecialchars($error) ?>
            </div>
            <div class="text-center mt-3">
                <a href="courses.php" class="btn btn-outline-light btn-lg">
                    <i class="bx bx-arrow-back me-1"></i> Back to Courses
                </a>
            </div>
        <?php elseif ($course && $user): ?>
            <div class="row">
                <div class="col-md-7">
                    <!-- Enhanced Course Preview -->
                    <div class="course-preview">
                        <div class="course-thumbnail" style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '../assets/img/courses/default.jpg') ?>')"></div>
                        
                        <div class="course-content-wrapper">
                            <h4><?= htmlspecialchars($course['title']) ?></h4>
                            
                            <!-- Enhanced Instructor Info -->
                            <div class="instructor-info">
                                <div class="instructor-avatar">
                                    <?= strtoupper(substr($course['teacher_first_name'], 0, 1) . substr($course['teacher_last_name'], 0, 1)) ?>
                                </div>
                                <div class="instructor-details">
                                    <h6><?= htmlspecialchars($course['teacher_first_name'] . ' ' . $course['teacher_last_name']) ?></h6>
                                    <div class="instructor-title">Course Instructor</div>
                                </div>
                            </div>
                            
                            <!-- Course Meta Information -->
                            <div class="course-meta">
                                <div class="meta-item">
                                    <i class="bx bx-category"></i>
                                    <span><?= htmlspecialchars($course['category_name']) ?></span>
                                </div>
                                <div class="meta-item">
                                    <i class="bx bx-time"></i>
                                    <span>Lifetime Access</span>
                                </div>
                                <div class="meta-item">
                                    <i class="bx bx-laptop"></i>
                                    <span>Online Learning</span>
                                </div>
                            </div>
                            
                            <!-- Course Description -->
                            <div class="course-description">
                                <?= htmlspecialchars(substr($course['description'], 0, 300)) ?>...
                            </div>
                            
                            <!-- Course Features -->
                            <div class="course-info-tags">
                                <span class="course-tag"><i class='bx bx-video'></i> HD Video Content</span>
                                <span class="course-tag"><i class='bx bx-download'></i> Downloadable Resources</span>
                                <span class="course-tag"><i class='bx bx-badge-check'></i> Certificate of Completion</span>
                                <span class="course-tag"><i class='bx bx-support'></i> Instructor Support</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-5">
                    <!-- Enhanced Price Breakdown -->
                    <div class="price-breakdown">
                        <h5><i class="bx bx-receipt"></i>Payment Summary</h5>
                        
                        <?php if ($course['has_discount']): ?>
                            <div class="price-item">
                                <span><i class="bx bx-purchase-tag me-2"></i>Course Price:</span>
                                <span class="price-value text-decoration-line-through text-muted">R<?= number_format($course['price'], 2) ?></span>
                            </div>
                            
                            <div class="price-item">
                                <span><i class="bx bx-gift me-2"></i>Discounted Price:</span>
                                <span class="price-value text-success fw-bold">R<?= number_format($course['discounted_price'], 2) ?></span>
                            </div>
                            
                            <div class="price-item">
                                <span><i class="bx bx-badge me-2"></i>You Save:</span>
                                <span class="text-success fw-bold">R<?= number_format($course['savings'], 2) ?> (<?= number_format($course['discount_percentage'], 1) ?>% OFF)</span>
                            </div>
                        <?php else: ?>
                            <div class="price-item">
                                <span><i class="bx bx-purchase-tag me-2"></i>Course Price:</span>
                                <span class="price-value">R<?= number_format($course['price'], 2) ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <div class="price-item">
                            <span><i class="bx bx-credit-card me-2"></i>Processing Fee:</span>
                            <span class="text-success fw-bold">FREE</span>
                        </div>
                        
                        <div class="price-total">
                            <span><i class="bx bx-money me-2"></i>Total Amount:</span>
                            <span class="price-value">R<?= number_format($course['discounted_price'], 2) ?></span>
                        </div>
                    </div>
                    
                    <!-- Enhanced Security Info -->
                    <div class="security-info">
                        <h6><i class="bx bx-shield-check"></i>Secure Payment</h6>
                        <p>Your payment is processed securely through PayFast, South Africa's leading payment gateway. Your personal and financial information is protected with bank-level security.</p>
                    </div>
                    
                    <!-- Enhanced PayFast Payment Form -->
                    <div class="payfast-form">
                        <h5><i class="bx bx-credit-card"></i>Complete Your Purchase</h5>
                        
                        <?php if (!empty($paymentFormData)): ?>
                            <form action="<?= $paymentFormData['url'] ?>" method="post" id="payfast-form" class="payment-form-container">
                                <?php foreach ($paymentFormData['data'] as $key => $value): ?>
                                    <input type="hidden" name="<?= htmlspecialchars($key) ?>" value="<?= htmlspecialchars($value) ?>">
                                <?php endforeach; ?>
                                
                                <div class="mb-3 mt-4">
                                    <p class="small text-muted">
                                        By clicking "Pay Now", you agree to our Terms of Service and confirm that you want to purchase this course.
                                    </p>
                                </div>
                                
                                <div class="position-relative">
                                    <button type="submit" class="payment-button">
                                        <i class="bx bx-credit-card me-2"></i>
                                        Pay Now - R<?= number_format($course['discounted_price'], 2) ?>
                                    </button>
                                    <div class="payment-security-badge">
                                        <img src="../assets/img/payfast-logo.png" alt="PayFast" width="40" onerror="this.onerror=null; this.src='https://www.payfast.co.za/assets/images/logos/payfast-badge.svg';">
                                    </div>
                                </div>
                            </form>
                            
                            <div class="text-center mt-3">
                                <a href="courses.php" class="back-link">
                                    <i class="bx bx-arrow-back"></i> Back to Courses
                                </a>
                            </div>
                        <?php else: ?>
                            <div class="alert alert-warning">
                                <i class="bx bx-warning me-2"></i>
                                Payment system is currently unavailable. Please try again later.
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Store course information in localStorage before submitting payment form
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('payfast-form');
    if (form) {
        // Store course info in localStorage for potential recovery
        localStorage.setItem('last_payment_course_id', '<?= $course_id ?>');
        localStorage.setItem('last_payment_course_title', '<?= addslashes($course['title']) ?>');
        localStorage.setItem('last_payment_amount', '<?= $course['price'] ?>');
        
        // Add loading state to button when clicked
        form.addEventListener('submit', function() {
            const button = form.querySelector('button[type="submit"]');
            button.innerHTML = '<i class="bx bx-loader-alt bx-spin me-2"></i>Processing...';
            button.disabled = true;
        });
    }
});
</script>

<?php include_once 'components/footer.php'; ?>
