<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Decrypt and validate course ID
$course_id = getSecureId('id');
if (!$course_id) {
    $_SESSION['error'] = "Invalid or corrupted course ID";
    header('Location: courses.php');
    exit();
}

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

// Get course details
$courseQuery = "
    SELECT c.*, cat.category_name, 
           u.first_name, u.last_name,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as enrolled_students
    FROM courses c
    JOIN categories cat ON c.category_id = cat.category_id
    JOIN users u ON c.teacher_id = u.user_id
    WHERE c.course_id = ?
";

$courseStmt = $conn->prepare($courseQuery);
$courseStmt->bind_param("i", $course_id);
$courseStmt->execute();
$courseResult = $courseStmt->get_result();

if ($courseResult->num_rows == 0) {
    $_SESSION['error'] = "Course not found";
    header('Location: courses.php');
    exit();
}

$course = $courseResult->fetch_assoc();
$page_title = $course['title'] . " - Quizzes";

// Include header
include __DIR__ . '/components/header.php';
?>

<!-- Add direct CSS inclusion for quizzes page -->
<link rel="stylesheet" href="assets/css/pages/course-quizzes.css?v=<?= time(); ?>">

<div class="page-content">
    <div class="container-fluid pt-4">
        <div class="row">
            <div class="col-12">
                <div class="course-banner">
                    <div class="course-banner-content">
                        <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="course-title"><?= htmlspecialchars($course['title']) ?></h2>
                            <p class="course-instructor">
                                <i class='bx bx-user'></i> Instructor: <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>
                            </p>
                            <div class="mt-2">
                                <span class="course-category"><?= htmlspecialchars($course['category_name'] ?? 'General') ?></span>
                                <?php if (!empty($course['level'])): ?>
                                <span class="course-category"><?= htmlspecialchars(ucfirst($course['level'])) ?> Level</span>
                                <?php endif; ?>
                                <?php if (!empty($course['enrolled_students'])): ?>
                                <span class="course-category"><i class='bx bx-group'></i> <?= $course['enrolled_students'] ?> students</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if ($course && $is_enrolled) { ?>
<!-- Course Navigation Menu -->
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="course-nav">
                <!-- Course content tabs -->
                <div class="course-tabs">
                    <a href="course-detail.php?id=<?= $course_id ?>" class="course-tab">
                        <i class='bx bx-book-open'></i> Course Content
                    </a>
                    <a href="course-materials.php?id=<?= $course_id ?>" class="course-tab">
                        <i class='bx bx-file'></i> Materials
                    </a>
                    <a href="course-assignments.php?id=<?= $course_id ?>" class="course-tab">
                        <i class='bx bx-task'></i> Assignments
                    </a>
                    <a href="course-quizzes.php?id=<?= $course_id ?>" class="course-tab active">
                        <i class='bx bx-question-mark'></i> Quizzes
                    </a>
                    <a href="course-discussions.php?id=<?= $course_id ?>" class="course-tab">
                        <i class='bx bx-chat'></i> Discussions
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Quizzes Content -->
<div class="container-fluid mb-5">
    <div class="row">
        <div class="col-12">
            <div class="content-card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Course Quizzes</h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="quiz-filters mb-4">
                        <button type="button" class="quiz-filter-btn active" data-filter="all">All Quizzes</button>
                        <button type="button" class="quiz-filter-btn" data-filter="available">Available</button>
                        <button type="button" class="quiz-filter-btn" data-filter="completed">Completed</button>
                        <button type="button" class="quiz-filter-btn" data-filter="upcoming">Upcoming</button>
                    </div>
                    
                    <?php
                    // Check if quizzes table exists
                    $quizzesTableQuery = "SHOW TABLES LIKE 'quizzes'";
                    $quizzesTableResult = $conn->query($quizzesTableQuery);
                    $quizzesTableExists = $quizzesTableResult->num_rows > 0;
                    
                    // Check if quiz_attempts table exists
                    $attemptsTableQuery = "SHOW TABLES LIKE 'quiz_attempts'";
                    $attemptsTableResult = $conn->query($attemptsTableQuery);
                    $attemptsTableExists = $attemptsTableResult->num_rows > 0;
                    
                    if ($quizzesTableExists) {
                        // Check if status column exists in quizzes table
                        $statusColumnQuery = "SHOW COLUMNS FROM quizzes LIKE 'status'";
                        $statusColumnResult = $conn->query($statusColumnQuery);
                        $statusColumnExists = $statusColumnResult->num_rows > 0;
                        
                        // Check if start_time column exists in quizzes table
                        $startTimeColumnQuery = "SHOW COLUMNS FROM quizzes LIKE 'start_time'";
                        $startTimeColumnResult = $conn->query($startTimeColumnQuery);
                        $startTimeColumnExists = $startTimeColumnResult->num_rows > 0;
                        
                        // Get course quizzes with attempt status
                        if ($attemptsTableExists) {
                            // Simple, clean query to get all quiz data
                            $quizzesQuery = "
                                SELECT 
                                    q.quiz_id,
                                    q.title,
                                    q.description,
                                    q.time_limit,
                                    q.max_attempts,
                                    q.passing_score,
                                    q.is_published,
                                    q.created_at,
                                    COALESCE(MAX(qa.score), 0) as best_score,
                                    COUNT(DISTINCT qa.attempt_id) as attempt_count,
                                    COUNT(DISTINCT qq.question_id) as question_count,
                                    COALESCE(SUM(DISTINCT qq.points), 0) as total_points
                                FROM quizzes q
                                LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                                LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
                                WHERE q.course_id = ? AND q.is_published = 1
                                GROUP BY q.quiz_id, q.title, q.description, q.time_limit, q.max_attempts, q.passing_score, q.is_published, q.created_at
                                ORDER BY q.created_at ASC
                            ";
                        } else {
                            // If quiz_attempts table doesn't exist, just query quizzes
                            // Build WHERE clause and ORDER BY based on column existence
                            $whereClause = "WHERE q.course_id = ?";
                            if ($statusColumnExists) {
                                $whereClause .= " AND q.status = 'active'";
                            }
                            
                            $orderByClause = "ORDER BY " . ($startTimeColumnExists ? "q.start_time" : "q.quiz_id") . " ASC";
                            
                            $quizzesQuery = "
                                SELECT q.*, 
                                       NULL as best_score,
                                       0 as attempt_count,
                                       NULL as last_attempt_date
                                FROM quizzes q
                                {$whereClause}
                                {$orderByClause}
                            ";
                        }
                        
                        $quizStmt = $conn->prepare($quizzesQuery);
                    if ($quizStmt) {
                        if ($attemptsTableExists) {
                            $quizStmt->bind_param("ii", $student_id, $course_id);
                        } else {
                            $quizStmt->bind_param("i", $course_id);
                        }
                        $quizStmt->execute();
                        $quizzesResult = $quizStmt->get_result();
                        
                        if ($quizzesResult->num_rows > 0) {
                            // Display quizzes in table
                            echo '<div class="table-responsive">';
                            echo '<table class="quiz-table table">';
                            echo '<thead>';
                            echo '<tr>';
                            echo '<th style="width: 25%">Quiz</th>';
                            echo '<th style="width: 15%">Available</th>';
                            echo '<th style="width: 10%">Time Limit</th>';
                            echo '<th style="width: 8%">Questions</th>';
                            echo '<th style="width: 8%">Points</th>';
                            echo '<th style="width: 10%">Attempts</th>';
                            echo '<th style="width: 10%">Best Score</th>';
                            echo '<th style="width: 14%">Action</th>';
                            echo '</tr>';
                            echo '</thead>';
                            echo '<tbody>';
                            
                            // Current time for comparison
                            $now = new DateTime();
                            
                            while ($quiz = $quizzesResult->fetch_assoc()) {
                                // Determine quiz status based on attempts
                                $quizStatus = 'available';
                                if ($quiz['attempt_count'] > 0) {
                                    $quizStatus = 'completed';
                                }
                                
                                // Calculate percentage score safely
                                $percentageScore = '';
                                if ($quiz['best_score'] > 0 && $quiz['total_points'] > 0) {
                                    $percentageScore = round(($quiz['best_score'] / $quiz['total_points']) * 100) . '%';
                                }
                                
                                // Time limit display
                                $timeLimit = $quiz['time_limit'] ? $quiz['time_limit'] . ' minutes' : 'No limit';
                                
                                // Question count - ensure it shows correct number
                                $questionCount = ($quiz['question_count'] > 0) ? $quiz['question_count'] : '0';
                                
                                // Format created date for display
                                $createdDate = date('M d, Y', strtotime($quiz['created_at']));
                                
                                // Display quiz row
                                echo "<tr class='quiz-item' data-filter='{$quizStatus}'>";
                                echo "<td>";
                                echo "<div class='quiz-title'>" . htmlspecialchars($quiz['title']) . "</div>";
                                echo "<span class='quiz-status {$quizStatus}'>" . ucfirst($quizStatus) . "</span>";
                                if (!empty($quiz['description'])) {
                                    echo "<div class='quiz-description small text-muted mt-1'>" . substr(htmlspecialchars($quiz['description']), 0, 100) . (strlen($quiz['description']) > 100 ? '...' : '') . "</div>";
                                }
                                
                                echo "</td>";
                                echo "<td>";
                                echo "<small>Created: {$createdDate}</small>";
                                echo "</td>";
                                echo "<td>{$timeLimit}</td>";
                                echo "<td>{$questionCount}</td>";
                                echo "<td>" . (isset($quiz['total_points']) ? $quiz['total_points'] : '0') . "</td>";
                                
                                // Safe display of attempts with null checks
                                $attemptCount = isset($quiz['attempt_count']) ? $quiz['attempt_count'] : '0';
                                $maxAttempts = isset($quiz['max_attempts']) ? $quiz['max_attempts'] : '∞';
                                echo "<td>{$attemptCount} / {$maxAttempts}</td>";
                                echo "<td>" . ($percentageScore ? $percentageScore : '-') . "</td>";
                                
                                echo "<td>";
                                // Action button based on status
                                $maxAttempts = isset($quiz['max_attempts']) ? $quiz['max_attempts'] : PHP_INT_MAX;
                                $attemptCount = isset($quiz['attempt_count']) ? $quiz['attempt_count'] : 0;
                                
                                if ($quizStatus === 'available' && ($attemptCount < $maxAttempts)) {
                                    echo "<a href='take_quiz.php?id={$quiz['quiz_id']}' class='quiz-action-btn primary'>
                                            <i class='bx bx-play-circle'></i> Start Quiz
                                          </a>";
                                } else if ($quizStatus === 'completed') {     
                                    echo "<a href='quiz_results.php?quiz_id={$quiz['quiz_id']}' class='quiz-action-btn info'>
                                           <i class='bx bx-bar-chart'></i> View Results
                                         </a>";
                                } else {
                                    echo "<span class='quiz-action-btn disabled'>
                                           <i class='bx bx-lock'></i> Unavailable
                                         </span>";
                                }
                                echo "</td>";
                                echo "</tr>";
                                
                                // Optional: Description row if it exists
                                if (!empty($quiz['description'])) {
                                    echo "<tr class='quiz-description-row' data-filter='{$quizStatus}'>";
                                    echo "<td colspan='8' class='text-muted small'>" . htmlspecialchars($quiz['description']) . "</td>";
                                    echo "</tr>";
                                }
                            }
                            
                            echo '</tbody>';
                            echo '</table>';
                            echo '</div>';
                        } else {
                            echo '<div class="quiz-empty">';
                            echo '<i class="bx bx-question-mark-circle"></i>';
                            echo '<h5 class="mt-3">No Quizzes Available</h5>';
                            echo '<p>No quizzes have been added to this course yet.</p>';
                            echo '</div>';
                        }
                    }
                    ?>
                    
                    <script>
                    // Filter quizzes by status
                    document.addEventListener('DOMContentLoaded', function() {
                        // Filter quizzes by status
                        document.querySelectorAll('.quiz-filter-btn').forEach(button => {
                            button.addEventListener('click', () => {
                                // Set active class
                                document.querySelectorAll('.quiz-filter-btn').forEach(btn => btn.classList.remove('active'));
                                button.classList.add('active');
                                
                                // Get filter value
                                const filterValue = button.getAttribute('data-filter');
                                
                                // Track if we have items for this filter
                                let visibleItems = 0;
                                
                                // Show/hide quizzes based on filter
                                document.querySelectorAll('.quiz-item').forEach(item => {
                                    if (filterValue === 'all' || item.getAttribute('data-filter') === filterValue) {
                                        item.style.display = '';
                                        visibleItems++;
                                    } else {
                                        item.style.display = 'none';
                                    }
                                });
                                
                                // Show empty state if no items for this filter
                                const tableContainer = document.querySelector('.table-responsive');
                                const emptyState = document.getElementById('quizzes-empty-state');
                                
                                if (visibleItems === 0 && filterValue !== 'all') {
                                    // Hide table, show empty state
                                    if (tableContainer) tableContainer.style.display = 'none';
                                    if (emptyState) {
                                        emptyState.style.display = 'block';
                                        const message = emptyState.querySelector('p');
                                        if (message) {
                                            message.textContent = `No ${filterValue} quizzes found in this course.`;
                                        }
                                    }
                                } else {
                                    // Show table, hide empty state
                                    if (tableContainer) tableContainer.style.display = 'block';
                                    if (emptyState) emptyState.style.display = 'none';
                                }
                            });
                        });
                    });
                    </script>
                    
                    <!-- Empty state for filtered quizzes -->
                    <div id="quizzes-empty-state" class="quiz-empty" style="display:none;">
                        <i class="bx bx-filter"></i>
                        <h5 class="mt-3">No Matching Quizzes</h5>
                        <p>No quizzes match the selected filter.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php 
    } // Close if ($quizzesTableExists)
    else {
        // Content for non-enrolled users
        ?>
        <div class="container-fluid mb-5">
            <div class="row">
                <div class="col-12">
                    <div class="card shadow-sm">
                        <div class="card-body p-4">
                            <div class="alert alert-warning">
                                <h5><i class="bx bx-lock me-2"></i> Course quizzes are only available to enrolled students</h5>
                                <p>To access this content, please enroll in this course.</p>
                                <a href="course-enrollment.php?id=<?= $course_id ?>" class="btn btn-primary mt-2">Enroll Now</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
    } // Close else
} // Close if ($course && $is_enrolled)

?>

<script>
function openDebugWindow(url) {
    window.open(url, 'debugWindow', 'width=800,height=600');
}
</script>
