<?php
/**
 * Student Courses Page - Clean Modern Implementation
 * Panadite Academy
 * 
 * Features:
 * - Enrolled courses with progress tracking
 * - Available courses with payment integration
 * - Modern responsive design
 * - Course filtering and statistics
 * - Clean error handling
 */

// Authentication and session
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');

// Initialize variables
$enrolledCourses = [];
$availableCourses = [];
$coursesInProgress = 0;
$completedCourses = 0;
$totalCourseHours = 0;

// FIXED: Use robust session handling like certificates.php (not simple fallback to 0)
$userId = null;

// Try multiple session key variations (same as certificates.php)
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
} elseif (isset($_SESSION['id'])) {
    $userId = $_SESSION['id'];
}

// Fallback: If session detection fails, get user by email (most reliable)
if (!$userId && isset($_SESSION['email'])) {
    // Will set up email lookup after database connection
}

// Database operations with comprehensive error handling
try {
    require_once '../config/db_connect.php';
    
    // Complete the email fallback (same as certificates.php)
    if (!$userId && isset($_SESSION['email']) && $conn) {
        $email_stmt = $conn->prepare("SELECT user_id, role FROM users WHERE email = ?");
        if ($email_stmt) {
            $email_stmt->bind_param("s", $_SESSION['email']);
            $email_stmt->execute();
            $email_result = $email_stmt->get_result();
            if ($email_result && $row = $email_result->fetch_assoc()) {
                $userId = $row['user_id'];
            }
            $email_stmt->close();
        }
    }
    
    if (isset($conn) && $userId) {
        // Fetch enrolled courses with FIXED progress calculation (same as certificate logic)
        $enrolledQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, 
                   u.first_name, u.last_name, e.enrolled_date, 
                   'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   (SELECT COUNT(*) FROM videos v WHERE v.course_id = c.course_id AND v.is_published = 1) as total_videos,
                   COALESCE(c.duration, (SELECT COUNT(*) FROM videos v WHERE v.course_id = c.course_id AND v.is_published = 1)) as course_duration
            FROM enrollments e
            JOIN courses c ON e.course_id = c.course_id
            JOIN users u ON c.teacher_id = u.user_id
            WHERE e.student_id = ?
            ORDER BY e.enrolled_date DESC
        ";
        
        $stmt = $conn->prepare($enrolledQuery);
        if (!$stmt) {
            error_log("Enrolled courses SQL prepare failed: " . $conn->error);
            error_log("Query was: " . $enrolledQuery);
            throw new Exception("Enrolled courses database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("Enrolled courses SQL execute failed: " . $stmt->error);
            throw new Exception("Enrolled courses database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $courseId = $row['id'];
            
            // Calculate REAL progress using FIXED logic (same as certificates and dashboard)
            
            // 1. Video completion (FIXED - uses user_video_completions)
            $videoQuery = $conn->prepare("
                SELECT 
                    COUNT(DISTINCT v.video_id) as total_videos,
                    COUNT(DISTINCT uvc.video_id) as completed_videos
                FROM videos v
                LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                    AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                WHERE v.course_id = ? AND v.is_published = 1
            ");
            
            $total_videos = 0;
            $completed_videos = 0;
            
            if ($videoQuery) {
                $videoQuery->bind_param("ii", $userId, $courseId);
                $videoQuery->execute();
                $videoResult = $videoQuery->get_result();
                if ($videoRow = $videoResult->fetch_assoc()) {
                    $total_videos = (int)$videoRow['total_videos'];
                    $completed_videos = (int)$videoRow['completed_videos'];
                }
                $videoQuery->close();
            }
            
            // 2. Quiz performance 
            $quizQuery = $conn->prepare("
                SELECT 
                    COUNT(DISTINCT q.quiz_id) as total_quizzes,
                    COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes
                FROM quizzes q
                LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                WHERE q.course_id = ? AND q.is_published = 1
            ");
            
            $total_quizzes = 0;
            $passed_quizzes = 0;
            
            if ($quizQuery) {
                $quizQuery->bind_param("ii", $userId, $courseId);
                $quizQuery->execute();
                $quizResult = $quizQuery->get_result();
                if ($quizRow = $quizResult->fetch_assoc()) {
                    $total_quizzes = (int)$quizRow['total_quizzes'];
                    $passed_quizzes = (int)$quizRow['passed_quizzes'];
                }
                $quizQuery->close();
            }
            
            // 🚨 CRITICAL PROFESSIONAL FIX: Use SAME logic as fixed course-detail.php
            // PREVENT courses with NO CONTENT from showing 100% completion!
            
            $has_any_content = ($total_videos > 0 || $total_quizzes > 0);
            
            if (!$has_any_content) {
                // 🚫 NO CONTENT = 0% progress (NOT 100%!)
                $video_progress = 0;
                $quiz_progress = 0;
                $overall_course_progress = 0;
                $video_completion_percentage = 0;
                $videos_completed = false;
                $certificate_eligible = false;
                $status = 'in_progress'; // Empty courses are NOT completed
                error_log("COURSES.PHP PROGRESS: Course ID $courseId has NO CONTENT - Progress set to 0%");
            } else {
                // Course has content - calculate normally
                $video_progress = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                $quiz_progress = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 0;
                
                // Only weight if both exist, otherwise use the available one
                if ($total_videos > 0 && $total_quizzes > 0) {
                    $overall_course_progress = round(($video_progress * 0.6) + ($quiz_progress * 0.4));
                } elseif ($total_videos > 0) {
                    $overall_course_progress = round($video_progress);
                } elseif ($total_quizzes > 0) {
                    $overall_course_progress = round($quiz_progress);
                } else {
                    $overall_course_progress = 0;
                }
                
                // Certificate eligibility calculations  
                $video_completion_percentage = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                $videos_completed = ($total_videos == 0 || $video_completion_percentage >= 100);
                
                // Get course passing grade
                $passingGradeQuery = $conn->prepare("SELECT course_passing_grade FROM courses WHERE course_id = ?");
                $course_passing_grade = 60; // Default
                if ($passingGradeQuery) {
                    $passingGradeQuery->bind_param("i", $courseId);
                    $passingGradeQuery->execute();
                    $passingResult = $passingGradeQuery->get_result();
                    if ($passingRow = $passingResult->fetch_assoc()) {
                        $course_passing_grade = (float)$passingRow['course_passing_grade'] ?: 60;
                    }
                    $passingGradeQuery->close();
                }
                
                // 🚨 CRITICAL FIX: Quizzes must exist AND be passed for completion
                if ($total_quizzes == 0) {
                    $quizzes_passed = false; // NO quizzes = NOT complete
                } else {
                    $quiz_pass_rate = ($passed_quizzes / $total_quizzes) * 100;
                    $quizzes_passed = ($quiz_pass_rate >= $course_passing_grade);
                }
                
                // Course is "completed" only if it has content AND meets requirements
                $certificate_eligible = ($has_any_content && $videos_completed && $quizzes_passed);
                $status = $certificate_eligible ? 'completed' : 'in_progress';
            }
            
            $enrolledCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                'progress' => $overall_course_progress,
                'total_modules' => $total_videos, // Real video count
                'completed_modules' => $completed_videos, // Real completed videos
                'start_date' => $row['enrolled_date'],
                'category' => $row['category'],
                'status' => $status,
                'students' => $row['total_students'],
                'duration' => $row['course_duration']
            ];
            
            if ($status === 'in_progress') {
                $coursesInProgress++;
            } else {
                $completedCourses++;
            }
        }
        
        // Calculate real total course hours from actual course durations
        $totalCourseHours = 0;
        foreach ($enrolledCourses as $course) {
            $totalCourseHours += $course['duration'] ?? 0; // Use real duration from database
        }
        
        // Calculate real overall progress percentage for completion rate (same logic as dashboard)
        $overallProgress = 0;
        $totalEnrolledCount = count($enrolledCourses);
        if ($totalEnrolledCount > 0) {
            $progressQuery = $conn->prepare("
                SELECT 
                    AVG(COALESCE(
                        (SELECT 
                            CASE 
                                WHEN COUNT(v.video_id) = 0 THEN 0
                                ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                            END
                        FROM videos v
                        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                        WHERE v.course_id = e.course_id AND v.is_published = 1), 0
                    )) as avg_progress
                FROM enrollments e
                WHERE e.student_id = ?
            ");
            
            if ($progressQuery) {
                $progressQuery->bind_param("i", $userId);
                if ($progressQuery->execute()) {
                    $result = $progressQuery->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $overallProgress = round($row['avg_progress'] ?? 0);
                    }
                }
                $progressQuery->close();
            }
        }
        
        // Also get real completed courses count (courses with 100% video completion)
        $realCompletedCourses = 0;
        if ($totalEnrolledCount > 0) {
            $completedQuery = $conn->prepare("
                SELECT COUNT(*) as count 
                FROM enrollments e
                WHERE e.student_id = ? 
                AND (
                    SELECT 
                        CASE 
                            WHEN COUNT(v.video_id) = 0 THEN 0
                            ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                        END
                    FROM videos v
                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                    WHERE v.course_id = e.course_id AND v.is_published = 1
                ) >= 100
            ");
            
            if ($completedQuery) {
                $completedQuery->bind_param("i", $userId);
                if ($completedQuery->execute()) {
                    $result = $completedQuery->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $realCompletedCourses = $row['count'];
                    }
                }
                $completedQuery->close();
            }
        }
        
        // Fetch NEW courses (uploaded in past 7 days, not enrolled) 🆕
        $availableQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, c.price,
                   u.first_name, u.last_name, 'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   COALESCE(4.5, 4.5) as rating,
                   '10 hours' as duration, 
                   'Beginner' as level,
                   c.created_at
            FROM courses c
            JOIN users u ON c.teacher_id = u.user_id
            WHERE c.is_published = 1
            AND c.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            AND c.course_id NOT IN (
                SELECT course_id FROM enrollments WHERE student_id = ? AND course_id IS NOT NULL
            )
            ORDER BY c.created_at DESC
        ";
        
        $stmt = $conn->prepare($availableQuery);
        if (!$stmt) {
            error_log("SQL prepare failed: " . $conn->error);
            error_log("Query was: " . $availableQuery);
            throw new Exception("Database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("SQL execute failed: " . $stmt->error);
            throw new Exception("Database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        // FETCH USER'S AVAILABLE DISCOUNTS
        $available_discounts = [];
        $discountsQuery = "SELECT * FROM student_referral_discounts 
                          WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                          ORDER BY discount_percentage DESC LIMIT 1";
        $discountStmt = $conn->prepare($discountsQuery);
        if ($discountStmt) {
            $discountStmt->bind_param("i", $userId);
            $discountStmt->execute();
            $discountResult = $discountStmt->get_result();
            if ($discountResult->num_rows > 0) {
                $available_discounts = $discountResult->fetch_assoc();
            }
        }
        
        while ($row = $result->fetch_assoc()) {
            $original_price = $row['price'];
            $discounted_price = $original_price;
            $has_discount = false;
            
            // APPLY DISCOUNT IF AVAILABLE
            if (!empty($available_discounts) && $original_price > 0) {
                $discount_amount = min(
                    ($original_price * $available_discounts['discount_percentage'] / 100),
                    $available_discounts['discount_amount']
                );
                $discounted_price = $original_price - $discount_amount;
                $has_discount = true;
            }
            
            $availableCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                'category' => $row['category'],
                'students' => $row['total_students'],
                'rating' => $row['rating'],
                'duration' => $row['duration'],
                'level' => $row['level'],
                'price' => $original_price,
                'discounted_price' => $discounted_price,
                'has_discount' => $has_discount,
                'discount_percentage' => $has_discount ? $available_discounts['discount_percentage'] : 0,
                'savings' => $has_discount ? ($original_price - $discounted_price) : 0
            ];
        }
        
        // 🎲 RANDOM SHUFFLE: Display new courses in random order each page load
        if (!empty($availableCourses)) {
            shuffle($availableCourses);
        }
    }
} catch (Exception $e) {
    error_log("Courses page error: " . $e->getMessage());
}

// Calculate statistics using REAL progress data (same logic as dashboard)
$totalEnrolled = count($enrolledCourses);

// Calculate REAL completed courses using certificate eligibility (same as dashboard)
$realCompletedCourses = 0;
if ($totalEnrolled > 0) {
    foreach ($enrolledCourses as $course) {
        if ($course['status'] === 'completed') {
            $realCompletedCourses++;
        }
    }
}

// Calculate REAL completion rate
$completionRate = $totalEnrolled > 0 ? round(($realCompletedCourses / $totalEnrolled) * 100) : 0;
$completedCourses = $realCompletedCourses;

// Handle session notifications
$enrollmentSuccess = $_SESSION['enrollment_success'] ?? null;
$enrollmentError = $_SESSION['enrollment_error'] ?? null;
unset($_SESSION['enrollment_success'], $_SESSION['enrollment_error']);

include_once 'components/header.php';
?>

<!-- Enrollment Notifications -->
<?php if ($enrollmentSuccess): ?>
<div class="container-fluid pt-3">
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class='bx bx-check-circle me-2'></i>
        <?= htmlspecialchars($enrollmentSuccess) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
</div>
<?php endif; ?>

<?php if ($enrollmentError): ?>
<div class="container-fluid pt-3">
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class='bx bx-error-circle me-2'></i>
        <?= htmlspecialchars($enrollmentError) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
</div>
<?php endif; ?>

<!-- Learning Dashboard Header - Outside Container -->
<div class="dashboard-header">
    <!-- Lots of floating decorative icons around header -->
    <div class="tiny-float-icon tiny-float-1"><i class='bx bx-star'></i></div>
    <div class="tiny-float-icon tiny-float-2"><i class='bx bx-heart'></i></div>
    <div class="tiny-float-icon tiny-float-3"><i class='bx bx-crown'></i></div>
    <div class="tiny-float-icon tiny-float-4"><i class='bx bx-trophy'></i></div>
    <div class="tiny-float-icon tiny-float-5"><i class='bx bx-diamond'></i></div>
    <div class="tiny-float-icon tiny-float-6"><i class='bx bx-bulb'></i></div>
    <div class="tiny-float-icon tiny-float-7"><i class='bx bx-rocket'></i></div>
    <div class="tiny-float-icon tiny-float-8"><i class='bx bx-zap'></i></div>
    <div class="tiny-float-icon tiny-float-9"><i class='bx bx-medal'></i></div>
    <div class="tiny-float-icon tiny-float-10"><i class='bx bx-gift'></i></div>
    <div class="tiny-float-icon tiny-float-11"><i class='bx bx-magic-wand'></i></div>
    <div class="tiny-float-icon tiny-float-12"><i class='bx bx-coin'></i></div>
    <div class="tiny-float-icon tiny-float-13"><i class='bx bx-gem'></i></div>
    <div class="tiny-float-icon tiny-float-14"><i class='bx bx-shield'></i></div>
    <div class="tiny-float-icon tiny-float-15"><i class='bx bx-bookmark'></i></div>
    <div class="tiny-float-icon tiny-float-16"><i class='bx bx-fire'></i></div>
    <div class="tiny-float-icon tiny-float-17"><i class='bx bx-target-lock'></i></div>
    <div class="tiny-float-icon tiny-float-18"><i class='bx bx-trending-up'></i></div>
    <div class="tiny-float-icon tiny-float-19"><i class='bx bx-check-circle'></i></div>
    <div class="tiny-float-icon tiny-float-20"><i class='bx bx-award'></i></div>
    <div class="tiny-float-icon tiny-float-21"><i class='bx bx-like'></i></div>
    <div class="tiny-float-icon tiny-float-22"><i class='bx bx-brain'></i></div>
    <div class="tiny-float-icon tiny-float-23"><i class='bx bx-compass'></i></div>
    
    <h1 class="dashboard-title">
        <i class='bx bx-graduation'></i>
        My Learning Dashboard
    </h1>
    <p class="dashboard-subtitle">
        Track your progress and continue your learning journey with our comprehensive course catalog
    </p>
</div>

<!-- Compact Learning Stats Banner - Outside Container -->
<div class="compact-stats-banner">
    <div class="stats-container">
        <!-- Active Learning Stat -->
        <div class="stat-item">
            <div class="stat-icon">
                <i class='bx bx-book-bookmark'></i>
            </div>
            <div class="stat-badge">
                <?php if ($coursesInProgress > 0): ?>
                    Active Learning
                <?php else: ?>
                    Ready to Start
                <?php endif; ?>
            </div>
            <div class="stat-number"><?= count($enrolledCourses) ?></div>
            <div class="stat-label">Enrolled Courses</div>
            <div class="stat-subtitle">Your active learning journey</div>
            <a href="#enrolled-courses" class="stat-action">
                View Courses <i class='bx bx-right-arrow-alt'></i>
            </a>
        </div>

        <!-- Completed Courses Stat -->
        <div class="stat-item">
            <div class="stat-icon">
                <i class='bx bx-check-circle'></i>
            </div>
            <div class="stat-badge">
                <?php if ($completedCourses > 0): ?>
                    Achieved
                <?php else: ?>
                    Getting Started
                <?php endif; ?>
            </div>
            <div class="stat-number"><?= $completedCourses ?></div>
            <div class="stat-label">Completed Courses</div>
            <div class="stat-subtitle">
                <?= $completionRate ?>% Completion Rate
            </div>
            <div class="courses-progress-bar-container">
                <div class="courses-progress-bar">
                    <div class="courses-progress-fill" style="width: <?= $completionRate ?>%"></div>
                </div>
                <div class="progress-percentage"><?= $completionRate ?>%</div>
            </div>
        </div>

        <!-- Learning Hours Stat -->
        <div class="stat-item">
            <div class="stat-icon">
                <i class='bx bx-time'></i>
            </div>
            <div class="stat-badge">Time Invested</div>
            <div class="stat-number"><?= $totalCourseHours ?></div>
            <div class="stat-label">Learning Hours</div>
            <div class="stat-subtitle">Keep growing your expertise</div>
        </div>
    </div>
</div>

<!-- Main Content -->
<div class="dashboard-content">

    <!-- Enrolled Courses Section -->
    <section class="modern-courses-section" id="enrolled-courses">
        <div class="modern-card full-width">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-book-open'></i> My Enrolled Courses</h3>
                    <span class="subtitle">Continue your learning journey</span>
                </div>
                <div class="course-filters">
                    <button class="filter-btn active" data-filter="all">
                        <i class='bx bx-list-ul'></i>
                        All Courses
                    </button>
                    <button class="filter-btn" data-filter="in_progress">
                        <i class='bx bx-play-circle'></i>
                        In Progress
                    </button>
                    <button class="filter-btn" data-filter="completed">
                        <i class='bx bx-check-circle'></i>
                        Completed
                    </button>
                </div>
            </div>
            <div class="card-content">
                <?php if (empty($enrolledCourses)): ?>
                    <div class="modern-empty-state">
                        <div class="empty-illustration">
                            <i class='bx bx-book-add'></i>
                        </div>
                        <div class="empty-content">
                            <h4>No Courses Enrolled Yet</h4>
                            <p>Start your learning journey by enrolling in courses that match your interests and career goals.</p>
                            <a href="#available-courses" class="modern-btn primary">
                                <i class='bx bx-search'></i>
                                Browse Available Courses
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Modern Course Catalog Style Enrolled Courses Grid -->
                    <div class="courses-grid" id="enrolledCoursesGrid">
                        <?php foreach ($enrolledCourses as $index => $course): ?>
                            <div class="course-card enrolled-course" 
                                 data-course-id="<?= $course['id'] ?>"
                                 data-status="<?= $course['status'] ?>" 
                                 style="animation-delay: <?= $index * 0.1 ?>s">
                                
                                <!-- Course Image -->
                                <div class="course-image-container">
                                    <img src="<?= $course['thumbnail'] ?>" 
                                         alt="<?= htmlspecialchars($course['title']) ?>" 
                                         class="course-image"
                                         onerror="this.src='./assets/img/courses/default.jpg'">
                                    
                                    <!-- Progress Badge -->
                                    <div class="course-badge progress-badge <?= $course['progress'] >= 100 ? 'completed' : ($course['progress'] > 0 ? 'in-progress' : 'not-started') ?>">
                                        <?php if ($course['progress'] >= 100): ?>
                                            <i class='bx bx-check-circle'></i> Complete
                                        <?php elseif ($course['progress'] > 0): ?>
                                            <?= $course['progress'] ?>%
                                        <?php else: ?>
                                            <i class='bx bx-play'></i> Start
                                        <?php endif; ?>
                                    </div>
                                    
                                    <!-- Enrollment Status -->
                                    <div class="enrollment-badge">
                                        <i class='bx bx-check-circle'></i> Enrolled
                                    </div>
                                </div>

                                <!-- Course Content -->
                                <div class="course-content">
                                    <!-- Course Title -->
                                    <h3 class="course-title" title="<?= htmlspecialchars($course['title']) ?>">
                                        <?= htmlspecialchars($course['title']) ?>
                                    </h3>
                                    
                                    <!-- Course Description -->
                                    <p class="course-description">
                                        <?= htmlspecialchars(substr($course['description'], 0, 60)) ?>...
                                    </p>
                                    
                                    <!-- Instructor Info -->
                                    <div class="instructor-info">
                                        <i class='bx bx-user'></i>
                                        <span><?= htmlspecialchars($course['instructor']) ?></span>
                                    </div>
                                    
                                    <!-- Progress Section -->
                                    <div class="progress-section">
                                        <div class="progress-info">
                                            <span class="progress-label">Progress</span>
                                            <span class="progress-text"><?= $course['progress'] ?>% complete</span>
                                        </div>
                                        <div class="course-progress-bar">
                                            <div class="progress-fill" style="width: <?= $course['progress'] ?>%"></div>
                                        </div>
                                    </div>
                                    
                                    <!-- Course Meta -->
                                    <div class="course-meta">
                                        <div class="meta-item">
                                            <i class='bx bx-book-open'></i>
                                            <span><?= $course['completed_modules'] ?>/<?= $course['total_modules'] ?> modules</span>
                                        </div>
                                        <?php if ($course['duration'] && $course['duration'] > 0): ?>
                                        <div class="meta-item">
                                             <i class='bx bx-time'></i>
                                             <span><?= $course['duration'] ?> hours</span>
                                         </div>
                                         <?php endif; ?>
                                    </div>
                                </div>
                                
                                <!-- Course Actions -->
                                <div class="course-actions">
                                    <a href="<?= generateSecureUrl('course-details.php', $course['id']) ?>" class="action-btn view-btn" title="View Course">
                                        <i class='bx bx-show'></i>
                                    </a>
                                    <a href="<?= generateSecureUrl('course-detail.php', $course['id']) ?>" class="action-btn continue-btn" title="<?= $course['progress'] >= 100 ? 'Review Course' : 'Continue Learning' ?>">
                                        <i class='bx bx-play-circle'></i>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </section>

    <!-- Available Courses Section -->
    <section class="modern-courses-section" id="available-courses">
        <div class="modern-card full-width">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-compass'></i> Discover New Courses</h3>
                    <span class="subtitle">Expand your skills with our curated course collection</span>
                </div>
                <a href="course-catalog.php" class="view-all-btn">
                    <span>View All Courses</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="card-content">
                <?php if (empty($availableCourses)): ?>
                    <div class="modern-empty-state">
                        <div class="empty-illustration">
                            <i class='bx bx-search-alt'></i>
                        </div>
                        <div class="empty-content">
                            <h4>More Courses Coming Soon</h4>
                            <p>We're constantly adding new courses to help you grow. Check back soon for exciting new learning opportunities!</p>
                            <a href="course-catalog.php" class="modern-btn">
                                <i class='bx bx-bell'></i>
                                Get Notified
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="udemy-style-courses-grid">
                        <?php foreach ($availableCourses as $index => $course): ?>
                            <div class="udemy-course-card" data-course-id="<?= IdEncryption::encrypt($course['id']) ?>" style="animation-delay: <?= $index * 0.1 ?>s">
                                <!-- Course Thumbnail -->
                                <div class="course-image-container">
                                    <img src="<?= $course['thumbnail'] ?>" alt="<?= htmlspecialchars($course['title']) ?>" class="course-image">
                                    <?php if ($course['price'] <= 0): ?>
                                        <div class="course-badge free-badge">FREE</div>
                                    <?php else: ?>
                                        <?php if ($course['has_discount']): ?>
                                            <div class="course-badge price-badge discount-active">
                                                <span class="discount-percent">-<?= $course['discount_percentage'] ?>%</span>
                                                <span class="original-price">R<?= number_format($course['price'], 0) ?></span>
                                                <span class="discounted-price">R<?= number_format($course['discounted_price'], 0) ?></span>
                                            </div>
                                        <?php else: ?>
                                            <div class="course-badge price-badge">R<?= number_format($course['price'], 0) ?></div>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Course Content -->
                                <div class="course-card-content">
                                    <!-- Course Title (single line with ellipsis) -->
                                    <h3 class="course-card-title" title="<?= htmlspecialchars($course['title']) ?>">
                                        <?= htmlspecialchars($course['title']) ?>
                                    </h3>
                                    
                                    <!-- Course Description (2 lines max with ellipsis) -->
                                    <p class="course-card-description" title="<?= htmlspecialchars($course['description']) ?>">
                                        <?= htmlspecialchars($course['description']) ?>
                                    </p>
                                    
                                    <!-- Instructor -->
                                    <div class="course-instructor-name">
                                        <?= htmlspecialchars($course['instructor']) ?>
                                    </div>
                                    
                                    <!-- Rating and Students -->
                                    <div class="course-rating-info">
                                        <div class="rating-stars">
                                            <span class="rating-value"><?= $course['rating'] ?></span>
                                            <div class="stars">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class='bx <?= $i <= $course['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                <?php endfor; ?>
                                            </div>
                                            <span class="rating-count">(<?= $course['students'] ?>)</span>
                                        </div>
                                    </div>
                                    
                                    <!-- Course Meta -->
                                    <div class="course-meta-info">
                                        <span class="course-duration"><?= htmlspecialchars($course['duration']) ?></span>
                                        <span class="course-level"><?= htmlspecialchars($course['level']) ?></span>
                                    </div>
                                    
                                    <!-- Price and Action Buttons -->
                                    <div class="course-card-footer">
                                        <div class="course-price-section">
                                            <?php if ($course['price'] > 0): ?>
                                                <?php if ($course['has_discount']): ?>
                                                    <div class="price-with-discount">
                                                        <span class="original-price-text">R<?= number_format($course['price'], 0) ?></span>
                                                        <span class="discounted-price-text">R<?= number_format($course['discounted_price'], 0) ?></span>
                                                    </div>
                                                <?php else: ?>
                                                    <span class="course-price">R<?= number_format($course['price'], 0) ?></span>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="course-price free">Free</span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="course-action-buttons">
                                            <a href="<?= generateSecureUrl('course-details.php', $course['id']) ?>" class="action-btn view-btn" title="View Course Details">
                                                <i class='bx bx-show'></i>
                                            </a>
                                            <button class="action-btn cart-btn" 
                                                    onclick="addToCart('<?= IdEncryption::encrypt($course['id']) ?>', this)" 
                                                    title="<?= $course['has_discount'] ? 'Add to Cart - R' . number_format($course['discounted_price'], 0) . ' (Save R' . number_format($course['savings'], 0) . '!)' : 'Add to Cart' ?>">
                                                <i class='bx bx-cart-add'></i>
                                            </button>
                                            <a href="course-payment.php?id=<?= IdEncryption::encrypt($course['id']) ?>" 
                                               class="action-btn enroll-btn" 
                                               title="<?= $course['has_discount'] ? 'Enroll Now - R' . number_format($course['discounted_price'], 0) . ' (Save R' . number_format($course['savings'], 0) . '!)' : 'Enroll Now' ?>">
                                                <i class='bx bx-user-plus'></i>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </section>
</div>

<!-- Clean Course Cards CSS -->
<style>
/* Page Background and Container */
.dashboard-content {
    background: whitesmoke;
    min-height: 100vh;
    padding: 2rem;
}

/* Dashboard Header */
.dashboard-header {
    text-align: center;
    margin-bottom: 2rem;
    padding: 2rem 1rem;
    background: transparent;
    width: 100%;
}

.dashboard-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0 0 1rem 0;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 1rem;
}

.dashboard-title i {
    font-size: 2.8rem;
    color: #f59e0b;
    background: linear-gradient(135deg, #f59e0b, #d97706);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.dashboard-subtitle {
    font-size: 1.2rem;
    color: #6b7280;
    margin: 0;
    font-weight: 400;
}

/* Learning Stats Banner - Clean No Shadow Design */
.compact-stats-banner {
    background: transparent;
    padding: 1.5rem;
    margin: 0 auto 3rem auto;
    max-width: 1200px;
    border: none;
    box-shadow: none;
    position: relative;
    overflow: visible;
}

/* Removed decorative ::before element */
.compact-stats-banner::before {
    display: none;
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(37, 99, 235, 0.02) 0%, rgba(245, 158, 11, 0.02) 100%);
    z-index: 1;
}

/* Clean Simple Stats Cards */
.stats-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 1rem;
}

.stat-item {
    background: white !important;
    padding: 1.5rem !important;
    border-radius: 16px !important;
    border: 1px solid #e5e7eb !important;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05) !important;
    text-align: center !important;
    transition: all 0.3s ease !important;
    display: block !important;
    width: auto !important;
    height: auto !important;
    min-height: auto !important;
    position: relative !important;
    overflow: hidden !important;
}

.stat-item:hover {
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.1);
    transform: translateY(-2px);
}

.stat-icon {
    width: 48px !important;
    height: 48px !important;
    background: #f3f4f6 !important;
    border-radius: 12px !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    margin: 0 auto 1rem auto !important;
}

.stat-icon i {
    font-size: 1.5rem !important;
    color: #2563eb !important;
}

/* Small visible floating decorative icons around header */
.tiny-float-icon {
    position: absolute !important;
    width: 14px !important;
    height: 14px !important;
    border-radius: 50% !important;
    background: rgba(37, 99, 235, 0.1) !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    opacity: 0.7 !important;
    animation: tinyFloatGently 6s ease-in-out infinite !important;
    z-index: 1 !important;
    border: 1px solid rgba(37, 99, 235, 0.15) !important;
}

.tiny-float-icon i {
    font-size: 7px !important;
    color: #2563eb !important;
}

/* Individual positioning around header area - lots of icons */
.tiny-float-1 { top: 10% !important; left: 8% !important; animation-delay: 0s !important; }
.tiny-float-2 { top: 15% !important; right: 12% !important; animation-delay: 0.3s !important; }
.tiny-float-3 { top: 20% !important; left: 25% !important; animation-delay: 0.6s !important; }
.tiny-float-4 { top: 25% !important; right: 30% !important; animation-delay: 0.9s !important; }
.tiny-float-5 { top: 30% !important; left: 5% !important; animation-delay: 1.2s !important; }
.tiny-float-6 { top: 35% !important; right: 8% !important; animation-delay: 1.5s !important; }
.tiny-float-7 { top: 40% !important; left: 35% !important; animation-delay: 1.8s !important; }
.tiny-float-8 { top: 45% !important; right: 35% !important; animation-delay: 2.1s !important; }
.tiny-float-9 { top: 50% !important; left: 12% !important; animation-delay: 2.4s !important; }
.tiny-float-10 { top: 55% !important; right: 18% !important; animation-delay: 2.7s !important; }
.tiny-float-11 { top: 60% !important; left: 30% !important; animation-delay: 3s !important; }
.tiny-float-12 { top: 65% !important; right: 25% !important; animation-delay: 3.3s !important; }
.tiny-float-13 { top: 70% !important; left: 18% !important; animation-delay: 3.6s !important; }
.tiny-float-14 { top: 75% !important; right: 15% !important; animation-delay: 3.9s !important; }
.tiny-float-15 { top: 80% !important; left: 22% !important; animation-delay: 4.2s !important; }
.tiny-float-16 { top: 85% !important; right: 22% !important; animation-delay: 4.5s !important; }
.tiny-float-17 { top: 90% !important; left: 15% !important; animation-delay: 4.8s !important; }
.tiny-float-18 { top: 12% !important; left: 45% !important; animation-delay: 5.1s !important; }
.tiny-float-19 { top: 28% !important; right: 45% !important; animation-delay: 5.4s !important; }
.tiny-float-20 { top: 48% !important; left: 45% !important; animation-delay: 5.7s !important; }
.tiny-float-21 { top: 68% !important; right: 40% !important; animation-delay: 6s !important; }
.tiny-float-22 { top: 82% !important; left: 40% !important; animation-delay: 6.3s !important; }
.tiny-float-23 { top: 92% !important; right: 5% !important; animation-delay: 6.6s !important; }

/* Gentle floating animation for visible icons */
@keyframes tinyFloatGently {
    0%, 100% { transform: translateY(0px) rotate(0deg) !important; opacity: 0.5 !important; }
    25% { transform: translateY(-2px) rotate(2deg) !important; opacity: 0.7 !important; }
    50% { transform: translateY(-1px) rotate(-1deg) !important; opacity: 0.9 !important; }
    75% { transform: translateY(-3px) rotate(1deg) !important; opacity: 0.6 !important; }
}

/* Dashboard header positioning for floating icons */
.dashboard-header {
    position: relative !important;
}

.stat-badge {
    font-size: 0.75rem !important;
    font-weight: 600 !important;
    text-transform: uppercase !important;
    letter-spacing: 0.5px !important;
    margin: 0 0 1rem 0 !important;
    padding: 0 !important;
    color: #6b7280 !important;
    display: block !important;
}

.stat-number {
    font-size: 3rem !important;
    font-weight: 700 !important;
    color: #1f2937 !important;
    margin: 0 0 0.5rem 0 !important;
    padding: 0 !important;
    display: block !important;
    line-height: 1 !important;
}

.stat-label {
    font-size: 1.25rem !important;
    font-weight: 600 !important;
    color: #374151 !important;
    margin: 0 0 0.5rem 0 !important;
    padding: 0 !important;
    display: block !important;
    line-height: 1.2 !important;
}

.stat-subtitle {
    font-size: 0.875rem !important;
    color: #6b7280 !important;
    margin: 0 0 1.5rem 0 !important;
    padding: 0 !important;
    display: block !important;
    line-height: 1.4 !important;
}

.stat-action {
    display: inline-block;
    background: #2563eb;
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    margin: 1rem 0 0 0;
}

.stat-action:hover {
    background: #1d4ed8;
    color: white;
    text-decoration: none;
}

.courses-progress-bar-container {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin: 1rem 0 1.5rem 0;
    padding: 0;
}

.courses-progress-bar {
    flex: 1;
    height: 6px;
    background: #e5e7eb;
    border-radius: 3px;
}

.courses-progress-fill {
    height: 100%;
    background: #10b981;
    border-radius: 3px;
    transition: width 0.3s ease;
}

.progress-percentage {
    font-size: 0.875rem;
    font-weight: 600;
    color: #10b981;
}

/* Simple responsive design */
@media screen and (max-width: 768px) {
    .stats-container {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .stat-item {
        padding: 1.5rem;
    }
    
    .stat-number {
        font-size: 2.5rem;
    }
    
    .stat-label {
        font-size: 1.125rem;
    }
}

@media screen and (max-width: 480px) {
    .stats-container {
        padding: 0 0.5rem;
    }
    
    .stat-item {
        padding: 1.25rem;
    }
    
    .stat-number {
        font-size: 2rem;
    }
    
    .stat-label {
        font-size: 1rem;
    }
    
    .stat-action {
        padding: 0.625rem 1.25rem;
        font-size: 0.8rem;
    }
}

/* Course Sections */
.modern-courses-section {
    margin-bottom: 3rem;
}

.modern-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid #e5e7eb;
    overflow: hidden;
}

.card-header {
    padding: 2rem 2rem 1rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e5e7eb;
}

.header-content h3 {
    font-size: 1.8rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-content h3 i {
    color: #2563eb;
    font-size: 2rem;
}

.subtitle {
    color: #6b7280;
    font-size: 1rem;
}

.view-all-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #2563eb;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.view-all-btn:hover {
    color: #1d4ed8;
    transform: translateX(5px);
    text-decoration: none;
}

.card-content {
    padding: 2rem;
}

/* Filter Buttons */
.course-filters {
    display: flex;
    gap: 0.5rem;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
    justify-content: center;
}

.filter-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.6rem 1.2rem;
    background: white;
    color: #6b7280;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.filter-btn:hover {
    background: #f8fafc;
    border-color: #2563eb;
    color: #2563eb;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.15);
}

.filter-btn.active {
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    color: white;
    border-color: #2563eb;
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
}

.filter-btn.active:hover {
    background: linear-gradient(135deg, #1d4ed8, #1e40af);
    transform: translateY(-2px);
}

.filter-btn i {
    font-size: 1rem;
}

/* Enrolled Courses Grid - Modern Course Catalog Style */
.courses-grid {
    display: grid !important;
    grid-template-columns: repeat(5, 1fr) !important;
    grid-template-rows: auto !important;
    gap: 0.8rem !important;
    padding: 1rem 0 !important;
    max-width: 100% !important;
    width: 100% !important;
    margin: 0 !important;
}

/* Responsive Grid - Consistent with course catalog */
@media (max-width: 1200px) {
    .courses-grid {
        grid-template-columns: repeat(4, 1fr) !important;
        gap: 1rem !important;
    }
}

@media (max-width: 900px) {
    .courses-grid {
        grid-template-columns: repeat(3, 1fr) !important;
        gap: 1rem !important;
    }
}

/* Mobile: 2 cards per row */
@media screen and (max-width: 768px) {
    .courses-grid {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        grid-gap: 0.75rem !important;
        gap: 0.75rem !important;
        padding: 1rem 0.5rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 !important;
        box-sizing: border-box !important;
    }
}

@media screen and (max-width: 480px) {
    .courses-grid {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        grid-gap: 0.5rem !important;
        gap: 0.5rem !important;
        padding: 1rem 0.25rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 !important;
        box-sizing: border-box !important;
    }
}

/* Modern Course Card - Matching catalog design */
.course-card {
    background: #ffffff;
    border-radius: 10px;
    border: 1px solid #e5e7eb;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: slideUp 0.6s ease forwards;
    box-shadow: 0 2px 6px rgba(0, 0, 0, 0.05);
    display: flex;
    flex-direction: column;
    height: 380px;
    max-width: 100%;
    min-height: 380px;
}

.course-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12);
    border-color: #cbd5e1;
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Course Image */
.course-image-container {
    position: relative;
    width: 100%;
    height: 120px;
    overflow: hidden;
    flex-shrink: 0;
}

.course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.course-card:hover .course-image {
    transform: scale(1.05);
}

/* Course badges */
.course-badge {
    position: absolute;
    top: 0.75rem;
    right: 0.75rem;
    padding: 0.25rem 0.5rem;
    border-radius: 0.375rem;
    font-size: 0.75rem;
    font-weight: 600;
    z-index: 2;
    display: flex;
    align-items: center;
    gap: 0.25rem;
    max-width: calc(50% - 0.5rem);
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.course-badge.progress-badge.completed {
    background: #10b981;
    color: white;
}

.course-badge.progress-badge.in-progress {
    background: #f59e0b;
    color: white;
}

.course-badge.progress-badge.not-started {
    background: #6b7280;
    color: white;
}

/* Enrollment Badge */
.enrollment-badge {
    position: absolute;
    top: 0.75rem;
    left: 0.75rem;
    background: rgba(16, 185, 129, 0.9);
    color: white;
    padding: 0.25rem 0.5rem;
    border-radius: 0.375rem;
    font-size: 0.7rem;
    font-weight: 600;
    z-index: 2;
    display: flex;
    align-items: center;
    gap: 0.25rem;
    max-width: calc(50% - 0.5rem);
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Mobile badge adjustments */
@media screen and (max-width: 768px) {
    .course-badge {
        padding: 0.2rem 0.4rem !important;
        font-size: 0.65rem !important;
        top: 0.5rem !important;
        right: 0.5rem !important;
        max-width: calc(45% - 0.25rem) !important;
    }
    
    .enrollment-badge {
        padding: 0.2rem 0.4rem !important;
        font-size: 0.6rem !important;
        top: 0.5rem !important;
        left: 0.5rem !important;
        max-width: calc(45% - 0.25rem) !important;
    }
}

@media screen and (max-width: 480px) {
    .course-badge {
        padding: 0.15rem 0.3rem !important;
        font-size: 0.6rem !important;
        top: 0.4rem !important;
        right: 0.4rem !important;
        max-width: calc(42% - 0.2rem) !important;
    }
    
    .enrollment-badge {
        padding: 0.15rem 0.3rem !important;
        font-size: 0.55rem !important;
        top: 0.4rem !important;
        left: 0.4rem !important;
        max-width: calc(42% - 0.2rem) !important;
    }
}

/* Course Content */
.course-content {
    padding: 0.75rem;
    display: flex;
    flex-direction: column;
    gap: 0.4rem;
    height: calc(380px - 120px - 44px);
    overflow: hidden;
    justify-content: flex-start;
}

.course-title {
    font-size: 0.9rem;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 0.3rem 0;
    line-height: 1.2;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    width: 100%;
}

.course-description {
    font-size: 0.8rem;
    color: #6b7280;
    line-height: 1.3;
    margin: 0 0 0.4rem 0;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
    max-height: 2.6em;
}

.instructor-info {
    display: flex;
    align-items: center;
    gap: 0.4rem;
    font-size: 0.8rem;
    color: #4b5563;
    margin-bottom: 0.5rem;
}

.instructor-info i {
    color: #6b7280;
    font-size: 1rem;
}

/* Progress Section for Enrolled Courses */
.progress-section {
    margin-bottom: 0.5rem;
}

.progress-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.2rem;
}

.progress-label {
    font-size: 0.7rem;
    color: #6b7280;
    font-weight: 500;
}

.progress-text {
    font-size: 0.7rem;
    color: #4b5563;
    font-weight: 600;
}

.course-progress-bar {
    width: 100%;
    height: 4px;
    background: #f1f5f9;
    border-radius: 2px;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #3b82f6, #1d4ed8);
    border-radius: 2px;
    transition: width 0.3s ease;
}

.course-meta {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 0.5rem;
    margin-top: auto;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.2rem;
    font-size: 0.7rem;
    color: #6b7280;
    white-space: nowrap;
}

.meta-item i {
    font-size: 0.8rem;
    color: #9ca3af;
}

.course-instructor {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    color: #6b7280;
    font-size: 0.75rem;
    margin: 0;
}

.course-instructor i {
    color: #2563eb;
    font-size: 0.8rem;
}

/* Progress Section */
.progress-section {
    margin: 0;
}

.progress-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.25rem;
    font-size: 0.7rem;
}

.text-muted {
    color: #6b7280;
}

.progress-text {
    color: #2563eb;
    font-weight: 600;
}

.modern-progress-bar {
    width: 100%;
    height: 4px;
    background: #f1f5f9;
    border-radius: 2px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #2563eb, #60a5fa);
    border-radius: 4px;
    transition: width 0.8s ease-out;
    position: relative;
}

.progress-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

/* Modules Stats */
.modules-stats {
    display: flex;
    justify-content: space-between;
    margin: 0;
    padding: 0;
    font-size: 0.6rem;
    line-height: 1;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 0.15rem;
    color: #6b7280;
    padding: 0;
    margin: 0;
}

.stat-item i {
    color: #2563eb;
    font-size: 0.65rem;
    flex-shrink: 0;
}

.stat-item span {
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Course Actions - Matching catalog design */
.course-actions {
    display: flex !important;
    flex-direction: row !important;
    justify-content: center !important;
    align-items: center !important;
    padding: 0.6rem 0.75rem !important;
    border-top: 1px solid #f1f5f9 !important;
    background: #fafafa !important;
    margin-top: auto !important;
    gap: 0.4rem !important;
    flex-wrap: nowrap !important;
    width: 100% !important;
    box-sizing: border-box !important;
    flex-shrink: 0 !important;
    min-height: 44px !important;
}

.action-btn {
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    width: 32px !important;
    height: 32px !important;
    border-radius: 6px !important;
    border: none !important;
    cursor: pointer !important;
    font-size: 14px !important;
    transition: all 0.2s ease !important;
    flex-shrink: 0 !important;
    text-decoration: none !important;
}

.action-btn:hover {
    transform: translateY(-2px) !important;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15) !important;
}

.continue-btn {
    background: #8b5cf6 !important;
    color: white !important;
}

.continue-btn:hover {
    background: #7c3aed !important;
    color: white !important;
}

.view-btn {
    background: #6b7280 !important;
    color: white !important;
}

.view-btn:hover {
    background: #4b5563 !important;
    color: white !important;
}

.heart-btn {
    background: #ef4444 !important;
    color: white !important;
}

.heart-btn:hover {
    background: #dc2626 !important;
    color: white !important;
}

/* Mobile responsive actions */
@media screen and (max-width: 768px) {
    .course-actions {
        padding: 0.5rem 0.75rem !important;
        gap: 0.4rem !important;
    }
    
    .action-btn {
        width: 30px !important;
        height: 30px !important;
        font-size: 13px !important;
    }
}

@media screen and (max-width: 480px) {
    .course-actions {
        padding: 0.4rem 0.5rem !important;
        gap: 0.3rem !important;
    }
    
    .action-btn {
        width: 26px !important;
        height: 26px !important;
        font-size: 11px !important;
    }
}

/* Available Courses Grid */
.udemy-style-courses-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1rem;
    margin-top: 1rem;
}

/* Available Course Cards */
.udemy-course-card {
    background: white;
    border-radius: 12px;
    border: 1px solid #e5e7eb;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: fadeInUp 0.6s ease forwards;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
}

.udemy-course-card:hover {
    transform: translateY(-6px);
    box-shadow: 0 12px 32px rgba(0, 0, 0, 0.15);
    border-color: #d1d5db;
}

/* Course Image */
.course-image-container {
    position: relative;
    width: 100%;
    height: 120px;
    overflow: hidden;
}

.course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.udemy-course-card:hover .course-image {
    transform: scale(1.05);
}

.course-badge {
    position: absolute;
    top: 12px;
    right: 12px;
    padding: 6px 12px;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 700;
    text-transform: uppercase;
}

.free-badge {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
}

.price-badge {
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    color: white;
}

.price-badge.discount-active {
    background: linear-gradient(135deg, #dc2626, #b91c1c);
    color: white;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

/* Course Content */
.course-card-content {
    padding: 0.75rem;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.course-card-title {
    font-size: 0.9rem;
    font-weight: 700;
    color: #1f2937;
    line-height: 1.2;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
}

.course-card-title:hover {
    color: #2563eb;
}

.course-card-description {
    display: none;
}

.course-instructor-name {
    font-size: 0.75rem;
    color: #9ca3af;
    font-weight: 500;
}

/* Rating and Meta - Hidden */
.course-rating-info {
    display: none;
}

.rating-value {
    display: none;
}

.stars {
    display: none;
}

.stars i {
    display: none;
}

.rating-count {
    display: none;
}

.course-meta-info {
    display: none;
}

/* Course Footer */
.course-card-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: auto;
    padding-top: 0.5rem;
    border-top: 1px solid #f1f5f9;
}

.course-price {
    font-size: 0.9rem;
    font-weight: 800;
    color: #1f2937;
}

.course-price.free {
    color: #10b981;
}

.price-with-discount {
    display: flex;
    flex-direction: column;
    align-items: flex-start;
    gap: 1px;
}

.original-price-text {
    font-size: 0.7rem;
    color: #9ca3af;
    text-decoration: line-through;
}

.discounted-price-text {
    font-size: 0.9rem;
    color: #dc2626;
    font-weight: 800;
}

/* Action Buttons */
.course-action-buttons {
    display: flex;
    gap: 0.25rem;
    align-items: center;
}

.action-btn {
    width: 28px;
    height: 28px;
    border: none;
    border-radius: 6px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.75rem;
}

.view-btn {
    background: #f1f5f9;
    color: #64748b;
}

.view-btn:hover {
    background: #e2e8f0;
    color: #475569;
    transform: translateY(-1px);
}

.cart-btn {
    background: linear-gradient(135deg, #fef3c7, #fde68a);
    color: #d97706;
}

.cart-btn:hover {
    background: linear-gradient(135deg, #fde68a, #fcd34d);
    color: #92400e;
    transform: translateY(-1px);
}

.enroll-btn {
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    color: white;
}

.enroll-btn:hover {
    background: linear-gradient(135deg, #1d4ed8, #1e40af);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
}

/* Empty States */
.modern-empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-illustration i {
    font-size: 5rem;
    color: #e5e7eb;
    margin-bottom: 2rem;
}

.empty-content h4 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 1rem;
}

.empty-content p {
    color: #6b7280;
    font-size: 1rem;
    margin-bottom: 2rem;
    max-width: 500px;
    margin-left: auto;
    margin-right: auto;
}

.modern-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.modern-btn:hover {
    background: linear-gradient(135deg, #1d4ed8, #1e40af);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
    color: white;
    text-decoration: none;
}

/* Responsive Design */
@media (max-width: 1200px) {
    .modern-courses-marketplace-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 1rem;
    }
    
    .udemy-style-courses-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 1rem;
    }
}

@media (max-width: 992px) {
    .modern-courses-marketplace-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 0.75rem;
    }
    
    .udemy-style-courses-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 0.75rem;
    }
}

@media (max-width: 768px) {
    .dashboard-content {
        padding: 1rem;
    }
    
    .dashboard-title {
        font-size: 2rem;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .dashboard-title i {
        font-size: 2.2rem;
    }
    
    .compact-stats-banner {
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }
    
    .stats-container {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .stat-item {
        padding: 1rem;
    }
    
    .card-header {
        padding: 1.5rem 1rem 1rem 1rem;
    }
    
    .card-content {
        padding: 1rem;
    }
    
    /* Mobile: 2 cards per row */
    .modern-courses-marketplace-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .udemy-style-courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .course-thumbnail-container {
        height: 100px;
    }
    
    .course-image-container {
        height: 100px;
    }
}

@media (max-width: 480px) {
    .dashboard-content {
        padding: 0.75rem;
    }
    
    .compact-stats-banner {
        padding: 1rem;
    }
    
    .card-header {
        padding: 1rem 0.75rem 0.75rem 0.75rem;
    }
    
    .card-content {
        padding: 0.75rem;
    }
    
    /* Small mobile: Still 2 cards per row but smaller */
    .modern-courses-marketplace-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.5rem;
    }
    
    .udemy-style-courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.5rem;
    }
    
    .course-thumbnail-container {
        height: 80px;
    }
    
    .course-image-container {
        height: 80px;
    }
    
    .course-title {
        font-size: 0.8rem;
    }
    
    .course-card-title {
        font-size: 0.8rem;
    }
    
    .course-instructor {
        font-size: 0.7rem;
    }
    
    .course-instructor-name {
        font-size: 0.7rem;
    }
    
    .course-price {
        font-size: 0.8rem;
    }
    
    .btn-primary-action {
        font-size: 0.7rem;
        padding: 0.3rem 0.5rem;
        min-height: 24px;
    }
    
    .action-btn {
        width: 24px;
        height: 24px;
        font-size: 0.7rem;
    }
}

/* Cart Notification Styles */
.cart-notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
    min-width: 300px;
    max-width: 400px;
}

.cart-notification .alert {
    margin-bottom: 10px;
    border-radius: 10px;
    padding: 12px 16px;
    font-weight: 500;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    animation: slideInRight 0.3s ease-out;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

/* Loading Spinner */
.bx-spin {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}
</style>

<!-- Course Filtering and Interaction JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize cart system
    initializeCart();
    
    // Course filtering functionality
    const filterButtons = document.querySelectorAll('.filter-btn');
    const courseCards = document.querySelectorAll('.enrolled-course');
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            filterButtons.forEach(btn => btn.classList.remove('active'));
            // Add active class to clicked button
            this.classList.add('active');
            
            const filter = this.dataset.filter;
            
            courseCards.forEach(card => {
                if (filter === 'all' || card.dataset.status === filter) {
                    card.style.display = 'block';
                    card.style.animation = 'slideUp 0.6s ease forwards';
                } else {
                    card.style.display = 'none';
                }
            });
        });
    });
    
    // Smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // Enhanced button hover effects
    document.querySelectorAll('.modern-btn').forEach(btn => {
        btn.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px) scale(1.02)';
        });
        
        btn.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
    
    // Progress ring animation
    document.querySelectorAll('.progress-ring-circle').forEach(circle => {
        const progressRing = circle.closest('.course-progress-ring');
        if (progressRing) {
            const progress = progressRing.style.getPropertyValue('--progress');
            if (progress) {
                setTimeout(() => {
                    circle.style.strokeDashoffset = `calc(163.36 - (163.36 * ${progress}) / 100)`;
                }, 500);
            }
        }
    });
});

// Course Action Functions
function viewCourse(courseId) {
    // Navigate to course detail page
    window.location.href = `course-detail.php?id=${courseId}`;
}



// Update navbar cart count
function updateNavbarCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cart.length;
    
    // Update cart badge in navbar
    const cartBadge = document.querySelector('.cart-count');
    if (cartBadge) {
        cartBadge.textContent = cartCount;
        cartBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    // Update cart icon badge (alternative selector)
    const cartIconBadge = document.querySelector('#cart-count');
    if (cartIconBadge) {
        cartIconBadge.textContent = cartCount;
        cartIconBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    console.log('Cart count updated:', cartCount);
}

// Initialize cart on page load
function initializeCart() {
    // Create notification container if it doesn't exist
    if (!document.querySelector('.cart-notification')) {
        const notificationContainer = document.createElement('div');
        notificationContainer.className = 'cart-notification';
        document.body.appendChild(notificationContainer);
    }
    
    // Update cart count on page load
    updateNavbarCartCount();
    
    // Mark courses already in cart
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    cart.forEach(cartItem => {
        // Since data-course-id now contains encrypted IDs, we need to check all course cards
        // and compare the raw course ID from the cart with each card
        document.querySelectorAll('[data-course-id]').forEach(courseCard => {
            const cartButton = courseCard.querySelector('.cart-btn');
            if (cartButton) {
                // Check if this course is in the cart by comparing raw IDs
                // The cart stores raw course IDs but the HTML uses encrypted ones
                // We'll rely on the course matching logic in the addToCart function
                // For now, let the addToCart function handle the "already in cart" check
            }
        });
    });
}

// Enhanced notification system
function showNotification(message, type = 'info', duration = 4000) {
    const container = document.querySelector('.cart-notification') || createNotificationContainer();
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible`;
    alert.style.opacity = '0';
    
    const alertTypes = {
        'success': 'alert-success',
        'error': 'alert-danger', 
        'warning': 'alert-warning',
        'info': 'alert-info'
    };
    
    alert.className = `alert ${alertTypes[type] || 'alert-info'} alert-dismissible`;
    
    alert.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="flex-grow-1">${message}</div>
            <button type="button" class="btn-close btn-close-white" aria-label="Close"></button>
        </div>
    `;
    
    container.appendChild(alert);
    
    // Animate in
    setTimeout(() => {
        alert.style.opacity = '1';
    }, 10);
    
    // Auto dismiss
    const timeoutId = setTimeout(() => {
        dismissNotification(alert);
    }, duration);
    
    // Manual dismiss
    const closeBtn = alert.querySelector('.btn-close');
    closeBtn.addEventListener('click', () => {
        clearTimeout(timeoutId);
        dismissNotification(alert);
    });
}

function createNotificationContainer() {
    const container = document.createElement('div');
    container.className = 'cart-notification';
    document.body.appendChild(container);
    return container;
}

function dismissNotification(alert) {
    alert.style.animation = 'slideOutRight 0.3s ease-in';
    setTimeout(() => {
        if (alert.parentNode) {
            alert.parentNode.removeChild(alert);
        }
    }, 300);
}

function addToCart(courseId, buttonElement) {
    console.log('addToCart called with courseId:', courseId, 'buttonElement:', buttonElement);
    
    // Get the cart button - either passed as parameter or find by course ID
    let cartButton = buttonElement;
    
    if (!cartButton) {
        // Try to find the button by course ID in the card
        cartButton = document.querySelector(`[data-course-id="${courseId}"] .cart-btn`);
        console.log('Searched for cart button with selector:', `[data-course-id="${courseId}"] .cart-btn`);
    }
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        console.log('Available course cards with data-course-id:');
        document.querySelectorAll('[data-course-id]').forEach(card => {
            console.log('- Card ID:', card.getAttribute('data-course-id'));
        });
        showNotification('❌ Unable to add course to cart - button not found', 'error');
        return;
    }
    
    console.log('Cart button found:', cartButton);
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-check"></i>';
                cartButton.style.background = '#10b981';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                cartButton.classList.add('in-cart');
                cartButton.title = 'Added to Cart';
            }, 1500);
            
        } else if (data.duplicate) {
            // Handle duplicate course (already in cart)
            cartButton.innerHTML = '<i class="bx bx-info-circle"></i>';
            cartButton.style.background = '#f59e0b';
            cartButton.style.color = 'white';
            cartButton.disabled = true;
            cartButton.classList.add('in-cart');
            cartButton.title = 'Already in Cart';
            
            // Show duplicate warning
            showNotification(`📚 ${data.message}`, 'warning');
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i>';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

function enrollCourse(courseId) {
    // Direct enrollment functionality
    const enrollButton = document.querySelector(`[data-course-id="${courseId}"] .enroll-btn`);
    
    // Visual feedback
    const originalHTML = enrollButton.innerHTML;
    enrollButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    enrollButton.disabled = true;
    
    // Check if course is free or paid
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    const priceElement = courseCard.querySelector('.course-price');
    const isFree = priceElement.textContent.toLowerCase().includes('free');
    
    if (isFree) {
        // Direct enrollment for free courses
        fetch('api/enroll-course.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                course_id: courseId,
                action: 'enroll_free'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Success feedback
                enrollButton.innerHTML = '<i class="bx bx-check"></i>';
                enrollButton.style.background = '#10b981';
                
                showNotification('Successfully enrolled in course!', 'success');
                
                // Redirect to course or refresh page after delay
                setTimeout(() => {
                    window.location.href = `course-detail.php?id=${courseId}`;
                }, 1500);
            } else {
                handleEnrollmentError(enrollButton, originalHTML, data.message);
            }
        })
        .catch(error => {
            console.error('Enrollment error:', error);
            handleEnrollmentError(enrollButton, originalHTML, 'An error occurred. Please try again.');
        });
    } else {
        // Redirect to payment page for paid courses
        enrollButton.innerHTML = originalHTML;
        enrollButton.disabled = false;
        window.location.href = `course-payment.php?id=${courseId}`;
    }
}

function handleEnrollmentError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    
    showNotification(message || 'Failed to enroll in course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
    }, 2000);
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `course-notification ${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles if not already present
    if (!document.querySelector('#notification-styles')) {
        const style = document.createElement('style');
        style.id = 'notification-styles';
        style.textContent = `
            .course-notification {
                position: fixed;
                top: 100px;
                right: 20px;
                background: white;
                padding: 16px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
                border-left: 4px solid;
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 400px;
            }
            .course-notification.success { border-left-color: #10b981; }
            .course-notification.error { border-left-color: #ef4444; }
            .course-notification.info { border-left-color: #3b82f6; }
            .notification-content {
                display: flex;
                align-items: center;
                gap: 12px;
                font-size: 14px;
                font-weight: 500;
            }
            .notification-content i {
                font-size: 18px;
                color: inherit;
            }
            .course-notification.success .notification-content i { color: #10b981; }
            .course-notification.error .notification-content i { color: #ef4444; }
            .course-notification.info .notification-content i { color: #3b82f6; }
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOutRight {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 4 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 4000);
}

// Encrypted URL mapping - populated server-side
const encryptedUrls = {};
<?php foreach ($enrolledCourses as $course): ?>
encryptedUrls[<?= $course['id'] ?>] = '<?= generateSecureUrl('course-detail.php', $course['id']) ?>';
<?php endforeach; ?>

// Enrolled Course Action Functions
function continueCourse(courseId) {
    console.log('Continue course clicked for ID:', courseId);
    console.log('Available encrypted URLs:', encryptedUrls);
    
    // Navigate to course detail page with encrypted URL
    const encryptedUrl = encryptedUrls[courseId];
    console.log('Found encrypted URL:', encryptedUrl);
    
    if (encryptedUrl) {
        console.log('Navigating to:', encryptedUrl);
        window.location.href = encryptedUrl;
    } else {
        console.log('No encrypted URL found, using fallback');
        // Fallback for safety - try the correct page name
        window.location.href = 'courses.php';
    }
}

function unenrollCourse(courseId) {
    // Confirm unenrollment
    if (!confirm('Are you sure you want to unenroll from this course? You will lose access to all course materials and your progress will be saved but not accessible until you re-enroll.')) {
        return;
    }
    
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    if (!courseCard) {
        console.error('Course card not found for ID:', courseId);
        showNotification('Error: Course not found', 'error');
        return;
    }
    
    const unenrollButton = courseCard.querySelector('.unenroll-btn');
    if (!unenrollButton) {
        console.error('Unenroll button not found for course:', courseId);
        showNotification('Error: Button not found', 'error');
        return;
    }
    
    // Visual feedback
    const originalHTML = unenrollButton.innerHTML;
    unenrollButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    unenrollButton.disabled = true;
    
    // Simulate API call
    fetch('api/unenroll-course.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            course_id: courseId,
            action: 'unenroll'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Success feedback
            unenrollButton.innerHTML = '<i class="bx bx-check"></i>';
            unenrollButton.style.background = '#10b981';
            unenrollButton.style.color = 'white';
            
            showNotification('Successfully unenrolled from course!', 'success');
            
            // Remove course card with animation
            setTimeout(() => {
                courseCard.style.animation = 'slideOut 0.5s ease forwards';
                setTimeout(() => {
                    courseCard.remove();
                    
                    // Check if no enrolled courses left
                    const remainingCourses = document.querySelectorAll('.enrolled-course-card');
                    if (remainingCourses.length === 0) {
                        location.reload(); // Reload to show empty state
                    }
                }, 500);
            }, 1000);
        } else {
            handleUnenrollmentError(unenrollButton, originalHTML, data.message);
        }
    })
    .catch(error => {
        console.error('Unenrollment error:', error);
        handleUnenrollmentError(unenrollButton, originalHTML, 'An error occurred. Please try again.');
    });
}

function deleteCourse(courseId) {
    // Confirm deletion
    if (!confirm('Are you sure you want to delete this course from your account? This will permanently remove all your progress and course data. This action cannot be undone.')) {
        return;
    }
    
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    if (!courseCard) {
        console.error('Course card not found for ID:', courseId);
        showNotification('Error: Course not found', 'error');
        return;
    }
    
    const deleteButton = courseCard.querySelector('.delete-btn');
    if (!deleteButton) {
        console.error('Delete button not found for course:', courseId);
        showNotification('Error: Button not found', 'error');
        return;
    }
    
    // Visual feedback
    const originalHTML = deleteButton.innerHTML;
    deleteButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    deleteButton.disabled = true;
    
    // Simulate API call
    fetch('api/delete-course-enrollment.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            course_id: courseId,
            action: 'delete'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Success feedback
            deleteButton.innerHTML = '<i class="bx bx-check"></i>';
            deleteButton.style.background = '#10b981';
            deleteButton.style.color = 'white';
            
            showNotification('Course deleted successfully!', 'success');
            
            // Remove course card with animation
            setTimeout(() => {
                courseCard.style.animation = 'slideOut 0.5s ease forwards';
                setTimeout(() => {
                    courseCard.remove();
                    
                    // Check if no enrolled courses left
                    const remainingCourses = document.querySelectorAll('.enrolled-course-card');
                    if (remainingCourses.length === 0) {
                        location.reload(); // Reload to show empty state
                    }
                }, 500);
            }, 1000);
        } else {
            handleDeletionError(deleteButton, originalHTML, data.message);
        }
    })
    .catch(error => {
        console.error('Deletion error:', error);
        handleDeletionError(deleteButton, originalHTML, 'An error occurred. Please try again.');
    });
}

function handleUnenrollmentError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    button.style.color = 'white';
    
    showNotification(message || 'Failed to unenroll from course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
        button.style.color = '';
    }, 2000);
}

function handleDeletionError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    button.style.color = 'white';
    
    showNotification(message || 'Failed to delete course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
        button.style.color = '';
    }, 2000);
}

// Add slide out animation for card removal
document.addEventListener('DOMContentLoaded', function() {
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideOut {
            0% {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
            100% {
                opacity: 0;
                transform: translateY(-20px) scale(0.9);
            }
        }
    `;
    document.head.appendChild(style);
});
</script>

<style>
