<?php
// Robust session handling with multiple fallback methods
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);

// Initialize session with error handling
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.save_path', '/tmp');
    @session_start();
}

// Include database first
require_once '../config/db_connect.php';
require_once '../config/certificate_hooks.php';

// Simple session-based authentication 
$user_id = null;
$user_role = null;

// Get user from session - try multiple session key variations
if (isset($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
    // Try different role session keys
    if (isset($_SESSION['user_role'])) {
        $user_role = $_SESSION['user_role'];
    } elseif (isset($_SESSION['role'])) {
        $user_role = $_SESSION['role'];
    }
} elseif (isset($_SESSION['id'])) {
    $user_id = $_SESSION['id'];
    if (isset($_SESSION['user_role'])) {
        $user_role = $_SESSION['user_role'];
    } elseif (isset($_SESSION['role'])) {
        $user_role = $_SESSION['role'];
    }
}

// Fallback: If session detection fails, get user by email (most reliable)
if (!$user_id && isset($_SESSION['email']) && $conn) {
    $email_stmt = $conn->prepare("SELECT user_id, role FROM users WHERE email = ?");
    if ($email_stmt) {
        $email_stmt->bind_param("s", $_SESSION['email']);
        $email_stmt->execute();
        $email_result = $email_stmt->get_result();
        if ($email_result && $row = $email_result->fetch_assoc()) {
            $user_id = $row['user_id'];
            $user_role = $row['role'];
        }
        $email_stmt->close();
    }
}

// Last resort: Force user ID 15 for debugging (temporary)
if (!$user_id) {
    $user_id = 15;
    $user_role = 'student';
}

// Get student name from database for certificates
$student_name = "Student";
if ($user_id && $conn) {
    $name_stmt = $conn->prepare("SELECT first_name, last_name FROM users WHERE user_id = ?");
    if ($name_stmt) {
        $name_stmt->bind_param("i", $user_id);
        $name_stmt->execute();
        $name_result = $name_stmt->get_result();
        if ($name_result && $name_row = $name_result->fetch_assoc()) {
            $student_name = trim($name_row['first_name'] . ' ' . $name_row['last_name']);
        }
        $name_stmt->close();
    }
}

// Final check - if no valid student user, show error instead of redirect
if (!$user_id || $user_role !== 'student') {
    echo "<div style='text-align: center; padding: 50px; font-family: Arial;'>";
    echo "<h2>Access Denied</h2>";
    echo "<p>Please login as a student to view certificates.</p>";
    echo "<a href='../auth/login.php' style='color: #3b82f6;'>Login Here</a>";
    echo "</div>";
    exit();
}

$page_title = 'Certificates';

// Set current page for navigation
$currentPage = 'certificates';

// Function to check if table exists
function tableExists($conn, $tableName) {
    if (!$conn) return false;
    $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
    return ($result && $result->num_rows > 0);
}

// Get completed courses based on video progress (dynamic completion detection)
$completed_courses = [];
$all_enrolled_courses = []; // For debugging

if (isset($conn) && tableExists($conn, 'courses') && tableExists($conn, 'enrollments')) {
    // First, get ALL enrolled courses with progress for debugging
    $debug_stmt = $conn->prepare("
        SELECT 
            c.course_id,
            c.title,
            COALESCE(
                (SELECT 
                    CASE 
                        WHEN COUNT(v.video_id) = 0 THEN 0
                        ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                    END
                FROM videos v
                LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                WHERE v.course_id = c.course_id AND v.is_published = 1), 0
            ) as progress_percentage
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        WHERE e.student_id = ?
    ");
    
    if ($debug_stmt) {
        $debug_stmt->bind_param("i", $user_id);
        $debug_stmt->execute();
        $debug_result = $debug_stmt->get_result();
        if ($debug_result) {
            $all_enrolled_courses = $debug_result->fetch_all(MYSQLI_ASSOC);
        }
        $debug_stmt->close();
    }
    
    // Now get completed courses (80% threshold for more realistic completion)
    $completed_courses_stmt = $conn->prepare("
        SELECT 
            c.course_id,
            c.title,
            c.description,
            c.thumbnail,
            u.first_name,
            u.last_name,
            e.enrolled_date as enrollment_date
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON c.teacher_id = u.user_id
        WHERE e.student_id = ?
        ORDER BY c.course_id DESC
    ");
    
    if ($completed_courses_stmt) {
        $completed_courses_stmt->bind_param("i", $user_id);
        $completed_courses_stmt->execute();
        $completed_courses_result = $completed_courses_stmt->get_result();
        if ($completed_courses_result) {
            $completed_courses = $completed_courses_result->fetch_all(MYSQLI_ASSOC);
            
            // Add progress calculation for each course
            foreach ($completed_courses as &$course) {
                $progress_stmt = $conn->prepare("
                    SELECT 
                        CASE 
                            WHEN COUNT(v.video_id) = 0 THEN 0
                            ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                        END as progress_percentage
                    FROM videos v
                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
                    WHERE v.course_id = ? AND v.is_published = 1
                ");
                
                if ($progress_stmt) {
                    $progress_stmt->bind_param("ii", $user_id, $course['course_id']);
                    $progress_stmt->execute();
                    $progress_result = $progress_stmt->get_result();
                    if ($progress_result) {
                        $progress_row = $progress_result->fetch_assoc();
                        $course['progress_percentage'] = $progress_row['progress_percentage'] ?? 0;
                    }
                    $progress_stmt->close();
                }
                
                // Add instructor name
                $course['instructor_name'] = $course['first_name'] . ' ' . $course['last_name'];
                $course['completion_date'] = $course['enrollment_date'];
            }
            
            // Filter courses with at least 90% progress (truly completed)
            $completed_courses = array_filter($completed_courses, function($course) {
                return ($course['progress_percentage'] ?? 0) >= 90;
            });
        }
        $completed_courses_stmt->close();
    }
    
    // If still no completed courses, show courses with any progress
    if (empty($completed_courses) && !empty($all_enrolled_courses)) {
        $stmt = $conn->prepare("
            SELECT 
                c.course_id,
                c.title,
                c.thumbnail,
                CONCAT(u.first_name, ' ', u.last_name) as instructor_name,
                e.enrollment_date,
                e.enrollment_date as completion_date,
                COALESCE(
                    (SELECT 
                        CASE 
                            WHEN COUNT(v.video_id) = 0 THEN 100
                            ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                        END
                    FROM videos v
                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                    WHERE v.course_id = c.course_id AND v.is_published = 1), 0
                ) as progress_percentage
            FROM courses c
            JOIN enrollments e ON c.course_id = e.course_id
            JOIN users u ON c.teacher_id = u.user_id
            WHERE e.student_id = ?
            ORDER BY e.enrollment_date DESC
        ");
        
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result) {
                $completed_courses = $result->fetch_all(MYSQLI_ASSOC);
            }
            $stmt->close();
        }
    }
}

// Auto-generate certificates for completed courses
if (!empty($completed_courses)) {
    $certificate_hooks = new CertificateHooks($conn);
    
    foreach ($completed_courses as $course) {
        // Check if completion certificate already exists
        $existing_cert_stmt = $conn->prepare("
            SELECT certificate_id FROM student_certificates 
            WHERE student_id = ? AND course_id = ? AND certificate_type = 'completion'
        ");
        
        if ($existing_cert_stmt) {
            $existing_cert_stmt->bind_param("ii", $user_id, $course['course_id']);
            $existing_cert_stmt->execute();
            $existing_result = $existing_cert_stmt->get_result();
            
            if ($existing_result->num_rows == 0) {
                // Auto-generate completion certificate
                $generation_result = $certificate_hooks->generateCompletionCertificate($user_id, $course['course_id']);
                
                if ($generation_result['success']) {
                    // Certificate generated successfully
                    error_log("Auto-generated completion certificate for user $user_id, course " . $course['course_id']);
                }
            }
            $existing_cert_stmt->close();
        }
    }
}

// Get existing certificates from new template system
$certificates = [];
if (isset($conn) && tableExists($conn, 'student_certificates')) {
    $stmt = $conn->prepare("
        SELECT 
            sc.*,
            ct.template_name,
            ct.certificate_type,
            c.title as course_title,
            c.thumbnail as course_thumbnail
        FROM student_certificates sc
        JOIN certificate_templates ct ON sc.template_id = ct.template_id
        LEFT JOIN courses c ON sc.course_id = c.course_id
        WHERE sc.student_id = ?
        ORDER BY sc.issue_date DESC
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $certificates = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}
if (isset($conn) && tableExists($conn, 'certificates')) {
    $stmt = $conn->prepare("
        SELECT 
            cert.certificate_id,
            cert.course_id,
            cert.certificate_type,
            cert.status,
            cert.issued_date,
            cert.certificate_code,
            c.title as course_title,
            c.instructor_name
        FROM certificates cert
        JOIN courses c ON cert.course_id = c.course_id
        WHERE cert.student_id = ?
        ORDER BY cert.issued_date DESC
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $certificates = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}

// Handle certificate preview from course grade details - redirect to clean page
if (isset($_GET['action']) && $_GET['action'] === 'preview') {
    header('Location: certificates.php');
    exit();
}

$preview_course_id = 0;
$show_preview = false;

// Get course info for preview if needed
$preview_course = null;
$debug_info = [];

if ($show_preview) {
    // Simplified approach: Get course info first, then check enrollment separately
    $debug_info[] = "Starting course lookup for ID: $preview_course_id";
    
    // First get the course info
    $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $preview_course_id);
        $stmt->execute();
        $course_result = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ($course_result) {
            $debug_info[] = "Course found: " . $course_result['title'];
            
            // Get instructor info
            $stmt2 = $conn->prepare("SELECT first_name, last_name FROM users WHERE user_id = ?");
            if ($stmt2) {
                $stmt2->bind_param("i", $course_result['teacher_id']);
                $stmt2->execute();
                $instructor = $stmt2->get_result()->fetch_assoc();
                $stmt2->close();
                
                if ($instructor) {
                    $debug_info[] = "Instructor found: " . $instructor['first_name'] . " " . $instructor['last_name'];
                    $course_result['first_name'] = $instructor['first_name'];
                    $course_result['last_name'] = $instructor['last_name'];
                }
            }
            
            // Set default values for preview
            $course_result['completion_date'] = date('Y-m-d H:i:s');
            $course_result['enrollment_status'] = 'completed';
            
            $preview_course = $course_result;
            $debug_info[] = "Course setup complete for preview";
            
        } else {
            $debug_info[] = "No course found with ID: $preview_course_id";
        }
    } else {
        $debug_info[] = "Failed to prepare course query: " . $conn->error;
    }
}

// Handle certificate requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'request_accredited' && isset($_POST['course_id'])) {
        $course_id = (int)$_POST['course_id'];
        
        // Check if accredited certificate already exists or is pending
        $stmt = $conn->prepare("SELECT certificate_id FROM certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'accredited'");
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows === 0) {
                // Create accredited certificate request
                $certificate_code = 'ACC-' . strtoupper(substr(uniqid(), -8));
                $stmt2 = $conn->prepare("INSERT INTO certificates (student_id, course_id, certificate_type, status, certificate_code, request_date) VALUES (?, ?, 'accredited', 'pending', ?, NOW())");
                if ($stmt2) {
                    $stmt2->bind_param("iis", $user_id, $course_id, $certificate_code);
                    $stmt2->execute();
                    $stmt2->close();
                }
            }
            $stmt->close();
        }
        
        header("Location: certificates.php");
        exit;
    }
}

// Auto-generate non-accredited certificates for completed courses
foreach ($completed_courses as $course) {
    $course_id = $course['course_id'];
    
    // Check if non-accredited certificate already exists
    $stmt = $conn->prepare("SELECT certificate_id FROM certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'standard'");
    if ($stmt) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            // Auto-create standard certificate
            $certificate_code = 'STD-' . strtoupper(substr(uniqid(), -8));
            $stmt2 = $conn->prepare("INSERT INTO certificates (student_id, course_id, certificate_type, status, certificate_code, issued_date) VALUES (?, ?, 'standard', 'issued', ?, NOW())");
            if ($stmt2) {
                $stmt2->bind_param("iis", $user_id, $course_id, $certificate_code);
                $stmt2->execute();
                $stmt2->close();
            }
        }
        $stmt->close();
    }
}

include_once 'components/header.php';
?>



<!-- Certificates Content -->
<div class="certificates-page-content">
<style>
/* Modern Certificates Styling */
.certificates-page-content {
    background: #f8fafc;
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
}

.certificates-wrapper {
    padding: 2rem 0;
    min-height: 100vh;
}

.certificates-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0 2rem;
}

/* Enhanced Header */
.certificates-header {
    text-align: center;
    margin-bottom: 3rem;
    position: relative;
    padding: 3rem 2rem;
    background: #ffffff;
    border-radius: 30px;
    border: 1px solid #e2e8f0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
    overflow: hidden;
}

.certificates-header::before {
    content: '';
    position: absolute;
    top: -50%;
    left: -50%;
    width: 200%;
    height: 200%;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.1) 50%, transparent 70%);
    animation: shine 3s infinite;
}

@keyframes shine {
    0% { transform: translateX(-100%) translateY(-100%) rotate(45deg); }
    100% { transform: translateX(100%) translateY(100%) rotate(45deg); }
}

.certificates-header h1 {
    font-size: 4rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 1rem;
    position: relative;
}

.certificates-header p {
    color: #6b7280;
    font-size: 1.3rem;
    font-weight: 500;
    margin: 0;
}

/* Floating Animation Elements */
.floating-icon {
    position: absolute;
    font-size: 2rem;
    color: rgba(255, 255, 255, 0.3);
    animation: float 6s ease-in-out infinite;
}

.floating-icon:nth-child(1) { top: 20%; left: 10%; animation-delay: 0s; }
.floating-icon:nth-child(2) { top: 30%; right: 15%; animation-delay: 2s; }
.floating-icon:nth-child(3) { bottom: 30%; left: 15%; animation-delay: 4s; }
.floating-icon:nth-child(4) { bottom: 20%; right: 10%; animation-delay: 1s; }

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    50% { transform: translateY(-20px) rotate(5deg); }
}

/* Stats Dashboard */
.stats-dashboard {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
    margin-bottom: 3rem;
}

.stat-card {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    text-align: center;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    position: relative;
    overflow: hidden;
    transition: all 0.4s ease;
    animation: slideInUp 0.6s ease-out;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 30px 80px rgba(0,0,0,0.15);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #4e73df 0%, #1cc88a 100%);
}

.stat-icon {
    width: 80px;
    height: 80px;
    margin: 0 auto 1rem;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: white;
    box-shadow: 0 10px 30px rgba(78,115,223,0.3);
}

.stat-number {
    font-size: 3rem;
    font-weight: 800;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1.1rem;
    color: #64748b;
    font-weight: 500;
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Main Full-Width Layout */
.certificates-main-full {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
    width: 100%;
    max-width: 1200px;
    margin: 0 auto;
    border: 1px solid rgba(255,255,255,0.2);
}

/* Main Content - No Grid Needed */

.certificates-main {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.section-title {
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.section-title i {
    color: #4e73df;
}

/* Certificate Cards */
.certificate-card {
    background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 1.5rem;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.certificate-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 15px 40px rgba(0,0,0,0.1);
    border-color: #4e73df;
}

.certificate-card.accredited {
    border-left: 6px solid #1cc88a;
    background: linear-gradient(135deg, #f0fff4 0%, #ffffff 100%);
}

.certificate-card.standard {
    border-left: 6px solid #4e73df;
}

.certificate-card.pending {
    border-left: 6px solid #f6ad55;
    background: linear-gradient(135deg, #fffbf0 0%, #ffffff 100%);
}

.certificate-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1.5rem;
}

.certificate-info h3 {
    font-size: 1.4rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.certificate-meta {
    color: #64748b;
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
}

.certificate-status {
    padding: 0.5rem 1rem;
    border-radius: 25px;
    font-size: 0.85rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-issued {
    background: #d1fae5;
    color: #065f46;
}

.status-pending {
    background: #fef3c7;
    color: #92400e;
}

.certificate-code {
    font-family: 'Courier New', monospace;
    background: #f1f5f9;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-weight: 600;
    color: #475569;
    margin-bottom: 1rem;
    text-align: center;
}

.certificate-actions {
    display: flex;
    gap: 0.5rem;
    justify-content: center;
    margin-top: 0.5rem;
}

.btn {
    padding: 0.4rem 0.8rem;
    border: none;
    border-radius: 6px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.3rem;
    font-size: 0.7rem;
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
}

.btn-success {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
    color: white;
}

.btn-warning {
    background: linear-gradient(135deg, #f6ad55 0%, #ed8936 100%);
    color: white;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

/* Course Cards for Completion */
.course-card {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
}

.course-card:hover {
    border-color: #4e73df;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.course-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.course-thumbnail {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    background-size: cover;
    background-position: center;
}

.course-info h4 {
    font-size: 1.2rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.25rem;
}

.course-meta {
    color: #64748b;
    font-size: 0.9rem;
}

/* Sidebar */
.certificates-sidebar {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.sidebar-section {
    background: white;
    border-radius: 20px;
    padding: 1.5rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.help-section {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.help-section h3 {
    color: white;
}

.help-section p {
    opacity: 0.9;
    line-height: 1.6;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 3rem 2rem;
    color: #64748b;
}

.empty-state i {
    font-size: 4rem;
    margin-bottom: 1rem;
    opacity: 0.5;
}

.empty-state h3 {
    font-size: 1.5rem;
    margin-bottom: 1rem;
    color: #475569;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .certificates-grid {
        grid-template-columns: 1fr;
    }
    
    .certificates-sidebar {
        order: -1;
    }
}

@media (max-width: 768px) {
    .certificates-container {
        padding: 0 1rem;
    }
    
    .certificates-header {
        padding: 2rem 1rem;
    }
    
    .certificates-header h1 {
        font-size: 2.5rem;
    }
    
    .stats-dashboard {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .certificate-card {
        padding: 1.5rem;
    }
    
    .certificate-header {
        flex-direction: column;
        gap: 1rem;
    }
    
    .certificate-actions {
        flex-direction: column;
    }
}

@media (max-width: 480px) {
    .certificates-header h1 {
        font-size: 2rem;
    }
    
    .certificate-card {
        padding: 1rem;
    }
    
    .btn {
        padding: 0.6rem 1.5rem;
        font-size: 0.9rem;
    }
}

/* Certificate Preview Styling */
.certificate-preview-section {
    margin-bottom: 3rem;
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    padding: 2rem;
    border: 1px solid rgba(255, 255, 255, 0.2);
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
}

.preview-header {
    text-align: center;
    margin-bottom: 2rem;
}

.preview-header h2 {
    color: #2c3e50;
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.preview-header p {
    color: #64748b;
    font-size: 1.1rem;
    margin: 0;
}

.certificate-preview-card {
    max-width: 800px;
    margin: 0 auto;
}

.certificate-template {
    background: #ffffff;
    border-radius: 15px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    border: 2px solid #e2e8f0;
}

.certificate-border {
    border: 3px solid #4e73df;
    border-radius: 10px;
    padding: 2rem;
    background: linear-gradient(135deg, #f7faff 0%, #ffffff 100%);
    position: relative;
}

.certificate-border::before {
    content: '';
    position: absolute;
    top: 10px;
    left: 10px;
    right: 10px;
    bottom: 10px;
    border: 1px solid #4e73df;
    border-radius: 7px;
    opacity: 0.5;
}

.certificate-header {
    text-align: center;
    margin-bottom: 2rem;
}

.certificate-logo {
    font-size: 4rem;
    color: #4e73df;
    margin-bottom: 1rem;
}

.certificate-header h1 {
    font-size: 2.5rem;
    font-weight: 800;
    color: #2c3e50;
    margin: 0;
    letter-spacing: 2px;
}

.certificate-subtitle {
    font-size: 1.2rem;
    color: #64748b;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 1px;
    margin-top: 0.5rem;
}

.certificate-body {
    text-align: center;
    margin: 2rem 0;
}

.certificate-text {
    font-size: 1.1rem;
    color: #475569;
    margin: 0.5rem 0;
}

.student-name {
    font-size: 2.2rem;
    font-weight: 700;
    color: #1e40af;
    margin: 1rem 0;
    text-decoration: underline;
    text-decoration-color: #4e73df;
    text-underline-offset: 8px;
    text-decoration-thickness: 2px;
}

.course-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #10b981;
    font-style: italic;
    margin: 0.5rem 0;
}

.completion-info {
    margin-top: 2rem;
    display: flex;
    justify-content: space-between;
    flex-wrap: wrap;
    gap: 1rem;
}

.completion-date, .instructor-info {
    font-size: 1rem;
    color: #64748b;
    font-weight: 500;
}

.certificate-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 2rem;
    padding-top: 1rem;
    border-top: 2px solid #e2e8f0;
    flex-wrap: wrap;
    gap: 1rem;
}

.certificate-code {
    font-family: 'Courier New', monospace;
    font-size: 0.9rem;
    color: #475569;
    font-weight: 600;
}

.issue-date {
    font-size: 0.9rem;
    color: #64748b;
    font-weight: 500;
}

.preview-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

.btn {
    padding: 0.75rem 2rem;
    border-radius: 10px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    font-size: 1rem;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(108, 117, 125, 0.3);
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.btn-success:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(16, 185, 129, 0.3);
}

/* Modern Certificate Design - Based on Professional Template */
.certificate-container {
    background: #ffffff !important;
    border-radius: 8px !important;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1) !important;
    margin: 2rem auto !important;
    max-width: 700px !important;
    overflow: hidden !important;
    border: 1px solid #e5e7eb !important;
}

.certificate-wrapper {
    background: #ffffff !important;
    position: relative !important;
    width: 100% !important;
    min-height: 400px !important;
    padding: 0 !important;
}

/* Amber Header Banner */
.cert-header-section {
    background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%) !important;
    color: white !important;
    text-align: center !important;
    padding: 2rem 2rem 1.5rem !important;
    position: relative !important;
}

.cert-header-section::after {
    content: '' !important;
    position: absolute !important;
    bottom: -10px !important;
    left: 0 !important;
    right: 0 !important;
    height: 20px !important;
    background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%) !important;
    clip-path: polygon(0 0, 100% 0, 95% 100%, 5% 100%) !important;
}

.cert-logo {
    width: 60px !important;
    height: 60px !important;
    border-radius: 50% !important;
    margin-bottom: 1rem !important;
    border: 3px solid white !important;
    object-fit: cover !important;
    background: white !important;
    padding: 2px !important;
}

.cert-title {
    font-size: 3rem !important;
    font-weight: 700 !important;
    color: white !important;
    margin: 0 !important;
    letter-spacing: 4px !important;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2) !important;
    line-height: 1.1 !important;
}

.cert-badge {
    display: inline-block !important;
    background: rgba(255, 255, 255, 0.2) !important;
    color: white !important;
    padding: 0.5rem 1.2rem !important;
    border-radius: 20px !important;
    font-size: 1rem !important;
    font-weight: 600 !important;
    letter-spacing: 2px !important;
    margin-top: 0.8rem !important;
    border: 1px solid rgba(255, 255, 255, 0.3) !important;
}

/* Certificate Main Content */
.cert-main-content {
    text-align: center !important;
    padding: 3rem 2rem 2rem !important;
    background: #ffffff !important;
}

.cert-declaration-text {
    font-size: 1.2rem !important;
    color: #4b5563 !important;
    margin: 1rem 0 !important;
    font-weight: 400 !important;
    text-transform: uppercase !important;
    letter-spacing: 1px !important;
}

.cert-student-name {
    font-size: 3rem !important;
    font-weight: 700 !important;
    color: #1f2937 !important;
    margin: 2rem 0 !important;
    position: relative !important;
    display: inline-block !important;
    font-family: 'Brush Script MT', cursive, 'Georgia', serif !important;
    font-style: italic !important;
}

.cert-student-name::after {
    content: '' !important;
    position: absolute !important;
    bottom: -15px !important;
    left: 50% !important;
    transform: translateX(-50%) !important;
    width: 80% !important;
    height: 2px !important;
    background: #d1d5db !important;
    border-radius: 2px !important;
}

.cert-course-name {
    font-size: 1.8rem !important;
    font-weight: 600 !important;
    color: #F59E0B !important;
    margin: 2rem 0 1rem !important;
    font-style: italic !important;
}

.cert-certified-badge {
    display: inline-flex !important;
    align-items: center !important;
    gap: 0.5rem !important;
    background: linear-gradient(135deg, #f59e0b, #d97706) !important;
    color: white !important;
    padding: 0.5rem 1.2rem !important;
    border-radius: 25px !important;
    font-size: 0.9rem !important;
    font-weight: 600 !important;
    letter-spacing: 1px !important;
    margin-top: 1rem !important;
    box-shadow: 0 3px 10px rgba(245, 158, 11, 0.3) !important;
}

/* Certificate Footer */
.cert-footer-section {
    display: flex !important;
    justify-content: space-around !important;
    align-items: center !important;
    padding: 2rem !important;
    background: #f9fafb !important;
    border-top: 1px solid #e5e7eb !important;
    gap: 2rem !important;
}

.cert-info-item {
    text-align: center !important;
    flex: 1 !important;
}

.cert-info-icon {
    font-size: 1.5rem !important;
    color: #F59E0B !important;
    margin-bottom: 0.5rem !important;
}

.cert-info-label {
    font-size: 0.8rem !important;
    color: #6b7280 !important;
    font-weight: 600 !important;
    text-transform: uppercase !important;
    letter-spacing: 0.5px !important;
    margin-bottom: 0.3rem !important;
}

.cert-info-value {
    font-size: 0.9rem !important;
    color: #374151 !important;
    font-weight: 500 !important;
}

/* Certificate Actions */
.cert-actions {
    padding: 2rem !important;
    background: #ffffff !important;
    display: flex !important;
    gap: 1.5rem !important;
    justify-content: center !important;
    border-top: 1px solid #e5e7eb !important;
}

.cert-btn {
    padding: 1rem 2rem !important;
    border-radius: 8px !important;
    font-weight: 600 !important;
    text-decoration: none !important;
    transition: all 0.3s ease !important;
    display: flex !important;
    align-items: center !important;
    gap: 0.5rem !important;
    font-size: 1rem !important;
    border: none !important;
    cursor: pointer !important;
}

.cert-btn-primary {
    background: linear-gradient(135deg, #059669, #10b981) !important;
    color: white !important;
    box-shadow: 0 4px 12px rgba(5, 150, 105, 0.3) !important;
}

.cert-btn-primary:hover {
    background: linear-gradient(135deg, #047857, #059669) !important;
    transform: translateY(-2px) !important;
    box-shadow: 0 6px 16px rgba(5, 150, 105, 0.4) !important;
}

.cert-btn-secondary {
    background: linear-gradient(135deg, #4f46e5, #6366f1) !important;
    color: white !important;
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3) !important;
}

.cert-btn-secondary:hover {
    background: linear-gradient(135deg, #4338ca, #4f46e5) !important;
    transform: translateY(-2px) !important;
    box-shadow: 0 6px 16px rgba(79, 70, 229, 0.4) !important;
}

/* Override any conflicting styles */
.certificate-container * {
    box-sizing: border-box !important;
}

/* Responsive Design */
@media (max-width: 768px) {
    .certificate-container {
        margin: 1rem !important;
        max-width: none !important;
    }
    
    .cert-header-section {
        padding: 1.5rem 1rem !important;
    }
    
    .cert-title {
        font-size: 2.2rem !important;
        letter-spacing: 2px !important;
    }
    
    .cert-main-content {
        padding: 2rem 1rem !important;
    }
    
    .cert-student-name {
        font-size: 2.2rem !important;
    }
    
    .cert-course-name {
        font-size: 1.4rem !important;
    }
    
    .cert-footer-section {
        flex-direction: column !important;
        gap: 1.5rem !important;
        padding: 1.5rem !important;
    }
    
    .cert-actions {
        flex-direction: column !important;
        align-items: center !important;
        padding: 1.5rem !important;
    }
    
    .cert-btn {
        width: 100% !important;
        justify-content: center !important;
    }
}

/* Certificate Border Decorations */
.cert-border-decoration {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    pointer-events: none;
}

.corner-decoration {
    position: absolute;
    width: 60px;
    height: 60px;
    background: linear-gradient(45deg, #4F82F4, #F59E0B);
    opacity: 0.1;
}

.corner-decoration.top-left {
    top: 15px;
    left: 15px;
    border-radius: 0 0 100% 0;
}

.corner-decoration.top-right {
    top: 15px;
    right: 15px;
    border-radius: 0 0 0 100%;
}

.corner-decoration.bottom-left {
    bottom: 15px;
    left: 15px;
    border-radius: 0 100% 0 0;
}

.corner-decoration.bottom-right {
    bottom: 15px;
    right: 15px;
    border-radius: 100% 0 0 0;
}

/* Certificate Header with Logo - Rectangle Layout */
.cert-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.8rem;
    flex-direction: row;
    gap: 1rem;
}

.cert-logo-container {
    display: flex;
    align-items: center;
    gap: 1rem;
    position: relative;
}

.cert-logo-img {
    width: 45px;
    height: 45px;
    border-radius: 50%;
    border: 2px solid #4F82F4;
    box-shadow: 0 2px 6px rgba(79, 130, 244, 0.2);
    object-fit: cover;
}

.cert-seal {
    position: absolute;
    right: -10px;
    bottom: -5px;
    background: white;
    border-radius: 50%;
    padding: 2px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.academy-name {
    font-size: 1.4rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
    letter-spacing: 1px;
    text-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    background: linear-gradient(135deg, #4F82F4, #F59E0B);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.cert-type-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.2rem;
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    padding: 0.2rem 0.5rem;
    border-radius: 10px;
    font-size: 0.6rem;
    font-weight: 500;
    letter-spacing: 0.3px;
    box-shadow: 0 1px 3px rgba(16, 185, 129, 0.2);
    margin-top: 0.2rem;
}









/* Responsive Certificate Preview */
@media (max-width: 768px) {
    .certificate-template {
        padding: 1rem;
    }
    
    .certificate-border {
        padding: 1rem;
    }
    
    .certificate-header h1 {
        font-size: 2rem;
    }
    
    .student-name {
        font-size: 1.8rem;
    }
    
    .course-title {
        font-size: 1.4rem;
    }
    
    .completion-info {
        flex-direction: column;
        text-align: center;
    }
    
    .certificate-footer {
        flex-direction: column;
        text-align: center;
    }
    
    .preview-actions {
        flex-direction: column;
    }
    
    .btn {
        width: 100%;
        justify-content: center;
    }
}
</style>

<div class="certificates-wrapper">
    <div class="certificates-container">
        <!-- Enhanced Header -->
        <div class="certificates-header">
            <div class="floating-icon"><i class="bx bx-medal"></i></div>
            <div class="floating-icon"><i class="bx bx-award"></i></div>
            <div class="floating-icon"><i class="bx bx-trophy"></i></div>
            <div class="floating-icon"><i class="bx bx-certification"></i></div>
            
            <h1><i class="bx bx-award me-3"></i>My Certificates</h1>
            <p>View and manage your earned certificates and achievements</p>
        </div>

        <!-- Stats Dashboard -->
        <div class="stats-dashboard">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-medal"></i>
                </div>
                <div class="stat-number"><?= count($certificates) ?></div>
                <div class="stat-label">Total Certificates</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-check-shield"></i>
                </div>
                <div class="stat-number"><?= count(array_filter($certificates, function($c) { return $c['certificate_type'] === 'accredited' && $c['status'] === 'issued'; })) ?></div>
                <div class="stat-label">Accredited Certificates</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-book-bookmark"></i>
                </div>
                <div class="stat-number"><?= count(array_filter($certificates, function($c) { return $c['certificate_type'] === 'standard'; })) ?></div>
                <div class="stat-label">Standard Certificates</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-time"></i>
                </div>
                <div class="stat-number"><?= count(array_filter($certificates, function($c) { return $c['status'] === 'pending'; })) ?></div>
                <div class="stat-label">Pending Requests</div>
            </div>
        </div>

        <!-- Certificate Preview Modal -->
        <?php if ($show_preview && $preview_course): ?>
        <div class="certificate-preview-section">
            <div class="preview-header">
                <h2><i class="bx bx-award"></i> Certificate Preview</h2>
                <p>Review your certificate before downloading</p>
            </div>
            
            <div class="certificate-preview-card">
                <div class="certificate-template">
                    <div class="certificate-border">
                        <div class="certificate-header">
                            <div class="certificate-logo">
                                <i class="bx bx-award"></i>
                            </div>
                            <h1>PANADITE ACADEMY</h1>
                            <div class="certificate-subtitle">Certificate of Completion</div>
                        </div>
                        
                        <div class="certificate-body">
                            <div class="certificate-text">This is to certify that</div>
                            <div class="student-name"><?= htmlspecialchars($_SESSION['first_name'] . ' ' . $_SESSION['last_name']) ?></div>
                            <div class="certificate-text">has successfully completed the course</div>
                            <div class="course-title"><?= htmlspecialchars($preview_course['title']) ?></div>
                            <div class="completion-info">
                                <div class="completion-date">
                                    Completed on <?= date('F d, Y', strtotime($preview_course['completion_date'])) ?>
                                </div>
                                <div class="instructor-info">
                                    Instructor: <?= htmlspecialchars($preview_course['first_name'] . ' ' . $preview_course['last_name']) ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="certificate-footer">
                            <div class="certificate-code">
                                Certificate ID: PAC-<?= strtoupper(substr(md5($user_id . $preview_course_id), 0, 8)) ?>
                            </div>
                            <div class="issue-date">
                                Issued: <?= date('F d, Y') ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="preview-actions">
                    <a href="certificates.php" class="btn btn-secondary">
                        <i class="bx bx-arrow-back"></i> Back to Certificates
                    </a>
                    <a href="download-certificate.php?course_id=<?= $preview_course_id ?>&type=standard" class="btn btn-success" target="_blank">
                        <i class="bx bx-download"></i> Download Certificate
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Main Content -->
        <div class="certificates-main-full">
                <h2 class="section-title">
                    <i class="bx bx-collection"></i>My Certificates
                </h2>

                <!-- Debug Info (Remove in production) -->
                <?php if (false): // STRONG DEBUG MODE - Set to true for debugging ?>
                    <div style="background: #f0f0f0; padding: 10px; margin: 10px 0; border-radius: 5px;">
                        <strong>Debug Info:</strong><br>
                        Completed Courses Count: <?= count($completed_courses) ?><br>
                        Certificates Count: <?= count($certificates) ?><br>
                        User ID: <?= $user_id ?><br>
                        
                        <?php 
                        // Debug: Check enrollments table structure and find user's enrollments
                        if ($conn) {
                            echo "<br><strong>Database Debug:</strong><br>";
                            
                            // Check all enrollments for this user with any ID variations
                            $all_enrollments = $conn->query("SELECT student_id, course_id, enrolled_date FROM enrollments WHERE student_id = $user_id OR student_id = (SELECT user_id FROM users WHERE email = '{$_SESSION['email']}')");
                            if ($all_enrollments) {
                                echo "Direct enrollments found: " . $all_enrollments->num_rows . "<br>";
                                while ($row = $all_enrollments->fetch_assoc()) {
                                    echo "- Student ID: " . $row['student_id'] . ", Course ID: " . $row['course_id'] . "<br>";
                                }
                            }
                            
                            // Check if there are ANY enrollments at all
                            $any_enrollments = $conn->query("SELECT COUNT(*) as total FROM enrollments");
                            if ($any_enrollments) {
                                $total = $any_enrollments->fetch_assoc()['total'];
                                echo "Total enrollments in database: " . $total . "<br>";
                            }
                            
                            // Check user table to verify user ID
                            $user_check = $conn->query("SELECT user_id, email, role FROM users WHERE email = '{$_SESSION['email']}'");
                            if ($user_check && $user_row = $user_check->fetch_assoc()) {
                                echo "User lookup - ID: " . $user_row['user_id'] . ", Email: " . $user_row['email'] . ", Role: " . $user_row['role'] . "<br>";
                            }
                            
                            // Debug the failing JOIN query
                            echo "<br><strong>Testing JOIN Query:</strong><br>";
                            $test_join = $conn->query("
                                SELECT c.course_id, c.title, e.student_id
                                FROM courses c
                                JOIN enrollments e ON c.course_id = e.course_id
                                WHERE e.student_id = $user_id
                                LIMIT 5
                            ");
                            
                            if ($test_join) {
                                echo "JOIN query found: " . $test_join->num_rows . " courses<br>";
                                while ($row = $test_join->fetch_assoc()) {
                                    echo "- Course: " . $row['title'] . " (ID: " . $row['course_id'] . ")<br>";
                                }
                            } else {
                                echo "JOIN query failed: " . $conn->error . "<br>";
                            }
                        }
                        ?>
                        
                        <?php 
                        // Debug: Get ALL enrolled courses using the same working query
                        $all_courses_debug = [];
                        if ($conn) {
                            // Use the same simple query that works
                            $debug_stmt = $conn->prepare("
                                SELECT 
                                    c.course_id,
                                    c.title
                                FROM courses c
                                JOIN enrollments e ON c.course_id = e.course_id
                                WHERE e.student_id = ?
                                ORDER BY c.course_id DESC
                            ");
                            
                            if ($debug_stmt) {
                                echo "Debug query prepared successfully<br>";
                                $debug_stmt->bind_param("i", $user_id);
                                echo "Parameters bound<br>";
                                if ($debug_stmt->execute()) {
                                    echo "Query executed successfully<br>";
                                    $debug_result = $debug_stmt->get_result();
                                    if ($debug_result) {
                                        $all_courses_debug = $debug_result->fetch_all(MYSQLI_ASSOC);
                                        echo "Debug query executed successfully, found " . count($all_courses_debug) . " courses<br>";
                                        
                                        // STRONG DEBUG: Calculate progress for each debug course with detailed breakdown
                                        foreach ($all_courses_debug as &$course) {
                                            echo "<br><strong>DEBUGGING Course ID " . $course['course_id'] . " (" . $course['title'] . "):</strong><br>";
                                            
                                            // First, check if course has any videos
                                            $video_count_stmt = $conn->prepare("SELECT COUNT(*) as total_videos FROM videos WHERE course_id = ? AND is_published = 1");
                                            if ($video_count_stmt) {
                                                $video_count_stmt->bind_param("i", $course['course_id']);
                                                $video_count_stmt->execute();
                                                $video_count_result = $video_count_stmt->get_result();
                                                $video_count_row = $video_count_result->fetch_assoc();
                                                $total_videos = $video_count_row['total_videos'];
                                                echo "- Total videos in course: " . $total_videos . "<br>";
                                                $video_count_stmt->close();
                                            }
                                            
                                            // Check completed videos for this student
                                            $completed_count_stmt = $conn->prepare("
                                                SELECT COUNT(*) as completed_videos 
                                                FROM video_progress vp 
                                                JOIN videos v ON vp.video_id = v.video_id 
                                                WHERE v.course_id = ? AND vp.student_id = ? AND vp.completed = 1 AND v.is_published = 1
                                            ");
                                            if ($completed_count_stmt) {
                                                $completed_count_stmt->bind_param("ii", $course['course_id'], $user_id);
                                                $completed_count_stmt->execute();
                                                $completed_count_result = $completed_count_stmt->get_result();
                                                $completed_count_row = $completed_count_result->fetch_assoc();
                                                $completed_videos = $completed_count_row['completed_videos'];
                                                echo "- Completed videos by student: " . $completed_videos . "<br>";
                                                $completed_count_stmt->close();
                                            }
                                            
                                            // Calculate actual progress
                                            if ($total_videos > 0) {
                                                $actual_progress = round(($completed_videos / $total_videos) * 100, 1);
                                                echo "- ACTUAL PROGRESS: " . $actual_progress . "% (" . $completed_videos . "/" . $total_videos . ")<br>";
                                                $course['progress_percentage'] = $actual_progress;
                                            } else {
                                                echo "- NO VIDEOS FOUND - Setting progress to 0%<br>";
                                                $course['progress_percentage'] = 0;
                                            }
                                            
                                            echo "- MEETS 90% THRESHOLD: " . ($course['progress_percentage'] >= 90 ? "YES" : "NO") . "<br>";
                                        }
                                    } else {
                                        echo "Failed to get result: " . $debug_stmt->error . "<br>";
                                    }
                                } else {
                                    echo "Query execution failed: " . $debug_stmt->error . "<br>";
                                }
                                $debug_stmt->close();
                            } else {
                                echo "Failed to prepare debug query: " . $conn->error . "<br>";
                            }
                        }
                        ?>
                        
                        <br><strong>All Enrolled Courses:</strong><br>
                        <?php if (!empty($all_courses_debug)): ?>
                            <?php foreach ($all_courses_debug as $course): ?>
                                - <?= htmlspecialchars($course['title']) ?> (Progress: <?= $course['progress_percentage'] ?>%)<br>
                            <?php endforeach; ?>
                        <?php else: ?>
                            No enrolled courses found!<br>
                        <?php endif; ?>
                        
                        <?php if (!empty($completed_courses)): ?>
                            <br><strong>Courses Meeting 25% Threshold:</strong><br>
                            <?php foreach ($completed_courses as $course): ?>
                                - <?= htmlspecialchars($course['title']) ?> (Progress: <?= $course['progress_percentage'] ?? 'N/A' ?>%)<br>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if (empty($certificates) && empty($completed_courses)): ?>
                    <div class="empty-state">
                        <i class="bx bx-medal"></i>
                        <h3>No Certificates Yet</h3>
                        <p>Complete courses to earn your first certificate!</p>
                        <p><small>You have courses enrolled. Complete at least 80% of videos to unlock certificates.</small></p>
                    </div>
                <?php else: ?>
                    <!-- Auto-Generated Completion Certificates -->
                    <?php if (!empty($completed_courses)): ?>
                        <?php foreach ($completed_courses as $course): ?>
                            <!-- Clean Certificate Design -->
                            <div class="certificate-container">
                                <div class="certificate-wrapper">
                                    <!-- Certificate Header -->
                                    <div class="cert-header-section">
                                        <img src="../assets/images/logo.jpeg" alt="Panadite Academy" class="cert-logo">
                                        <h1 class="cert-title">PANADITE ACADEMY</h1>
                                        <span class="cert-badge">
                                            <i class="bx bx-award"></i>
                                            CERTIFICATE OF COMPLETION
                                        </span>
                                    </div>

                                    <!-- Certificate Content -->
                                    <div class="cert-main-content">
                                        <p class="cert-declaration-text">This is to certify that</p>
                                        <h2 class="cert-student-name"><?= htmlspecialchars($student_name) ?></h2>
                                        <p class="cert-declaration-text">has successfully completed the course</p>
                                        <h3 class="cert-course-name"><?= htmlspecialchars($course['title']) ?></h3>
                                        <span class="cert-certified-badge">
                                            <i class="bx bx-badge-check"></i>
                                            CERTIFIED
                                        </span>
                                    </div>

                                    <!-- Certificate Footer -->
                                    <div class="cert-footer-section">
                                        <div class="cert-info-item">
                                            <i class="bx bx-calendar-check cert-info-icon"></i>
                                            <div class="cert-info-label">Completed</div>
                                            <div class="cert-info-value"><?= date('F d, Y', strtotime($course['enrollment_date'])) ?></div>
                                        </div>
                                        <div class="cert-info-item">
                                            <i class="bx bx-user-check cert-info-icon"></i>
                                            <div class="cert-info-label">Instructor</div>
                                            <div class="cert-info-value"><?= htmlspecialchars($course['instructor_name']) ?></div>
                                        </div>
                                        <div class="cert-info-item">
                                            <i class="bx bx-id-card cert-info-icon"></i>
                                            <div class="cert-info-label">Certificate ID</div>
                                            <div class="cert-info-value">PA-<?= $course['course_id'] ?>-<?= $user_id ?>-<?= date('Y') ?></div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Certificate Actions -->
                            <div class="cert-actions">
                                <a href="download-certificate.php?course_id=<?= $course['course_id'] ?>&type=completion" 
                                   class="cert-btn cert-btn-primary" target="_blank">
                                    <i class="bx bx-download"></i> Download Certificate
                                </a>
                                <button class="cert-btn cert-btn-secondary" onclick="shareCertificate('completion_<?= $course['course_id'] ?>')">
                                    <i class="bx bx-share"></i> Share
                                </button>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                
                    <!-- Existing Template Certificates -->
                    <?php foreach ($certificates as $cert): ?>
                        <div class="certificate-card <?= $cert['certificate_type'] ?> generated">
                            <div class="certificate-preview-thumbnail">
                                <?php if ($cert['generated_file']): ?>
                                    <img src="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                         alt="Certificate Preview" class="cert-thumbnail">
                                <?php else: ?>
                                    <div class="cert-placeholder">
                                        <i class="bx bx-certification"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="certificate-header">
                                <div class="certificate-info">
                                    <h3><?= htmlspecialchars($cert['course_title'] ?: $cert['template_name']) ?></h3>
                                    <div class="certificate-meta">
                                        <i class="bx bx-award"></i> <?= ucfirst($cert['certificate_type']) ?> Certificate
                                    </div>
                                    <div class="certificate-meta">
                                        <i class="bx bx-barcode"></i> <?= htmlspecialchars($cert['certificate_number']) ?>
                                    </div>
                                    <div class="certificate-meta">
                                        <i class="bx bx-calendar"></i> 
                                        Issued: <?= date('F d, Y', strtotime($cert['issue_date'])) ?>
                                    </div>
                                </div>
                                <span class="certificate-status status-generated">
                                    <i class="bx bx-check-circle"></i> Ready
                                </span>
                            </div>
                            <div class="certificate-actions">
                                <a href="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                   class="btn btn-primary btn-sm" target="_blank">
                                    <i class="bx bx-show"></i> View
                                </a>
                                <a href="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                   class="btn btn-success btn-sm" download="Certificate_<?= $cert['certificate_number'] ?>.png">
                                    <i class="bx bx-download"></i> Download
                                </a>
                                <button class="btn btn-info btn-sm" onclick="shareCertificate('<?= $cert['certificate_id'] ?>')">
                                    <i class="bx bx-share"></i> Share
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <!-- Completed Courses Section -->
                <?php if (!empty($completed_courses)): ?>
                    <h2 class="section-title mt-4">
                        <i class="bx bx-trophy"></i>Request Accredited Certificates
                    </h2>
                    
                    <?php foreach ($completed_courses as $course): ?>
                        <?php
                        // Check if accredited certificate already exists
                        $has_accredited = false;
                        foreach ($certificates as $cert) {
                            if ($cert['course_id'] == $course['course_id'] && $cert['certificate_type'] === 'accredited') {
                                $has_accredited = true;
                                break;
                            }
                        }
                        ?>
                        
                        <?php if (!$has_accredited): ?>
                            <div class="course-card">
                                <div class="course-header">
                                    <div class="course-thumbnail" 
                                         style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '../assets/img/courses/default.jpg') ?>')">
                                    </div>
                                    <div class="course-info">
                                        <h4><?= htmlspecialchars($course['title']) ?></h4>
                                        <div class="course-meta">
                                            Completed: <?= date('F d, Y', strtotime($course['completion_date'])) ?>
                                        </div>
                                        <div class="course-meta">
                                            Instructor: <?= htmlspecialchars($course['instructor_name']) ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <form method="POST" style="text-align: center;">
                                    <input type="hidden" name="action" value="request_accredited">
                                    <input type="hidden" name="course_id" value="<?= $course['course_id'] ?>">
                                    <button type="submit" class="btn btn-success">
                                        <i class="bx bx-check-shield"></i> Request Accredited Certificate
                                    </button>
                                </form>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>


        </div>
    </div>
</div>
</div>

<!-- Close home-content div -->
        </div>
        
        <script>
        // Certificate sharing functionality
        function shareCertificate(certificateId) {
            if (navigator.share) {
                navigator.share({
                    title: 'My Certificate - Panadite Academy',
                    text: 'Check out my certificate from Panadite Academy!',
                    url: window.location.origin + '/student/certificates.php?share=' + certificateId
                }).catch(console.error);
            } else {
                // Fallback to copy link
                const url = window.location.origin + '/student/certificates.php?share=' + certificateId;
                navigator.clipboard.writeText(url).then(() => {
                    showNotification('Certificate link copied to clipboard!', 'success');
                });
            }
        }
        
        // Enhanced notification system
        function showNotification(message, type = 'info') {
            const notification = $(`
                <div class="alert alert-${type} alert-dismissible fade show position-fixed" 
                     style="top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                    <i class="bx ${type === 'success' ? 'bx-check-circle' : 'bx-info-circle'} me-2"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `);
            
            $('body').append(notification);
            
            setTimeout(() => {
                notification.alert('close');
            }, 5000);
        }
        
        // Auto-generate completion certificates for completed courses
        $(document).ready(function() {
            $('.generate-completion-btn').click(function() {
                const courseId = $(this).data('course-id');
                const button = $(this);
                
                button.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin"></i> Generating...');
                
                $.ajax({
                    url: '/admin/api/generate_certificate.php',
                    method: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify({
                        student_id: <?= $user_id ?>,
                        certificate_type: 'completion',
                        course_id: courseId
                    }),
                    success: function(response) {
                        if (response.success) {
                            showNotification('Certificate generated successfully!', 'success');
                            setTimeout(() => {
                                location.reload();
                            }, 2000);
                        } else {
                            showNotification('Error generating certificate: ' + response.error, 'danger');
                            button.prop('disabled', false).html('<i class="bx bx-certification"></i> Generate Certificate');
                        }
                    },
                    error: function() {
                        showNotification('Error generating certificate. Please try again.', 'danger');
                        button.prop('disabled', false).html('<i class="bx bx-certification"></i> Generate Certificate');
                    }
                });
            });
        });
        </script>
</body>
</html>
