<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');
require_once '../config/db_connect.php';
require_once '../includes/course_progress_service_quiz_only.php';

$page_title = 'Course Detail';
$user_id = $_SESSION['user_id'];

// Initialize progress service for comprehensive course completion checking
$progressService = new QuizOnlyProgressService($conn);

// Decrypt the course ID from URL
$course_id = getSecureId('id');

// Redirect if invalid or missing encrypted ID
if (!$course_id) {
    header('Location: courses.php?error=invalid_course');
    exit();
}

// Get course information
$course = null;
if (isset($conn) && $course_id > 0) {
    $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        $stmt->close();
    }
}

if (!$course) {
    header("Location: courses.php");
    exit;
}

// Check enrollment
$is_enrolled = false;
$enrollment_data = null;
if (isset($conn)) {
    $stmt = $conn->prepare("SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?");
    if ($stmt) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $enrollment_data = $result->fetch_assoc();
        $is_enrolled = (bool)$enrollment_data;
        $stmt->close();
    }
}

// Get course lessons/videos from database
$lessons = [];

// First try to get real videos from the videos table
if (isset($conn)) {
    $video_query = "
        SELECT v.*, 
               COALESCE(vp.completion_percentage, 0) as watch_progress,
               CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
               CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
        FROM videos v
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
        WHERE v.course_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
    ";
    
    if ($stmt = $conn->prepare($video_query)) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            // Handle both file uploads and URL videos
            if (!empty($row['external_url'])) {
                // URL video - convert to proper embed format
                $external_url = $row['external_url'];
                
                // Convert YouTube URLs to embed format
                if (strpos($external_url, 'youtube.com/watch') !== false) {
                    // Convert: https://www.youtube.com/watch?v=VIDEO_ID to https://www.youtube.com/embed/VIDEO_ID
                    parse_str(parse_url($external_url, PHP_URL_QUERY), $query);
                    if (isset($query['v'])) {
                        $row['video_url'] = 'https://www.youtube.com/embed/' . $query['v'];
                    } else {
                        $row['video_url'] = $external_url;
                    }
                } elseif (strpos($external_url, 'youtu.be') !== false) {
                    // Convert: https://youtu.be/VIDEO_ID to https://www.youtube.com/embed/VIDEO_ID
                    $video_id = basename(parse_url($external_url, PHP_URL_PATH));
                    $row['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
                } elseif (strpos($external_url, 'youtube.com/embed') !== false) {
                    // Already in embed format
                    $row['video_url'] = $external_url;
                } else {
                    // Other external URLs (Vimeo, etc.)
                    $row['video_url'] = $external_url;
                }
                
                $row['is_external'] = true;
            } elseif (!empty($row['video_url'])) {
                // File upload - fix video URL path for web access
                if (strpos($row['video_url'], 'http') === 0) {
                    $row['video_url'] = $row['video_url'];
                } else {
                    // Convert relative path to full URL
                    $row['video_url'] = '/' . ltrim($row['video_url'], './');
                }
                $row['is_external'] = false;
            } else {
                // No video source available
                $row['video_url'] = '';
                $row['is_external'] = false;
            }
            
            // Fix thumbnail path for web access
            if (!empty($row['thumbnail'])) {
                // If it's already a full URL (starts with http), keep it
                if (strpos($row['thumbnail'], 'http') === 0) {
                    $row['thumbnail'] = $row['thumbnail'];
                } else {
                    // Convert relative path to full URL
                    $row['thumbnail'] = '/' . ltrim($row['thumbnail'], './');
                }
            } else {
                // Set default placeholder thumbnail if none exists
                $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
            }
            
            // Check for related content
            $row['has_assignment'] = false;
            $row['has_quiz'] = false;
            $row['has_materials'] = false;
            
            // Check assignments for this course
            $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE course_id = ?");
            if ($assignment_check) {
                $assignment_check->bind_param("i", $course_id);
                $assignment_check->execute();
                $assignment_result = $assignment_check->get_result();
                $assignment_data = $assignment_result->fetch_assoc();
                $row['has_assignment'] = $assignment_data['count'] > 0;
                $assignment_check->close();
            }
            
            // Check quizzes for this course
            $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
            if ($quiz_check) {
                $quiz_check->bind_param("i", $course_id);
                $quiz_check->execute();
                $quiz_result = $quiz_check->get_result();
                $quiz_data = $quiz_result->fetch_assoc();
                $row['has_quiz'] = $quiz_data['count'] > 0;
                $quiz_check->close();
            }
            
            // Check materials for this course
            $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE course_id = ?");
            if ($material_check) {
                $material_check->bind_param("i", $course_id);
                $material_check->execute();
                $material_result = $material_check->get_result();
                $material_data = $material_result->fetch_assoc();
                $row['has_materials'] = $material_data['count'] > 0;
                $material_check->close();
            }
            
            $row['lesson_id'] = $row['video_id']; // For compatibility
            $lessons[] = $row;
        }
        $stmt->close();
    }
}

// If no videos found, create structured lessons based on available course content
if (empty($lessons)) {
    // Get real assignments for this course
    $assignments = [];
    if (isset($conn)) {
        $assignment_query = "SELECT title, assignment_id, due_date, points FROM assignments WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($assignment_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $assignments[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real materials for this course
    $materials = [];
    if (isset($conn)) {
        $materials_query = "SELECT title, material_id, category FROM course_materials WHERE course_id = ? ORDER BY date_added ASC";
        if ($stmt = $conn->prepare($materials_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $materials[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real quizzes for this course
    $quizzes = [];
    if (isset($conn)) {
        $quiz_query = "SELECT title, quiz_id FROM quizzes WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($quiz_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $quizzes[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Create lessons based on actual course assignments and materials
    $course_title = $course['title'] ?? 'Course';
    
    // Create lessons based on real assignments first
    $lesson_count = 0;
    foreach ($assignments as $assignment) {
        $lesson_count++;
        $lessons[] = [
            'lesson_id' => $lesson_count,
            'video_id' => $lesson_count,
            'title' => $assignment['title'] . " - Tutorial",
            'duration' => rand(10, 30) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
            'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
            'video_path' => '',
            'thumbnail' => '',
            'description' => "Learn about " . $assignment['title'],
            'completed' => $lesson_count <= 2 ? true : false,
            'has_assignment' => true,
            'has_quiz' => count($quizzes) > ($lesson_count - 1),
            'has_materials' => count($materials) > ($lesson_count - 1),
            'watch_progress' => $lesson_count <= 2 ? 100 : ($lesson_count == 3 ? 65 : 0),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'video_order' => $lesson_count,
            'is_published' => 1,
            'last_position' => 0
        ];
    }
    
    // If no assignments, create lessons based on materials
    if (empty($assignments) && !empty($materials)) {
        foreach ($materials as $index => $material) {
            $lesson_count++;
            $lessons[] = [
                'lesson_id' => $lesson_count,
                'video_id' => $lesson_count,
                'title' => $material['title'],
                'duration' => rand(15, 35) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
                'video_path' => '',
                'thumbnail' => '',
                'description' => $material['description'] ?? "Learn about " . $material['title'],
                'completed' => $index < 2 ? true : false,
                'has_assignment' => false,
                'has_quiz' => count($quizzes) > $index,
                'has_materials' => true,
                'watch_progress' => $index < 2 ? 100 : ($index == 2 ? 65 : 0),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
                'video_order' => $lesson_count,
                'is_published' => 1,
                'last_position' => 0
            ];
        }
    }
    
    // No demo data - keep lessons empty if no real content exists
}

// Fix lesson ID detection - use first available lesson if none specified
$current_lesson_id = isset($_GET['lesson']) ? (int)$_GET['lesson'] : null;
$current_lesson = null;

// Only process lessons if they exist
if (!empty($lessons)) {
    if ($current_lesson_id) {
        // Look for specific lesson
        foreach ($lessons as $lesson) {
            if ($lesson['lesson_id'] == $current_lesson_id) {
                $current_lesson = $lesson;
                break;
            }
        }
    }
    
    // If no specific lesson found, use first lesson
    if (!$current_lesson) {
        $current_lesson = $lessons[0];
        $current_lesson_id = $current_lesson['lesson_id']; // Set to actual lesson ID
    }
} else {
    // No lessons available, set to null to avoid JavaScript errors
    $current_lesson_id = null;
}

// Debug: Log the lesson ID being used
error_log("COURSE DETAIL DEBUG: current_lesson_id = " . ($current_lesson_id ?? 'NULL'));
error_log("COURSE DETAIL DEBUG: Available lessons: " . json_encode(array_column($lessons, 'lesson_id')));

// â­ CRITICAL FIX: Calculate course progress percentage (aligned with backend)
$course_progress_percentage = 0;
$total_lessons = count($lessons);
$completed_lessons = 0;

if ($total_lessons > 0 && $user_id) {
    // Count completed lessons from video_progress table (same as backend)
    $progress_stmt = $conn->prepare("
        SELECT COUNT(DISTINCT vp.video_id) as completed_count
        FROM video_progress vp
        JOIN videos v ON vp.video_id = v.video_id 
        WHERE vp.student_id = ? AND v.course_id = ? AND vp.completed = 1
    ");
    
    if ($progress_stmt) {
        $progress_stmt->bind_param("ii", $user_id, $course_id);
        $progress_stmt->execute();
        $progress_result = $progress_stmt->get_result();
        $progress_data = $progress_result->fetch_assoc();
        $completed_lessons = intval($progress_data['completed_count'] ?? 0);
        $progress_stmt->close();
        
        // Use SAME calculation as frontend banner and backend AJAX
        $course_progress_percentage = round(($completed_lessons / $total_lessons) * 100);
        
        error_log("COURSE PROGRESS DEBUG: Total: $total_lessons, Completed: $completed_lessons, Percentage: $course_progress_percentage%");
    }
}

include_once 'components/header.php';

// ===== COMPREHENSIVE REVIEW SYSTEM QUERIES =====

// Get review statistics for this course
$review_stats = [
    'average_rating' => 0,
    'total_reviews' => 0,
    'rating_breakdown' => [5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0]
];

if (isset($conn)) {
    $stats_query = "SELECT 
        COUNT(*) as total_reviews,
        ROUND(AVG(rating), 1) as average_rating,
        SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
        SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
        SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
        SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
        SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
    FROM course_reviews 
    WHERE course_id = ? AND is_approved = TRUE";
    
    if ($stmt = $conn->prepare($stats_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($stats = $result->fetch_assoc()) {
            $review_stats['total_reviews'] = $stats['total_reviews'];
            $review_stats['average_rating'] = $stats['average_rating'] ?: 0;
            $review_stats['rating_breakdown'] = [
                5 => $stats['five_star'],
                4 => $stats['four_star'],
                3 => $stats['three_star'],
                2 => $stats['two_star'],
                1 => $stats['one_star']
            ];
        }
        $stmt->close();
    }
}

// Get all reviews for this course with user details
$reviews = [];
if (isset($conn)) {
    $reviews_query = "SELECT 
        cr.review_id,
        cr.rating,
        cr.review_title,
        cr.review_text,
        cr.helpful_count,
        cr.created_at,
        u.first_name,
        u.last_name,
        u.profile_picture,
        (SELECT COUNT(*) FROM review_likes WHERE review_id = cr.review_id AND is_helpful = TRUE) as helpful_votes,
        (SELECT COUNT(*) FROM review_replies WHERE review_id = cr.review_id) as reply_count
    FROM course_reviews cr
    JOIN users u ON cr.student_id = u.user_id
    WHERE cr.course_id = ? AND cr.is_approved = TRUE
    ORDER BY cr.created_at DESC
    LIMIT 20";
    
    if ($stmt = $conn->prepare($reviews_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $reviews[] = $row;
        }
        $stmt->close();
    }
}

// Check if current user has already reviewed this course
$user_has_reviewed = false;
$user_review = null;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $user_review_query = "SELECT * FROM course_reviews WHERE course_id = ? AND student_id = ?";
    if ($stmt = $conn->prepare($user_review_query)) {
        $stmt->bind_param("ii", $course_id, $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($user_review = $result->fetch_assoc()) {
            $user_has_reviewed = true;
        }
        $stmt->close();
    }
}

// Check if user is enrolled (for review eligibility)
$user_enrolled = false;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $enrollment_query = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
    if ($stmt = $conn->prepare($enrollment_query)) {
        $stmt->bind_param("ii", $_SESSION['user_id'], $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_enrolled = $result->num_rows > 0;
        $stmt->close();
    }
}

// For backward compatibility with existing code
$average_rating = $review_stats['average_rating'];
$rating_count = $review_stats['total_reviews'];

?>

<!-- ✅ EXTRACTED CSS BLOCK 1 - Now External (with cache-busting) -->
<link rel="stylesheet" href="css/course-detail-main.css?v=<?= time() ?>">
<link rel="stylesheet" href="css/course-detail-secondary.css?v=<?= time() ?>">
<link rel="stylesheet" href="css/course-banner.css?v=<?= time() ?>">

<!-- Course Banner - 8 Columns Width (Outside Container) -->
<div class="container-fluid">
    <div class="row">
        <div class="col-lg-8">
            <div class="course-profile-card">
                <!-- Course Header: Thumbnail + Video Title -->
                <div class="course-header-row">
                    <div class="course-image" style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '') ?>')">
                        <?php if (empty($course['thumbnail'])): ?>
                            <i class="bx bx-book-open"></i>
                        <?php endif; ?>
                    </div>
                    <h1 id="main-lesson-title" class="video-title">
                        <?= $current_lesson ? htmlspecialchars($current_lesson['title']) : 'Select a lesson to get started' ?>
                    </h1>
                </div>
                
                <div class="course-profile-info">
                    <!-- Small Course Title -->
                    <h3 style="font-size: 16px; font-weight: 600; margin: 0 0 5px 0; color: #000; opacity: 0.8;">
                        <?= htmlspecialchars($course['title']) ?>
                    </h3>
        
        <!-- Now Watching Label with Bouncing Play Icon -->
        <div style="margin: 0 0 15px 0; display: flex; align-items: center; gap: 8px;">
            <div class="bouncing-play-icon" style="
                display: inline-flex; 
                align-items: center; 
                justify-content: center; 
                width: 20px; 
                height: 20px; 
                background: #007bff; 
                border-radius: 50%; 
                animation: bounce 1.5s infinite;
                box-shadow: 0 2px 4px rgba(0, 123, 255, 0.3);
            ">
                <i class="bx bx-play" style="color: white; font-size: 10px; margin-left: 1px;"></i>
            </div>
            <span style="font-size: 14px; color: #666; font-weight: 500;">Now Watching:</span>
        </div>
        
        <!-- CSS Animation for Bouncing Play Icon -->
        <style>
            @keyframes bounce {
                0%, 20%, 50%, 80%, 100% {
                    transform: translateY(0);
                }
                40% {
                    transform: translateY(-4px);
                }
                60% {
                    transform: translateY(-2px);
                }
            }
        </style>
        
        <!-- Course Description -->
        <p class="course-description-compact" style="margin: 8px 0 15px 0; color: #666; font-size: 14px; line-height: 1.4;">
            <?= htmlspecialchars($course['description']) ?>
        </p>
        
        <!-- First Row: Basic Course Info -->
        <div class="course-profile-meta">
            <div class="meta-item">
                <i class="bx bx-user"></i>
                <?php 
                // Get instructor info from database
                $instructor_name = 'Panadite Academy';
                if (isset($conn) && $course['teacher_id']) {
                    $instructor_query = "SELECT first_name, last_name FROM teachers WHERE teacher_id = ?";
                    if ($stmt = $conn->prepare($instructor_query)) {
                        $stmt->bind_param("i", $course['teacher_id']);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        if ($instructor = $result->fetch_assoc()) {
                            $instructor_name = trim($instructor['first_name'] . ' ' . $instructor['last_name']);
                        }
                        $stmt->close();
                    }
                }
                echo htmlspecialchars($instructor_name);
                ?>
            </div>
            <div class="meta-item">
                <i class="bx bx-time"></i>
                <?= count($lessons) ?> Lessons
            </div>
            <div class="meta-item">
                <i class="bx bx-group"></i>
                <?php 
                // Get real enrollment count
                $enrollment_count = 0;
                if (isset($conn)) {
                    $enrollment_query = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = ?";
                    if ($stmt = $conn->prepare($enrollment_query)) {
                        $stmt->bind_param("i", $course_id);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        if ($enrollment_data = $result->fetch_assoc()) {
                            $enrollment_count = $enrollment_data['count'];
                        }
                        $stmt->close();
                    }
                }
                echo $enrollment_count;
                ?> Students
            </div>
            <div class="meta-item">
                <i class="bx bx-star"></i>
                <?php 
                // Get real course rating
                $average_rating = 0;
                $rating_count = 0;
                if (isset($conn)) {
                    $rating_query = "SELECT AVG(rating) as avg_rating, COUNT(*) as count FROM ratings WHERE course_id = ?";
                    if ($stmt = $conn->prepare($rating_query)) {
                        $stmt->bind_param("i", $course_id);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        if ($rating_data = $result->fetch_assoc()) {
                            $average_rating = round($rating_data['avg_rating'], 1);
                            $rating_count = $rating_data['count'];
                        }
                        $stmt->close();
                    }
                }
                
                // Show rating or default
                if ($rating_count > 0) {
                    echo $average_rating . " (" . $rating_count . " reviews)";
                } else {
                    echo "New Course";
                }
                ?>
            </div>
        </div>
        
        <!-- Second Row: Progress and Grade -->
        <div class="course-profile-meta mt-2">
            <!-- Course Progress in Profile Card -->
            <div class="meta-item">
                <i class="bx bx-trending-up"></i>
                <div class="course-progress-compact">
                    <div class="progress-header-compact">
                        <span>Course Progress</span>
                        <span class="progress-percentage" id="main-course-progress-percentage">
                            <?php 
                            // Calculate real course progress
                            $total_lessons = count($lessons);
                            $completed_lessons = 0;
                            foreach ($lessons as $lesson) {
                                if ($lesson['completed']) {
                                    $completed_lessons++;
                                }
                            }
                            $course_progress_percentage = $total_lessons > 0 ? round(($completed_lessons / $total_lessons) * 100) : 0;
                            
                            // Calculate comprehensive course completion and overall grade
                            $comprehensive_progress = $progressService->calculateCourseProgress($user_id, $course_id);
                            $certificate_eligibility = $progressService->isCertificateEligible($user_id, $course_id);
                            $overall_course_mark = $comprehensive_progress['overall_percentage'];
                            $videos_completed = $course_progress_percentage >= 100; // All videos watched
                            $quiz_requirements_met = $certificate_eligibility['eligible']; // Quiz passing grade met
                            $full_course_completed = $videos_completed && $quiz_requirements_met;
                            
                            echo $course_progress_percentage;
                            ?>%
                        </span>
                    </div>
                    <div class="progress-bar" style="margin-top: 4px; height: 4px;">
                        <div class="progress-fill" id="main-course-progress-bar" style="width: <?= $course_progress_percentage ?>%"></div>
                    </div>
                </div>
            </div>
            
            <!-- Overall Course Grade in Profile Card -->
            <?php if ($overall_course_mark > 0): ?>
            <div class="meta-item">
                <i class="bx bx-medal"></i>
                <div class="grade-info-compact">
                    <span class="grade-label">Overall Grade:</span>
                    <span class="grade-value <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                        <?= number_format($overall_course_mark, 1) ?>%
                        <?php if ($quiz_requirements_met): ?>
                            <i class="bx bx-check-circle text-success ms-1"></i>
                        <?php endif; ?>
                    </span>
                    <span class="grade-status" style="font-size: 0.8em; color: #666;">
                        (Passing: <?= $certificate_eligibility['minimum_required'] ?>%)
                    </span>
                </div>
            </div>
            <?php endif; ?>
                </div> <!-- End course-profile-info -->
            </div> <!-- End course-profile-card -->
        </div>
    </div>
</div>

<div class="course-wrapper">
    <!-- Bootstrap Container for Grid Layout -->
    <div class="container-fluid mt-4">
        <div class="row">
            <!-- Course Content Accordion Sidebar (25% width) -->
            <div class="col-md-3">
                <div class="course-content-sidebar">
                    <div class="accordion" id="courseContentAccordion">
                        
                        <?php if (empty($lessons)): ?>
                            <!-- No Lessons Available -->
                            <div class="accordion-item">
                                <div class="empty-state p-4">
                                    <div class="text-center text-muted">
                                        <i class="bx bx-video-off fs-1"></i>
                                        <p class="mt-2 mb-0">No lessons available</p>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <!-- Individual Lesson Accordions -->
                            <?php foreach ($lessons as $index => $lesson): ?>
                                <?php 
                                // Get video-specific content for badges and links
                                $hasVideoQuiz = false;
                                $hasVideoAssignment = false; 
                                $hasVideoMaterials = false;
                                $hasVideoQuiz = false;
                                $videoAssignments = [];
                                $videoMaterials = [];
                                $videoQuizzes = []; // Changed to array to handle multiple quizzes
                                
                                if (isset($conn) && !empty($lesson['lesson_id'])) {
                                    // Get ALL quizzes for this video (removed LIMIT 1)
                                    $quiz_check = $conn->prepare("SELECT quiz_id, title, time_limit, passing_score FROM quizzes WHERE video_id = ? AND is_published = 1");
                                    if ($quiz_check) {
                                        $quiz_check->bind_param("i", $lesson['lesson_id']);
                                        $quiz_check->execute();
                                        $quiz_result = $quiz_check->get_result();
                                        
                                        // Process ALL quizzes for this video
                                        while ($quiz_data = $quiz_result->fetch_assoc()) {
                                            $hasVideoQuiz = true;
                                            
                                            // Get student's attempt data for this specific quiz
                                            $attempt_check = $conn->prepare("
                                                SELECT attempt_id, score, passed, completed_at 
                                                FROM quiz_attempts 
                                                WHERE quiz_id = ? AND student_id = ? 
                                                ORDER BY completed_at DESC 
                                                LIMIT 1
                                            ");
                                            if ($attempt_check) {
                                                $attempt_check->bind_param("ii", $quiz_data['quiz_id'], $user_id);
                                                $attempt_check->execute();
                                                $attempt_result = $attempt_check->get_result();
                                                if ($attempt_data = $attempt_result->fetch_assoc()) {
                                                    // Merge attempt data with quiz data
                                                    $quiz_data = array_merge($quiz_data, $attempt_data);
                                                }
                                                $attempt_check->close();
                                            }
                                            
                                            // Add this quiz to the array
                                            $videoQuizzes[] = $quiz_data;
                                        }
                                        $quiz_check->close();
                                    }
                                    
                                    // Get video-specific assignments data
                                    $assignment_check = $conn->prepare("SELECT assignment_id, title, due_date FROM assignments WHERE video_id = ?");
                                    if ($assignment_check) {
                                        $assignment_check->bind_param("i", $lesson['lesson_id']);
                                        $assignment_check->execute();
                                        $assignment_result = $assignment_check->get_result();
                                        while ($assignment_data = $assignment_result->fetch_assoc()) {
                                            $videoAssignments[] = $assignment_data;
                                        }
                                        $hasVideoAssignment = count($videoAssignments) > 0;
                                        $assignment_check->close();
                                    }
                                    
                                    // Get video-specific materials data
                                    $materials_check = $conn->prepare("SELECT material_id, title, file_path, file_type FROM course_materials WHERE video_id = ?");
                                    if ($materials_check) {
                                        $materials_check->bind_param("i", $lesson['lesson_id']);
                                        $materials_check->execute();
                                        $materials_result = $materials_check->get_result();
                                        while ($materials_data = $materials_result->fetch_assoc()) {
                                            $videoMaterials[] = $materials_data;
                                        }
                                        $hasVideoMaterials = count($videoMaterials) > 0;
                                        $materials_check->close();
                                    }
                                    
                                    // Check for video-specific notes
                                    $notes_check = $conn->prepare("SELECT COUNT(*) as count FROM lesson_notes WHERE video_id = ?");
                                    if ($notes_check) {
                                        $notes_check->bind_param("i", $lesson['lesson_id']);
                                        $notes_check->execute();
                                        $notes_result = $notes_check->get_result();
                                        if ($notes_data = $notes_result->fetch_assoc()) {
                                            $hasVideoNotes = $notes_data['count'] > 0;
                                        }
                                        $notes_check->close();
                                    }
                                }
                                ?>
                                
                                <?php
                                // Calculate REAL progress from database BEFORE using it in HTML
                                $real_progress = $lesson['watch_progress'] ?? 0;
                                
                                // DEBUG: Check completion data for this lesson
                                $debug_completion_data = [
                                    'lesson_id' => $lesson['lesson_id'],
                                    'video_id' => $lesson['video_id'] ?? 'none',
                                    'watch_progress' => $real_progress,
                                    'completed_from_query' => $lesson['completed'] ?? 'not_set',
                                    'completed_bool' => isset($lesson['completed']) ? (bool)$lesson['completed'] : 'not_set',
                                    'is_current_lesson' => ($lesson['lesson_id'] == $current_lesson_id),
                                ];
                                echo "<!-- DEBUG LESSON {$lesson['lesson_id']}: " . json_encode($debug_completion_data) . " -->";
                                ?>
                                <?php if (isset($conn) && isset($_SESSION['user_id'])) {
                                    $progress_query = "SELECT completion_percentage FROM video_progress WHERE video_id = ? AND student_id = ?";
                                    if ($stmt = $conn->prepare($progress_query)) {
                                        $stmt->bind_param("ii", $lesson['lesson_id'], $_SESSION['user_id']);
                                        $stmt->execute();
                                        $result = $stmt->get_result();
                                        if ($row = $result->fetch_assoc()) {
                                            $real_progress = round($row['completion_percentage']);
                                        }
                                        $stmt->close();
                                    }
                                }
                                ?>
                                
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingLesson<?= $lesson['lesson_id'] ?>">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseLesson<?= $lesson['lesson_id'] ?>" aria-expanded="false" aria-controls="collapseLesson<?= $lesson['lesson_id'] ?>">
                                            <div class="d-flex align-items-center w-100">
                                                <div class="lesson-thumbnail-small me-3">
                                                    <?php if (!empty($lesson['thumbnail'])): ?>
                                                        <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson <?= $index + 1 ?>" class="rounded" style="width: 40px; height: 30px; object-fit: cover;">
                                                    <?php else: ?>
                                                        <div class="thumbnail-placeholder-small d-flex align-items-center justify-content-center" style="width: 40px; height: 30px; background: #f8f9fa; border-radius: 4px;">
                                                            <i class="bx bx-play text-muted"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if ($lesson['completed']): ?>
                                                        <div class="completion-check position-absolute" style="top: -5px; right: -5px;">
                                                            <i class="bx bx-check-circle text-success fs-6"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="flex-grow-1" style="min-width: 0; max-width: calc(100% - 60px);">
                                                    <div class="fw-semibold accordion-title-text" style="white-space: nowrap; overflow: hidden; text-overflow: ellipsis; max-width: 100%; padding-right: 15px;"><?= htmlspecialchars($lesson['title']) ?></div>
                                                    <div class="d-flex align-items-center justify-content-between mb-1">
                                                        <small class="text-muted">
                                                            <i class="bx bx-time"></i> <?= !empty($lesson['duration']) ? htmlspecialchars($lesson['duration']) : 'Duration not available' ?>
                                                        </small>
                                                        <small class="text-muted" id="progress-display-<?= $lesson['lesson_id'] ?>" style="font-size: 0.7rem; display: none;">
                                                            <span id="progress-percentage-<?= $lesson['lesson_id'] ?>">0</span>% watched
                                                        </small>
                                                    </div>
                                                    <!-- Badge rendering removed - will be replaced with unified system -->
                                                    <div class="d-flex align-items-center flex-wrap" id="lesson-badges-<?= $lesson['lesson_id'] ?>" style="display: none;">
                                                        <!-- Watching and Completed badges removed -->
                                                        <?php if ($hasVideoQuiz): ?>
                                                            <span class="badge badge-quiz-accordion me-1" style="font-size: 0.6rem;" title="Interactive Quiz Available">
                                                                <i class="bx bx-help-circle"></i> Quiz
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoAssignment): ?>
                                                            <span class="badge bg-warning me-1" style="font-size: 0.7rem;" title="Assignment Available">
                                                                <i class="bx bx-task"></i> Task
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoMaterials): ?>
                                                            <span class="badge badge-files-accordion me-1" style="font-size: 0.6rem;" title="Materials Available">
                                                                <i class="bx bx-folder-open"></i> Files
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoNotes): ?>
                                                            <span class="badge bg-secondary me-1" style="font-size: 0.7rem;" title="Notes Available">
                                                                <i class="bx bx-note"></i> Notes
                                                            </span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseLesson<?= $lesson['lesson_id'] ?>" class="accordion-collapse collapse" aria-labelledby="headingLesson<?= $lesson['lesson_id'] ?>" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <!-- Watch Button -->
                                            <div class="d-grid gap-2 mb-3">
                                                <button type="button" class="btn btn-watch-lesson" onclick="event.preventDefault(); event.stopPropagation(); return loadLesson(<?= $lesson['lesson_id'] ?>);">
                                                    <i class="bx bx-play-circle me-2"></i>
                                                    <?php 
                                                    // ðŸŽ¯ FIX: Show proper status based on completion, not just current lesson
                                                    if ($lesson['completed']) {
                                                        echo 'Rewatch Video';
                                                    } else if ($lesson['lesson_id'] == $current_lesson_id) {
                                                        echo 'Currently Watching';
                                                    } else {
                                                        echo 'Watch Lesson';
                                                    }
                                                    ?>
                                                </button>
                                            </div>
                                            
                                            <!-- 🎨 Dynamic Range-Style Progress Bar -->
                                            <div class="dynamic-progress-container mb-3">
                                                <!-- Progress Label -->
                                                <div class="text-center text-muted small mb-2 progress-label">
                                                    Progress: <span id="progress-percentage-<?= $lesson['lesson_id'] ?>" class="progress-percentage-text"><?= $real_progress ?></span>% complete
                                                </div>
                                                
                                                <!-- Range-Style Progress Track -->
                                                <div class="progress-track clean-progress-track">
                                                    <!-- Progress Fill -->
                                                    <div 
                                                        class="progress-bar clean-progress-fill" 
                                                        id="accordion-progress-bar-<?= $lesson['lesson_id'] ?>"
                                                        style="width: <?= $real_progress ?>%;"
                                                    ></div>
                                                    
                                                    <!-- Progress Thumb/Handle -->
                                                    <div 
                                                        class="progress-thumb clean-progress-thumb <?php if ($real_progress > 0): ?>active<?php endif; ?>" 
                                                        id="progress-thumb-<?= $lesson['lesson_id'] ?>"
                                                        style="left: <?= $real_progress ?>%;"
                                                        "
                                                    ></div>
                                                    
                                                    <!-- Completion Sparkle Effect (only when 100%) -->
                                                    <?php if ($real_progress >= 100): ?>
                                                    <div class="completion-sparkle" style="
                                                        position: absolute;
                                                        top: 50%;
                                                        right: 10px;
                                                        transform: translateY(-50%);
                                                        color: #fbbf24;
                                                        font-size: 16px;
                                                        animation: sparkle 1.5s ease-in-out infinite;
                                                    ">✨</div>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <!-- Progress Milestones (Optional) -->
                                                <div class="progress-milestones" style="
                                                    position: relative;
                                                    margin-top: 4px;
                                                    height: 4px;
                                                ">
                                                    <?php for ($i = 25; $i <= 75; $i += 25): ?>
                                                    <div style="
                                                        position: absolute;
                                                        left: <?= $i ?>%;
                                                        width: 2px;
                                                        height: 4px;
                                                        background: <?= $real_progress >= $i ? '#2563eb' : 'rgba(245, 158, 11, 0.5)' ?>;
                                                        border-radius: 1px;
                                                        transform: translateX(-50%);
                                                    "></div>
                                                    <?php endfor; ?>
                                                </div>
                                            </div>
                                            
                                            <!-- Nested Accordions for Quizzes and Materials in Stacked Layout -->
                                            <?php if ($hasVideoQuiz || $hasVideoMaterials || $hasVideoAssignment || $hasVideoNotes): ?>
                                                <div class="lesson-content-container">
                                                    <!-- Quizzes Section (Full Width) -->
                                                    <?php if ($hasVideoQuiz && !empty($videoQuizzes)): ?>
                                                        <div class="nested-accordion mb-3">
                                                            <div class="nested-accordion-header" data-toggle="collapse" data-target="#quizzes-<?= $lesson['lesson_id'] ?>" aria-expanded="false">
                                                                <h6 class="mb-0 d-flex align-items-center">
                                                                    <i class="bx bx-help-circle me-2 text-primary"></i>
                                                                    <span>Quizzes</span>
                                                                    <span class="badge bg-primary ms-2"><?= count($videoQuizzes) ?></span>
                                                                </h6>
                                                                <i class="bx bx-chevron-down"></i>
                                                            </div>
                                                            <div id="quizzes-<?= $lesson['lesson_id'] ?>" class="collapse nested-accordion-body">
                                                                <div class="nested-content p-2">
                                                                    <?php foreach ($videoQuizzes as $index => $videoQuiz): ?>
                                                                        <?php 
                                                                        // Encrypt quiz ID
                                                                        $encrypted_quiz_id = '';
                                                                        try {
                                                                            if (class_exists('IdEncryption') && method_exists('IdEncryption', 'encrypt')) {
                                                                                $encrypted_quiz_id = IdEncryption::encrypt($videoQuiz['quiz_id']);
                                                                                if (empty($encrypted_quiz_id) || $encrypted_quiz_id == $videoQuiz['quiz_id']) {
                                                                                    $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                                }
                                                                            } else {
                                                                                $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                            }
                                                                        } catch (Exception $e) {
                                                                            $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                        }
                                                                        
                                                                        // Check if student has completed the quiz
                                                                        $hasCompleted = !empty($videoQuiz['attempt_id']);
                                                                        $studentScore = $videoQuiz['score'] ?? 0;
                                                                        $hasPassed = $videoQuiz['passed'] ?? false;
                                                                        $completedAt = $videoQuiz['completed_at'] ?? null;
                                                                        ?>
                                                                        
                                                                        <div class="compact-quiz-item mb-1">
                                                                            <div class="d-flex align-items-center justify-content-between py-1 px-2">
                                                                                <div class="flex-grow-1">
                                                                                    <div class="d-flex align-items-center gap-2">
                                                                                        <i class="bx bx-help-circle text-primary" style="font-size: 0.9rem;"></i>
                                                                                        <span class="fw-bold" style="font-size: 0.8rem;">
                                                                                            <?= htmlspecialchars($videoQuiz['title']) ?>
                                                                                        </span>
                                                                                    </div>
                                                                                    <div class="d-flex align-items-center gap-2 mt-1">
                                                                                        <small class="text-muted" style="font-size: 0.7rem;"><?= $videoQuiz['time_limit'] ?>min</small>
                                                                                        <?php if ($hasCompleted): ?>
                                                                                            <small class="text-<?= $hasPassed ? 'success' : 'warning' ?>" style="font-size: 0.7rem;">
                                                                                                <?= round($studentScore) ?>%
                                                                                            </small>
                                                                                        <?php endif; ?>
                                                                                        <span class="badge bg-<?= $hasCompleted ? ($hasPassed ? 'success' : 'warning') : 'secondary' ?>" style="font-size: 0.6rem; padding: 2px 6px;">
                                                                                            <?php if ($hasCompleted): ?>
                                                                                                <?= $hasPassed ? 'PASSED' : 'FAILED' ?>
                                                                                            <?php else: ?>
                                                                                                PENDING
                                                                                            <?php endif; ?>
                                                                                        </span>
                                                                                    </div>
                                                                                </div>
                                                                                <a href="take_quiz.php?id=<?= htmlspecialchars($encrypted_quiz_id) ?>" 
                                                                                   class="btn btn-outline-primary" style="font-size: 0.7rem; padding: 4px 8px;">
                                                                                    <?= $hasCompleted ? 'Retry' : 'Start' ?>
                                                                                </a>
                                                                            </div>
                                                                        </div>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>

                                                    <!-- Materials Section (Full Width) -->
                                                    <?php if ($hasVideoMaterials && !empty($videoMaterials)): ?>
                                                        <div class="nested-accordion mb-3">
                                                            <div class="nested-accordion-header" data-toggle="collapse" data-target="#materials-<?= $lesson['lesson_id'] ?>" aria-expanded="false">
                                                                <h6 class="mb-0 d-flex align-items-center">
                                                                    <i class="bx bx-folder me-2 text-success"></i>
                                                                    <span>Materials</span>
                                                                    <span class="badge bg-success ms-2"><?= count($videoMaterials) ?></span>
                                                                </h6>
                                                                <i class="bx bx-chevron-down"></i>
                                                            </div>
                                                            <div id="materials-<?= $lesson['lesson_id'] ?>" class="collapse nested-accordion-body">
                                                                <div class="nested-content p-2">
                                                                    <?php foreach ($videoMaterials as $material): ?>
                                                                        <?php
                                                                        // Determine file type icon with standardized colors
                                                                        $fileType = strtolower($material['file_type'] ?? '');
                                                                        $fileIcon = 'bx-file';
                                                                        $fileColor = 'info';
                                                                        
                                                                        if (strpos($fileType, 'pdf') !== false) {
                                                                            $fileIcon = 'bx-file-pdf';
                                                                            $fileColor = 'danger';
                                                                        } elseif (strpos($fileType, 'doc') !== false) {
                                                                            $fileIcon = 'bx-file-doc';
                                                                            $fileColor = 'primary';
                                                                        } elseif (strpos($fileType, 'excel') !== false || strpos($fileType, 'sheet') !== false) {
                                                                            $fileIcon = 'bx-file-excel';
                                                                            $fileColor = 'success';
                                                                        } elseif (strpos($fileType, 'powerpoint') !== false || strpos($fileType, 'presentation') !== false) {
                                                                            $fileIcon = 'bx-file-ppt';
                                                                            $fileColor = 'warning';
                                                                        } elseif (strpos($fileType, 'image') !== false) {
                                                                            $fileIcon = 'bx-image';
                                                                            $fileColor = 'secondary';
                                                                        }
                                                                        ?>
                                                                        
                                                                        <div class="compact-material-item mb-1">
                                                                            <div class="d-flex align-items-center justify-content-between py-1 px-2">
                                                                                <div class="flex-grow-1">
                                                                                    <div class="d-flex align-items-center gap-2">
                                                                                        <i class="bx <?= $fileIcon ?> text-<?= $fileColor ?>" style="font-size: 0.9rem;"></i>
                                                                                        <span class="fw-bold" style="font-size: 0.8rem;">
                                                                                            <?= htmlspecialchars($material['title']) ?>
                                                                                        </span>
                                                                                    </div>
                                                                                    <div class="mt-1">
                                                                                        <small class="text-muted" style="font-size: 0.7rem;">Material</small>
                                                                                    </div>
                                                                                </div>
                                                                                <?php if (!empty($material['file_path'])): ?>
                                                                                    <a href="../<?= htmlspecialchars($material['file_path']) ?>" 
                                                                                       class="btn btn-outline-<?= $fileColor ?>" 
                                                                                       style="font-size: 0.7rem; padding: 4px 8px;"
                                                                                       target="_blank"
                                                                                       title="Download <?= htmlspecialchars($material['title']) ?>">
                                                                                        Download
                                                                                    </a>
                                                                                <?php endif; ?>
                                                                            </div>
                                                                        </div>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <!-- Additional Content Row (Full Width) -->
                                                    <?php if ($hasVideoAssignment || $hasVideoNotes): ?>
                                                        <div class="row mt-2">
                                                            <div class="col-12">
                                                                <div class="additional-content">
                                                                    <?php if ($hasVideoAssignment): ?>
                                                                        <div class="mb-2">
                                                                            <a href="#" class="btn btn-outline-warning btn-sm w-100">
                                                                                <i class="bx bx-task me-2"></i>Complete Assignment
                                                                            </a>
                                                                        </div>
                                                                    <?php endif; ?>
                                                                    
                                                                    <?php if ($hasVideoNotes): ?>
                                                                        <div class="mb-2">
                                                                            <a href="#" class="btn btn-outline-secondary btn-sm w-100">
                                                                                <i class="bx bx-note me-2"></i>View Lecture Notes
                                                                            </a>
                                                                        </div>
                                                                    <?php endif; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center text-muted">
                                                    <small>No additional content for this lesson</small>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>

                    </div>
                </div>
            </div>

            <!-- Main Content Area (75% width) -->
            <div class="col-md-9">



            <!-- Debug Panel -->
            <div id="debug-panel" style="position: fixed; top: 10px; right: 10px; background: rgba(0,0,0,0.8); color: white; padding: 15px; border-radius: 8px; max-width: 400px; z-index: 9999; font-family: monospace; font-size: 12px; display: none;">
                <div style="display: flex; justify-content: between; align-items: center; margin-bottom: 10px;">
                    <h6 style="color: #ffd700; margin: 0;">ðŸ› Video Debug Panel</h6>
                    <button onclick="toggleDebugPanel()" style="background: #ff4444; color: white; border: none; padding: 2px 6px; border-radius: 4px; cursor: pointer;">Ã—</button>
                </div>
                <div id="debug-content">
                    <div id="debug-progress">Progress: 0%</div>
                    <div id="debug-time">Time: 0s / 0s</div>
                    <div id="debug-completion">Completion Status: Not Started</div>
                    <div id="debug-ajax">AJAX Status: Ready</div>
                    <div id="debug-badges">Badges: None</div>
                    <button onclick="forceCompletion()" style="background: #28a745; color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; margin-top: 10px;">ðŸ”§ Force Complete</button>
                    <button onclick="debugVideoCompletion()" style="background: #007bff; color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; margin-top: 5px;">ðŸ“Š Check DB</button>
                </div>
            </div>

            <!-- Video Player -->
            <div class="video-container">
                <div class="video-player" id="videoPlayer">
                    <?php if (empty($lessons)): ?>
                        <!-- No lessons available -->
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af;">
                            <div style="text-align: center;">
                                <i class="bx bx-video-off" style="font-size: 4rem; margin-bottom: 1rem; color: #6b7280;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;">No Content Available</h3>
                                <p style="margin: 0 0 0.5rem 0;">This course doesn't have any lessons or videos yet.</p>
                                <small>Please check back later or contact your instructor</small>
                            </div>
                        </div>
                    <?php elseif (!empty($current_lesson['video_url'])): ?>
                        <?php if (!empty($current_lesson['is_external']) && $current_lesson['is_external']): ?>
                            <!-- External video (YouTube, etc.) with automatic iframe API progress tracking -->
                            <?php 
                            $videoUrl = htmlspecialchars($current_lesson['video_url']);
                            // Add YouTube API parameters for automatic progress tracking
                            if (strpos($videoUrl, 'youtube.com/embed') !== false) {
                                $separator = strpos($videoUrl, '?') !== false ? '&' : '?';
                                $videoUrl .= $separator . 'autoplay=0&rel=0&showinfo=0&controls=1&enablejsapi=1&origin=' . urlencode($_SERVER['HTTP_HOST']);
                            } else {
                                $separator = strpos($videoUrl, '?') !== false ? '&' : '?';
                                $videoUrl .= $separator . 'autoplay=0&rel=0&showinfo=0&controls=1';
                            }
                            ?>
                            <iframe id="lessonVideo" 
                                    src="<?= $videoUrl ?>" 
                                    style="width: 100%; height: 100%; border: none; background: #000;"
                                    frameborder="0" 
                                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                                    allowfullscreen>
                            </iframe>
                        <?php else: ?>
                            <!-- Local video file -->
                            <video id="lessonVideo" preload="metadata" 
                                   style="width: 100%; height: 100%; background: #000;"
                                   <?php if (!empty($current_lesson['thumbnail'])): ?>
                                   poster="<?= htmlspecialchars($current_lesson['thumbnail']) ?>"
                                   <?php endif; ?>>
                                <source src="<?= htmlspecialchars($current_lesson['video_url']) ?>" type="video/mp4">
                                Your browser does not support the video tag.
                            </video>
                        <?php endif; ?>
                    <?php else: ?>
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af; position: relative;">
                            <!-- Show thumbnail if available even when video is not available -->
                            <?php if (!empty($current_lesson['thumbnail'])): ?>
                                <img src="<?= htmlspecialchars($current_lesson['thumbnail']) ?>" 
                                     alt="<?= htmlspecialchars($current_lesson['title']) ?>"
                                     style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; opacity: 0.3;">
                            <?php endif; ?>
                            
                            <div style="position: relative; z-index: 1; text-align: center;">
                                <i class="bx bx-video" style="font-size: 4rem; margin-bottom: 1rem;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;"><?= htmlspecialchars($current_lesson['title']) ?></h3>
                                <p style="margin: 0 0 0.5rem 0;">Video not available</p>
                                <small>Please contact your instructor</small>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if (empty($current_lesson['is_external']) || !$current_lesson['is_external']): ?>
                <!-- Custom video controls - only show for local video files, not external videos -->
                <div class="video-controls">
                    <div class="time-bar" onclick="seekVideo(event)">
                        <div class="time-progress" id="timeProgress"></div>
                    </div>
                    <div class="control-buttons">
                        <div class="control-left">
                            <button id="playPauseBtn" onclick="togglePlayPause()" class="control-btn">
                                <i class="bx bx-play"></i>
                            </button>
                            <button onclick="skipBackward()" class="control-btn" title="Skip back 10s">
                                <i class="bx bx-rewind"></i>
                            </button>
                            <button onclick="skipForward()" class="control-btn" title="Skip forward 10s">
                                <i class="bx bx-fast-forward"></i>
                            </button>
                            <div class="time-display">
                                <span id="currentTime">0:00</span> / <span id="totalTime"><?= $current_lesson['duration'] ?></span>
                            </div>
                        </div>
                        <div class="control-center">
                            <!-- Playback Speed -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleSpeedMenu()" title="Playback Speed">
                                    <span id="speedLabel">1x</span>
                                </button>
                                <div class="dropdown-menu" id="speedMenu">
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.5)">0.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.75)">0.75x</div>
                                    <div class="dropdown-item active" onclick="setPlaybackSpeed(1)">1x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.25)">1.25x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.5)">1.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(2)">2x</div>
                                </div>
                            </div>

                            <!-- Quality Settings -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleQualityMenu()" title="Video Quality">
                                    <i class="bx bx-cog"></i>
                                </button>
                                <div class="dropdown-menu" id="qualityMenu">
                                    <div class="dropdown-item active" onclick="setVideoQuality('auto')">Auto</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('1080p')">1080p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('720p')">720p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('480p')">480p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('360p')">360p</div>
                                </div>
                            </div>

                            <!-- Subtitles -->
                            <button class="control-btn" onclick="toggleSubtitles()" title="Subtitles" id="subtitleBtn">
                                <i class="bx bx-captions"></i>
                            </button>

                            <!-- Notes -->
                            <button class="control-btn" onclick="addNote()" title="Add Note at Current Time">
                                <i class="bx bx-note"></i>
                            </button>
                        </div>
                        <div class="control-right">
                            <!-- Volume Control -->
                            <div class="volume-control">
                                <button onclick="toggleMute()" class="control-btn">
                                    <i class="bx bx-volume-full" id="volumeIcon"></i>
                                </button>
                                <input type="range" id="volumeSlider" class="volume-slider" min="0" max="100" value="100" onchange="setVolume(this.value)">
                            </div>
                            
                            <!-- Picture in Picture -->
                            <button onclick="togglePictureInPicture()" class="control-btn" title="Picture in Picture">
                                <i class="bx bx-dock-top"></i>
                            </button>
                            
                            <!-- Fullscreen -->
                            <button onclick="toggleFullscreen()" class="control-btn">
                                <i class="bx bx-fullscreen"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Course Content Tabs -->
            <div class="course-tabs-container">
                <!-- Tab Navigation -->
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="description">
                        <i class="bx bx-book-open"></i>
                        Course Description
                    </button>
                    <button class="tab-btn" data-tab="reviews">
                        <i class="bx bx-star"></i>
                        Reviews
                    </button>
                    <button class="tab-btn" data-tab="notes">
                        <i class="bx bx-note"></i>
                        Notes
                    </button>
                    <button class="tab-btn" data-tab="certificate">
                        <i class="bx bx-award"></i>
                        Certificate
                    </button>
                    <button class="tab-btn" data-tab="tools">
                        <i class="bx bx-wrench"></i>
                        Learning Tools
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="tab-content">
                    <!-- Enhanced Course Description Tab -->
                    <div class="tab-pane active" id="description">
                        <div class="enhanced-course-description">
                            
                            <!-- Course Header Section -->
                            <div class="course-header-section">
                                <div class="row g-4">
                                    <div class="col-lg-8">
                                        <div class="course-title-section">
                                            <h2 class="course-main-title"><?= htmlspecialchars($course['title']) ?></h2>
                                            <p class="course-subtitle"><?= htmlspecialchars($course['description']) ?></p>
                                        </div>
                                        
                                        <!-- Course Meta Information -->
                                        <div class="course-meta-grid">
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-play-circle"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value"><?= count($lessons) ?></div>
                                                    <div class="meta-label">Video Lessons</div>
                                                </div>
                                            </div>
                                            

                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-time"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php 
                                                        // Calculate total course duration
                                                        $total_minutes = 0;
                                                        foreach ($lessons as $lesson) {
                                                            if (!empty($lesson['duration'])) {
                                                                // Parse duration (assumes format like "5:30" or "10:15")
                                                                $time_parts = explode(':', $lesson['duration']);
                                                                if (count($time_parts) == 2) {
                                                                    $total_minutes += intval($time_parts[0]) * 60 + intval($time_parts[1]);
                                                                }
                                                            }
                                                        }
                                                        $hours = floor($total_minutes / 60);
                                                        $minutes = $total_minutes % 60;
                                                        echo $hours > 0 ? $hours . 'h ' . $minutes . 'm' : $minutes . ' mins';
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Total Duration</div>
                                                </div>
                                            </div>
                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-medal"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php
                                                        // Get course level/difficulty
                                                        $course_level = !empty($course['level']) ? htmlspecialchars($course['level']) : 'Beginner';
                                                        echo $course_level;
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Difficulty Level</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Instructor Information -->
                                    <div class="col-lg-4">
                                        <div class="instructor-card">
                                            <div class="instructor-header">
                                                <h5><i class="bx bx-user-circle"></i> Your Instructor</h5>
                                            </div>
                                            <div class="instructor-content">
                                                <div class="instructor-avatar">
                                                    <?php if (!empty($course['instructor_profile_image'])): ?>
                                                        <img src="<?= htmlspecialchars($course['instructor_profile_image']) ?>" alt="Instructor">
                                                    <?php else: ?>
                                                        <div class="avatar-placeholder">
                                                            <i class="bx bx-user"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="instructor-details">
                                                    <h6 class="instructor-name"><?= htmlspecialchars($instructor_name) ?></h6>
                                                    <p class="instructor-title">
                                                        <?php
                                                        // Get instructor title/bio from database
                                                        $instructor_title = "Course Instructor";
                                                        if (isset($conn)) {
                                                            $instructor_query = "SELECT bio, title FROM users WHERE user_id = ?";
                                            if ($stmt = $conn->prepare($instructor_query)) {
                                                                $stmt->bind_param("i", $course['teacher_id']);
                                                                $stmt->execute();
                                                                $result = $stmt->get_result();
                                                                if ($instructor_info = $result->fetch_assoc()) {
                                                                    $instructor_title = !empty($instructor_info['title']) ? htmlspecialchars($instructor_info['title']) : $instructor_title;
                                                                }
                                                                $stmt->close();
                                                            }
                                                        }
                                                        echo $instructor_title;
                                                        ?>
                                                    </p>
                                                    <div class="instructor-stats">
                                                        <div class="stat-item">
                                                            <i class="bx bx-book"></i>
                                                            <span>
                                                                <?php
                                                                // Get instructor's total courses
                                                                $instructor_courses = 0;
                                                                if (isset($conn)) {
                                                                    $courses_query = "SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?";
                                                                    if ($stmt = $conn->prepare($courses_query)) {
                                                                        $stmt->bind_param("i", $course['teacher_id']);
                                                                        $stmt->execute();
                                                                        $result = $stmt->get_result();
                                                                        if ($courses_data = $result->fetch_assoc()) {
                                                                            $instructor_courses = $courses_data['count'];
                                                                        }
                                                                        $stmt->close();
                                                                    }
                                                                }
                                                                echo $instructor_courses . ' Courses';
                                                                ?>
                                                            </span>
                                                        </div>

                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Learning Objectives Section -->
                            <div class="learning-objectives-section">
                                <h4><i class="bx bx-target-lock"></i> What You'll Learn</h4>
{{ ... }}
                                <div class="objectives-grid">
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Master the fundamental concepts and principles</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Apply practical skills through hands-on exercises</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Build confidence in the subject matter</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Complete interactive quizzes and assessments</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Gain industry-relevant knowledge and expertise</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Develop problem-solving and critical thinking skills</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Course Requirements Section -->
                            <div class="course-requirements-section">
                                <div class="row g-4">
                                    <div class="col-md-6">
                                        <div class="requirements-card">
                                            <h5><i class="bx bx-list-check"></i> Prerequisites</h5>
                                            <ul class="requirements-list">
                                                <li><i class="bx bx-check"></i> Basic computer literacy</li>
                                                <li><i class="bx bx-check"></i> Internet connection for online content</li>
                                                <li><i class="bx bx-check"></i> Motivation to learn and practice</li>
                                                <li><i class="bx bx-check"></i> No prior experience required</li>
                                            </ul>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="course-info-card">
                                            <h5><i class="bx bx-info-circle"></i> Course Information</h5>
                                            <div class="info-list">
                                                <div class="info-item">
                                                    <span class="info-label">Created:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Last Updated:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['updated_at'] ?? $course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Language:</span>
                                                    <span class="info-value">English</span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Access:</span>
                                                    <span class="info-value">Lifetime Access</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>


                        </div>
                    </div>

                    <!-- Enhanced Reviews Tab -->
                    <div class="tab-pane" id="reviews">
                        <div class="enhanced-reviews-content">
                            
                            <!-- Reviews Summary Header -->
                            <div class="reviews-summary-section">
                                <div class="row g-4">
                                    <div class="col-lg-4">
                                        <div class="overall-rating-card">
                                            <div class="rating-display">
                                                <div class="rating-number"><?= $review_stats['average_rating'] > 0 ? number_format($review_stats['average_rating'], 1) : 'New' ?></div>
                                                <div class="rating-stars-large">
                                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                                        <i class='bx <?= $i <= $review_stats['average_rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                    <?php endfor; ?>
                                                </div>
                                                <div class="rating-text"><?= $review_stats['total_reviews'] ?> student review<?= $review_stats['total_reviews'] != 1 ? 's' : '' ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-lg-8">
                                        <div class="rating-breakdown-card">
                                            <h6 class="breakdown-title">Rating Breakdown</h6>
                                            <?php for ($star = 5; $star >= 1; $star--): ?>
                                                <?php 
                                                $count = $review_stats['rating_breakdown'][$star];
                                                $percentage = $review_stats['total_reviews'] > 0 ? ($count / $review_stats['total_reviews']) * 100 : 0;
                                                ?>
                                                <div class="rating-breakdown-row">
                                                    <span class="star-label"><?= $star ?> stars</span>
                                                    <div class="rating-bar">
                                                        <div class="rating-bar-fill" style="width: <?= $percentage ?>%"></div>
                                                    </div>
                                                    <span class="rating-count"><?= $count ?></span>
                                                </div>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Write Review Section -->
                            <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                            <div class="write-review-section">
                                <div class="write-review-card">
                                    <div class="write-review-header">
                                        <h5><i class="bx bx-edit"></i> Share Your Experience</h5>
                                        <p>Help other students by sharing your experience with this course</p>
                                    </div>
                                    
                                    <form id="reviewForm" class="review-form">
                                        <input type="hidden" name="course_id" value="<?= encryptId($course_id) ?>">
                                        <input type="hidden" name="user_id" value="<?= $_SESSION['user_id'] ?? '' ?>">
                                        <input type="hidden" name="user_token" value="<?= md5(($_SESSION['user_id'] ?? '') . 'panadite_review_token') ?>">
                                        
                                        <div class="form-group">
                                            <label class="form-label">Your Rating</label>
                                            <div class="star-rating-input">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class="bx bx-star rating-star" data-rating="<?= $i ?>"></i>
                                                <?php endfor; ?>
                                            </div>
                                            <input type="hidden" name="rating" id="selectedRating" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewTitle" class="form-label">Review Title <small class="text-muted">(minimum 3 characters)</small></label>
                                            <input type="text" class="form-control" id="reviewTitle" name="review_title" 
                                                   placeholder="Summarize your experience... (minimum 3 characters required)" 
                                                   minlength="3" maxlength="255" required>
                                            <div class="form-text">
                                                <span id="titleCharCount">0</span>/3 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewText" class="form-label">Your Review <small class="text-muted">(minimum 10 characters)</small></label>
                                            <textarea class="form-control" id="reviewText" name="review_text" rows="4" 
                                                      placeholder="Share details about your experience with this course... (minimum 10 characters required)" 
                                                      minlength="10" required></textarea>
                                            <div class="form-text">
                                                <span id="charCount">0</span>/10 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-actions">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bx bx-send"></i> Submit Review
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="cancelReview()">
                                                Cancel
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            <?php elseif (isset($_SESSION['user_id']) && $user_has_reviewed): ?>
                            <div class="user-review-section">
                                <div class="user-review-card">
                                    <div class="user-review-header">
                                        <h6><i class="bx bx-check-circle"></i> Your Review</h6>
                                        <div class="review-actions">
                                            <button class="btn btn-sm btn-outline-primary" onclick="editReview(<?= $user_review['review_id'] ?>, '<?= htmlspecialchars($user_review['review_title'], ENT_QUOTES) ?>', '<?= htmlspecialchars($user_review['review_text'], ENT_QUOTES) ?>', <?= $user_review['rating'] ?>)">
                                                <i class="bx bx-edit"></i> Edit
                                            </button>
                                        </div>
                                    </div>
                                    <div class="user-review-content">
                                        <div class="review-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <i class='bx <?= $i <= $user_review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                            <?php endfor; ?>
                                        </div>
                                        <h6 class="review-title"><?= htmlspecialchars($user_review['review_title']) ?></h6>
                                        <p class="review-text"><?= htmlspecialchars($user_review['review_text']) ?></p>
                                        <small class="review-date">Reviewed on <?= date('M d, Y', strtotime($user_review['created_at'])) ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php elseif (!isset($_SESSION['user_id'])): ?>
                            <div class="login-prompt-section">
                                <div class="login-prompt-card">
                                    <i class="bx bx-user-circle" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Sign in to leave a review</h6>
                                    <p>Share your experience and help other students</p>
                                    <a href="../login.php" class="btn btn-primary btn-sm">
                                        <i class="bx bx-log-in"></i> Sign In
                                    </a>
                                </div>
                            </div>
                            <?php elseif (!$user_enrolled): ?>
                            <div class="enrollment-prompt-section">
                                <div class="enrollment-prompt-card">
                                    <i class="bx bx-lock-alt" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Enroll to leave a review</h6>
                                    <p>Only enrolled students can leave reviews</p>
                                </div>
                            </div>
                            <?php endif; ?>

                            <!-- Reviews List -->
                            <div class="reviews-list-section">
                                <div class="reviews-list-header">
                                    <h5><i class="bx bx-message-square-detail"></i> Student Reviews</h5>
                                    <?php if ($review_stats['total_reviews'] > 0): ?>
                                        <div class="dropdown">
                                            <button class="btn btn-outline-secondary dropdown-toggle" type="button" id="reviewSortDropdown" data-toggle="dropdown" aria-expanded="false">
                                                Newest First
                                            </button>
                                            <ul class="dropdown-menu" aria-labelledby="reviewSortDropdown">
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('newest')">Newest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('oldest')">Oldest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('highest')">Highest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('lowest')">Lowest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('helpful')">Most Helpful</a></li>
                                            </ul>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <?php if (empty($reviews)): ?>
                                <div class="no-reviews-placeholder">
                                    <div class="no-reviews-icon">
                                        <i class="bx bx-message-square-dots"></i>
                                    </div>
                                    <h6>No Reviews Yet</h6>
                                    <p>Be the first to share your experience with this course!</p>
                                    <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                                        <button class="btn btn-primary btn-sm" onclick="showReviewForm()">
                                            <i class="bx bx-plus"></i> Write First Review
                                        </button>
                                    <?php endif; ?>
                                </div>
                                <?php else: ?>
                                <div class="reviews-list" id="reviewsList">
                                    <?php foreach ($reviews as $review): ?>
                                    <div class="review-item-enhanced">
                                        <div class="review-avatar">
                                            <?php if (!empty($review['profile_picture'])): ?>
                                                <img src="<?= htmlspecialchars($review['profile_picture']) ?>" alt="Reviewer">
                                            <?php else: ?>
                                                <div class="avatar-placeholder">
                                                    <i class="bx bx-user"></i>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="review-content">
                                            <div class="review-header">
                                                <div class="reviewer-info">
                                                    <h6 class="reviewer-name"><?= htmlspecialchars($review['first_name'] . ' ' . substr($review['last_name'], 0, 1) . '.') ?></h6>
                                                    <div class="review-meta">
                                                        <div class="review-rating">
                                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                                <i class='bx <?= $i <= $review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                            <?php endfor; ?>
                                                        </div>
                                                        <span class="review-date"><?= date('M d, Y', strtotime($review['created_at'])) ?></span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="review-body">
                                                <h6 class="review-title"><?= htmlspecialchars($review['review_title']) ?></h6>
                                                <p class="review-text"><?= nl2br(htmlspecialchars($review['review_text'])) ?></p>
                                            </div>
                                            
                                            <div class="review-actions">
                                                <button class="review-action-btn helpful-btn" data-review-id="<?= encryptId($review['review_id']) ?>">
                                                    <i class="bx bx-like"></i>
                                                    <span class="helpful-count">Helpful (<?= $review['helpful_votes'] ?>)</span>
                                                </button>
                                                <?php if ($review['reply_count'] > 0): ?>
                                                    <button class="review-action-btn" onclick="showReplies('<?= encryptId($review['review_id']) ?>')">
                                                        <i class="bx bx-message"></i>
                                                        <span>Instructor Reply</span>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <?php if (count($reviews) >= 20): ?>
                                <div class="load-more-section">
                                    <button class="btn btn-outline-primary" onclick="loadMoreReviews()">
                                        <i class="bx bx-download"></i> Load More Reviews
                                    </button>
                                </div>
                                <?php endif; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- SITA-Compliant Accredited Certificate Tab -->
                    <div class="tab-pane" id="certificate">
                        <?php 
                        // Include accreditation logic
                        include_once 'includes/accreditation_logic.php';
                        ?>
                        
                        <div class="accreditation-content">
                                <div class="accreditation-header border rounded mb-4 p-4" style="border-color: #e2e8f0;">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h3 class="mb-2 text-dark"><i class="bx bx-award me-2 text-primary"></i> SITA-Accredited Professional Certificate</h3>
                                            <p class="mb-0 text-muted">Request an official, industry-recognized accredited certificate from Panadite Academy</p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <div class="accreditation-badge bg-white rounded p-3 text-center" style="border: 1px solid #e2e8f0;">
                                                <img src="../assets/images/sita-logo.png" alt="SITA Accredited" class="sita-logo mb-2" style="height: 40px;" onerror="this.style.display='none'">
                                                <div class="badge bg-primary">SITA Compliant</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                            <?php if (!$accreditation_request): ?>
                            <!-- Initial Request Stage -->
                            <div class="accreditation-stage" id="initialRequest">
                                <!-- SITA Information Accordion -->
                                <div class="accordion" id="sitaInfoAccordion">
                                    <!-- Requirements Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingRequirements">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseRequirements" aria-expanded="false" aria-controls="collapseRequirements">
                                                <i class="bx bx-task text-primary me-2"></i>
                                                <strong>Certificate Requirements</strong>
                                                <small class="text-muted ms-2">(<?= $videos_completed && $quiz_requirements_met ? 'Ready' : 'In Progress' ?>)</small>
                                            </button>
                                        </h2>
                                        <div id="collapseRequirements" class="accordion-collapse collapse" aria-labelledby="headingRequirements" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="requirements-grid">
                                                    <div class="requirement-card <?= $videos_completed ? 'completed' : '' ?>">
                                                        <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Complete all course videos (<?= $course_progress_percentage ?>% done)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $quiz_requirements_met ? 'completed' : '' ?>">
                                                        <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Achieve passing grade in quizzes (<?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>%)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $is_enrolled ? 'completed' : '' ?>">
                                                        <i class="bx <?= $is_enrolled ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Active course enrollment</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Pass SITA assessment (minimum 70%)</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Submit required documents</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- About SITA Accreditation -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingAbout">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseAbout" aria-expanded="false" aria-controls="collapseAbout">
                                                <i class="bx bx-shield-check text-primary me-2"></i>
                                                <strong>About SITA Accreditation</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseAbout" class="accordion-collapse collapse" aria-labelledby="headingAbout" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <p class="text-muted">SITA (Skills and Industry Training Authority) accreditation ensures your certificate meets professional industry standards and is recognized by employers worldwide. This certification carries significant weight in the professional market and demonstrates your commitment to excellence.</p>
                                                
                                                <div class="alert alert-info">
                                                    <div class="d-flex align-items-center">
                                                        <i class="bx bx-time text-info me-2" style="font-size: 1.5rem;"></i>
                                                        <div>
                                                            <h6 class="alert-heading mb-1">Processing Timeline</h6>
                                                            <p class="mb-0"><strong>Total Processing Time:</strong> Up to 5 months from initial request submission</p>
                                                            <small class="text-muted">SITA accreditation is a comprehensive process that ensures the highest professional standards.</small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Process Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingProcess">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseProcess" aria-expanded="false" aria-controls="collapseProcess">
                                                <i class="bx bx-list-check text-primary me-2"></i>
                                                <strong>6-Step Process Overview</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseProcess" class="accordion-collapse collapse" aria-labelledby="headingProcess" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="process-steps">
                                                    <div class="step-item">
                                                        <span class="step-number">1</span>
                                                        <div class="step-content">
                                                            <strong>Request Submission:</strong> Submit your accreditation request
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">2</span>
                                                        <div class="step-content">
                                                            <strong>Admin Review:</strong> Our team reviews your eligibility
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">3</span>
                                                        <div class="step-content">
                                                            <strong>SITA Assessment:</strong> Complete a comprehensive final assessment
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">4</span>
                                                        <div class="step-content">
                                                            <strong>Document Submission:</strong> Upload CV, qualifications, ID, and assignment work
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">5</span>
                                                        <div class="step-content">
                                                            <strong>Final Review:</strong> Admin team validates all requirements
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">6</span>
                                                        <div class="step-content">
                                                            <strong>Certificate Issuance:</strong> Receive your official accredited certificate
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <?php if ($full_course_completed && $is_enrolled): ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-header bg-primary text-white">
                                        <h5 class="card-title mb-0"><i class="bx bx-send me-2"></i> Request Accredited Certificate</h5>
                                    </div>
                                    <div class="card-body p-4">
                                        <form id="accreditationRequestForm">
                                            <input type="hidden" name="course_id" value="<?= $course_id ?>">
                                            <input type="hidden" name="student_id" value="<?= $user_id ?>">
                                            
                                            <div class="mb-3">
                                                <label class="form-label">Course</label>
                                                <input type="text" class="form-control" value="<?= htmlspecialchars($course['title']) ?>" readonly>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="motivationText" class="form-label">Motivation Statement <small class="text-muted">(Optional)</small></label>
                                                <textarea class="form-control" id="motivationText" name="motivation" rows="4" placeholder="Briefly explain why you want this accredited certificate and how it will benefit your career..."></textarea>
                                            </div>
                                            
                                            <div class="alert alert-warning mb-3">
                                                <h6 class="alert-heading"><i class="bx bx-info-circle me-2"></i>Important Notice</h6>
                                                <p class="mb-2">By submitting this request, you acknowledge:</p>
                                                <ul class="mb-0">
                                                    <li>This is a comprehensive accreditation process taking up to 5 months</li>
                                                    <li>You will need to complete a SITA assessment and submit professional documents</li>
                                                    <li>The certificate fee and assessment requirements will be communicated after approval</li>
                                                </ul>
                                            </div>
                                            
                                            <div class="form-check mb-4">
                                                <input class="form-check-input" type="checkbox" id="agreeTerms" required>
                                                <label class="form-check-label fw-bold" for="agreeTerms">
                                                    I understand and agree to the accreditation process requirements and timeline.
                                                </label>
                                            </div>
                                            
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary btn-lg">
                                                    <i class="bx bx-send"></i> Submit Accreditation Request
                                                </button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-body text-center py-5">
                                        <div class="mb-3">
                                            <i class="bx bx-lock-alt text-warning" style="font-size: 4rem;"></i>
                                        </div>
                                        <h5 class="text-warning mb-3">Complete All Course Requirements</h5>
                                        <p class="text-muted mb-4">You must complete all videos AND achieve the passing grade in quizzes to earn your certificate.</p>
                                        
                                        <!-- Comprehensive Requirements Progress -->
                                        <div class="requirements-progress mb-4">
                                            <div class="row text-start">
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Video Completion:</strong>
                                                                <div class="progress mt-1" style="height: 8px;">
                                                                    <div class="progress-bar bg-primary" role="progressbar" style="width: <?= $course_progress_percentage ?>%" id="certificate-video-progress-bar">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted" id="certificate-video-progress-text"><?= $course_progress_percentage ?>% videos watched</small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Quiz Performance:</strong>
                                                                <div class="progress mt-1" style="height: 8px;">
                                                                    <div class="progress-bar <?= $quiz_requirements_met ? 'bg-success' : 'bg-warning' ?>" role="progressbar" style="width: <?= min(100, ($overall_course_mark / $certificate_eligibility['minimum_required']) * 100) ?>%">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted">
                                                                    <?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>% required
                                                                </small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Overall Course Grade Display -->
                                        <div class="overall-grade-display mb-4">
                                            <div class="card bg-light">
                                                <div class="card-body py-3">
                                                    <h6 class="mb-1">Overall Course Grade</h6>
                                                    <div class="grade-display">
                                                        <span class="h4 <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                                                            <?= number_format($overall_course_mark, 1) ?>%
                                                        </span>
                                                        <small class="text-muted ms-2">
                                                            (Passing: <?= $certificate_eligibility['minimum_required'] ?>%)
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Next Steps -->
                                        <div class="next-steps">
                                            <?php if (!$videos_completed && !$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Steps:</strong> Complete all videos and improve your quiz scores</p>
                                                <div class="d-flex gap-2 justify-content-center">
                                                    <a href="#lessons" class="btn btn-outline-primary btn-sm">Continue Videos</a>
                                                    <a href="#quizzes" class="btn btn-outline-success btn-sm">Take Quizzes</a>
                                                </div>
                                            <?php elseif (!$videos_completed): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Complete all remaining videos</p>
                                                <a href="#lessons" class="btn btn-outline-primary">Continue Learning</a>
                                            <?php elseif (!$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Improve your quiz performance</p>
                                                <a href="#quizzes" class="btn btn-outline-success">Take Quizzes</a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php else: ?>
                            <!-- Request Status and Progress -->
                            <div class="accreditation-progress">
                                <div class="status-header mb-4">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h5>Your Accreditation Request</h5>
                                            <p class="mb-0">Request ID: #<?= str_pad($accreditation_request['request_id'], 6, '0', STR_PAD_LEFT) ?></p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <?= getStatusBadge($accreditation_request['status']) ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="status-timeline mb-4">
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['pending', 'approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : 'active' ?>">
                                        <div class="timeline-marker"><i class="bx bx-send"></i></div>
                                        <div class="timeline-content">
                                            <h6>Request Submitted</h6>
                                            <small><?= date('M d, Y \a\t H:i', strtotime($accreditation_request['request_date'])) ?></small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : ($accreditation_request['status'] == 'pending' ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-check-circle"></i></div>
                                        <div class="timeline-content">
                                            <h6>Admin Approval</h6>
                                            <?php if ($accreditation_request['approved_date']): ?>
                                                <small><?= date('M d, Y \a\t H:i', strtotime($accreditation_request['approved_date'])) ?></small>
                                            <?php else: ?>
                                                <small>Pending review</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['assessment_created']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-brain"></i></div>
                                        <div class="timeline-content">
                                            <h6>SITA Assessment</h6>
                                            <?php if ($assessment_attempt && $assessment_attempt['end_time']): ?>
                                                <small>Completed: <?= $assessment_attempt['percentage_score'] ?>%</small>
                                            <?php else: ?>
                                                <small>Awaiting completion</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['documents_pending']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-folder-plus"></i></div>
                                        <div class="timeline-content">
                                            <h6>Document Submission</h6>
                                            <small><?= count($submitted_documents) ?> document(s) uploaded</small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['under_review', 'approved_final']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-award"></i></div>
                                        <div class="timeline-content">
                                            <h6>Certificate Issued</h6>
                                            <?php if ($issued_certificate): ?>
                                                <small><?= date('M d, Y', strtotime($issued_certificate['issue_date'])) ?></small>
                                            <?php else: ?>
                                                <small>Pending</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="current-status-card mb-4">
                                    <div class="card-body">
                                        <h6><i class="bx bx-info-circle"></i> Current Status</h6>
                                        <p><?= getStatusDescription($accreditation_request['status']) ?></p>
                                        
                                        <?php if ($accreditation_request['admin_notes']): ?>
                                        <div class="admin-notes">
                                            <h6><i class="bx bx-message-detail"></i> Admin Notes</h6>
                                            <p class="text-primary"><?= htmlspecialchars($accreditation_request['admin_notes']) ?></p>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="next-steps">
                                            <h6><i class="bx bx-navigation"></i> Next Steps</h6>
                                            <p><?= getNextSteps($accreditation_request['status'], $assessment_attempt) ?></p>
                                        </div>
                                    </div>
                                </div>

                                <!-- SITA Assessment Interface -->
                                <?php if ($accreditation_request['status'] == 'assessment_created' && $current_assessment): ?>
                                <div class="assessment-interface-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-brain"></i> SITA Assessment - <?= htmlspecialchars($current_assessment['title']) ?></h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="assessment-info mb-4">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-time"></i>
                                                        <span><strong>Time Limit:</strong> <?= $current_assessment['time_limit_minutes'] ?> minutes</span>
                                                    </div>
                                                    <div class="info-item">
                                                        <i class="bx bx-target-lock"></i>
                                                        <span><strong>Pass Mark:</strong> <?= $current_assessment['pass_mark_percentage'] ?>%</span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-refresh"></i>
                                                        <span><strong>Attempts Allowed:</strong> <?= $current_assessment['max_attempts'] ?></span>
                                                    </div>
                                                    <?php if ($current_assessment['deadline']): ?>
                                                    <div class="info-item">
                                                        <i class="bx bx-calendar"></i>
                                                        <span><strong>Deadline:</strong> <?= date('M d, Y', strtotime($current_assessment['deadline'])) ?></span>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <?php if ($current_assessment['instructions']): ?>
                                        <div class="assessment-instructions mb-4">
                                            <h6><i class="bx bx-info-circle"></i> Instructions</h6>
                                            <div class="instructions-content">
                                                <?= nl2br(htmlspecialchars($current_assessment['instructions'])) ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($assessment_attempt && $assessment_attempt['status'] == 'completed'): ?>
                                        <div class="assessment-result mb-4">
                                            <div class="result-card <?= $assessment_attempt['passed'] ? 'success' : 'warning' ?>">
                                                <div class="result-header">
                                                    <i class="bx <?= $assessment_attempt['passed'] ? 'bx-check-circle' : 'bx-x-circle' ?>"></i>
                                                    <h6><?= $assessment_attempt['passed'] ? 'Assessment Passed!' : 'Assessment Not Passed' ?></h6>
                                                </div>
                                                <div class="result-details">
                                                    <div class="score-display">
                                                        <span class="score-number"><?= $assessment_attempt['percentage_score'] ?>%</span>
                                                        <span class="score-total">Score: <?= $assessment_attempt['total_score'] ?> points</span>
                                                    </div>
                                                    <div class="attempt-info">
                                                        <small>Completed: <?= date('M d, Y \a\t H:i', strtotime($assessment_attempt['end_time'])) ?></small>
                                                        <small>Time Taken: <?= $assessment_attempt['time_taken_minutes'] ?> minutes</small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="assessment-actions">
                                            <?php if (!$assessment_attempt): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-primary btn-lg">
                                                <i class="bx bx-play-circle"></i> Start SITA Assessment
                                            </a>
                                            <?php elseif ($assessment_attempt['status'] == 'in_progress'): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>&resume=<?= $assessment_attempt['attempt_id'] ?>" class="btn btn-warning btn-lg">
                                                <i class="bx bx-play"></i> Resume Assessment
                                            </a>
                                            <?php elseif (!$assessment_attempt['passed'] && $assessment_attempt['attempt_number'] < $current_assessment['max_attempts']): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-secondary btn-lg">
                                                <i class="bx bx-refresh"></i> Retake Assessment (Attempt <?= $assessment_attempt['attempt_number'] + 1 ?>)
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Document Upload Interface -->
                                <?php if ($accreditation_request['status'] == 'documents_pending' || $accreditation_request['status'] == 'documents_submitted'): ?>
                                <div class="document-upload-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-folder-plus"></i> Required Document Submission</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="document-requirements mb-4">
                                            <h6><i class="bx bx-list-check"></i> Required Documents</h6>
                                            <div class="requirements-grid">
                                                <?php 
                                                $required_docs = [
                                                    'cv' => ['name' => 'Curriculum Vitae (CV)', 'icon' => 'bx-user', 'formats' => 'PDF, DOC, DOCX'],
                                                    'qualifications' => ['name' => 'Educational Qualifications', 'icon' => 'bx-certification', 'formats' => 'PDF, JPG, PNG'],
                                                    'id_document' => ['name' => 'Government ID Document', 'icon' => 'bx-id-card', 'formats' => 'PDF, JPG, PNG'],
                                                    'assignment' => ['name' => 'Course Assignment Work', 'icon' => 'bx-file', 'formats' => 'PDF, DOC, DOCX, ZIP']
                                                ];
                                                
                                                foreach ($required_docs as $doc_type => $doc_info):
                                                    $uploaded = array_filter($submitted_documents, function($doc) use ($doc_type) {
                                                        return $doc['document_type'] == $doc_type;
                                                    });
                                                    $has_upload = !empty($uploaded);
                                                ?>
                                                <div class="requirement-card <?= $has_upload ? 'completed' : 'pending' ?>">
                                                    <div class="requirement-header">
                                                        <i class="bx <?= $doc_info['icon'] ?> <?= $has_upload ? 'text-success' : 'text-muted' ?>"></i>
                                                        <h6><?= $doc_info['name'] ?></h6>
                                                        <?php if ($has_upload): ?>
                                                        <span class="badge bg-success">âœ“ Uploaded</span>
                                                        <?php endif; ?>
                                                    </div>
                                                    <p class="requirement-formats">Accepted: <?= $doc_info['formats'] ?></p>
                                                    <?php if ($has_upload): ?>
                                                    <div class="uploaded-files">
                                                        <?php foreach ($uploaded as $doc): ?>
                                                        <div class="uploaded-file">
                                                            <i class="bx bx-file"></i>
                                                            <span><?= htmlspecialchars($doc['document_name']) ?></span>
                                                            <small class="verification-status <?= $doc['verification_status'] ?>">
                                                                <?= ucfirst($doc['verification_status']) ?>
                                                            </small>
                                                        </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                        
                                        <?php if ($accreditation_request['status'] == 'documents_pending'): ?>
                                        <form id="documentUploadForm" enctype="multipart/form-data">
                                            <input type="hidden" name="request_id" value="<?= $accreditation_request['request_id'] ?>">
                                            
                                            <div class="mb-3">
                                                <label for="documentType" class="form-label">Document Type</label>
                                                <select class="form-select" id="documentType" name="document_type" required>
                                                    <option value="">Select document type...</option>
                                                    <option value="cv">Curriculum Vitae (CV)</option>
                                                    <option value="qualifications">Educational Qualifications</option>
                                                    <option value="id_document">Government ID Document</option>
                                                    <option value="assignment">Course Assignment Work</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentFiles" class="form-label">Select Files</label>
                                                <input type="file" class="form-control" id="documentFiles" name="documents[]" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.zip" required>
                                                <div class="form-text">
                                                    <i class="bx bx-info-circle"></i> 
                                                    You can upload multiple files. Maximum 50MB per file.
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentNotes" class="form-label">Additional Notes <small class="text-muted">(Optional)</small></label>
                                                <textarea class="form-control" id="documentNotes" name="notes" rows="3" placeholder="Any additional information about these documents..."></textarea>
                                            </div>
                                            
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="bx bx-upload"></i> Upload Documents
                                                </button>
                                            </div>
                                        </form>
                                        <?php else: ?>
                                        <div class="submission-complete">
                                            <div class="alert alert-info">
                                                <i class="bx bx-check-circle"></i>
                                                <strong>Documents Submitted Successfully!</strong>
                                                Your documents are currently under admin review. You will be notified of the outcome.
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Certificate Download Interface -->
                                <?php if ($accreditation_request['status'] == 'certificate_issued' && $issued_certificate): ?>
                                <div class="certificate-issued-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-award"></i> Your SITA-Accredited Certificate</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="certificate-preview mb-4">
                                            <div class="text-center py-4">
                                                <i class="bx bx-award text-warning" style="font-size: 4rem;"></i>
                                                <h4 class="text-success">ðŸŽ‰ Congratulations!</h4>
                                                <p class="lead">Your SITA-accredited certificate has been successfully issued.</p>
                                                
                                                <div class="certificate-details">
                                                    <div class="row">
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Certificate Number:</strong><br>
                                                                <code><?= $issued_certificate['certificate_number'] ?></code>
                                                            </div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Issue Date:</strong><br>
                                                                <?= date('F d, Y', strtotime($issued_certificate['issue_date'])) ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div class="row mt-3">
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Verification Code:</strong><br>
                                                                <code><?= $issued_certificate['verification_code'] ?></code>
                                                            </div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Certificate Type:</strong><br>
                                                                SITA Accredited Professional
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="certificate-actions">
                                            <div class="row g-3">
                                                <div class="col-md-4">
                                                    <a href="download-certificate.php?id=<?= $issued_certificate['certificate_id'] ?>" class="btn btn-primary w-100">
                                                        <i class="bx bx-download"></i> Download PDF
                                                    </a>
                                                </div>
                                                <div class="col-md-4">
                                                    <button class="btn btn-outline-primary w-100" onclick="shareCertificate('<?= $issued_certificate['certificate_number'] ?>')">
                                                        <i class="bx bx-share-alt"></i> Share Certificate
                                                    </button>
                                                </div>
                                                <div class="col-md-4">
                                                    <a href="verify-certificate.php?code=<?= $issued_certificate['verification_code'] ?>" class="btn btn-outline-secondary w-100" target="_blank">
                                                        <i class="bx bx-check-shield"></i> Verify Online
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="certificate-note mt-4">
                                            <div class="alert alert-info">
                                                <i class="bx bx-info-circle"></i>
                                                <strong>Important:</strong> This certificate is digitally signed and can be verified online using the verification code above. 
                                                Share this achievement on your professional profiles!
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <!-- Rejection Notice -->
                                <?php if ($accreditation_request['status'] == 'rejected'): ?>
                                <div class="rejection-notice-card mb-4">
                                    <div class="card-header bg-danger text-white">
                                        <h5><i class="bx bx-x-circle"></i> Request Rejected</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="alert alert-danger">
                                            <h6>Unfortunately, your accreditation request has been rejected.</h6>
                                            <?php if ($accreditation_request['rejection_reason']): ?>
                                            <p><strong>Reason:</strong> <?= htmlspecialchars($accreditation_request['rejection_reason']) ?></p>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="support-contact">
                                            <h6><i class="bx bx-support"></i> Need Help?</h6>
                                            <p>If you believe this rejection was made in error or would like to reapply, please contact our support team:</p>
                                            <a href="mailto:accreditation@panaditeacademy.com" class="btn btn-outline-primary">
                                                <i class="bx bx-envelope"></i> Contact Support
                                            </a>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Notes Tab -->
                    <div class="tab-pane" id="notes">
                        <div class="notes-content">
                            <!-- Notes Header -->
                            <div class="notes-header">
                                <div class="notes-title">
                                    <i class="bx bx-note"></i>
                                    My Notes
                                </div>
                                <button class="btn btn-primary btn-sm" onclick="focusNoteInput()" title="Add new note at current time">
                                    <i class="bx bx-plus"></i> Add Note
                                </button>
                            </div>
                            
                            <!-- Notes Input Area -->
                            <div class="notes-input-area" style="margin-bottom: 2rem;">
                                <div class="input-group">
                                    <textarea class="form-control note-input" 
                                              placeholder="Add a note at the current video time..." 
                                              rows="3"
                                              style="resize: vertical; min-height: 80px;"></textarea>
                                    <button class="btn btn-primary save-note-btn" type="button">
                                        <i class="bx bx-save"></i> Save Note
                                    </button>
                                </div>
                                <small class="text-muted">
                                    <i class="bx bx-info-circle"></i> 
                                    Notes will be saved with the current video timestamp
                                </small>
                            </div>
                            
                            <!-- Notes List -->
                            <div class="notes-list" id="notesList">
                                <div class="no-notes text-center py-4">
                                    <i class="bx bx-note text-muted" style="font-size: 2rem;"></i>
                                    <p class="text-muted mt-2 mb-0">No notes yet. Add your first note while watching!</p>
                                </div>
                            </div>
                            
                            <!-- Notes Actions -->
                            <div class="notes-actions">
                                <button class="btn btn-outline-primary btn-sm" onclick="exportNotes()" title="Export all notes">
                                    <i class="bx bx-export"></i> Export Notes
                                </button>
                                <button class="btn btn-outline-secondary btn-sm" onclick="printNotes()" title="Print notes">
                                    <i class="bx bx-printer"></i> Print Notes
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Learning Tools Tab -->
                    <div class="tab-pane" id="tools">
                        <div class="learning-tools-content">
                            <!-- Smart Study Productivity Hub -->
                            <div class="productivity-header mb-4">
                                <div class="row align-items-center">
                                    <div class="col-md-8">
                                        <h3 class="mb-2">
                                            <i class="bx bx-brain text-primary me-2"></i>
                                            Smart Study Hub
                                        </h3>
                                        <p class="text-muted mb-0">Powerful tools to enhance your learning experience and productivity</p>
                                    </div>
                                    <div class="col-md-4 text-md-end">
                                        <div class="productivity-stats">
                                            <div class="stat-badge">
                                                <i class="bx bx-time-five text-success"></i>
                                                <span class="ms-1">Study Smart</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Two Main Tools Row -->
                            <div class="row g-4">
                                <!-- Smart Study Calendar -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-calendar-event text-primary me-2"></i>
                                                        Smart Study Calendar
                                                    </h5>
                                                    <p class="text-muted small mb-0">Set lesson reminders with email notifications</p>
                                                </div>
                                                <div class="calendar-actions">
                                                    <button class="btn btn-sm btn-primary" onclick="createQuickReminder()">
                                                        <i class="bx bx-plus"></i> Quick Reminder
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <!-- Calendar Navigation -->
                                            <div class="calendar-nav p-3 border-bottom">
                                                <div class="d-flex align-items-center justify-content-between">
                                                    <button class="btn btn-outline-primary btn-sm" onclick="previousMonth()">
                                                        <i class="bx bx-chevron-left"></i>
                                                    </button>
                                                    <h6 class="mb-0 fw-bold" id="currentMonthYear">November 2025</h6>
                                                    <button class="btn btn-outline-primary btn-sm" onclick="nextMonth()">
                                                        <i class="bx bx-chevron-right"></i>
                                                    </button>
                                                </div>
                                            </div>
                                            
                                            <!-- Calendar Grid -->
                                            <div class="calendar-container p-3">
                                                <div class="calendar-grid" id="calendarGrid">
                                                    <!-- Calendar will be rendered here by JavaScript -->
                                                </div>
                                            </div>
                                            
                                            <!-- Upcoming Reminders -->
                                            <div class="upcoming-reminders p-3 border-top bg-light">
                                                <h6 class="mb-2">
                                                    <i class="bx bx-bell text-warning me-1"></i>
                                                    Upcoming Reminders
                                                </h6>
                                                <div id="upcomingRemindersList">
                                                    <div class="text-muted small text-center py-2">
                                                        <i class="bx bx-calendar-plus me-1"></i>
                                                        No reminders set yet
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Picture-in-Picture Video Player -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-video text-success me-2"></i>
                                                        Picture-in-Picture Player
                                                    </h5>
                                                    <p class="text-muted small mb-0">Continue watching while browsing</p>
                                                </div>
                                                <div class="pip-status">
                                                    <span class="badge bg-secondary" id="pipStatus">Ready</span>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body">
                                            <!-- PiP Controls -->
                                            <div class="pip-controls mb-4">
                                                <div class="row g-3">
                                                    <div class="col-6">
                                                        <button class="btn btn-success w-100" onclick="enablePictureInPicture()" id="pipEnableBtn">
                                                            <i class="bx bx-dock-top me-2"></i>
                                                            Enable PiP Mode
                                                        </button>
                                                    </div>
                                                    <div class="col-6">
                                                        <button class="btn btn-outline-secondary w-100" onclick="disablePictureInPicture()" id="pipDisableBtn" disabled>
                                                            <i class="bx bx-dock-bottom me-2"></i>
                                                            Exit PiP Mode
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Current Video Info -->
                                            <div class="current-video-info">
                                                <div class="video-thumbnail-container mb-3">
                                                    <div class="mini-video-preview" id="miniVideoPreview">
                                                        <?php if ($current_lesson): ?>
                                                            <img src="<?= !empty($current_lesson['thumbnail']) ? htmlspecialchars($current_lesson['thumbnail']) : '/assets/images/video-placeholder.jpg' ?>" 
                                                                 alt="Current Video" class="img-fluid rounded">
                                                            <div class="play-overlay">
                                                                <i class="bx bx-play-circle"></i>
                                                            </div>
                                                        <?php else: ?>
                                                            <div class="no-video-placeholder">
                                                                <i class="bx bx-video-off text-muted"></i>
                                                                <p class="text-muted mb-0">No video selected</p>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                
                                                <div class="video-details">
                                                    <h6 class="mb-2" id="currentVideoTitle">
                                                        <?= $current_lesson ? htmlspecialchars($current_lesson['title']) : 'No video selected' ?>
                                                    </h6>
                                                    <div class="video-meta text-muted small">
                                                        <span><i class="bx bx-time me-1"></i>Duration: 
                                                            <span id="videoDuration"><?= $current_lesson && !empty($current_lesson['duration']) ? htmlspecialchars($current_lesson['duration']) : 'N/A' ?></span>
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- PiP Features -->
                                            <div class="pip-features mt-4">
                                                <h6 class="mb-3">Features:</h6>
                                                <div class="feature-list">
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Watch while browsing other tabs</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Resizable floating window</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Always on top display</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Full playback controls</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Browser Support Info -->
                                            <div class="browser-support mt-4">
                                                <div class="alert alert-info small mb-0">
                                                    <i class="bx bx-info-circle me-1"></i>
                                                    <strong>Browser Support:</strong> Works best in Chrome, Firefox, Safari, and Edge. 
                                                    Make sure your browser allows Picture-in-Picture mode.
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>



                            <!-- Keyboard Shortcuts -->
                            <div class="shortcuts-section mt-4">
                                <h5><i class="bx bx-keyboard"></i> Enhanced Keyboard Shortcuts</h5>
                                <div class="shortcuts-grid">
                                    <div class="shortcut-item">
                                        <span>Play/Pause</span>
                                        <span class="shortcut-key">Space</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Forward</span>
                                        <span class="shortcut-key">â†’</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Backward</span>
                                        <span class="shortcut-key">â†</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Add Note</span>
                                        <span class="shortcut-key">N</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Fullscreen</span>
                                        <span class="shortcut-key">F</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Volume Up/Down</span>
                                        <span class="shortcut-key">â†‘ / â†“</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Custom Note Modal -->
            <div id="customNoteModal" class="custom-modal" style="display: none;">
                <div class="custom-modal-overlay" onclick="closeNoteModal()"></div>
                <div class="custom-modal-container">
                    <div class="custom-modal-header">
                        <div class="custom-modal-title">
                            <i class="bx bx-note"></i>
                            <span id="modalTitleText">Add Note at</span>
                            <span id="modalTimestamp">0:00</span>
                        </div>
                        <button class="custom-modal-close" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                        </button>
                    </div>
                    <div class="custom-modal-body">
                        <div class="note-input-section">
                            <label for="customNoteTextarea">Your note:</label>
                            <textarea id="customNoteTextarea" 
                                placeholder="Add your thoughts, questions, or important points about this moment in the video..."></textarea>
                        </div>
                        <div class="note-tips-section">
                            <i class="bx bx-lightbulb"></i>
                            <span><strong>Tips:</strong> Your notes will be saved with the timestamp so you can easily jump back to this moment later.</span>
                        </div>
                    </div>
                    <div class="custom-modal-footer">
                        <button class="custom-btn custom-btn-secondary" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                            Cancel
                        </button>
                        <button class="custom-btn custom-btn-primary" id="customSaveNoteBtn">
                            <i class="bx bx-check"></i>
                            <span id="saveButtonText">Save Note</span>
                        </button>
                    </div>
                </div>
            </div>



          

            <!-- Dynamic Question Section -->
            <div class="question-section" id="questionSection">
                <div class="question-header">
                    <div class="question-icon">
                        <i class="bx bx-help-circle"></i>
                    </div>
                    <div class="question-text">
                        <h4>Need Help with This Lesson?</h4>
                        <p>Ask your facilitator any questions about the content you just watched</p>
                    </div>
                </div>
                <form class="question-form" onsubmit="submitQuestion(event)">
                    <textarea class="question-input" 
                              placeholder="Type your question here... What didn't you understand about this lesson?"
                              maxlength="500" required></textarea>
                    <button type="submit" class="question-submit">
                        <i class="bx bx-send"></i>
                        Send Question to Facilitator
                    </button>
                </form>
            </div>
        </div>
<!-- Current Lesson-->
                </div>
        </div>
    </div>

<!-- ✅ EXTRACTED JAVASCRIPT BLOCK 1 - Now External (CLEAN VERSION) -->
<script src="js/course-detail-main-clean.js"></script>

<!-- ===== BOOTSTRAP JAVASCRIPT LIBRARIES - CRITICAL FOR ACCORDION FUNCTIONALITY ===== -->
<!-- jQuery (required for Bootstrap 4) -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

<!-- Bootstrap 4 JavaScript Bundle (includes Popper.js) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>













<script src="js/course-detail-youtube.js"></script>

<!-- 🎯 PANADITE ACADEMY - ISOLATED PROGRESS TRACKING SYSTEM v2.0 -->
<!-- Clean, modular progress tracking that prevents file bloat -->
<script>
    // Pass PHP data to JavaScript for the isolated progress tracking system
    window.currentVideoId = <?= json_encode($current_lesson['video_id'] ?? null) ?>;
    window.currentLessonId = <?= json_encode($current_lesson_id ?? null) ?>;
    window.currentCourseId = <?= json_encode($course_id ?? null) ?>;
    window.currentStudentId = <?= json_encode($_SESSION['user_id'] ?? $_SESSION['student_id'] ?? 15) ?>; // Use session or fallback ID
    window.progressDebug = true; // Enable debug mode
</script>

<!-- ✅ NATURAL ACCORDION LAYOUT - NO ARTIFICIAL CONSTRAINTS -->
<style>
/* Natural accordion layout - let Bootstrap handle positioning */
.accordion-button {
    display: flex !important;
    align-items: center !important;
    justify-content: space-between !important;
    padding: 1rem !important;
    width: 100% !important;
}

/* Keep arrow in natural Bootstrap position but ensure it doesn't overlap */
.accordion-button::after {
    flex-shrink: 0 !important;
    margin-left: 0.5rem !important; /* Reduced from 1rem - arrows closer to content */
    position: static !important; /* Let it flow naturally */
}

/* Allow content to use available space naturally */
.accordion-button .d-flex {
    flex: 1 !important; /* Take available space */
    min-width: 0 !important; /* Allow shrinking if needed */
    margin-right: 1rem !important; /* Space before arrow */
}

/* Let titles use available space with smart truncation */
.accordion-title-text {
    overflow: hidden !important;
    text-overflow: ellipsis !important;
    white-space: nowrap !important;
    line-height: 1.4 !important;
    flex: 1 !important; /* Use available space */
}

/* Ensure flex container works properly */
.accordion-button .flex-grow-1 {
    min-width: 0 !important; /* Essential for flex text truncation */
    flex: 1 !important;
}

/* Mobile adjustments */
@media (max-width: 768px) {
    .accordion-button {
        padding: 0.75rem !important;
    }
    
    .accordion-button::after {
        margin-left: 0.75rem !important;
    }
    
    .accordion-button .d-flex {
        margin-right: 0.75rem !important;
    }
}
</style>

<!-- Load isolated progress tracking components -->
<!-- ✅ ISOLATED PROGRESS TRACKING SYSTEM - TEMPORARILY DISABLED TO PREVENT CONFLICTS -->
<!-- <script src="js/video-progress-tracker.js"></script> -->
<!-- <script src="js/progress-ui-integration.js"></script> -->
<!-- <script src="js/progress-integration-init.js"></script> -->
<!-- ✅ YOUTUBE IFRAME API INTEGRATION - Automatic progress tracking -->
<script src="js/youtube-progress-integration.js"></script>

<!-- Note: video-progress-handler.php is a PHP AJAX endpoint, NOT a JavaScript include -->

<!-- âœ… Clean Isolated Progress Tracking System - No more expansion needed -->
<script>
// ðŸŽ¯ CLEAN ISOLATED PROGRESS TRACKING - Initialize the new system
document.addEventListener('DOMContentLoaded', function() {
    console.log('âœ… Isolated Progress Tracking System Loading...');
    
    // Pass PHP data to isolated system in expected format
    window.currentVideoId = <?= json_encode($current_lesson['video_id'] ?? null) ?>;
    window.currentLessonId = <?= json_encode($current_lesson_id) ?>;
    window.currentCourseId = <?= json_encode($course_id) ?>;
    window.currentStudentId = <?= json_encode($_SESSION['user_id'] ?? $_SESSION['student_id'] ?? null) ?>;
    window.progressDebug = true; // Enable debug mode for testing
    
    // Additional data for external videos
    window.currentVideoUrl = <?= json_encode($current_lesson['video_url'] ?? '') ?>;
    window.isExternalVideo = <?= json_encode(!empty($current_lesson['is_external'])) ?>;
    
    console.log('ðŸ“‹ Data passed to isolated system:', {
        videoId: window.currentVideoId,
        lessonId: window.currentLessonId,
        courseId: window.currentCourseId,
        studentId: window.currentStudentId,
        videoUrl: window.currentVideoUrl,
        isExternal: window.isExternalVideo
    });
    
    // ðŸ” CRITICAL DEBUG: Check if all required data is present
    console.log('ðŸ” VALIDATION CHECK:', {
        'videoId exists': !!window.currentVideoId,
        'courseId exists': !!window.currentCourseId,
        'studentId exists': !!window.currentStudentId,
        'videoId value': window.currentVideoId,
        'courseId value': window.currentCourseId,  
        'studentId value': window.currentStudentId
    });
    
    // ðŸŽ¬ INITIALIZE YOUTUBE PROGRESS TRACKING
    if (window.currentVideoUrl && window.initializeYouTubeProgressTracking) {
        console.log('ðŸŽ¬ YOUTUBE: Calling initializeYouTubeProgressTracking...');
        console.log('ðŸŽ¬ YOUTUBE: Video URL:', window.currentVideoUrl);
        initializeYouTubeProgressTracking(
            window.currentVideoUrl,
            window.currentVideoId,
            window.currentLessonId,
            window.currentCourseId
        );
    } else {
        console.log('ðŸŽ¬ YOUTUBE: Skipping - No video URL or function not loaded');
        console.log('ðŸŽ¬ YOUTUBE: Video URL available:', !!window.currentVideoUrl);
        console.log('🎬 YOUTUBE: Function available:', !!window.initializeYouTubeProgressTracking);
    }
});
</script>
</body>
</html>
