<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if required files exist before including them
$required_files = [
    '../auth/functions.php',
    '../config/db_connect.php', 
    '../includes/course_progress_service_quiz_only.php',
    '../includes/certificate_automation.php'
];

foreach ($required_files as $file) {
    if (!file_exists($file)) {
        die("Required file missing: $file");
    }
}

require_once '../auth/functions.php';
require_once '../config/db_connect.php';
require_once '../includes/course_progress_service_quiz_only.php';
require_once '../includes/certificate_automation.php';

// Check if user is logged in as student
try {
    requireRole('student');
} catch (Exception $e) {
    die("Authentication error: " . $e->getMessage());
}

$student_id = $_SESSION['user_id'];
$selected_course_id = isset($_GET['course_id']) ? intval($_GET['course_id']) : null;

// Initialize progress service
$progressService = new QuizOnlyProgressService($conn);

$debug_results = [];
$courses = [];
$certificate_requirements = [];

try {
    // Use enrollments table (course_enrollments table doesn't exist)
    $enrollment_found = false;
    
    // Try enrollments table (main enrollment system)
    $table_check = $conn->query("SHOW TABLES LIKE 'enrollments'");
    if ($table_check && $table_check->num_rows > 0) {
        $enrollmentQuery = "
            SELECT DISTINCT
                c.course_id, 
                c.title as course_title,
                c.description,
                COUNT(DISTINCT q.quiz_id) as total_quizzes
            FROM courses c
            INNER JOIN enrollments e ON c.course_id = e.course_id
            LEFT JOIN quizzes q ON c.course_id = q.course_id
            WHERE e.student_id = ?
            GROUP BY c.course_id
            ORDER BY COUNT(DISTINCT q.quiz_id) DESC, c.title
        ";
        
        $stmt = $conn->prepare($enrollmentQuery);
        if ($stmt) {
            $stmt->bind_param("i", $student_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            while ($course = $result->fetch_assoc()) {
                $courses[] = $course;
            }
            $enrollment_found = true;
        }
    }
    
    // Fallback: Get all courses with quizzes first, then others
    if (!$enrollment_found) {
        $enrollmentQuery = "
            SELECT DISTINCT
                c.course_id, 
                c.title as course_title,
                c.description,
                COUNT(DISTINCT q.quiz_id) as total_quizzes
            FROM courses c
            LEFT JOIN quizzes q ON c.course_id = q.course_id
            GROUP BY c.course_id
            ORDER BY COUNT(DISTINCT q.quiz_id) DESC, c.title
            LIMIT 10
        ";
        
        $stmt = $conn->prepare($enrollmentQuery);
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            
            while ($course = $result->fetch_assoc()) {
                $courses[] = $course;
            }
        }
    }
    
    // If no course selected, select first enrolled course
    if (!$selected_course_id && !empty($courses)) {
        $selected_course_id = $courses[0]['course_id'];
    }
    
    // Get detailed progress for selected course
    if ($selected_course_id) {
        $debug_results = [
            'course_details' => null,
            'progress' => ['overall_percentage' => 0, 'quizzes' => ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0]],
            'certificate_eligibility' => ['eligible' => false, 'current_percentage' => 0],
            'quiz_report' => ['overall_stats' => ['total_quizzes' => 0, 'passed_quizzes' => 0, 'failed_quizzes' => 0, 'average_score' => 0], 'quiz_details' => []],
            'existing_certificate' => null
        ];
        
        try {
            // Get course details first
            $courseQuery = "SELECT * FROM courses WHERE course_id = ?";
            $courseStmt = $conn->prepare($courseQuery);
            if ($courseStmt) {
                $courseStmt->bind_param("i", $selected_course_id);
                $courseStmt->execute();
                $debug_results['course_details'] = $courseStmt->get_result()->fetch_assoc();
            }
            
            // Try to get course progress using our quiz-only system (with error handling)
            try {
                $progress = $progressService->calculateCourseProgress($student_id, $selected_course_id);
                $debug_results['progress'] = $progress;
            } catch (Exception $e) {
                error_log("Progress calculation error: " . $e->getMessage());
            }
            
            try {
                $certificate_eligibility = $progressService->isCertificateEligible($student_id, $selected_course_id);
                $debug_results['certificate_eligibility'] = $certificate_eligibility;
            } catch (Exception $e) {
                error_log("Certificate eligibility error: " . $e->getMessage());
            }
            
            try {
                $quiz_report = $progressService->getQuizPerformanceReport($student_id, $selected_course_id);
                
                // Debug: Check what we got from quiz report
                error_log("Quiz Report Debug - Student ID: $student_id, Course ID: $selected_course_id");
                error_log("Quiz Report Result: " . print_r($quiz_report, true));
                
                $debug_results['quiz_report'] = $quiz_report;
                
                // If quiz report is empty, force fallback to basic info
                if (empty($quiz_report['quiz_details']) && empty($quiz_report['overall_stats']['total_quizzes'])) {
                    error_log("Quiz report returned empty, using fallback");
                    $debug_results['quiz_report'] = getBasicQuizInfo($conn, $selected_course_id, $student_id);
                }
                
            } catch (Exception $e) {
                error_log("Quiz report error: " . $e->getMessage());
                // Fallback: Get basic quiz info directly
                $debug_results['quiz_report'] = getBasicQuizInfo($conn, $selected_course_id, $student_id);
            }
            
            // Check if certificate already exists (with error handling)
            $cert_table_check = $conn->query("SHOW TABLES LIKE 'certificates'");
            if ($cert_table_check && $cert_table_check->num_rows > 0) {
                $certExistsQuery = "
                    SELECT certificate_id, certificate_number, issue_date, file_path 
                    FROM certificates 
                    WHERE student_id = ? AND course_id = ?
                    ORDER BY issue_date DESC 
                    LIMIT 1
                ";
                $certStmt = $conn->prepare($certExistsQuery);
                if ($certStmt) {
                    $certStmt->bind_param("ii", $student_id, $selected_course_id);
                    $certStmt->execute();
                    $debug_results['existing_certificate'] = $certStmt->get_result()->fetch_assoc();
                }
            }
            
        } catch (Exception $e) {
            $debug_results['error'] = "Progress calculation error: " . $e->getMessage();
        }
    }

} catch (Exception $e) {
    $debug_results['error'] = $e->getMessage();
}

/**
 * Fallback function to get basic quiz information when main service fails
 */
function getBasicQuizInfo($conn, $course_id, $student_id) {
    $quiz_info = [
        'overall_stats' => [
            'total_quizzes' => 0,
            'passed_quizzes' => 0,
            'failed_quizzes' => 0,
            'average_score' => 0
        ],
        'quiz_details' => []
    ];
    
    try {
        // Check if quizzes table exists
        $table_check = $conn->query("SHOW TABLES LIKE 'quizzes'");
        if (!$table_check || $table_check->num_rows === 0) {
            return $quiz_info;
        }
        
        // Get basic quiz information
        $quizQuery = "
            SELECT 
                q.quiz_id,
                q.title,
                q.passing_score,
                0 as student_score,
                'not_attempted' as status,
                NULL as attempt_date,
                NULL as attempt_number
            FROM quizzes q
            WHERE q.course_id = ?
            ORDER BY q.quiz_id
        ";
        
        $stmt = $conn->prepare($quizQuery);
        if ($stmt) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            while ($quiz = $result->fetch_assoc()) {
                $quiz_info['quiz_details'][] = $quiz;
                $quiz_info['overall_stats']['total_quizzes']++;
            }
            
            // Try to get student's quiz attempts if quiz_attempts table exists
            $attempts_check = $conn->query("SHOW TABLES LIKE 'quiz_attempts'");
            if ($attempts_check && $attempts_check->num_rows > 0) {
                
                foreach ($quiz_info['quiz_details'] as &$quiz) {
                    $attemptQuery = "
                        SELECT score, attempt_date, attempt_number
                        FROM quiz_attempts
                        WHERE quiz_id = ? AND student_id = ?
                        ORDER BY attempt_id DESC
                        LIMIT 1
                    ";
                    
                    $attemptStmt = $conn->prepare($attemptQuery);
                    if ($attemptStmt) {
                        $attemptStmt->bind_param("ii", $quiz['quiz_id'], $student_id);
                        $attemptStmt->execute();
                        $attempt = $attemptStmt->get_result()->fetch_assoc();
                        
                        if ($attempt) {
                            $quiz['student_score'] = $attempt['score'];
                            $quiz['attempt_date'] = $attempt['attempt_date'];
                            $quiz['attempt_number'] = $attempt['attempt_number'];
                            
                            if ($attempt['score'] >= $quiz['passing_score']) {
                                $quiz['status'] = 'passed';
                                $quiz_info['overall_stats']['passed_quizzes']++;
                            } else {
                                $quiz['status'] = 'failed';
                                $quiz_info['overall_stats']['failed_quizzes']++;
                            }
                        }
                    }
                }
            }
        }
        
    } catch (Exception $e) {
        error_log("Basic quiz info error: " . $e->getMessage());
    }
    
    return $quiz_info;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Certificate Progress Debug - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .debug-container {
            max-width: 1200px;
            margin: 20px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
        .progress-section {
            margin: 20px;
            padding: 20px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            background: #f8f9fa;
        }
        .requirement-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin: 10px 0;
            border-left: 4px solid #28a745;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .completed-card {
            border-left-color: #28a745;
            background: #f0fff4;
        }
        .pending-card {
            border-left-color: #ffc107;
            background: #fffbf0;
        }
        .missing-card {
            border-left-color: #dc3545;
            background: #fff5f5;
        }
        .quiz-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            margin: 10px 0;
            background: white;
            border-radius: 8px;
            border-left: 4px solid #ddd;
        }
        .quiz-passed {
            border-left-color: #28a745;
            background: #f0fff4;
        }
        .quiz-failed {
            border-left-color: #dc3545;
            background: #fff5f5;
        }
        .quiz-not-attempted {
            border-left-color: #6c757d;
            background: #f8f9fa;
        }
        .progress-bar-custom {
            height: 30px;
            border-radius: 15px;
            overflow: hidden;
            background: #e9ecef;
        }
        .certificate-preview {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 2px dashed #28a745;
            padding: 30px;
            text-align: center;
            border-radius: 10px;
            margin: 20px 0;
        }
        .certificate-ready {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            border-color: #28a745;
        }
    </style>
</head>
<body>
    <div class="debug-container">
        <div class="header">
            <i class="bx bx-certification bx-lg mb-2"></i>
            <h1>Certificate Progress Debug</h1>
            <p class="mb-0">Track your progress toward earning course certificates</p>
            <small>Student: <?= htmlspecialchars($_SESSION['first_name'] . ' ' . $_SESSION['last_name']) ?></small>
        </div>
        
        <div class="p-4">
            <!-- Course Selection -->
            <div class="progress-section">
                <h4><i class="bx bx-book text-primary me-2"></i>Select Course to Debug</h4>
                <div class="row">
                    <div class="col-md-8">
                        <select class="form-select" onchange="selectCourse(this.value)">
                            <option value="">Choose a course...</option>
                            <?php foreach ($courses as $course): ?>
                            <option value="<?= $course['course_id'] ?>" <?= $selected_course_id == $course['course_id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($course['course_title']) ?> (<?= $course['total_quizzes'] ?> quizzes)
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <button onclick="location.reload()" class="btn btn-outline-primary">
                            <i class="bx bx-refresh me-1"></i>Refresh Status
                        </button>
                    </div>
                </div>
            </div>
            
            <?php if ($selected_course_id && !empty($debug_results)): ?>
            
            <!-- Overall Progress Status -->
            <div class="progress-section">
                <h4><i class="bx bx-chart-line text-success me-2"></i>Certificate Requirements Progress</h4>
                
                <div class="requirement-card <?= $debug_results['certificate_eligibility']['eligible'] ? 'completed-card' : 'pending-card' ?>">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="mb-1">
                                <?php if ($debug_results['certificate_eligibility']['eligible']): ?>
                                <i class="bx bx-check-circle text-success me-2"></i>Certificate Ready!
                                <?php else: ?>
                                <i class="bx bx-time text-warning me-2"></i>Certificate In Progress
                                <?php endif; ?>
                            </h5>
                            <p class="mb-0">
                                <strong>Course:</strong> <?= htmlspecialchars($debug_results['course_details']['title']) ?><br>
                                <strong>System:</strong> Quiz-Only Completion (Achieve <?= $debug_results['certificate_eligibility']['course_passing_grade'] ?? '60' ?>% average to earn certificate)<br>
                                <strong>Current Progress:</strong> <?= $debug_results['quiz_report']['overall_stats']['average_score'] ?>% quiz average
                            </p>
                        </div>
                        <div class="col-md-4">
                            <div class="progress-bar-custom">
                                <div class="progress-bar bg-success" style="width: <?= $debug_results['progress']['overall_percentage'] ?>%; line-height: 30px;">
                                    <?= $debug_results['progress']['overall_percentage'] ?>%
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Quiz Requirements Breakdown -->
                <!-- Debug Information Section -->
                <div class="mt-3 mb-4">
                    <div class="alert alert-secondary">
                        <h6><i class="bx bx-bug me-2"></i>Diagnostic Information</h6>
                        <small>
                            <strong>Student ID:</strong> <?= $student_id ?><br>
                            <strong>Course ID:</strong> <?= $selected_course_id ?><br>
                            <strong>Main Progress Method:</strong> <?= $debug_results['progress']['overall_percentage'] ?>% completed<br>
                            <strong>Quiz Report Method:</strong> <?= $debug_results['quiz_report']['overall_stats']['total_quizzes'] ?> total quizzes found<br>
                            <strong>Certificate Eligible:</strong> <?= $debug_results['certificate_eligibility']['eligible'] ? 'Yes' : 'No' ?><br>
                            <?php if (!empty($debug_results['quiz_report']['quiz_details'])): ?>
                                <strong>Quiz Details Found:</strong> <?= count($debug_results['quiz_report']['quiz_details']) ?> quiz(es)<br>
                            <?php endif; ?>
                        </small>
                    </div>
                </div>
                
                <div class="mt-4">
                    <h6>Quiz Completion Requirements:</h6>
                    <div class="row">
                        <div class="col-md-3">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h3 class="text-primary"><?= $debug_results['quiz_report']['overall_stats']['total_quizzes'] ?></h3>
                                    <small class="text-muted">Total Quizzes</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h3 class="text-success"><?= $debug_results['quiz_report']['overall_stats']['passed_quizzes'] ?></h3>
                                    <small class="text-muted">Passed Quizzes</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h3 class="text-danger"><?= $debug_results['quiz_report']['overall_stats']['failed_quizzes'] ?></h3>
                                    <small class="text-muted">Failed/Pending</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h3 class="text-info"><?= $debug_results['quiz_report']['overall_stats']['average_score'] ?>%</h3>
                                    <small class="text-muted">Average Score</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Quiz-by-Quiz Breakdown -->
            <div class="progress-section">
                <h4><i class="bx bx-list-check text-info me-2"></i>Individual Quiz Status</h4>
                
                <?php if (!empty($debug_results['quiz_report']['quiz_details'])): ?>
                <?php foreach ($debug_results['quiz_report']['quiz_details'] as $quiz): ?>
                <div class="quiz-item quiz-<?= $quiz['status'] ?>">
                    <div class="quiz-info">
                        <h6 class="mb-1"><?= htmlspecialchars($quiz['title']) ?></h6>
                        <small class="text-muted">
                            Passing Score: <?= $quiz['passing_score'] ?>% | 
                            Your Score: <?= $quiz['student_score'] ?? 'Not attempted' ?>%
                            <?php if ($quiz['attempt_date']): ?>
                            | Attempted: <?= date('M d, Y', strtotime($quiz['attempt_date'])) ?>
                            <?php endif; ?>
                        </small>
                    </div>
                    <div class="quiz-status">
                        <?php if ($quiz['status'] == 'passed'): ?>
                        <span class="badge bg-success">
                            <i class="bx bx-check"></i> Passed
                        </span>
                        <?php elseif ($quiz['status'] == 'failed'): ?>
                        <span class="badge bg-danger">
                            <i class="bx bx-x"></i> Failed - Retake Required
                        </span>
                        <?php else: ?>
                        <span class="badge bg-secondary">
                            <i class="bx bx-time"></i> Not Attempted
                        </span>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
                <?php else: ?>
                <div class="alert alert-info">
                    <i class="bx bx-info-circle me-2"></i>
                    <strong>No Quizzes Available</strong><br>
                    This course currently has no quizzes configured for certificate completion.
                    <br><br>
                    <div class="mt-3">
                        <strong>📚 Courses with active quizzes:</strong><br>
                        <ul class="list-unstyled mt-2">
                            <li>🎓 <strong>Computer science</strong> - 3 quizzes available</li>
                            <li>🎓 <strong>Technoprenuer</strong> - 2 quizzes available</li>
                        </ul>
                        <small class="text-muted">Select a course with quizzes to track your certificate progress.</small>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Certificate Status -->
            <div class="progress-section">
                <h4><i class="bx bx-certification text-warning me-2"></i>Certificate Status</h4>
                
                <?php if ($debug_results['existing_certificate']): ?>
                <div class="certificate-preview certificate-ready">
                    <i class="bx bx-check-circle bx-lg text-success mb-3"></i>
                    <h5 class="text-success">Certificate Issued!</h5>
                    <p class="mb-3">
                        <strong>Certificate Number:</strong> <?= htmlspecialchars($debug_results['existing_certificate']['certificate_number']) ?><br>
                        <strong>Issue Date:</strong> <?= date('F j, Y', strtotime($debug_results['existing_certificate']['issue_date'])) ?>
                    </p>
                    <a href="<?= htmlspecialchars($debug_results['existing_certificate']['file_path']) ?>" 
                       target="_blank" class="btn btn-success">
                        <i class="bx bx-download me-1"></i>Download Certificate
                    </a>
                </div>
                <?php elseif ($debug_results['certificate_eligibility']['eligible']): ?>
                <div class="certificate-preview certificate-ready">
                    <i class="bx bx-certification bx-lg text-success mb-3"></i>
                    <h5 class="text-success">Ready for Certificate!</h5>
                    <p class="mb-3">
                        Congratulations! You have completed all requirements.<br>
                        Certificate should be generated automatically within a few minutes.
                    </p>
                    <button onclick="triggerCertificateGeneration()" class="btn btn-success">
                        <i class="bx bx-refresh me-1"></i>Trigger Certificate Generation
                    </button>
                </div>
                <?php else: ?>
                <div class="certificate-preview">
                    <i class="bx bx-time bx-lg text-warning mb-3"></i>
                    <h5 class="text-warning">Certificate Pending</h5>
                    <p class="mb-3">
                        <strong>What you need to do:</strong><br>
                        Achieve <?= $debug_results['certificate_eligibility']['minimum_required'] ?? '60' ?>% quiz average to earn certificate.<br>
                        Current average: <?= $debug_results['quiz_report']['overall_stats']['average_score'] ?>% (<?= $debug_results['quiz_report']['overall_stats']['passed_quizzes'] ?>/<?= $debug_results['quiz_report']['overall_stats']['total_quizzes'] ?> quizzes passed)
                    </p>
                    <div class="alert alert-info">
                        <strong>Quiz-Only System:</strong> This course uses a quiz-only completion system. 
                        You need <?= $debug_results['certificate_eligibility']['minimum_required'] ?? '60' ?>% quiz average to earn your certificate. Assignments and videos are optional.
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Action Items -->
            <div class="progress-section">
                <h4><i class="bx bx-list-ol text-primary me-2"></i>Next Steps</h4>
                
                <?php
                // Check if student is eligible for certificate based on course passing grade
                $is_certificate_eligible = $debug_results['certificate_eligibility']['eligible'];
                $pending_quizzes = array_filter($debug_results['quiz_report']['quiz_details'], function($quiz) {
                    return $quiz['status'] !== 'passed';
                });
                ?>
                
                <?php if ($is_certificate_eligible): ?>
                <div class="alert alert-success">
                    <h6><i class="bx bx-check-circle me-2"></i>Certificate Requirements Met!</h6>
                    <p class="mb-2">
                        <strong>🎉 Congratulations!</strong> Your quiz average of <strong><?= $debug_results['quiz_report']['overall_stats']['average_score'] ?>%</strong> 
                        meets the course requirement of <strong><?= $debug_results['certificate_eligibility']['minimum_required'] ?? '60' ?>%</strong>.
                    </p>
                    <div class="bg-light p-3 rounded mt-2">
                        <small class="text-muted">
                            <i class="bx bx-info-circle me-1"></i>
                            <strong>Optional:</strong> You can still take remaining quizzes to improve your knowledge, but your certificate is ready for generation!
                        </small>
                    </div>
                </div>
                <?php elseif (!empty($pending_quizzes)): ?>
                <div class="alert alert-warning">
                    <h6><i class="bx bx-info-circle me-2"></i>To earn your certificate:</h6>
                    <p class="mb-2">You need <strong><?= $debug_results['certificate_eligibility']['minimum_required'] ?? '60' ?>%</strong> quiz average. Current: <strong><?= $debug_results['quiz_report']['overall_stats']['average_score'] ?>%</strong></p>
                    <div class="mt-2">
                        <strong>Suggested actions:</strong>
                        <ol class="mb-0 mt-1">
                            <?php foreach ($pending_quizzes as $quiz): ?>
                            <li>
                                <strong><?= htmlspecialchars($quiz['title']) ?></strong> - 
                                <?php if ($quiz['status'] == 'not_attempted'): ?>
                                Take this quiz and score at least <?= $quiz['passing_score'] ?>%
                                <?php else: ?>
                                Retake this quiz and score at least <?= $quiz['passing_score'] ?>% (current: <?= $quiz['student_score'] ?>%)
                                <?php endif; ?>
                            </li>
                            <?php endforeach; ?>
                        </ol>
                    </div>
                </div>
                <?php else: ?>
                <div class="alert alert-success">
                    <h6><i class="bx bx-check-circle me-2"></i>All quizzes completed!</h6>
                    <p class="mb-0">You have passed all quizzes. Your certificate should be generated automatically.</p>
                </div>
                <?php endif; ?>
            </div>
            
            <?php elseif ($selected_course_id): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error me-2"></i>
                <strong>Debug Error:</strong> <?= htmlspecialchars($debug_results['error'] ?? 'Unable to load course progress data') ?>
            </div>
            <?php else: ?>
            <div class="alert alert-info">
                <i class="bx bx-info-circle me-2"></i>
                Please select a course above to view your certificate progress.
            </div>
            <?php endif; ?>
            
            <div class="text-center mt-4">
                <a href="courses.php" class="btn btn-outline-secondary me-2">
                    <i class="bx bx-arrow-back me-1"></i>Back to Courses
                </a>
                <a href="certificates.php" class="btn btn-outline-primary">
                    <i class="bx bx-certification me-1"></i>View Certificates
                </a>
            </div>
        </div>
    </div>

    <script>
        function selectCourse(courseId) {
            if (courseId) {
                window.location.href = `debug_certificate_progress.php?course_id=${courseId}`;
            }
        }
        
        function triggerCertificateGeneration() {
            const courseId = <?= $selected_course_id ?>;
            
            fetch('../includes/certificate_automation.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'check_completion',
                    course_id: courseId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Certificate generation triggered! Refreshing page...');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'Failed to trigger certificate generation'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Network error. Please try again.');
            });
        }
        
        // Auto-refresh every 30 seconds to check for certificate updates
        setInterval(() => {
            const eligible = <?= json_encode($debug_results['certificate_eligibility']['eligible'] ?? false) ?>;
            const hasCert = <?= json_encode(!empty($debug_results['existing_certificate'])) ?>;
            
            if (eligible && !hasCert) {
                location.reload();
            }
        }, 30000);
    </script>
</body>
</html>
