<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');
require_once '../config/db_connect.php';

// Set the user ID and course ID to debug - CHANGED TO WEB DEVELOPMENT COURSE  
$user_id = $_SESSION['user_id']; // Current logged-in user (should be 45)
$course_id = 14;  // Web Development course where user has actual activity

echo "<h1>🐛 COURSE COMPLETION DEBUG SCRIPT - ENHANCED VERSION</h1>";
echo "<p><strong>User ID:</strong> {$user_id}</p>";
echo "<p><strong>Course ID:</strong> {$course_id}</p>";
echo "<p style='color: green;'><strong>✅ ENHANCED DEBUG - Showing Course Pass Marks & Quiz Requirements</strong></p>";

// ==========================================
// 0. FIND USER'S ACTUAL COURSES & COMPLETED VIDEOS
// ==========================================
echo "<h2>🔍 USER'S ACTUAL COURSE ENROLLMENTS & COMPLETIONS</h2>";

// Show all user enrollments - FIX: Check table structure first
echo "<h3>User Enrollments</h3>";
echo "<p><strong>DEBUG:</strong> Checking enrollments table structure...</p>";

// Check enrollments table structure
$describe_enrollments = "DESCRIBE enrollments";
if ($stmt = $conn->prepare($describe_enrollments)) {
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<h4>Enrollments Table Structure:</h4>";
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr><td><strong>{$row['Field']}</strong></td><td>{$row['Type']}</td><td>{$row['Null']}</td><td>{$row['Key']}</td></tr>";
    }
    echo "</table>";
    $stmt->close();
}

$enrollment_query = "
    SELECT e.course_id, c.title, e.enrolled_date, e.progress, e.student_id, e.user_id
    FROM enrollments e
    JOIN courses c ON e.course_id = c.course_id
    WHERE e.user_id = ?
    ORDER BY e.enrolled_date DESC
";
if ($stmt = $conn->prepare($enrollment_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Course ID</th><th>Course Title</th><th>Enrollment Date</th><th>Progress</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><a href='?course_id=" . base64_encode($row['course_id']) . "'>{$row['course_id']}</a></td>";
        echo "<td>" . htmlspecialchars($row['title']) . "</td>";
        echo "<td>{$row['enrolled_date']}</td>";
        echo "<td>" . ($row['progress'] ?? '0') . "%</td>";
        echo "</tr>";
    }
    echo "</table>";
    $stmt->close();
}

// Show all user video completions with course mapping
echo "<h3>All User Video Completions</h3>";
$completions_query = "
    SELECT 
        uvc.video_id,
        uvc.completion_percentage,
        uvc.completed_at,
        v.title as video_title,
        v.course_id,
        c.title as course_title
    FROM user_video_completions uvc
    LEFT JOIN videos v ON uvc.video_id = v.video_id
    LEFT JOIN courses c ON v.course_id = c.course_id
    WHERE uvc.user_id = ?
    ORDER BY uvc.completed_at DESC
";
if ($stmt = $conn->prepare($completions_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Video ID</th><th>Video Title</th><th>Course ID</th><th>Course Title</th><th>Completion %</th><th>Completed At</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>{$row['video_id']}</td>";
        echo "<td>" . htmlspecialchars($row['video_title'] ?? 'Unknown') . "</td>";
        echo "<td><a href='?course_id=" . base64_encode($row['course_id'] ?? 0) . "'>{$row['course_id']}</a></td>";
        echo "<td>" . htmlspecialchars($row['course_title'] ?? 'Unknown') . "</td>";
        echo "<td>{$row['completion_percentage']}%</td>";
        echo "<td>{$row['completed_at']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    $stmt->close();
}

// Show user quiz attempts across all courses - FIX: Check quiz table structures first
echo "<h3>All User Quiz Attempts</h3>";
echo "<p><strong>DEBUG:</strong> Checking quiz-related table structures...</p>";

// Check quizzes table structure
echo "<h4>Quizzes Table Structure:</h4>";
$describe_quizzes = "DESCRIBE quizzes";
if ($stmt = $conn->prepare($describe_quizzes)) {
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr><td><strong>{$row['Field']}</strong></td><td>{$row['Type']}</td><td>{$row['Null']}</td><td>{$row['Key']}</td></tr>";
    }
    echo "</table>";
    $stmt->close();
}

// Check quiz_attempts table structure
echo "<h4>Quiz Attempts Table Structure:</h4>";
$describe_quiz_attempts = "DESCRIBE quiz_attempts";
if ($stmt = $conn->prepare($describe_quiz_attempts)) {
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr><td><strong>{$row['Field']}</strong></td><td>{$row['Type']}</td><td>{$row['Null']}</td><td>{$row['Key']}</td></tr>";
    }
    echo "</table>";
    $stmt->close();
}

$quiz_attempts_query = "
    SELECT 
        qa.quiz_id,
        qa.score,
        qa.pass_status,
        qa.created_at,
        q.title as quiz_title,
        q.course_id,
        c.title as course_title,
        q.passing_score
    FROM quiz_attempts qa
    JOIN quizzes q ON qa.quiz_id = q.quiz_id
    LEFT JOIN courses c ON q.course_id = c.course_id
    WHERE qa.student_id = ?
    ORDER BY qa.created_at DESC
";
if ($stmt = $conn->prepare($quiz_attempts_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Quiz ID</th><th>Quiz Title</th><th>Course ID</th><th>Course Title</th><th>Score</th><th>Passing</th><th>Status</th><th>Date</th></tr>";
    while ($row = $result->fetch_assoc()) {
        $status_color = $row['pass_status'] ? 'background: #d4edda;' : 'background: #f8d7da;';
        echo "<tr style='{$status_color}'>";
        echo "<td>{$row['quiz_id']}</td>";
        echo "<td>" . htmlspecialchars($row['quiz_title']) . "</td>";
        echo "<td><a href='?course_id=" . base64_encode($row['course_id']) . "'>{$row['course_id']}</a></td>";
        echo "<td>" . htmlspecialchars($row['course_title'] ?? 'Unknown') . "</td>";
        echo "<td>{$row['score']}%</td>";
        echo "<td>{$row['passing_score']}%</td>";
        echo "<td>" . ($row['pass_status'] ? 'PASSED' : 'FAILED') . "</td>";
        echo "<td>{$row['created_at']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    $stmt->close();
}

// ==========================================
// 0.5. COURSE REQUIREMENTS & PASS MARKS (CRITICAL!)
// ==========================================
echo "<h2>📋 COURSE REQUIREMENTS & PASS MARKS</h2>";

// Show course details and pass marks - FIX: Use actual course table columns
echo "<h3>Course Details & Requirements</h3>";
$course_details_query = "
    SELECT 
        course_id,
        title,
        description,
        course_passing_grade,
        assignment_weight,
        quiz_weight,
        created_at,
        updated_at
    FROM courses 
    WHERE course_id = ?
";
echo "<p><strong>DEBUG:</strong> Checking actual courses table structure...</p>";

// First, let's see what columns actually exist in courses table
$describe_query = "DESCRIBE courses";
if ($stmt = $conn->prepare($describe_query)) {
    $stmt->execute();
    $result = $stmt->get_result();
    echo "<h4>Actual Courses Table Structure:</h4>";
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><strong>{$row['Field']}</strong></td>";
        echo "<td>{$row['Type']}</td>";
        echo "<td>{$row['Null']}</td>";
        echo "<td>{$row['Key']}</td>";
        echo "<td>" . ($row['Default'] ?? 'NULL') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    $stmt->close();
}
if ($stmt = $conn->prepare($course_details_query)) {
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($course_info = $result->fetch_assoc()) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Field</th><th>Value</th></tr>";
        echo "<tr><td><strong>Course ID</strong></td><td>{$course_info['course_id']}</td></tr>";
        echo "<tr><td><strong>Title</strong></td><td>" . htmlspecialchars($course_info['title']) . "</td></tr>";
        echo "<tr><td><strong>Course Passing Grade</strong></td><td><span style='color: green; font-weight: bold;'>{$course_info['course_passing_grade']}%</span></td></tr>";
        echo "<tr><td><strong>Assignment Weight</strong></td><td>{$course_info['assignment_weight']}%</td></tr>";
        echo "<tr><td><strong>Quiz Weight</strong></td><td>{$course_info['quiz_weight']}%</td></tr>";
        echo "<tr><td><strong>Created</strong></td><td>{$course_info['created_at']}</td></tr>";
        echo "</table>";
        
        // Store for later calculations - USE REAL COURSE PASSING GRADE
        $course_pass_mark = $course_info['course_passing_grade'] ?? 60;
        echo "<p style='color: blue; font-weight: bold;'>🎯 <strong>Effective Course Pass Mark: {$course_pass_mark}%</strong></p>";
    } else {
        echo "<p style='color: red;'>❌ Course {$course_id} not found!</p>";
        $course_pass_mark = 70; // Default
    }
    $stmt->close();
} else {
    echo "<p style='color: red;'>❌ Failed to query course details</p>";
    $course_pass_mark = 70; // Default
}

// Show all quizzes in this course with their pass marks
echo "<h3>All Quizzes in Course {$course_id} (With Pass Marks)</h3>";
$all_quizzes_query = "
    SELECT 
        q.quiz_id,
        q.title,
        q.course_id,
        q.video_id,
        q.passing_score,
        q.time_limit,
        q.max_attempts,
        q.is_published,
        v.title as video_title
    FROM quizzes q
    LEFT JOIN videos v ON q.video_id = v.video_id
    WHERE q.course_id = ?
    ORDER BY q.quiz_id
";
if ($stmt = $conn->prepare($all_quizzes_query)) {
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $all_course_quizzes = [];
    while ($row = $result->fetch_assoc()) {
        $all_course_quizzes[] = $row;
    }
    
    if (empty($all_course_quizzes)) {
        echo "<p style='color: orange;'>⚠️ NO QUIZZES found in course {$course_id}</p>";
    } else {
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Quiz ID</th><th>Quiz Title</th><th>Video ID</th><th>Video Title</th><th>Pass Mark</th><th>Time Limit</th><th>Max Attempts</th><th>Published</th></tr>";
        
        foreach ($all_course_quizzes as $quiz) {
            $published_color = $quiz['is_published'] ? '' : 'background: #fff3cd;';
            echo "<tr style='{$published_color}'>";
            echo "<td>{$quiz['quiz_id']}</td>";
            echo "<td>" . htmlspecialchars($quiz['title']) . "</td>";
            echo "<td>{$quiz['video_id']}</td>";
            echo "<td>" . htmlspecialchars($quiz['video_title'] ?? 'N/A') . "</td>";
            echo "<td><strong>{$quiz['passing_score']}%</strong></td>";
            echo "<td>{$quiz['time_limit']} min</td>";
            echo "<td>{$quiz['max_attempts']}</td>";
            echo "<td>" . ($quiz['is_published'] ? 'YES' : 'NO') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    $stmt->close();
}

echo "<hr>";
echo "<h2>📊 ANALYSIS FOR COURSE {$course_id}</h2>";

// ==========================================
// 1. VIDEO COMPLETION ANALYSIS
// ==========================================
echo "<h2>📹 VIDEO COMPLETION ANALYSIS</h2>";

// Check both tables to see which one has data
echo "<h3>Table: video_progress (OLD)</h3>";
$video_progress_query = "SELECT * FROM video_progress WHERE student_id = ? LIMIT 5";
if ($stmt = $conn->prepare($video_progress_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $video_progress_data = [];
    while ($row = $result->fetch_assoc()) {
        $video_progress_data[] = $row;
    }
    
    if (empty($video_progress_data)) {
        echo "<p style='color: red;'>❌ NO DATA in video_progress table for user {$user_id}</p>";
    } else {
        echo "<p style='color: green;'>✅ Found " . count($video_progress_data) . " records in video_progress</p>";
        echo "<pre>" . print_r(array_slice($video_progress_data, 0, 2), true) . "</pre>";
    }
    $stmt->close();
}

echo "<h3>Table: user_video_completions (NEW)</h3>";
$user_video_completions_query = "SELECT * FROM user_video_completions WHERE user_id = ? LIMIT 5";
if ($stmt = $conn->prepare($user_video_completions_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user_video_completions_data = [];
    while ($row = $result->fetch_assoc()) {
        $user_video_completions_data[] = $row;
    }
    
    if (empty($user_video_completions_data)) {
        echo "<p style='color: red;'>❌ NO DATA in user_video_completions table for user {$user_id}</p>";
    } else {
        echo "<p style='color: green;'>✅ Found " . count($user_video_completions_data) . " records in user_video_completions</p>";
        echo "<pre>" . print_r(array_slice($user_video_completions_data, 0, 2), true) . "</pre>";
    }
    $stmt->close();
}

// Check video completion for this specific course
echo "<h3>Course Videos Analysis</h3>";
$course_videos_query = "
    SELECT 
        v.video_id,
        v.title,
        v.course_id,
        -- OLD table
        vp.completion_percentage as old_progress,
        vp.completed as old_completed,
        -- NEW table  
        uvc.completed_at as new_completed_at,
        uvc.completion_percentage as new_progress
    FROM videos v
    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
    LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id AND uvc.user_id = ?
    WHERE v.course_id = ?
    ORDER BY v.video_order
";

if ($stmt = $conn->prepare($course_videos_query)) {
    $stmt->bind_param("iii", $user_id, $user_id, $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $course_videos = [];
    while ($row = $result->fetch_assoc()) {
        $course_videos[] = $row;
    }
    
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr><th>Video ID</th><th>Title</th><th>OLD Progress</th><th>OLD Completed</th><th>NEW Progress</th><th>NEW Completed</th></tr>";
    
    foreach ($course_videos as $video) {
        echo "<tr>";
        echo "<td>{$video['video_id']}</td>";
        echo "<td>" . htmlspecialchars($video['title']) . "</td>";
        echo "<td>" . ($video['old_progress'] ?? 'NULL') . "%</td>";
        echo "<td>" . ($video['old_completed'] ? 'YES' : 'NO') . "</td>";
        echo "<td>" . ($video['new_progress'] ?? 'NULL') . "%</td>";
        echo "<td>" . ($video['new_completed_at'] ? 'YES' : 'NO') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    $stmt->close();
}

// ==========================================
// 2. QUIZ COMPLETION ANALYSIS
// ==========================================
echo "<hr><h2>❓ QUIZ COMPLETION ANALYSIS</h2>";

// Get all quizzes for this course
echo "<h3>Course Quizzes</h3>";
$course_quizzes_query = "
    SELECT 
        q.quiz_id,
        q.title,
        q.video_id,
        q.course_id,
        q.passing_score,
        q.time_limit,
        -- Student attempts
        qa.attempt_id,
        qa.score,
        qa.pass_status,
        qa.created_at as attempt_date
    FROM quizzes q
    LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
    WHERE q.course_id = ?
    ORDER BY q.quiz_id, qa.created_at DESC
";

if ($stmt = $conn->prepare($course_quizzes_query)) {
    $stmt->bind_param("ii", $user_id, $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $course_quizzes = [];
    while ($row = $result->fetch_assoc()) {
        $course_quizzes[] = $row;
    }
    
    if (empty($course_quizzes)) {
        echo "<p style='color: red;'>❌ NO QUIZZES found for course {$course_id}</p>";
    } else {
        echo "<p style='color: green;'>✅ Found " . count($course_quizzes) . " quiz records</p>";
        
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Quiz ID</th><th>Title</th><th>Video ID</th><th>Passing Score</th><th>Student Score</th><th>Pass Status</th><th>Attempt Date</th></tr>";
        
        foreach ($course_quizzes as $quiz) {
            $status_color = '';
            if ($quiz['pass_status'] === '1' || $quiz['pass_status'] === 1) {
                $status_color = 'background: #d4edda;'; // Green for passed
            } elseif ($quiz['attempt_id']) {
                $status_color = 'background: #f8d7da;'; // Red for failed
            } else {
                $status_color = 'background: #fff3cd;'; // Yellow for pending
            }
            
            echo "<tr style='{$status_color}'>";
            echo "<td>{$quiz['quiz_id']}</td>";
            echo "<td>" . htmlspecialchars($quiz['title']) . "</td>";
            echo "<td>{$quiz['video_id']}</td>";
            echo "<td>{$quiz['passing_score']}%</td>";
            echo "<td>" . ($quiz['score'] ?? 'N/A') . "%</td>";
            echo "<td>" . ($quiz['pass_status'] ? 'PASSED' : ($quiz['attempt_id'] ? 'FAILED' : 'PENDING')) . "</td>";
            echo "<td>" . ($quiz['attempt_date'] ?? 'Never attempted') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    $stmt->close();
}

// ==========================================
// 3. COURSE COMPLETION CALCULATIONS
// ==========================================
echo "<hr><h2>📊 COURSE COMPLETION CALCULATIONS</h2>";

// Calculate video completion percentage
$total_videos = count($course_videos);
$completed_videos_old = 0;
$completed_videos_new = 0;

foreach ($course_videos as $video) {
    if ($video['old_completed']) $completed_videos_old++;
    if ($video['new_completed_at']) $completed_videos_new++;
}

$video_completion_old = $total_videos > 0 ? round(($completed_videos_old / $total_videos) * 100) : 0;
$video_completion_new = $total_videos > 0 ? round(($completed_videos_new / $total_videos) * 100) : 0;

echo "<h3>Video Completion</h3>";
echo "<p><strong>Total Videos:</strong> {$total_videos}</p>";
echo "<p><strong>Completed (OLD table):</strong> {$completed_videos_old} ({$video_completion_old}%)</p>";
echo "<p><strong>Completed (NEW table):</strong> {$completed_videos_new} ({$video_completion_new}%)</p>";

// Calculate quiz completion
$unique_quizzes = [];
$passed_quizzes = 0;
$attempted_quizzes = 0;

foreach ($course_quizzes as $quiz) {
    if (!isset($unique_quizzes[$quiz['quiz_id']])) {
        $unique_quizzes[$quiz['quiz_id']] = $quiz;
        if ($quiz['attempt_id']) $attempted_quizzes++;
        if ($quiz['pass_status']) $passed_quizzes++;
    }
}

$total_quizzes = count($unique_quizzes);
$quiz_completion = $total_quizzes > 0 ? round(($passed_quizzes / $total_quizzes) * 100) : 100; // 100% if no quizzes

echo "<h3>Quiz Completion</h3>";
echo "<p><strong>Total Quizzes:</strong> {$total_quizzes}</p>";
echo "<p><strong>Attempted:</strong> {$attempted_quizzes}</p>";
echo "<p><strong>Passed:</strong> {$passed_quizzes}</p>";
echo "<p><strong>Quiz Completion:</strong> {$quiz_completion}%</p>";

// Overall course completion
echo "<h3>Overall Course Completion</h3>";
echo "<p><strong>Videos Completed (OLD):</strong> {$video_completion_old}%</p>";
echo "<p><strong>Videos Completed (NEW):</strong> {$video_completion_new}%</p>";
echo "<p><strong>Quizzes Completed:</strong> {$quiz_completion}%</p>";

// Check minimum requirements - USE ACTUAL COURSE PASS MARK
$minimum_required = $course_pass_mark; // Use actual course pass mark from database
$videos_meet_requirement_old = $video_completion_old >= 100;
$videos_meet_requirement_new = $video_completion_new >= 100;
$quizzes_meet_requirement = $quiz_completion >= $minimum_required;

echo "<h3>Certificate Eligibility</h3>";
echo "<p><strong>Minimum Required:</strong> {$minimum_required}% + All Videos</p>";
echo "<p><strong>Videos Complete (OLD):</strong> " . ($videos_meet_requirement_old ? '✅ YES' : '❌ NO') . "</p>";
echo "<p><strong>Videos Complete (NEW):</strong> " . ($videos_meet_requirement_new ? '✅ YES' : '❌ NO') . "</p>";
echo "<p><strong>Quizzes Meet Requirement:</strong> " . ($quizzes_meet_requirement ? '✅ YES' : '❌ NO') . "</p>";

$certificate_eligible_old = $videos_meet_requirement_old && $quizzes_meet_requirement;
$certificate_eligible_new = $videos_meet_requirement_new && $quizzes_meet_requirement;

echo "<p style='font-size: 18px; font-weight: bold;'>";
echo "<strong>Certificate Eligible (OLD logic):</strong> " . ($certificate_eligible_old ? '✅ YES' : '❌ NO') . "<br>";
echo "<strong>Certificate Eligible (NEW logic):</strong> " . ($certificate_eligible_new ? '✅ YES' : '❌ NO');
echo "</p>";

// ==========================================
// 4. RECOMMENDED FIXES
// ==========================================
echo "<hr><h2>🔧 RECOMMENDED FIXES</h2>";
echo "<ol>";
echo "<li><strong>Fix Table Usage:</strong> Use 'user_video_completions' instead of 'video_progress'</li>";
echo "<li><strong>Implement Quiz Calculation:</strong> Add proper \$quiz_requirements_met calculation</li>";
echo "<li><strong>Fix Certificate Logic:</strong> Ensure both video AND quiz requirements are met</li>";
echo "<li><strong>Add Proper Validation:</strong> Check for actual quiz attempts and pass status</li>";
echo "</ol>";

echo "<h3>Current Issues:</h3>";
echo "<ul>";
if ($total_videos > 0 && $completed_videos_old != $completed_videos_new) {
    echo "<li>❌ Video completion mismatch between old and new tables</li>";
}
if ($total_quizzes > 0 && $passed_quizzes != $total_quizzes) {
    echo "<li>❌ Not all quizzes passed ({$passed_quizzes}/{$total_quizzes})</li>";
}
if (!$certificate_eligible_new) {
    echo "<li>❌ Student not eligible for certificate</li>";
} else {
    echo "<li>⚠️ Student appears eligible but certificate logic may be using wrong data</li>";
}
echo "</ul>";

?>
<style>
body { font-family: Arial, sans-serif; margin: 20px; }
table { margin: 10px 0; }
th { background: #f8f9fa; padding: 8px; }
td { padding: 8px; }
h1 { color: #007bff; }
h2 { color: #28a745; border-bottom: 2px solid #28a745; }
h3 { color: #6f42c1; }
</style>
