<?php
/**
 * Debug Script for Enrolled Courses API
 * Access this file in your browser to see what's causing the error
 */

// Start with basic PHP error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>🔍 Debug: Enrolled Courses API</h2>";
echo "<style>body{font-family:Arial,sans-serif;margin:20px;} .error{color:red;} .success{color:green;} .info{color:blue;} pre{background:#f5f5f5;padding:10px;border-radius:5px;}</style>";

// Step 1: Check if session and authentication work
echo "<h3>1. Authentication Check</h3>";
try {
    require_once '../auth/functions.php';
    echo "<div class='success'>✓ Auth functions loaded</div>";
    
    if (isset($_SESSION['user_id'])) {
        echo "<div class='success'>✓ User logged in: ID = " . $_SESSION['user_id'] . "</div>";
        echo "<div class='info'>→ Role: " . ($_SESSION['role'] ?? 'not set') . "</div>";
    } else {
        echo "<div class='error'>✗ No user session found</div>";
        echo "<div class='info'>→ Please log in first</div>";
        exit;
    }
} catch (Exception $e) {
    echo "<div class='error'>✗ Auth error: " . $e->getMessage() . "</div>";
    exit;
}

// Step 2: Check database connection
echo "<h3>2. Database Connection</h3>";
try {
    require_once '../config/db_connect.php';
    if (isset($conn)) {
        echo "<div class='success'>✓ Database connected</div>";
    } else {
        echo "<div class='error'>✗ Database connection variable not set</div>";
        exit;
    }
} catch (Exception $e) {
    echo "<div class='error'>✗ Database error: " . $e->getMessage() . "</div>";
    exit;
}

// Step 3: Check if grade weighting columns exist
echo "<h3>3. Database Schema Check</h3>";
try {
    $result = $conn->query("DESCRIBE courses");
    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    
    echo "<div class='info'>Courses table columns: " . implode(', ', $columns) . "</div>";
    
    $has_assignment_weight = in_array('assignment_weight', $columns);
    $has_quiz_weight = in_array('quiz_weight', $columns);
    
    if ($has_assignment_weight && $has_quiz_weight) {
        echo "<div class='success'>✓ Grade weighting columns exist</div>";
    } else {
        echo "<div class='error'>✗ Missing grade weighting columns:</div>";
        if (!$has_assignment_weight) echo "<div class='error'>  - assignment_weight column missing</div>";
        if (!$has_quiz_weight) echo "<div class='error'>  - quiz_weight column missing</div>";
        echo "<div class='info'>→ You need to run the migration script first!</div>";
        echo "<div class='info'>→ Access: <a href='../teacher/add_grade_weighting_columns.php'>add_grade_weighting_columns.php</a></div>";
    }
} catch (Exception $e) {
    echo "<div class='error'>✗ Schema check error: " . $e->getMessage() . "</div>";
}

// Step 4: Check enrollments
echo "<h3>4. Enrollment Check</h3>";
try {
    $student_id = $_SESSION['user_id'];
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ?");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    $enrollment_count = $result['count'];
    echo "<div class='info'>Total enrollments: $enrollment_count</div>";
    
    if ($enrollment_count > 0) {
        echo "<div class='success'>✓ Student has enrollments</div>";
        
        // Get actual enrollment details
        $query = "
            SELECT c.course_id, c.title, e.enrollment_date 
            FROM courses c 
            JOIN enrollments e ON c.course_id = e.course_id 
            WHERE e.student_id = ?
        ";
        
        $stmt = $conn->prepare($query);
        if (!$stmt) {
            echo "<div class='error'>✗ SQL prepare error: " . $conn->error . "</div>";
            echo "<div class='info'>Query was: <pre>$query</pre></div>";
        } else {
            $stmt->bind_param("i", $student_id);
            $stmt->execute();
            $enrollments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        }
        
        echo "<div class='info'>Enrolled courses:</div>";
        echo "<pre>" . print_r($enrollments, true) . "</pre>";
    } else {
        echo "<div class='error'>✗ No enrollments found for this student</div>";
    }
} catch (Exception $e) {
    echo "<div class='error'>✗ Enrollment check error: " . $e->getMessage() . "</div>";
}

// Step 5: Check CourseGradeCalculator
echo "<h3>5. Grade Calculator Check</h3>";
try {
    require_once '../includes/course_grade_calculator.php';
    echo "<div class='success'>✓ CourseGradeCalculator class loaded</div>";
    
    $calculator = new CourseGradeCalculator($conn);
    echo "<div class='success'>✓ Calculator instance created</div>";
} catch (Exception $e) {
    echo "<div class='error'>✗ Calculator error: " . $e->getMessage() . "</div>";
    echo "<div class='info'>→ This might be due to missing grade weighting columns</div>";
}

// Step 6: Test the actual API call
echo "<h3>6. API Test</h3>";
if ($has_assignment_weight && $has_quiz_weight && $enrollment_count > 0) {
    try {
        // Simulate what the API does
        $student_id = $_SESSION['user_id'];
        
        $query = "
            SELECT 
                c.course_id,
                c.title,
                c.thumbnail,
                c.assignment_weight,
                c.quiz_weight,
                u.first_name as instructor_first,
                u.last_name as instructor_last,
                e.enrollment_date,
                e.status as enrollment_status
            FROM courses c
            JOIN enrollments e ON c.course_id = e.course_id
            JOIN users u ON c.teacher_id = u.user_id
            WHERE e.student_id = ?
            ORDER BY e.enrollment_date DESC
        ";
        
        $stmt = $conn->prepare($query);
        if (!$stmt) {
            echo "<div class='error'>✗ API SQL prepare error: " . $conn->error . "</div>";
            echo "<div class='info'>Query was: <pre>$query</pre></div>";
            throw new Exception("SQL prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $courses = [];
        while ($course = $result->fetch_assoc()) {
            $courses[] = $course;
        }
        
        echo "<div class='success'>✓ API query successful</div>";
        echo "<div class='info'>Found " . count($courses) . " courses</div>";
        echo "<pre>" . print_r($courses, true) . "</pre>";
        
    } catch (Exception $e) {
        echo "<div class='error'>✗ API test error: " . $e->getMessage() . "</div>";
    }
} else {
    echo "<div class='error'>⚠ Skipping API test due to previous errors</div>";
}

echo "<h3>🎯 Next Steps</h3>";
if (!$has_assignment_weight || !$has_quiz_weight) {
    echo "<div class='error'>1. Run the migration script to add grade weighting columns</div>";
    echo "<div class='info'>   → Access: <a href='../teacher/add_grade_weighting_columns.php'>add_grade_weighting_columns.php</a></div>";
}
if ($enrollment_count == 0) {
    echo "<div class='error'>2. Enroll in some courses to test the system</div>";
}
echo "<div class='info'>3. After fixing issues, test the API directly: <a href='api/get_enrolled_courses.php'>api/get_enrolled_courses.php</a></div>";
?>
