<?php
/**
 * DEBUG VERSION - Mark Video Completed API Endpoint
 * Shows all errors for debugging purposes
 */

// ENABLE ALL ERROR REPORTING FOR DEBUGGING
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('html_errors', 1);

// Start output buffering to capture any errors
ob_start();

// Set response header to JSON
header('Content-Type: application/json');

// 🔐 START SESSION TO ACCESS USER AUTHENTICATION  
require_once '../includes/session_start.php';

// Include database connection
require_once '../config/db_connect.php';

// Debug output function
function debugLog($message, $data = null) {
    $output = "🐛 DEBUG: " . $message;
    if ($data !== null) {
        $output .= " - " . json_encode($data);
    }
    error_log($output);
    echo "<!-- " . $output . " -->\n";
}

debugLog("Debug script started");

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Method not allowed', 'debug' => 'Not POST request']);
    exit;
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'User not authenticated', 'debug' => 'No user_id in session']);
    exit;
}

$user_id = $_SESSION['user_id'];
debugLog("User ID from session", $user_id);

// Get and validate input data
$video_id = $_POST['video_id'] ?? null;
$course_id = $_POST['course_id'] ?? null;
$auto_complete = isset($_POST['auto_complete']) ? (int)$_POST['auto_complete'] : 0;

debugLog("Input data", [
    'video_id' => $video_id, 
    'course_id' => $course_id, 
    'auto_complete' => $auto_complete,
    'POST_data' => $_POST
]);

if (!$video_id || !$course_id) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters', 'debug' => 'video_id or course_id missing']);
    exit;
}

try {
    debugLog("Starting database operations");
    
    // Test database connection
    debugLog("Testing PDO connection", [
        'pdo_defined' => isset($pdo),
        'pdo_type' => isset($pdo) ? get_class($pdo) : 'null'
    ]);
    
    // Check if user is enrolled in the course
    debugLog("Checking enrollment");
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM enrollments WHERE user_id = ? AND course_id = ? LIMIT 1");
    $stmt->execute([$user_id, $course_id]);
    $enrollment_count = $stmt->fetchColumn();
    debugLog("Enrollment check result", $enrollment_count);
    
    if ($enrollment_count == 0) {
        echo json_encode(['success' => false, 'message' => 'User not enrolled in this course', 'debug' => 'Enrollment count is 0']);
        exit;
    }
    
    // Check if video belongs to course
    debugLog("Checking video belongs to course");
    $stmt = $pdo->prepare("SELECT video_id, title FROM videos WHERE video_id = ? AND course_id = ? LIMIT 1");
    $stmt->execute([$video_id, $course_id]);
    $video = $stmt->fetch(PDO::FETCH_ASSOC);
    debugLog("Video check result", $video);
    
    if (!$video) {
        echo json_encode(['success' => false, 'message' => 'Video does not belong to this course', 'debug' => 'Video not found']);
        exit;
    }
    
    // Check database tables
    debugLog("Checking database tables");
    $stmt = $pdo->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    debugLog("Available tables", $tables);
    
    // Check enrollments table structure
    debugLog("Checking enrollments table structure");
    $stmt = $pdo->query("DESCRIBE enrollments");
    $enrollments_structure = $stmt->fetchAll(PDO::FETCH_ASSOC);
    debugLog("Enrollments table structure", $enrollments_structure);
    
    // Check videos table structure  
    debugLog("Checking videos table structure");
    $stmt = $pdo->query("DESCRIBE videos");
    $videos_structure = $stmt->fetchAll(PDO::FETCH_ASSOC);
    debugLog("Videos table structure", $videos_structure);
    
    // Try to create user_video_completions table
    debugLog("Creating/checking user_video_completions table");
    $createTableQuery = "CREATE TABLE IF NOT EXISTS `user_video_completions` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `user_id` int(11) NOT NULL,
        `video_id` int(11) NOT NULL,
        `completed_at` datetime NOT NULL,
        `created_at` datetime NOT NULL,
        `updated_at` datetime NOT NULL,
        PRIMARY KEY (`id`),
        UNIQUE KEY `user_video_unique` (`user_id`, `video_id`),
        KEY `user_id` (`user_id`),
        KEY `video_id` (`video_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $result = $pdo->exec($createTableQuery);
    debugLog("Table creation result", $result);
    
    // Check user_video_completions table structure
    debugLog("Checking user_video_completions table structure");
    $stmt = $pdo->query("DESCRIBE user_video_completions");
    $completions_structure = $stmt->fetchAll(PDO::FETCH_ASSOC);
    debugLog("User_video_completions table structure", $completions_structure);
    
    echo json_encode([
        'success' => true, 
        'message' => 'Debug completed successfully',
        'debug_data' => [
            'user_id' => $user_id,
            'video_id' => $video_id,
            'course_id' => $course_id,
            'enrollment_count' => $enrollment_count,
            'video_found' => $video,
            'tables' => $tables,
            'enrollments_structure' => $enrollments_structure,
            'videos_structure' => $videos_structure,
            'completions_structure' => $completions_structure ?? null
        ]
    ]);
    
} catch (Exception $e) {
    debugLog("Exception caught", [
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ]);
    
    echo json_encode([
        'success' => false, 
        'message' => 'Exception: ' . $e->getMessage(),
        'debug' => [
            'error_message' => $e->getMessage(),
            'error_file' => $e->getFile(),
            'error_line' => $e->getLine(),
            'stack_trace' => $e->getTraceAsString()
        ]
    ]);
} catch (PDOException $e) {
    debugLog("PDO Exception caught", [
        'message' => $e->getMessage(),
        'code' => $e->getCode()
    ]);
    
    echo json_encode([
        'success' => false, 
        'message' => 'Database error: ' . $e->getMessage(),
        'debug' => [
            'error_message' => $e->getMessage(),
            'error_code' => $e->getCode()
        ]
    ]);
}

// Capture any output and add to response
$output = ob_get_clean();
if ($output) {
    debugLog("Captured output", $output);
}
?>
