<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quiz Enrollment Debug - Panadite Academy</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .debug-section { background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .debug-section h2 { color: #2563eb; border-bottom: 2px solid #2563eb; padding-bottom: 10px; }
        .debug-section h3 { color: #dc2626; }
        .success { color: #059669; font-weight: bold; }
        .error { color: #dc2626; font-weight: bold; }
        .warning { color: #d97706; font-weight: bold; }
        table { width: 100%; border-collapse: collapse; margin: 10px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .highlight { background-color: #fef3c7; }
    </style>
</head>
<body>
    <h1>🔍 Quiz & Enrollment Debug Analysis</h1>
    <p><strong>Current Student ID:</strong> <?= $student_id ?></p>
    <p><strong>Debug Time:</strong> <?= date('Y-m-d H:i:s') ?></p>
    
    <div class="debug-section">
        <h2>1. Student Enrollment Check</h2>
        <?php
        // First, check if connection exists
        if (!$conn) {
            echo "<p class='error'>❌ Database connection failed!</p>";
        } else {
            echo "<p class='success'>✅ Database connected</p>";
        }
        
        // Simple enrollment query
        $enrollment_query = "SELECT e.enrollment_id, e.course_id, c.title as course_title, e.enrolled_at 
                           FROM enrollments e 
                           JOIN courses c ON e.course_id = c.course_id 
                           WHERE e.student_id = " . intval($student_id) . " 
                           ORDER BY e.enrolled_at DESC";
        
        echo "<p><strong>Query:</strong> " . $enrollment_query . "</p>";
        
        $result = $conn->query($enrollment_query);
        
        if (!$result) {
            echo "<p class='error'>❌ Query failed: " . $conn->error . "</p>";
        } else {
        
        if ($result->num_rows > 0) {
            echo "<p class='success'>✅ Student is enrolled in " . $result->num_rows . " course(s):</p>";
            echo "<table>";
            echo "<tr><th>Enrollment ID</th><th>Course ID</th><th>Course Title</th><th>Status</th><th>Enrolled Date</th></tr>";
            
            $enrolled_courses = [];
            while($row = $result->fetch_assoc()) {
                $enrolled_courses[] = $row['course_id'];
                echo "<tr class='highlight'>";
                echo "<td>" . $row['enrollment_id'] . "</td>";
                echo "<td>" . $row['course_id'] . "</td>";
                echo "<td>" . $row['course_title'] . "</td>";
                echo "<td>" . $row['status'] . "</td>";
                echo "<td>" . $row['enrolled_at'] . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p class='error'>❌ Student is NOT enrolled in any courses!</p>";
            $enrolled_courses = [];
        }
        }
        ?>
    </div>

    <div class="debug-section">
        <h2>2. All Published Quizzes in Database</h2>
        <?php
        $all_quizzes_query = "SELECT q.quiz_id, q.title, q.course_id, c.title as course_title, q.is_published, u.first_name, u.last_name
                             FROM quizzes q 
                             JOIN courses c ON q.course_id = c.course_id 
                             JOIN users u ON c.teacher_id = u.user_id
                             WHERE q.is_published = 1
                             ORDER BY q.course_id, q.quiz_id";
        
        echo "<p><strong>Query:</strong> " . $all_quizzes_query . "</p>";
        $result = $conn->query($all_quizzes_query);
        
        if (!$result) {
            echo "<p class='error'>❌ Query failed: " . $conn->error . "</p>";
        } else {
        
        if ($result->num_rows > 0) {
            echo "<p class='success'>✅ Found " . $result->num_rows . " published quiz(es) in database:</p>";
            echo "<table>";
            echo "<tr><th>Quiz ID</th><th>Quiz Title</th><th>Course ID</th><th>Course Title</th><th>Teacher</th><th>Student Can Access?</th></tr>";
            
            while($quiz = $result->fetch_assoc()) {
                $can_access = in_array($quiz['course_id'], $enrolled_courses) ? 'YES' : 'NO';
                $row_class = $can_access == 'YES' ? 'highlight' : '';
                
                echo "<tr class='$row_class'>";
                echo "<td>" . $quiz['quiz_id'] . "</td>";
                echo "<td>" . $quiz['title'] . "</td>";
                echo "<td>" . $quiz['course_id'] . "</td>";
                echo "<td>" . $quiz['course_title'] . "</td>";
                echo "<td>" . $quiz['first_name'] . ' ' . $quiz['last_name'] . "</td>";
                echo "<td class='" . ($can_access == 'YES' ? 'success' : 'error') . "'>" . $can_access . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p class='error'>❌ No published quizzes found in database!</p>";
        }
        }
        ?>
    </div>

    <div class="debug-section">
        <h2>3. Quizzes Student Should See (Enrolled Courses Only)</h2>
        <?php
        if (!empty($enrolled_courses)) {
            $course_ids_str = implode(',', array_map('intval', $enrolled_courses));
            $student_quizzes_query = "SELECT q.quiz_id, q.title, q.course_id, c.title as course_title, q.time_limit, q.passing_score
                                    FROM quizzes q 
                                    JOIN courses c ON q.course_id = c.course_id 
                                    WHERE q.is_published = 1 AND q.course_id IN ($course_ids_str)
                                    ORDER BY q.course_id, q.quiz_id";
            
            echo "<p><strong>Query:</strong> " . $student_quizzes_query . "</p>";
            $result = $conn->query($student_quizzes_query);
            
            if (!$result) {
                echo "<p class='error'>❌ Query failed: " . $conn->error . "</p>";
            } else {
            
            if ($result->num_rows > 0) {
                echo "<p class='success'>✅ Student should see " . $result->num_rows . " quiz(es):</p>";
                echo "<table>";
                echo "<tr><th>Quiz ID</th><th>Quiz Title</th><th>Course</th><th>Time Limit</th><th>Pass Score</th></tr>";
                
                while($quiz = $result->fetch_assoc()) {
                    echo "<tr class='highlight'>";
                    echo "<td>" . $quiz['quiz_id'] . "</td>";
                    echo "<td>" . $quiz['title'] . "</td>";
                    echo "<td>" . $quiz['course_title'] . " (ID: " . $quiz['course_id'] . ")</td>";
                    echo "<td>" . $quiz['time_limit'] . " min</td>";
                    echo "<td>" . $quiz['passing_score'] . "%</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p class='warning'>⚠️ No quizzes found in enrolled courses!</p>";
                echo "<p>This means:</p>";
                echo "<ul>";
                echo "<li>Either no quizzes have been created for the courses you're enrolled in</li>";
                echo "<li>Or the quizzes in your courses are not published yet</li>";
                echo "</ul>";
            }
            }
        } else {
            echo "<p class='error'>❌ Cannot check - student not enrolled in any courses!</p>";
        }
        ?>
    </div>

    <div class="debug-section">
        <h2>4. Current Quiz Page Query Test</h2>
        <?php
        // Test the exact query used in all_quizzes.php (simplified)
        $current_query = "
            SELECT 
                q.quiz_id,
                q.title,
                q.course_id,
                c.title as course_title,
                u.first_name,
                u.last_name
            FROM quizzes q
            JOIN courses c ON q.course_id = c.course_id
            JOIN users u ON c.teacher_id = u.user_id
            JOIN enrollments e ON c.course_id = e.course_id AND e.student_id = " . intval($student_id) . "
            WHERE q.is_published = 1
            ORDER BY q.created_at DESC
        ";
        
        echo "<p><strong>Query:</strong> " . $current_query . "</p>";
        $result = $conn->query($current_query);
        
        if (!$result) {
            echo "<p class='error'>❌ Query failed: " . $conn->error . "</p>";
        } else {
        
        echo "<p><strong>Current all_quizzes.php query returns:</strong> " . $result->num_rows . " quiz(es)</p>";
        
        if ($result->num_rows > 0) {
            echo "<table>";
            echo "<tr><th>Quiz ID</th><th>Title</th><th>Course</th><th>Teacher</th></tr>";
            
            while($quiz = $result->fetch_assoc()) {
                echo "<tr>";
                echo "<td>" . $quiz['quiz_id'] . "</td>";
                echo "<td>" . $quiz['title'] . "</td>";
                echo "<td>" . $quiz['course_title'] . " (ID: " . $quiz['course_id'] . ")</td>";
                echo "<td>" . $quiz['first_name'] . ' ' . $quiz['last_name'] . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p class='warning'>⚠️ Current query returns no results!</p>";
        }
        }
        ?>
    </div>

    <div class="debug-section">
        <h2>5. Problem Analysis & Recommendation</h2>
        <?php
        echo "<h3>Summary:</h3>";
        echo "<ul>";
        
        if (empty($enrolled_courses)) {
            echo "<li class='error'>❌ CRITICAL: Student not enrolled in any courses</li>";
            echo "<li>➡️ <strong>Solution:</strong> Enroll student in courses first</li>";
        } else {
            echo "<li class='success'>✅ Student enrolled in " . count($enrolled_courses) . " course(s): " . implode(', ', $enrolled_courses) . "</li>";
            
            // Check if enrolled courses have quizzes
            $course_ids_str = implode(',', array_map('intval', $enrolled_courses));
            $quiz_count_query = "SELECT COUNT(*) as quiz_count FROM quizzes WHERE is_published = 1 AND course_id IN ($course_ids_str)";
            $quiz_count_result = $conn->query($quiz_count_query);
            $quiz_count = $quiz_count_result->fetch_assoc()['quiz_count'];
            
            if ($quiz_count == 0) {
                echo "<li class='warning'>⚠️ No published quizzes in enrolled courses</li>";
                echo "<li>➡️ <strong>Solution:</strong> Teachers need to create and publish quizzes for enrolled courses</li>";
            } else {
                echo "<li class='success'>✅ Found $quiz_count published quiz(es) in enrolled courses</li>";
                echo "<li>➡️ Quiz system should work correctly</li>";
            }
        }
        
        echo "</ul>";
        ?>
    </div>

    <p><a href="all_quizzes.php" style="background: #2563eb; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;">← Back to Quiz Page</a></p>
</body>
</html>
