<?php
/**
 * ISOLATED DATABASE TABLE INSPECTOR
 * Shows exact table columns and structure for video completion debugging
 */

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include database connection
require_once '../config/db_connect.php';

echo "<h1>🔍 DATABASE TABLE STRUCTURE INSPECTOR</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; margin: 20px; }
    .table-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; background: #f9f9f9; }
    .table-name { color: #007bff; font-weight: bold; font-size: 1.2em; }
    table { width: 100%; border-collapse: collapse; margin: 10px 0; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background-color: #007bff; color: white; }
    .success { color: green; font-weight: bold; }
    .error { color: red; font-weight: bold; }
    .info { color: blue; }
</style>";

// Tables to inspect
$tables_to_check = [
    'user_video_completions',
    'video_progress', 
    'videos',
    'enrollments',
    'courses',
    'users'
];

foreach ($tables_to_check as $table_name) {
    echo "<div class='table-section'>";
    echo "<h2 class='table-name'>📊 TABLE: $table_name</h2>";
    
    // Check if table exists
    $check_table = $conn->query("SHOW TABLES LIKE '$table_name'");
    if ($check_table->num_rows == 0) {
        echo "<p class='error'>❌ TABLE DOES NOT EXIST</p>";
        echo "</div>";
        continue;
    }
    
    echo "<p class='success'>✅ Table exists</p>";
    
    // Get table structure
    $describe_query = "DESCRIBE $table_name";
    $result = $conn->query($describe_query);
    
    if ($result) {
        echo "<table>";
        echo "<tr><th>Column Name</th><th>Data Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td><strong>" . htmlspecialchars($row['Field']) . "</strong></td>";
            echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Default']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Extra']) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Get row count
        $count_query = "SELECT COUNT(*) as count FROM $table_name";
        $count_result = $conn->query($count_query);
        if ($count_result) {
            $count_data = $count_result->fetch_assoc();
            echo "<p class='info'>📈 Total rows: " . $count_data['count'] . "</p>";
        }
        
        // Show sample data for completion tables
        if (in_array($table_name, ['user_video_completions', 'video_progress'])) {
            echo "<h4>📝 Sample Data (Last 5 rows):</h4>";
            $sample_query = "SELECT * FROM $table_name ORDER BY id DESC LIMIT 5";
            $sample_result = $conn->query($sample_query);
            if ($sample_result && $sample_result->num_rows > 0) {
                echo "<table>";
                $first_row = true;
                while ($sample_row = $sample_result->fetch_assoc()) {
                    if ($first_row) {
                        echo "<tr>";
                        foreach (array_keys($sample_row) as $header) {
                            echo "<th>$header</th>";
                        }
                        echo "</tr>";
                        $first_row = false;
                    }
                    echo "<tr>";
                    foreach ($sample_row as $value) {
                        echo "<td>" . htmlspecialchars($value ?? 'NULL') . "</td>";
                    }
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p class='info'>No data found in table</p>";
            }
        }
        
    } else {
        echo "<p class='error'>❌ Error describing table: " . $conn->error . "</p>";
    }
    
    echo "</div>";
}

// Additional debugging queries
echo "<div class='table-section'>";
echo "<h2 class='table-name'>🔍 SPECIFIC DEBUGGING QUERIES</h2>";

// Check for video completion data for current course
echo "<h4>📊 Video Completion Data Analysis:</h4>";

// Check if we can find any completion records
$completion_check = "
SELECT 
    'user_video_completions' as source_table,
    COUNT(*) as total_records,
    COUNT(DISTINCT user_id) as unique_users,
    COUNT(DISTINCT video_id) as unique_videos
FROM user_video_completions 
WHERE 1=1
UNION ALL
SELECT 
    'video_progress' as source_table,
    COUNT(*) as total_records,
    COUNT(DISTINCT student_id) as unique_users,
    COUNT(DISTINCT video_id) as unique_videos
FROM video_progress 
WHERE 1=1
";

$completion_result = $conn->query($completion_check);
if ($completion_result) {
    echo "<table>";
    echo "<tr><th>Source Table</th><th>Total Records</th><th>Unique Users</th><th>Unique Videos</th></tr>";
    while ($row = $completion_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($row['source_table']) . "</td>";
        echo "<td>" . htmlspecialchars($row['total_records']) . "</td>";
        echo "<td>" . htmlspecialchars($row['unique_users']) . "</td>";
        echo "<td>" . htmlspecialchars($row['unique_videos']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p class='error'>❌ Error running completion analysis: " . $conn->error . "</p>";
}

// Check videos table structure for YouTube detection
echo "<h4>📹 YouTube Video Detection Analysis:</h4>";
$youtube_check = "
SELECT 
    video_id,
    title,
    video_url,
    video_path,
    CASE 
        WHEN video_url LIKE '%youtube.com%' OR video_url LIKE '%youtu.be%' THEN 'YouTube'
        WHEN video_url LIKE '%vimeo.com%' THEN 'Vimeo'
        WHEN video_url IS NOT NULL AND video_url != '' THEN 'External URL'
        WHEN video_path IS NOT NULL AND video_path != '' THEN 'Local File'
        ELSE 'No Video Source'
    END as detected_type,
    video_url,
    video_path
FROM videos 
WHERE course_id = (SELECT course_id FROM courses LIMIT 1)
LIMIT 10
";

$youtube_result = $conn->query($youtube_check);
if ($youtube_result) {
    echo "<table>";
    echo "<tr><th>Video ID</th><th>Title</th><th>Detected Type</th><th>Video URL</th><th>Video Path</th></tr>";
    while ($row = $youtube_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($row['video_id']) . "</td>";
        echo "<td>" . htmlspecialchars(substr($row['title'], 0, 30) . '...') . "</td>";
        echo "<td><strong>" . htmlspecialchars($row['detected_type']) . "</strong></td>";
        echo "<td>" . htmlspecialchars(substr($row['video_url'] ?? 'NULL', 0, 50) . '...') . "</td>";
        echo "<td>" . htmlspecialchars($row['video_path'] ?? 'NULL') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p class='error'>❌ Error running YouTube analysis: " . $conn->error . "</p>";
}

echo "</div>";

// Connection info
echo "<div class='table-section'>";
echo "<h2 class='table-name'>🔧 CONNECTION INFORMATION</h2>";
echo "<p class='info'>Connected to database: " . $conn->server_info . "</p>";
echo "<p class='info'>Current database: " . (isset($database) ? $database : 'Unknown') . "</p>";
echo "<p class='info'>Script executed at: " . date('Y-m-d H:i:s') . "</p>";
echo "</div>";

// Close connection
$conn->close();

echo "<p style='text-align: center; color: #007bff; font-weight: bold; margin-top: 30px;'>
🔍 Table inspection completed! Use this information to debug video completion issues.
</p>";
?>
