<?php
// Video Debug Script - Comprehensive Analysis
session_start();
require_once 'includes/config.php';

echo "<h1>Video System Debug Analysis</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; margin: 20px; }
    .section { background: #f8f9fa; padding: 15px; margin: 10px 0; border-radius: 8px; }
    .success { color: #28a745; }
    .error { color: #dc3545; }
    .warning { color: #ffc107; }
    table { width: 100%; border-collapse: collapse; margin: 10px 0; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background: #e9ecef; }
    .json-data { background: #f8f9fa; padding: 10px; border-radius: 4px; white-space: pre-wrap; }
</style>";

// 1. Database Connection Check
echo "<div class='section'>";
echo "<h2>1. Database Connection Status</h2>";
if ($conn) {
    echo "<p class='success'>✅ Database connection successful</p>";
    echo "<p>Connection info: " . $conn->host_info . "</p>";
} else {
    echo "<p class='error'>❌ Database connection failed</p>";
    exit();
}
echo "</div>";

// 2. Videos Table Analysis
echo "<div class='section'>";
echo "<h2>2. Videos Table Structure & Data</h2>";

// Check if videos table exists
$table_check = $conn->query("SHOW TABLES LIKE 'videos'");
if ($table_check && $table_check->num_rows > 0) {
    echo "<p class='success'>✅ Videos table exists</p>";
    
    // Show table structure
    echo "<h3>Table Structure:</h3>";
    $structure = $conn->query("DESCRIBE videos");
    if ($structure) {
        echo "<table>";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
        while ($row = $structure->fetch_assoc()) {
            echo "<tr>";
            foreach ($row as $value) {
                echo "<td>" . htmlspecialchars($value ?? 'NULL') . "</td>";
            }
            echo "</tr>";
        }
        echo "</table>";
    }
    
    // Count videos
    $count_result = $conn->query("SELECT COUNT(*) as total FROM videos");
    $count = $count_result->fetch_assoc()['total'];
    echo "<p><strong>Total videos in database:</strong> " . $count . "</p>";
    
    // Show all videos data
    if ($count > 0) {
        echo "<h3>All Videos Data:</h3>";
        $videos_result = $conn->query("SELECT * FROM videos ORDER BY created_at DESC LIMIT 20");
        if ($videos_result && $videos_result->num_rows > 0) {
            echo "<table>";
            echo "<tr><th>ID</th><th>Course ID</th><th>Title</th><th>Video Path</th><th>Video URL</th><th>Duration</th><th>Created</th><th>Published</th></tr>";
            while ($video = $videos_result->fetch_assoc()) {
                echo "<tr>";
                echo "<td>" . $video['video_id'] . "</td>";
                echo "<td>" . $video['course_id'] . "</td>";
                echo "<td>" . htmlspecialchars($video['title']) . "</td>";
                echo "<td>" . htmlspecialchars($video['video_path'] ?? 'NULL') . "</td>";
                echo "<td>" . htmlspecialchars($video['video_url'] ?? 'NULL') . "</td>";
                echo "<td>" . htmlspecialchars($video['duration'] ?? 'NULL') . "</td>";
                echo "<td>" . $video['created_at'] . "</td>";
                echo "<td>" . ($video['is_published'] ? 'Yes' : 'No') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    } else {
        echo "<p class='warning'>⚠️ No videos found in database</p>";
    }
    
} else {
    echo "<p class='error'>❌ Videos table does not exist</p>";
}
echo "</div>";

// 3. Course-Video Relationships
echo "<div class='section'>";
echo "<h2>3. Course-Video Relationships</h2>";

// Check specific course (ID 12)
$course_id = 12;
echo "<h3>Analysis for Course ID: $course_id</h3>";

// Get course info
$course_query = "SELECT * FROM courses WHERE course_id = ?";
$stmt = $conn->prepare($course_query);
$stmt->bind_param("i", $course_id);
$stmt->execute();
$course_result = $stmt->get_result();

if ($course_result && $course_result->num_rows > 0) {
    $course = $course_result->fetch_assoc();
    echo "<p class='success'>✅ Course found: " . htmlspecialchars($course['title']) . "</p>";
    echo "<p><strong>Course Details:</strong></p>";
    echo "<div class='json-data'>" . json_encode($course, JSON_PRETTY_PRINT) . "</div>";
    
    // Check for videos for this course
    $video_query = "SELECT * FROM videos WHERE course_id = ?";
    $video_stmt = $conn->prepare($video_query);
    $video_stmt->bind_param("i", $course_id);
    $video_stmt->execute();
    $video_result = $video_stmt->get_result();
    
    if ($video_result && $video_result->num_rows > 0) {
        echo "<p class='success'>✅ Found " . $video_result->num_rows . " videos for this course</p>";
        echo "<h4>Videos for Course ID $course_id:</h4>";
        echo "<table>";
        echo "<tr><th>Video ID</th><th>Title</th><th>Video Path</th><th>Video URL</th><th>Order</th><th>Published</th></tr>";
        while ($video = $video_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . $video['video_id'] . "</td>";
            echo "<td>" . htmlspecialchars($video['title']) . "</td>";
            echo "<td>" . htmlspecialchars($video['video_path'] ?? 'NULL') . "</td>";
            echo "<td>" . htmlspecialchars($video['video_url'] ?? 'NULL') . "</td>";
            echo "<td>" . ($video['video_order'] ?? 'NULL') . "</td>";
            echo "<td>" . ($video['is_published'] ? 'Yes' : 'No') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p class='warning'>⚠️ No videos found for course ID $course_id</p>";
    }
    
} else {
    echo "<p class='error'>❌ Course ID $course_id not found</p>";
}
echo "</div>";

// 4. File System Analysis
echo "<div class='section'>";
echo "<h2>4. File System Video Storage</h2>";

$video_directories = [
    'uploads/videos' => realpath(__DIR__ . '/uploads/videos'),
    'videos' => realpath(__DIR__ . '/videos'),
    'assets/videos' => realpath(__DIR__ . '/assets/videos'),
    'teacher/uploads' => realpath(__DIR__ . '/teacher/uploads'),
    'uploads' => realpath(__DIR__ . '/uploads')
];

echo "<h3>Checking Video Directories:</h3>";
foreach ($video_directories as $path_name => $full_path) {
    echo "<h4>Directory: $path_name</h4>";
    if ($full_path && is_dir($full_path)) {
        echo "<p class='success'>✅ Directory exists: $full_path</p>";
        
        // List files in directory
        $files = scandir($full_path);
        $video_files = array_filter($files, function($file) {
            $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            return in_array($ext, ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm']);
        });
        
        if (count($video_files) > 0) {
            echo "<p class='success'>Found " . count($video_files) . " video files:</p>";
            echo "<ul>";
            foreach ($video_files as $file) {
                $file_path = $full_path . '/' . $file;
                $file_size = filesize($file_path);
                $file_size_mb = round($file_size / 1024 / 1024, 2);
                echo "<li><strong>$file</strong> ({$file_size_mb} MB)</li>";
            }
            echo "</ul>";
        } else {
            echo "<p class='warning'>⚠️ No video files found in this directory</p>";
        }
        
        // Show all files for context
        $all_files = array_filter($files, function($file) {
            return $file !== '.' && $file !== '..';
        });
        if (count($all_files) > 0) {
            echo "<p><strong>All files in directory:</strong></p>";
            echo "<ul>";
            foreach ($all_files as $file) {
                echo "<li>$file</li>";
            }
            echo "</ul>";
        }
    } else {
        echo "<p class='error'>❌ Directory does not exist: $path_name</p>";
    }
}
echo "</div>";

// 5. Video URL Analysis
echo "<div class='section'>";
echo "<h2>5. Video URL & Path Analysis</h2>";

// Check what URLs are being generated in course-detail.php
$test_lesson_data = [
    'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
    'video_path' => '/uploads/videos/sample.mp4',
    'title' => 'Test Lesson'
];

echo "<h3>Sample Lesson Data Structure:</h3>";
echo "<div class='json-data'>" . json_encode($test_lesson_data, JSON_PRETTY_PRINT) . "</div>";

// Test URL accessibility
echo "<h3>URL Accessibility Tests:</h3>";
$test_urls = [
    'YouTube Embed' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
    'Local Video Path' => '/uploads/videos/sample.mp4',
    'Relative Path' => './uploads/videos/sample.mp4'
];

foreach ($test_urls as $label => $url) {
    echo "<p><strong>$label:</strong> $url</p>";
    if (strpos($url, 'http') === 0) {
        echo "<p class='success'>✅ External URL - should work if accessible</p>";
    } else {
        $local_path = realpath(__DIR__ . $url);
        if ($local_path && file_exists($local_path)) {
            echo "<p class='success'>✅ Local file exists: $local_path</p>";
        } else {
            echo "<p class='error'>❌ Local file not found: " . (__DIR__ . $url) . "</p>";
        }
    }
}
echo "</div>";

// 6. Current Session & User Context
echo "<div class='section'>";
echo "<h2>6. Current Session Context</h2>";
echo "<p><strong>Session Status:</strong> " . (session_status() === PHP_SESSION_ACTIVE ? 'Active' : 'Inactive') . "</p>";
if (isset($_SESSION['user_id'])) {
    echo "<p><strong>User ID:</strong> " . $_SESSION['user_id'] . "</p>";
    echo "<p><strong>User Role:</strong> " . ($_SESSION['user_role'] ?? 'Not set') . "</p>";
    
    // Check if user is enrolled in course 12
    $enrollment_check = $conn->prepare("SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?");
    $enrollment_check->bind_param("ii", $_SESSION['user_id'], $course_id);
    $enrollment_check->execute();
    $enrollment_result = $enrollment_check->get_result();
    
    if ($enrollment_result && $enrollment_result->num_rows > 0) {
        echo "<p class='success'>✅ User is enrolled in course $course_id</p>";
        $enrollment_data = $enrollment_result->fetch_assoc();
        echo "<div class='json-data'>" . json_encode($enrollment_data, JSON_PRETTY_PRINT) . "</div>";
    } else {
        echo "<p class='warning'>⚠️ User is NOT enrolled in course $course_id</p>";
    }
} else {
    echo "<p class='warning'>⚠️ No user session found</p>";
}
echo "</div>";

// 7. Recommendations
echo "<div class='section'>";
echo "<h2>7. Recommendations & Next Steps</h2>";
echo "<ol>";
echo "<li><strong>Video Storage:</strong> Videos should be stored in a dedicated directory like '/uploads/videos/' or '/videos/'</li>";
echo "<li><strong>Database References:</strong> Videos table should have proper file paths or URLs</li>";
echo "<li><strong>Fallback Content:</strong> When no real videos exist, use working YouTube embed URLs</li>";
echo "<li><strong>File Upload System:</strong> Implement proper video upload functionality in teacher panel</li>";
echo "<li><strong>Video Player:</strong> Ensure video player can handle both local files and external URLs</li>";
echo "</ol>";
echo "</div>";

echo "<div class='section'>";
echo "<h2>8. Quick Actions</h2>";
echo "<p><strong>To fix the immediate issue:</strong></p>";
echo "<ol>";
echo "<li>Upload sample videos to the file system</li>";
echo "<li>Insert proper video records in the database</li>";
echo "<li>Update course-detail.php to handle missing videos gracefully</li>";
echo "</ol>";
echo "<p><a href='course-detail.php?id=12' target='_blank'>Test Course Detail Page</a></p>";
echo "<p><a href='teacher/upload-video.php' target='_blank'>Upload New Video</a></p>";
echo "</div>";

// Close database connection
if ($conn) {
    $conn->close();
}
?>
