<?php
/**
 * Certificate Download Handler
 * Generates and downloads PDF certificates for completed courses
 */

require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;
$certificate_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$type = isset($_GET['type']) ? $_GET['type'] : 'standard';

// Handle ACCREDITED certificate download (using certificate ID)
if ($certificate_id > 0) {
    // Get accredited certificate details
    $cert_stmt = $conn->prepare("
        SELECT ac.*, ar.student_id, ar.course_id, c.title as course_title, 
               u.first_name, u.last_name
        FROM accreditation_certificates ac
        JOIN accreditation_requests ar ON ac.request_id = ar.request_id
        JOIN courses c ON ar.course_id = c.course_id
        JOIN users u ON ar.student_id = u.user_id
        WHERE ac.certificate_id = ? AND ar.student_id = ? AND ac.status = 'active'
    ");
    
    if ($cert_stmt) {
        $cert_stmt->bind_param("ii", $certificate_id, $student_id);
        $cert_stmt->execute();
        $cert_result = $cert_stmt->get_result();
        $cert_data = $cert_result->fetch_assoc();
        
        if ($cert_data && !empty($cert_data['certificate_path'])) {
            // Certificate found, serve the file directly
            $file_path = $_SERVER['DOCUMENT_ROOT'] . $cert_data['certificate_path'];
            
            // Check alternative paths if main path doesn't work
            if (!file_exists($file_path)) {
                $alt_paths = [
                    __DIR__ . '/../' . $cert_data['certificate_path'],
                    __DIR__ . '/..' . $cert_data['certificate_path'],
                    dirname(__DIR__) . $cert_data['certificate_path']
                ];
                
                foreach ($alt_paths as $alt_path) {
                    if (file_exists($alt_path)) {
                        $file_path = $alt_path;
                        break;
                    }
                }
            }
            
            if (file_exists($file_path)) {
                // Set headers for direct file download
                $filename = 'SITA_Certificate_' . $cert_data['certificate_number'] . '.pdf';
                
                // Clear any output buffer
                if (ob_get_level()) {
                    ob_end_clean();
                }
                
                header('Content-Type: application/pdf');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                header('Content-Length: ' . filesize($file_path));
                header('Cache-Control: private');
                header('Pragma: private');
                header('Expires: 0');
                
                // Output the file
                readfile($file_path);
                exit();
            } else {
                // File not found, show simple error
                echo "<!DOCTYPE html><html><head><title>Certificate Error</title></head><body>";
                echo "<h2>Certificate File Not Found</h2>";
                echo "<p>The certificate file could not be located on the server.</p>";
                echo "<p>Please contact support if this issue persists.</p>";
                echo "<p><a href='javascript:history.back()'>Go Back</a></p>";
                echo "</body></html>";
                exit();
            }
        } else {
            // Certificate not found or not authorized
            echo "<!DOCTYPE html><html><head><title>Certificate Access Error</title></head><body>";
            echo "<h2>Certificate Not Found</h2>";
            echo "<p>The requested certificate could not be found or you don't have access to it.</p>";
            echo "<p><a href='javascript:history.back()'>Go Back</a></p>";
            echo "</body></html>";
            exit();
        }
    } else {
        echo "Database error: " . $conn->error;
        exit();
    }
}

// Handle regular course certificates (existing functionality)
if (!$course_id) {
    header("Location: certificates.php");
    exit();
}

// Verify student completed the course
$stmt = $conn->prepare("
    SELECT c.*, u.first_name, u.last_name, u.email,
           e.completion_date, e.status as enrollment_status,
           s.first_name as student_first, s.last_name as student_last
    FROM courses c 
    JOIN users u ON c.teacher_id = u.user_id
    JOIN enrollments e ON c.course_id = e.course_id
    JOIN users s ON e.student_id = s.user_id
    WHERE c.course_id = ? AND e.student_id = ? AND e.status = 'completed'
");

if (!$stmt) {
    header("Location: certificates.php");
    exit();
}

$stmt->bind_param("ii", $course_id, $student_id);
$stmt->execute();
$result = $stmt->get_result();
$course_data = $result->fetch_assoc();

if (!$course_data) {
    header("Location: certificates.php");
    exit();
}

// Generate certificate ID
$certificate_id = 'PAC-' . strtoupper(substr(md5($student_id . $course_id), 0, 8));
$student_name = $course_data['student_first'] . ' ' . $course_data['student_last'];
$instructor_name = $course_data['first_name'] . ' ' . $course_data['last_name'];

// Include mPDF library for better PDF generation
require_once '../vendor/autoload.php'; // Assuming Composer autoload or manual include

// Set headers for PDF download
header('Content-Type: application/pdf');
header('Content-Disposition: attachment; filename="Panadite_Academy_Certificate_' . $certificate_id . '.pdf"');
header('Cache-Control: private, max-age=0, must-revalidate');
header('Pragma: public');

// Create HTML content matching the exact certificate design
$html = '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        @page {
            margin: 0;
            size: A4 landscape;
        }
        
        body {
            margin: 0;
            padding: 20px;
            font-family: "Inter", "Segoe UI", Roboto, sans-serif;
            background: #f8fafc;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
        }
        
        /* Certificate Container - Exact Match */
        .certificate-container {
            background: #ffffff;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
            margin: 0 auto;
            max-width: 700px;
            overflow: hidden;
            border: 2px solid #E5E7EB;
            position: relative;
        }

        .certificate-container::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(circle at 15% 85%, rgba(79, 130, 244, 0.04) 0%, transparent 40%),
                radial-gradient(circle at 85% 15%, rgba(245, 158, 11, 0.04) 0%, transparent 40%),
                radial-gradient(circle at 50% 20%, rgba(79, 130, 244, 0.02) 0%, transparent 30%),
                radial-gradient(circle at 20% 50%, rgba(245, 158, 11, 0.02) 0%, transparent 35%);
            pointer-events: none;
            z-index: 1;
        }

        .certificate-wrapper {
            background: #ffffff;
            position: relative;
            width: 100%;
            min-height: 400px;
            padding: 0;
            z-index: 2;
        }

        /* Amber Header Banner */
        .cert-header-section {
            background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%);
            color: white;
            text-align: center;
            padding: 2rem 2rem 1.5rem;
            position: relative;
        }

        .cert-header-section::after {
            content: "";
            position: absolute;
            bottom: -10px;
            left: 0;
            right: 0;
            height: 20px;
            background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%);
            clip-path: polygon(0 0, 100% 0, 95% 100%, 5% 100%);
        }

        .cert-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            margin-bottom: 1rem;
            border: 3px solid white;
            background: white;
            padding: 2px;
            display: block;
            margin-left: auto;
            margin-right: auto;
        }

        .cert-title {
            font-size: 3rem;
            font-weight: 700;
            color: white;
            margin: 0;
            letter-spacing: 4px;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            line-height: 1.1;
        }

        .cert-badge {
            display: inline-block;
            background: rgba(255, 255, 255, 0.2);
            color: white;
            padding: 0.5rem 1.2rem;
            border-radius: 20px;
            font-size: 1rem;
            font-weight: 600;
            letter-spacing: 2px;
            margin-top: 0.8rem;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        /* Certificate Main Content */
        .cert-main-content {
            text-align: center;
            padding: 3rem 2rem 2rem;
            background: #ffffff;
            position: relative;
        }

        .cert-main-content::before {
            content: "🎓";
            position: absolute;
            top: 30px;
            left: 40px;
            font-size: 2.5rem;
            opacity: 0.08;
            pointer-events: none;
            z-index: 1;
        }

        .cert-main-content::after {
            content: "⭐";
            position: absolute;
            bottom: 40px;
            right: 50px;
            font-size: 2rem;
            opacity: 0.06;
            pointer-events: none;
            z-index: 1;
        }

        .cert-declaration-text {
            font-size: 1.2rem;
            color: #4b5563;
            margin: 1rem 0;
            font-weight: 400;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .cert-student-name {
            font-size: 3rem;
            font-weight: 700;
            color: #1f2937;
            margin: 2rem 0;
            position: relative;
            display: inline-block;
            font-family: "Brush Script MT", cursive, "Georgia", serif;
            font-style: italic;
        }

        .cert-student-name::after {
            content: "";
            position: absolute;
            bottom: -15px;
            left: 50%;
            transform: translateX(-50%);
            width: 80%;
            height: 2px;
            background: #d1d5db;
            border-radius: 1px;
        }

        .cert-course-name {
            font-size: 1.8rem;
            font-weight: 600;
            color: #1f2937;
            margin: 2rem 0 1rem;
            font-style: italic;
        }

        .cert-certified-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            background: linear-gradient(135deg, #f59e0b, #d97706);
            color: white;
            padding: 0.5rem 1.2rem;
            border-radius: 25px;
            font-size: 0.9rem;
            font-weight: 600;
            letter-spacing: 1px;
            margin-top: 1rem;
            box-shadow: 0 3px 10px rgba(245, 158, 11, 0.3);
        }

        /* Certificate Footer */
        .cert-footer-section {
            display: flex;
            justify-content: space-around;
            align-items: center;
            padding: 2rem;
            background: #f9fafb;
            border-top: 1px solid #e5e7eb;
            gap: 2rem;
        }

        .cert-info-item {
            text-align: center;
            flex: 1;
        }

        .cert-info-icon {
            font-size: 1.5rem;
            color: #6b7280;
            margin-bottom: 0.5rem;
        }

        .cert-info-label {
            font-size: 0.8rem;
            color: #6b7280;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.3rem;
        }

        .cert-info-value {
            font-size: 0.9rem;
            color: #374151;
            font-weight: 500;
        }

        /* Override any conflicting styles */
        .certificate-container * {
            box-sizing: border-box;
        }
    </style>
</head>
<body>
    <div class="certificate-container">
        <div class="certificate-wrapper">
            <!-- Certificate Header -->
            <div class="cert-header-section">
                <img src="../assets/images/logo.jpeg" alt="Panadite Academy" class="cert-logo">
                <h1 class="cert-title">PANADITE ACADEMY</h1>
                <span class="cert-badge">
                    🏆 CERTIFICATE OF COMPLETION
                </span>
            </div>

            <!-- Certificate Content -->
            <div class="cert-main-content">
                <p class="cert-declaration-text">This is to certify that</p>
                <h2 class="cert-student-name">' . htmlspecialchars($student_name) . '</h2>
                <p class="cert-declaration-text">has successfully completed the course</p>
                <h3 class="cert-course-name">' . htmlspecialchars($course_data['title']) . '</h3>
                <span class="cert-certified-badge">
                    ✅ CERTIFIED
                </span>
            </div>

            <!-- Certificate Footer -->
            <div class="cert-footer-section">
                <div class="cert-info-item">
                    <div class="cert-info-icon">📅</div>
                    <div class="cert-info-label">Completed</div>
                    <div class="cert-info-value">' . date('F d, Y', strtotime($course_data['completion_date'])) . '</div>
                </div>
                <div class="cert-info-item">
                    <div class="cert-info-icon">👨‍🏫</div>
                    <div class="cert-info-label">Instructor</div>
                    <div class="cert-info-value">' . htmlspecialchars($instructor_name) . '</div>
                </div>
                <div class="cert-info-item">
                    <div class="cert-info-icon">🆔</div>
                    <div class="cert-info-label">Certificate ID</div>
                    <div class="cert-info-value">' . $certificate_id . '</div>
                </div>
            </div>
        </div>
    </div>
    </div>
</body>
</html>';

// Try to use DOMPdf or mPDF if available, otherwise use wkhtmltopdf or basic output
try {
    // Option 1: Try DOMPdf (most common)
    if (class_exists('Dompdf\Dompdf')) {
        require_once '../vendor/dompdf/dompdf/autoload.inc.php';
        
        $options = new \Dompdf\Options();
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isPhpEnabled', true);
        
        $dompdf = new \Dompdf\Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'landscape');
        $dompdf->render();
        
        echo $dompdf->output();
    }
    // Option 2: Try mPDF
    elseif (class_exists('Mpdf\Mpdf')) {
        require_once '../vendor/mpdf/mpdf/autoload.php';
        
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L',
            'orientation' => 'L',
            'margin_left' => 0,
            'margin_right' => 0,
            'margin_top' => 0,
            'margin_bottom' => 0,
        ]);
        
        $mpdf->WriteHTML($html);
        echo $mpdf->Output('', 'S');
    }
    // Option 3: Basic HTML output with print styles (fallback)
    else {
        // Add print-specific styles for better PDF conversion
        $printStyles = '
        <style media="print">
            @page {
                size: A4 landscape;
                margin: 0;
            }
            body {
                -webkit-print-color-adjust: exact;
                color-adjust: exact;
            }
        </style>';
        
        $html = str_replace('</head>', $printStyles . '</head>', $html);
        
        // Output HTML directly - browsers can print to PDF
        header('Content-Type: text/html; charset=utf-8');
        header('Content-Disposition: inline; filename="Panadite_Academy_Certificate_' . $certificate_id . '.html"');
        echo $html;
    }
    
} catch (Exception $e) {
    // Fallback: Output as HTML for manual PDF conversion
    header('Content-Type: text/html; charset=utf-8');
    header('Content-Disposition: inline; filename="Panadite_Academy_Certificate_' . $certificate_id . '.html"');
    
    // Add JavaScript to auto-trigger print dialog
    $html .= '
    <script>
        window.addEventListener("load", function() {
            setTimeout(function() {
                window.print();
            }, 1000);
        });
    </script>';
    
    echo $html;
}

$stmt->close();
$conn->close();
?>

// Check if we have a PDF library available or use HTML to PDF conversion
// For now, we'll use a simple HTML output with print-friendly CSS
// In production, you'd want to use a library like TCPDF, FPDF, or mPDF

// Simple fallback: serve as HTML with print styles for now
// This can be printed to PDF by the browser
header('Content-Type: text/html; charset=UTF-8');
header('Content-Disposition: inline; filename="certificate.html"');

echo $html;

// Add JavaScript to automatically trigger print dialog
echo '
<script>
window.onload = function() {
    setTimeout(function() {
        window.print();
    }, 1000);
}
</script>';
?>
