<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session but don't interfere with download headers
session_start();

// Store session data early to prevent logout
$session_user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
$session_role = isset($_SESSION['role']) ? $_SESSION['role'] : null;

// Debug: Check if files exist
if (!file_exists('../config/db_connect.php')) {
    die('ERROR: db_connect.php not found at ../config/db_connect.php');
}
if (!file_exists('../auth/functions.php')) {
    die('ERROR: functions.php not found at ../auth/functions.php');
}

require_once '../config/db_connect.php';
require_once '../auth/functions.php';

// Debug: Check database connection
if (!isset($conn) || $conn->connect_error) {
    die('ERROR: Database connection failed - ' . (isset($conn) ? $conn->connect_error : 'No connection object'));
}

// Check if user is logged in and is a student using stored session data
if (!$session_user_id || $session_role !== 'student') {
    // Instead of redirect, show JSON error for AJAX calls or simple error for direct access
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        http_response_code(401);
        echo json_encode(['error' => 'Authentication required']);
        exit;
    } else {
        // For direct access, show error instead of redirect to prevent logout
        http_response_code(401);
        die('<h3>Access Denied</h3><p>Please <a href="../auth/login.php">login</a> as a student to download files.</p>');
    }
}

$student_id = $session_user_id; // Use stored session data
$request_id = isset($_GET['request_id']) ? intval($_GET['request_id']) : 0;

if (!$request_id) {
    http_response_code(400);
    die('Invalid request ID');
}

try {
    // Verify that this accreditation request belongs to the current student
    $verify_query = "SELECT ar.request_id, ar.student_id, c.title as course_title 
                    FROM accreditation_requests ar 
                    JOIN courses c ON ar.course_id = c.course_id 
                    WHERE ar.request_id = ? AND ar.student_id = ? AND ar.status = 'assignment_created'";
    
    $verify_stmt = $conn->prepare($verify_query);
    if (!$verify_stmt) {
        throw new Exception('Database prepare error: ' . $conn->error);
    }
    
    $verify_stmt->bind_param("ii", $request_id, $student_id);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    
    if ($verify_result->num_rows === 0) {
        http_response_code(403);
        die('Access denied or assignment not available');
    }
    
    $request_data = $verify_result->fetch_assoc();
    $assignment_dir = "../uploads/assignments/request_{$request_id}/";
    
    // Check if assignment directory exists
    if (!is_dir($assignment_dir)) {
        http_response_code(404);
        die('Assignment files not found');
    }
    
    // Get all files in the assignment directory
    $files = glob($assignment_dir . "*");
    
    if (empty($files)) {
        http_response_code(404);
        die('No assignment files available');
    }
    
    // If there's only one file, download it directly
    if (count($files) == 1) {
        $file_path = $files[0];
        $file_name = basename($file_path);
        
        // Security check - ensure file is within the assignment directory
        if (realpath($file_path) !== false && strpos(realpath($file_path), realpath($assignment_dir)) === 0) {
            // Close session to prevent logout and session locking during download
            session_write_close();
            
            // Set headers for file download
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $file_name . '"');
            header('Content-Length: ' . filesize($file_path));
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            
            // Output file
            readfile($file_path);
            exit;
        } else {
            http_response_code(403);
            die('Invalid file path');
        }
    }
    
    // If multiple files, create a ZIP archive
    $zip_filename = "SITA_Assignment_" . $request_data['course_title'] . "_" . date('Y-m-d') . ".zip";
    $zip_path = sys_get_temp_dir() . '/' . $zip_filename;
    
    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
        http_response_code(500);
        die('Could not create ZIP file');
    }
    
    // Add all files to ZIP
    foreach ($files as $file_path) {
        if (is_file($file_path)) {
            $file_name = basename($file_path);
            // Security check
            if (realpath($file_path) !== false && strpos(realpath($file_path), realpath($assignment_dir)) === 0) {
                $zip->addFile($file_path, $file_name);
            }
        }
    }
    
    $zip->close();
    
    // Check if ZIP was created successfully
    if (!file_exists($zip_path)) {
        http_response_code(500);
        die('Failed to create ZIP file');
    }
    
    // Close session to prevent logout and session locking during download
    session_write_close();
    
    // Set headers for ZIP download
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $zip_filename . '"');
    header('Content-Length: ' . filesize($zip_path));
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    
    // Output ZIP file
    readfile($zip_path);
    
    // Clean up temporary ZIP file
    unlink($zip_path);
    exit;
    
} catch (Exception $e) {
    error_log("Assignment download error: " . $e->getMessage());
    http_response_code(500);
    die('An error occurred while downloading files');
}
?>
