<?php
require_once '../config/db_connect.php';

// Simple and reliable authentication check
$request_id = isset($_GET['request_id']) ? intval($_GET['request_id']) : 0;
$user_token = isset($_GET['token']) ? $_GET['token'] : '';

if (!$request_id) {
    http_response_code(400);
    die('Invalid request ID');
}

if (!$user_token) {
    http_response_code(400);
    die('Missing authentication token');
}

try {
    // Verify token and get user info
    $token_query = "SELECT u.user_id, u.role, ar.request_id, ar.student_id, c.title as course_title
                   FROM users u 
                   JOIN accreditation_requests ar ON u.user_id = ar.student_id
                   JOIN courses c ON ar.course_id = c.course_id
                   WHERE ar.request_id = ? 
                   AND u.remember_token = ? 
                   AND u.role = 'student' 
                   AND ar.status = 'assignment_created'";
    
    $stmt = $conn->prepare($token_query);
    if (!$stmt) {
        throw new Exception('Database prepare error: ' . $conn->error);
    }
    
    $stmt->bind_param("is", $request_id, $user_token);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        http_response_code(403);
        die('Access denied or invalid token');
    }
    
    $user_data = $result->fetch_assoc();
    $assignment_dir = "../uploads/assignments/request_{$request_id}/";
    
    // Check if assignment directory exists
    if (!is_dir($assignment_dir)) {
        http_response_code(404);
        die('Assignment files not found');
    }
    
    // Get all files in the assignment directory
    $files = glob($assignment_dir . "*");
    
    if (empty($files)) {
        http_response_code(404);
        die('No assignment files available');
    }
    
    // If there's only one file, download it directly
    if (count($files) == 1) {
        $file_path = $files[0];
        $file_name = basename($file_path);
        
        // Security check - ensure file is within the assignment directory
        if (realpath($file_path) !== false && strpos(realpath($file_path), realpath($assignment_dir)) === 0) {
            // Set headers for file download
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $file_name . '"');
            header('Content-Length: ' . filesize($file_path));
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            
            // Output file
            readfile($file_path);
            exit;
        } else {
            http_response_code(403);
            die('Invalid file path');
        }
    }
    
    // If multiple files, create a ZIP archive
    $zip_filename = "SITA_Assignment_" . preg_replace('/[^a-zA-Z0-9_-]/', '', $user_data['course_title']) . "_" . date('Y-m-d') . ".zip";
    $zip_path = sys_get_temp_dir() . '/' . $zip_filename;
    
    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
        http_response_code(500);
        die('Could not create ZIP file');
    }
    
    // Add all files to ZIP
    foreach ($files as $file_path) {
        if (is_file($file_path)) {
            $file_name = basename($file_path);
            // Security check
            if (realpath($file_path) !== false && strpos(realpath($file_path), realpath($assignment_dir)) === 0) {
                $zip->addFile($file_path, $file_name);
            }
        }
    }
    
    $zip->close();
    
    // Check if ZIP was created successfully
    if (!file_exists($zip_path)) {
        http_response_code(500);
        die('Failed to create ZIP file');
    }
    
    // Set headers for ZIP download
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $zip_filename . '"');
    header('Content-Length: ' . filesize($zip_path));
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    
    // Output ZIP file
    readfile($zip_path);
    
    // Clean up temporary ZIP file
    unlink($zip_path);
    exit;
    
} catch (Exception $e) {
    error_log("Secure assignment download error: " . $e->getMessage());
    http_response_code(500);
    die('An error occurred while downloading files');
}
?>
