<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Set the missing session variables that requireRole needs
if (!isset($_SESSION['logged_in']) && isset($_SESSION['user_id'])) {
    $_SESSION['logged_in'] = true;
}
if (!isset($_SESSION['user_role']) && isset($_SESSION['role'])) {
    $_SESSION['user_role'] = $_SESSION['role'];
}

requireRole('student');
require_once '../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$request_id = isset($_GET['request_id']) ? intval($_GET['request_id']) : 0;

if (!$request_id) {
    die('Error: Invalid request ID');
}

try {
    // Verify that this accreditation request belongs to the current student
    $verify_query = "SELECT ar.request_id, ar.student_id, c.title as course_title 
                    FROM accreditation_requests ar 
                    JOIN courses c ON ar.course_id = c.course_id 
                    WHERE ar.request_id = ? AND ar.student_id = ? AND ar.status = 'assignment_created'";
    
    $verify_stmt = $conn->prepare($verify_query);
    if (!$verify_stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }
    
    $verify_stmt->bind_param("ii", $request_id, $student_id);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    
    if ($verify_result->num_rows === 0) {
        die('Error: Assignment not found or access denied');
    }
    
    $request_data = $verify_result->fetch_assoc();
    $assignment_dir = "../uploads/assignments/request_{$request_id}/";
    
    // Check if assignment directory exists
    if (!is_dir($assignment_dir)) {
        die('Error: Assignment files directory not found');
    }
    
    // Get all files in the assignment directory
    $files = glob($assignment_dir . "*");
    
    if (empty($files)) {
        die('Error: No assignment files available for download');
    }
    
    // Close session to prevent logout during download
    session_write_close();
    
    // If there's only one file, download it directly
    if (count($files) == 1) {
        $file_path = $files[0];
        $file_name = basename($file_path);
        
        // Security check - ensure file is within the assignment directory
        $real_file_path = realpath($file_path);
        $real_dir_path = realpath($assignment_dir);
        
        if ($real_file_path && $real_dir_path && strpos($real_file_path, $real_dir_path) === 0) {
            // Set headers for file download
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . basename($file_name) . '"');
            header('Content-Length: ' . filesize($file_path));
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            
            // Clear any output buffers
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            // Output file
            readfile($file_path);
            exit;
        } else {
            die('Error: Invalid file path detected');
        }
    }
    
    // If multiple files, create a ZIP archive
    $safe_course_title = preg_replace('/[^a-zA-Z0-9_-]/', '_', $request_data['course_title']);
    $zip_filename = "SITA_Assignment_{$safe_course_title}_" . date('Y-m-d') . ".zip";
    $zip_path = sys_get_temp_dir() . '/' . uniqid('sita_') . '.zip';
    
    if (!class_exists('ZipArchive')) {
        die('Error: ZIP functionality not available on server');
    }
    
    $zip = new ZipArchive();
    $zip_result = $zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE);
    
    if ($zip_result !== TRUE) {
        die('Error: Could not create ZIP file (Code: ' . $zip_result . ')');
    }
    
    // Add all files to ZIP
    $files_added = 0;
    foreach ($files as $file_path) {
        if (is_file($file_path)) {
            $file_name = basename($file_path);
            $real_file_path = realpath($file_path);
            $real_dir_path = realpath($assignment_dir);
            
            // Security check
            if ($real_file_path && $real_dir_path && strpos($real_file_path, $real_dir_path) === 0) {
                if ($zip->addFile($file_path, $file_name)) {
                    $files_added++;
                }
            }
        }
    }
    
    $zip->close();
    
    if ($files_added === 0) {
        unlink($zip_path);
        die('Error: No valid files could be added to ZIP');
    }
    
    // Check if ZIP was created successfully
    if (!file_exists($zip_path) || filesize($zip_path) === 0) {
        if (file_exists($zip_path)) unlink($zip_path);
        die('Error: Failed to create ZIP file');
    }
    
    // Set headers for ZIP download
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $zip_filename . '"');
    header('Content-Length: ' . filesize($zip_path));
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    
    // Clear any output buffers
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // Output ZIP file
    readfile($zip_path);
    
    // Clean up temporary ZIP file
    unlink($zip_path);
    exit;
    
} catch (Exception $e) {
    error_log("Simple download error: " . $e->getMessage());
    die('Error: An unexpected error occurred while preparing download');
}
?>
