<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: download_error.php?type=access_denied&resource_id=0&file_name=Unknown&title=Unknown');
    exit();
}

// Check if resource ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $resourceId = $_GET['id'] ?? 'Invalid';
    header('Location: download_error.php?type=not_found&resource_id=' . urlencode($resourceId) . '&file_name=Unknown&title=Invalid Resource');
    exit();
}

// Debug: Show what resource ID we're looking for
if (isset($_GET['debug'])) {
    echo "Resource ID: " . $_GET['id'] . "<br>";
    echo "User ID: " . $_SESSION['user_id'] . "<br>";
}

$student_id = $_SESSION['user_id'];
$resource_id = (int)$_GET['id'];

// Include database connection
require_once '../config/db_connect.php';

// Helper function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Get library resource details
$resourceQuery = "SELECT * FROM library_resources WHERE resource_id = ?";

$stmt = $conn->prepare($resourceQuery);
if (!$stmt) {
    header('Location: download_error.php?type=server_error&resource_id=' . $resource_id . '&file_name=Unknown&title=Database Error');
    exit();
}

$stmt->bind_param("i", $resource_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result && $result->num_rows > 0) {
    $resource = $result->fetch_assoc();
    
    $filePath = $resource['file_path'];
    $resourceTitle = $resource['title'];
    $resourceType = $resource['type'];
    $resourceFormat = $resource['format'];
    
    // Determine the actual file path
    // Library resources are typically stored in materials directory like course materials
    if (strpos($filePath, '../uploads/materials/') === 0) {
        $actualFilePath = $filePath;
    } elseif (strpos($filePath, '../uploads/library/') === 0) {
        $actualFilePath = $filePath;
    } else {
        // Try multiple possible base directories
        $possiblePaths = [
            '../uploads/materials/' . $filePath,
            '../uploads/library/' . $filePath,
            '../uploads/resources/' . $filePath,
            '../assets/resources/' . $filePath
        ];
        
        $fileFound = false;
        foreach ($possiblePaths as $possiblePath) {
            if (file_exists($possiblePath)) {
                $actualFilePath = $possiblePath;
                $fileFound = true;
                break;
            }
        }
        
        if (!$fileFound) {
            // Try to find similar files if the exact path doesn't exist
            $fileName = basename($filePath);
            $searchDirs = ['../uploads/materials/', '../uploads/library/', 'uploads/materials/', 'uploads/library/'];
            $similarFiles = [];
            
            foreach ($searchDirs as $dir) {
                if (is_dir($dir)) {
                    $files = scandir($dir);
                    foreach ($files as $file) {
                        if ($file !== '.' && $file !== '..' && stripos($file, pathinfo($fileName, PATHINFO_FILENAME)) !== false) {
                            $similarFiles[] = $dir . $file;
                        }
                    }
                }
            }
            
            if (!empty($similarFiles)) {
                // Use the first similar file found
                $actualFilePath = $similarFiles[0];
                $fileFound = true;
            } else {
                header('Location: download_error.php?type=not_found&resource_id=' . $resource_id . '&file_name=' . urlencode($fileName) . '&title=' . urlencode($resourceTitle));
                exit();
            }
        }
    }
    
    // Final check if the file actually exists
    if (!file_exists($actualFilePath)) {
        header('Location: download_error.php?type=corrupted&resource_id=' . $resource_id . '&file_name=' . urlencode(basename($actualFilePath)) . '&title=' . urlencode($resourceTitle));
        exit();
    }
    
    // Get file info
    $fileSize = filesize($actualFilePath);
    $fileExtension = strtolower(pathinfo($actualFilePath, PATHINFO_EXTENSION));
    
    // Generate a clean filename for display to user
    $fileName = str_replace(' ', '_', strtolower($resourceTitle)) . '.' . $fileExtension;
    
    // Set up MIME types for different file types
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'doc' => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'xls' => 'application/vnd.ms-excel',
        'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'ppt' => 'application/vnd.ms-powerpoint',
        'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        'txt' => 'text/plain',
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'mp3' => 'audio/mpeg',
        'mp4' => 'video/mp4',
    ];
    
    // Set the appropriate content type
    if (array_key_exists($fileExtension, $mimeTypes)) {
        header('Content-Type: ' . $mimeTypes[$fileExtension]);
    } else {
        header('Content-Type: application/octet-stream');
    }
    
    // Properly encode filename for Content-Disposition header
    $encodedFilename = rawurlencode($fileName);
    header("Content-Disposition: attachment; filename*=UTF-8''$encodedFilename; filename=\"$fileName\"");
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . $fileSize);
    
    // Record the download in user_downloads table for history tracking
    $insertDownloadSql = "INSERT INTO user_downloads (user_id, material_id, material_title, material_type, file_size, course_id, course_name, file_path, category) 
                         VALUES (?, ?, ?, ?, ?, NULL, 'Library', ?, 'Library Resources')";
    
    $downloadStmt = $conn->prepare($insertDownloadSql);
    if ($downloadStmt) {
        $formattedSize = formatFileSize($fileSize);
        $fileType = 'Library - ' . strtoupper($fileExtension) . ' (' . $resourceType . ')';
        $downloadStmt->bind_param("iissss", $student_id, $resource_id, $resourceTitle, $fileType, $formattedSize, $filePath);
        $downloadStmt->execute();
        $downloadStmt->close();
    }
    
    // Update download count in library_resources table
    $updateDownloadCount = "UPDATE library_resources SET downloads = downloads + 1 WHERE resource_id = ?";
    $updateStmt = $conn->prepare($updateDownloadCount);
    if ($updateStmt) {
        $updateStmt->bind_param("i", $resource_id);
        $updateStmt->execute();
        $updateStmt->close();
    }
    
    // Disable output buffering
    while (ob_get_level()) ob_end_clean();
    
    // Output the file
    readfile($actualFilePath);
    exit();
    
} else {
    // Resource not found
    header('Location: download_error.php?type=not_found&resource_id=' . $resource_id . '&file_name=Unknown&title=Resource Not Found');
    exit();
}
?>
