<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Set custom CSS for this page
$customCss = 'downloads';

// Initialize variables
$downloads = [];
$categories = [];
$selectedCategory = $_GET['category'] ?? 'all';

// Connect to database and fetch real data
require_once '../config/db_connect.php';

// Helper function to get icon based on file type
function getIconForFileType($fileType) {
    $fileType = strtolower($fileType);
    
    // Special handling for assignments
    if (strpos($fileType, 'assignment') !== false) {
        if (strpos($fileType, 'pdf') !== false) return 'bx-task';
        return 'bx-task';
    }
    
    // Regular file type detection
    if (strpos($fileType, 'pdf') !== false) return 'bx-file-blank';
    if (strpos($fileType, 'image') !== false) return 'bx-image';
    if (strpos($fileType, 'video') !== false) return 'bx-video';
    if (strpos($fileType, 'audio') !== false) return 'bx-music';
    if (strpos($fileType, 'word') !== false || strpos($fileType, 'document') !== false) return 'bx-file-doc';
    if (strpos($fileType, 'excel') !== false || strpos($fileType, 'spreadsheet') !== false) return 'bx-spreadsheet';
    if (strpos($fileType, 'powerpoint') !== false || strpos($fileType, 'presentation') !== false) return 'bx-slideshow';
    return 'bx-file';
}

if ($conn) {
    // Get current user's ID from session
    $userId = $_SESSION['user_id'];
    
    // Get categories based on user's download history
    $categoryQuery = "SELECT DISTINCT category FROM user_downloads WHERE user_id = ? ORDER BY category";
    $categoryStmt = $conn->prepare($categoryQuery);
    $categoryStmt->bind_param('i', $userId);
    $categoryStmt->execute();
    $categoryResult = $categoryStmt->get_result();
    
    // Build categories array from user's download history
    $categoriesData = [];
    if ($categoryResult && mysqli_num_rows($categoryResult) > 0) {
        while ($row = mysqli_fetch_assoc($categoryResult)) {
            $categories[] = $row['category'];
            
            // Set appropriate icon based on category
            $categoryIcon = 'bx-download';
            if ($row['category'] === 'Assignment') {
                $categoryIcon = 'bx-task';
            } elseif ($row['category'] === 'Course Materials') {
                $categoryIcon = 'bx-book';
            } elseif ($row['category'] === 'Library Resources' || stripos($row['category'], 'library') !== false) {
                $categoryIcon = 'bx-library';
            }
            
            $categoriesData[$row['category']] = [
                'icon' => $categoryIcon,
                'id' => strtolower(str_replace(' ', '_', $row['category']))
            ];
        }
    }
    
    // Get user's download history (filtered by category if specified)
    if ($selectedCategory !== 'all') {
        $query = "SELECT * FROM user_downloads WHERE user_id = ? AND category = ? ORDER BY download_date DESC";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("is", $userId, $selectedCategory);
    } else {
        $query = "SELECT * FROM user_downloads WHERE user_id = ? ORDER BY download_date DESC";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("i", $userId);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $allDownloads = []; // For storing all downloads
    
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            // Add to downloads array
            $allDownloads[] = [
                'id' => $row['id'],
                'title' => $row['material_title'],
                'description' => 'Downloaded from: ' . $row['course_name'],
                'category' => $row['category'],
                'file_type' => $row['material_type'],
                'file_size' => $row['file_size'],
                'download_count' => 1, // Each record represents one download
                'date_added' => $row['download_date'],
                'icon' => getIconForFileType($row['material_type']),
                'file_path' => $row['file_path'],
                'material_id' => $row['material_id'],
                'course_name' => $row['course_name']
            ];
        }
    }
    
    // Set initial downloads
    $downloads = $allDownloads;
    
    // Handle search functionality
    if (isset($_GET['search']) && !empty($_GET['search'])) {
        $searchTerm = strtolower($_GET['search']);
        $downloads = array_filter($downloads, function($item) use ($searchTerm) {
            return (
                stripos(strtolower($item['title']), $searchTerm) !== false ||
                stripos(strtolower($item['description']), $searchTerm) !== false ||
                stripos(strtolower($item['category']), $searchTerm) !== false
            );
        });
    }
    
    // Count downloads by category for sidebar
    $categoryCounts = [];
    foreach ($categories as $category) {
        $countQuery = "SELECT COUNT(*) as count FROM user_downloads WHERE user_id = ? AND category = ?";
        $stmt = $conn->prepare($countQuery);
        $stmt->bind_param("is", $userId, $category);
        $stmt->execute();
        $countResult = $stmt->get_result();
        $count = $countResult->fetch_assoc();
        $categoryCounts[$category] = $count['count'];
    }
    
    // Also get total count for "All Downloads"
    $totalCountQuery = "SELECT COUNT(*) as count FROM user_downloads WHERE user_id = ?";
    $stmt = $conn->prepare($totalCountQuery);
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $totalResult = $stmt->get_result();
    $totalCount = $totalResult->fetch_assoc();
    $categoryCounts['all'] = $totalCount['count'];
} else {
    // Database connection error
    $error = "Could not connect to the database";
}

// Initialize page title for header
$page_title = 'Downloads';

// Add page-specific CSS
$page_specific_css = '<link rel="stylesheet" href="assets/css/pages/downloads.css">';

// Include header (also includes init, session checks, etc.)
include_once 'components/header.php';

// Force direct CSS inclusion for downloads page to bypass detection issues
echo '<link rel="stylesheet" href="assets/css/pages/downloads.css?v=' . time() . '">';
?>

<!-- Downloads Container -->
<div class="container-fluid">
    <!-- Page Header -->
    <div class="page-header">
        <div>
            <h1 class="page-title">Downloads</h1>
            <p class="page-subtitle">Access and download course materials, tools, and resources</p>
        </div>
    </div>
    
    <div class="downloads-wrapper">
        <!-- Filters Sidebar -->
        <div class="downloads-filters">
            <div class="filter-card">
                <div class="filter-header">
                    <h5><i class="bx bx-filter"></i>Categories</h5>
                </div>
                <div class="filter-body">
                    <a href="?category=all<?= isset($_GET['search']) ? '&search=' . htmlspecialchars($_GET['search']) : '' ?>" class="filter-item <?= $selectedCategory === 'all' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-download"></i></span>
                        <span class="filter-name">All Downloads</span>
                        <span class="filter-badge"><?= $categoryCounts['all'] ?? 0 ?></span>
                    </a>
                    
                    <?php foreach ($categories as $category): ?>
                    <a href="?category=<?= urlencode($category) ?><?= isset($_GET['search']) ? '&search=' . htmlspecialchars($_GET['search']) : '' ?>" class="filter-item <?= $selectedCategory === $category ? 'active' : '' ?>">
                        <span class="filter-icon">
                            <i class="bx <?= isset($categoriesData[$category]) ? $categoriesData[$category]['icon'] : 'bx-folder' ?>"></i>
                        </span>
                        <span class="filter-name"><?= $category ?></span>
                        <span class="filter-badge"><?= $categoryCounts[$category] ?? 0 ?></span>
                    </a>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Downloads Content -->
        <div class="downloads-content">
            <div class="downloads-header">
                <h5 class="downloads-title">
                    <?php
                    if ($selectedCategory === 'all') {
                        echo 'All Downloads';
                    } else {
                        echo htmlspecialchars($selectedCategory);
                    }
                    ?> <span class="text-muted">(<?= count($downloads) ?>)</span>
                </h5>
                <form class="search-form" action="downloads.php" method="get">
                    <?php if ($selectedCategory !== 'all'): ?>
                    <input type="hidden" name="category" value="<?= htmlspecialchars($selectedCategory) ?>">
                    <?php endif; ?>
                    <div class="search-input">
                        <i class="bx bx-search"></i>
                        <input type="text" class="form-control" name="search" placeholder="Search downloads..." value="<?= isset($_GET['search']) ? htmlspecialchars($_GET['search']) : '' ?>">
                        <button type="submit" class="btn btn-primary btn-sm">Search</button>
                    </div>
                </form>
            </div>
            
            <?php if (count($downloads) > 0): ?>
            <div class="download-list">
                <?php foreach ($downloads as $download): ?>
                <div class="download-item">
                    <div class="download-icon">
                        <i class="bx <?= $download['icon'] ?>"></i>
                    </div>
                    <div class="download-content">
                        <h3 class="download-title"><?= htmlspecialchars($download['title']) ?></h3>
                        <p class="download-description"><?= htmlspecialchars($download['description']) ?></p>
                        <div class="download-meta">
                            <div class="meta-item">
                                <i class="bx bx-folder"></i> <?= htmlspecialchars($download['category']) ?>
                            </div>
                            <div class="meta-item">
                                <i class="bx bx-file"></i> <?= $download['file_type'] ?>
                            </div>
                            <div class="meta-item">
                                <i class="bx bx-data"></i> <?= $download['file_size'] ?>
                            </div>
                            <?php if(isset($download['course_name'])): ?>
                            <div class="meta-item">
                                <i class="bx bx-book"></i> <?= htmlspecialchars($download['course_name']) ?>
                            </div>
                            <?php endif; ?>
                            <div class="meta-item">
                                <i class="bx bx-calendar"></i> Downloaded <?= date('M d, Y g:i A', strtotime($download['date_added'])) ?>
                            </div>
                        </div>
                    </div>
                    <div class="download-actions">
                        <?php if(isset($download['material_id'])): ?>
                            <?php if($download['category'] === 'Library Resources'): ?>
                            <a href="download_library_resource.php?id=<?= $download['material_id'] ?>" class="btn btn-primary rounded-pill download-button">
                                <i class="bx bx-download me-2"></i> Download Again
                            </a>
                            <?php elseif($download['category'] === 'Assignment'): ?>
                            <a href="download_assignment.php?id=<?= $download['material_id'] ?>" class="btn btn-primary rounded-pill download-button">
                                <i class="bx bx-download me-2"></i> Download Again
                            </a>
                            <?php else: ?>
                            <a href="download_resource.php?id=<?= $download['material_id'] ?>" class="btn btn-primary rounded-pill download-button">
                                <i class="bx bx-download me-2"></i> Download Again
                            </a>
                            <?php endif; ?>
                        <?php else: ?>
                        <button class="btn btn-secondary rounded-pill" disabled>
                            <i class="bx bx-info-circle me-2"></i> Original File
                        </button>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <div class="empty-state">
                <i class="bx bx-download"></i>
                <h4>No download history</h4>
                <p>You haven't downloaded any materials yet. Visit the library or course materials to start downloading resources.</p>
                <a href="library.php" class="btn btn-primary">Browse Library</a>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>
