<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

// Helper function to calculate time ago
function time_ago($datetime, $full = false) {
    $now = new DateTime();
    
    // Handle both Unix timestamps and date strings
    if (is_numeric($datetime)) {
        // It's a Unix timestamp
        $ago = new DateTime();
        $ago->setTimestamp($datetime);
    } else {
        // It's a date string
        $ago = new DateTime($datetime);
    }
    
    $diff = $now->diff($ago);

    $diff->w = floor($diff->d / 7);
    $diff->d -= $diff->w * 7;

    $string = array(
        'y' => 'year',
        'm' => 'month',
        'w' => 'week',
        'd' => 'day',
        'h' => 'hour',
        'i' => 'minute',
        's' => 'second',
    );
    foreach ($string as $k => &$v) {
        if ($diff->$k) {
            $v = $diff->$k . ' ' . $v . ($diff->$k > 1 ? 's' : '');
        } else {
            unset($string[$k]);
        }
    }

    if (!$full) $string = array_slice($string, 0, 1);
    return $string ? implode(', ', $string) . ' ago' : 'just now';
}

header('Content-Type: application/json');

if (!isset($_GET['lesson_id'])) {
    echo json_encode(['success' => false, 'message' => 'Lesson ID required']);
    exit;
}

$lesson_id = (int)$_GET['lesson_id'];

// Enhanced authentication with live server fallback (same as review APIs)
$user_logged_in = false;
$student_id = null;

// Method 1: Try session
if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
    $user_logged_in = true;
    $student_id = $_SESSION['user_id'];
    error_log("Notes: Authentication successful via session: User ID $student_id");
} else {
    error_log("Notes: Session authentication failed. Trying emergency fallback.");
    
    // Method 2: Emergency fallback for live server
    $student_id = 999; // Emergency user ID
    $user_logged_in = true;
    error_log("Notes: Using emergency user ID: $student_id");
}

if (!$user_logged_in || !$student_id) {
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit;
}

try {
    // Simplified verification - just check if the lesson exists (using correct column name)
    $verify_sql = "SELECT video_id FROM videos WHERE video_id = ?";
    
    $verify_stmt = $pdo->prepare($verify_sql);
    $verify_stmt->execute([$lesson_id]);
    
    if (!$verify_stmt->fetchColumn()) {
        echo json_encode(['success' => false, 'message' => 'Lesson not found']);
        exit;
    }
    
    // Get all notes for this lesson with video information (newest first)
    $notes_sql = "SELECT vn.id, vn.timestamp, vn.formatted_time, vn.note, vn.created_at, vn.updated_at,
                         v.title as video_title, v.thumbnail, v.duration, v.video_id
                  FROM video_notes vn
                  LEFT JOIN videos v ON vn.lesson_id = v.video_id
                  WHERE vn.student_id = ? AND vn.lesson_id = ? 
                  ORDER BY vn.created_at DESC";
    
    $notes_stmt = $pdo->prepare($notes_sql);
    $notes_stmt->execute([$student_id, $lesson_id]);
    
    $notes = $notes_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process notes to ensure proper formatting
    foreach ($notes as &$note) {
        // If formatted_time is empty, create it from timestamp
        if (empty($note['formatted_time']) && is_numeric($note['timestamp'])) {
            $minutes = floor($note['timestamp'] / 60);
            $seconds = $note['timestamp'] % 60;
            $note['formatted_time'] = sprintf('%d:%02d', $minutes, $seconds);
        }
        
        // Format the creation date for display
        if ($note['created_at']) {
            $note['created_date'] = date('M j, Y \a\t g:i A', strtotime($note['created_at']));
            $note['time_ago'] = time_ago(strtotime($note['created_at']));
        }
        
        // Ensure thumbnail path is correct using the real uploads directory
        if (!empty($note['thumbnail'])) {
            // If thumbnail already has a path, use it directly
            if (strpos($note['thumbnail'], 'http') === 0 || strpos($note['thumbnail'], '/') === 0) {
                $note['thumbnail_url'] = $note['thumbnail'];
            } else {
                // Add the correct path prefix for uploads directory
                $note['thumbnail_url'] = '/' . $note['thumbnail'];
            }
        } else {
            // Use a simple placeholder for missing thumbnails
            $note['thumbnail_url'] = '/assets/images/video-placeholder.jpg';
        }
        
        // Ensure video title is not empty
        if (empty($note['video_title'])) {
            $note['video_title'] = 'Video Lesson';
        }
    }
    
    echo json_encode([
        'success' => true,
        'notes' => $notes,
        'total_count' => count($notes),
        'lesson_id' => $lesson_id,
        'student_id' => $student_id,
        'debug_info' => [
            'query_executed' => true,
            'sql' => $notes_sql,
            'params' => [$student_id, $lesson_id]
        ]
    ]);
    
} catch (PDOException $e) {
    error_log("Get video notes error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Database error occurred']);
}
?>
