<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Set custom CSS for this page
$customCss = 'help';

// Initialize comprehensive help topics based on our platform features
$topics = [
    [
        'id' => 'getting-started',
        'title' => 'Getting Started',
        'icon' => 'bx-play-circle',
        'description' => 'Essential guides to begin your learning journey',
        'articles' => [
            [
                'title' => 'Complete Student Dashboard Overview',
                'content' => 'Navigate your dashboard to access courses, assignments, notifications, and progress tracking. Your dashboard is your learning command center.',
                'steps' => [
                    'Login to your account at Panadite Academy',
                    'Explore the main navigation: Dashboard, My Courses, Certificates',
                    'Check your notifications for important updates',
                    'Review your course progress and upcoming assignments',
                    'Use the search bar to quickly find courses or resources'
                ],
                'views' => 892
            ],
            [
                'title' => 'Setting Up Your Student Profile',
                'content' => 'Complete your profile to get personalized learning recommendations and connect with instructors.',
                'steps' => [
                    'Go to Profile section from the sidebar',
                    'Upload a professional profile photo',
                    'Fill in your educational background and interests',
                    'Set your learning goals and preferences',
                    'Configure your notification settings'
                ],
                'views' => 734
            ]
        ]
    ],
    [
        'id' => 'courses',
        'title' => 'Course Enrollment & Management',
        'icon' => 'bx-book-open',
        'description' => 'Everything about finding, enrolling, and managing your courses',
        'articles' => [
            [
                'title' => 'How to Enroll in Courses',
                'content' => 'Step-by-step guide to enroll in courses and start your learning journey.',
                'steps' => [
                    'Browse available courses from "My Courses" section',
                    'Click on any course to view detailed information',
                    'Review course syllabus, duration, and requirements',
                    'Click "Enroll Now" button to begin enrollment',
                    'Complete payment if required, or access free courses directly',
                    'Once enrolled, access course materials immediately'
                ],
                'tips' => [
                    'Check prerequisites before enrolling',
                    'Review instructor ratings and course reviews',
                    'Consider your schedule and time commitment'
                ],
                'views' => 1245
            ],
            [
                'title' => 'Accessing Course Materials',
                'content' => 'Learn how to access and download course materials, videos, and resources.',
                'steps' => [
                    'Go to your enrolled course from "My Courses"',
                    'Navigate to "Course Materials" tab',
                    'View available PDFs, videos, and documents',
                    'Click download button to save materials locally',
                    'Check your "Downloads" page to track download history'
                ],
                'views' => 967
            ],
            [
                'title' => 'Understanding Course Progress Tracking',
                'content' => 'Monitor your learning progress and completion status.',
                'steps' => [
                    'View progress bar on your course dashboard',
                    'Complete lessons in sequential order',
                    'Check completion percentage for each module',
                    'Track your overall course completion status'
                ],
                'views' => 543
            ]
        ]
    ],
    [
        'id' => 'library',
        'title' => 'Library & Resources',
        'icon' => 'bx-library',
        'description' => 'Access and manage your learning resources effectively',
        'articles' => [
            [
                'title' => 'Using the Digital Library',
                'content' => 'Access thousands of educational resources, books, and reference materials.',
                'steps' => [
                    'Navigate to "Library" from the main menu',
                    'Browse resources by category or use search',
                    'Click on any resource to view details',
                    'Use "Preview" to see content overview',
                    'Click "Download" to save resources to your device',
                    'All downloads are tracked in your "Downloads" history'
                ],
                'features' => [
                    'Filter by Technology, Business, Science, etc.',
                    'Search by title, author, or keywords',
                    'Preview functionality for supported file types',
                    'Download tracking and history'
                ],
                'views' => 678
            ],
            [
                'title' => 'Managing Your Downloads',
                'content' => 'Track and re-download all your course materials and library resources.',
                'steps' => [
                    'Visit "Downloads" page from navigation menu',
                    'View complete history of all your downloads',
                    'Filter downloads by category (Course Materials, Library Resources, Assignments)',
                    'Use search to find specific downloads',
                    'Click "Download Again" to re-download any file',
                    'See download date, file size, and course information'
                ],
                'views' => 445
            ]
        ]
    ],
    [
        'id' => 'assignments',
        'title' => 'Assignments & Submissions',
        'icon' => 'bx-task',
        'description' => 'Complete and submit your assignments successfully',
        'articles' => [
            [
                'title' => 'Submitting Assignments',
                'content' => 'Complete guide to submitting assignments and tracking your submissions.',
                'steps' => [
                    'Go to "Assignments" section or access via course page',
                    'Click on assignment to view requirements and details',
                    'Download assignment files if provided',
                    'Prepare your submission according to requirements',
                    'Click "Submit Assignment" and upload your files',
                    'Confirm submission and note the submission date'
                ],
                'important_notes' => [
                    'Pay attention to file format requirements',
                    'Submit before the deadline to avoid penalties',
                    'Keep backup copies of your submissions',
                    'Check file size limits before uploading'
                ],
                'views' => 834
            ],
            [
                'title' => 'Viewing Assignment Feedback',
                'content' => 'Access instructor feedback and grades on your submitted work.',
                'steps' => [
                    'Return to the assignment after instructor review',
                    'Check for grade and feedback in assignment details',
                    'Download any feedback files provided',
                    'Review comments for improvement suggestions'
                ],
                'views' => 567
            ]
        ]
    ],
    [
        'id' => 'communication',
        'title' => 'Communication & Support',
        'icon' => 'bx-chat',
        'description' => 'Connect with instructors and get help when needed',
        'articles' => [
            [
                'title' => 'Contacting Your Instructors',
                'content' => 'Various ways to communicate with your course instructors.',
                'methods' => [
                    [
                        'type' => 'Direct Messages',
                        'description' => 'Send private messages to instructors through the platform',
                        'steps' => [
                            'Go to "Messages" from the main navigation',
                            'Click "New Message" or find existing conversation',
                            'Select your instructor from contacts',
                            'Type your message and send',
                            'Check for replies in your message inbox'
                        ]
                    ],
                    [
                        'type' => 'Course Discussions',
                        'description' => 'Participate in course discussion forums',
                        'steps' => [
                            'Access course and go to "Discussions" tab',
                            'Browse existing topics or create new ones',
                            'Ask questions or contribute to discussions',
                            'Get responses from instructors and peers'
                        ]
                    ]
                ],
                'views' => 723
            ],
            [
                'title' => 'Getting Technical Support',
                'content' => 'When you need help with platform issues or technical problems.',
                'contact_methods' => [
                    [
                        'method' => 'Email Support',
                        'contact' => 'admin@panaditeacademy.com',
                        'description' => 'Send detailed emails for complex issues',
                        'response_time' => '24-48 hours',
                        'best_for' => 'Technical problems, account issues, billing questions'
                    ],
                    [
                        'method' => 'Help Center',
                        'description' => 'Search our knowledge base for quick answers',
                        'best_for' => 'Common questions, how-to guides, troubleshooting'
                    ],
                    [
                        'method' => 'Platform Messages',
                        'description' => 'Use the messaging system for course-related questions',
                        'best_for' => 'Course content, assignment help, instructor communication'
                    ]
                ],
                'when_to_contact' => [
                    'Login or access problems',
                    'Payment and billing issues',
                    'Technical errors or bugs',
                    'Course enrollment problems',
                    'Certificate generation issues',
                    'Download or streaming problems'
                ],
                'what_to_include' => [
                    'Detailed description of the problem',
                    'Steps you\'ve already tried',
                    'Browser and device information',
                    'Screenshots if applicable',
                    'Your account email or username'
                ],
                'views' => 892
            ]
        ]
    ],
    [
        'id' => 'notifications',
        'title' => 'Notifications & Updates',
        'icon' => 'bx-bell',
        'description' => 'Stay updated with important announcements and activities',
        'articles' => [
            [
                'title' => 'Managing Your Notifications',
                'content' => 'Control what notifications you receive and how you receive them.',
                'steps' => [
                    'Click on "Notifications" in the main navigation',
                    'View all your recent notifications',
                    'Filter notifications by type (Course, Assignment, Message, System)',
                    'Mark individual notifications as read',
                    'Use "Mark All as Read" for bulk actions',
                    'Access notification settings to customize preferences'
                ],
                'notification_types' => [
                    'Course announcements and updates',
                    'Assignment deadlines and feedback',
                    'New messages from instructors',
                    'System maintenance and updates',
                    'Certificate availability',
                    'Payment confirmations'
                ],
                'views' => 456
            ]
        ]
    ],
    [
        'id' => 'certificates',
        'title' => 'Certificates & Achievements',
        'icon' => 'bx-award',
        'description' => 'Earn and manage your course completion certificates',
        'articles' => [
            [
                'title' => 'Earning Course Certificates',
                'content' => 'Complete courses and earn digital certificates to showcase your achievements.',
                'steps' => [
                    'Complete all required course modules',
                    'Submit and pass all required assignments',
                    'Meet the minimum course completion percentage',
                    'Certificate will be automatically generated',
                    'Access certificates from "Certificates" page',
                    'Download PDF versions for your records'
                ],
                'requirements' => [
                    'Complete 100% of course content',
                    'Submit all required assignments',
                    'Pass any required quizzes or exams',
                    'Meet attendance requirements (if applicable)'
                ],
                'views' => 634
            ]
        ]
    ],
    [
        'id' => 'troubleshooting',
        'title' => 'Troubleshooting & FAQ',
        'icon' => 'bx-help-circle',
        'description' => 'Quick solutions to common problems',
        'articles' => [
            [
                'title' => 'Common Download Issues',
                'content' => 'Solve problems when downloading course materials or library resources.',
                'common_issues' => [
                    [
                        'problem' => 'Download button not working',
                        'solutions' => [
                            'Refresh the page and try again',
                            'Clear your browser cache and cookies',
                            'Try a different browser',
                            'Check your internet connection',
                            'Contact support if problem persists'
                        ]
                    ],
                    [
                        'problem' => 'File not found error',
                        'solutions' => [
                            'The file may have been moved or updated',
                            'Try accessing through the original course',
                            'Check your Downloads history for alternative access',
                            'Contact admin@panaditeacademy.com for assistance'
                        ]
                    ],
                    [
                        'problem' => 'Slow download speeds',
                        'solutions' => [
                            'Check your internet connection speed',
                            'Try downloading during off-peak hours',
                            'Close other applications using bandwidth',
                            'Try a different network if available'
                        ]
                    ]
                ],
                'views' => 789
            ],
            [
                'title' => 'Login and Access Problems',
                'content' => 'Troubleshoot account access and login issues.',
                'solutions' => [
                    'Verify your email and password are correct',
                    'Use password reset if you forgot your password',
                    'Clear browser cookies and cache',
                    'Try logging in from a different browser',
                    'Check if your account needs activation',
                    'Contact admin@panaditeacademy.com for account issues'
                ],
                'views' => 543
            ],
            [
                'title' => 'Browser Compatibility',
                'content' => 'Ensure optimal platform performance with supported browsers.',
                'recommended_browsers' => [
                    'Google Chrome (latest version)',
                    'Mozilla Firefox (latest version)',
                    'Microsoft Edge (latest version)',
                    'Safari (latest version for Mac)'
                ],
                'requirements' => [
                    'JavaScript enabled',
                    'Cookies enabled',
                    'Pop-up blocker disabled for our site',
                    'Minimum screen resolution: 1024x768'
                ],
                'views' => 321
            ]
        ]
    ]
];

// Get popular articles across all topics
$allArticles = [];
foreach ($topics as $topic) {
    foreach ($topic['articles'] as $article) {
        $allArticles[] = [
            'title' => $article['title'],
            'views' => $article['views'],
            'topic' => $topic['title']
        ];
    }
}

// Sort by views to get the most popular
usort($allArticles, function($a, $b) {
    return $b['views'] - $a['views'];
});

// Get top 5 popular articles
$popularArticles = array_slice($allArticles, 0, 5);

// Handle search functionality (if implemented)
$searchQuery = isset($_GET['search']) ? trim($_GET['search']) : '';
$searchResults = [];

if (!empty($searchQuery)) {
    foreach ($allArticles as $article) {
        if (stripos($article['title'], $searchQuery) !== false) {
            $searchResults[] = $article;
        }
    }
}

// Include header
include_once 'components/header.php';

// Force direct CSS inclusion for help page to bypass detection issues
echo '<link rel="stylesheet" href="assets/css/pages/help.css?v=' . time() . '">';

// Add comprehensive CSS for guide sections
echo '<style>
.guide-section {
    background: #ffffff;
    border: 1px solid #e3e6f0;
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: all 0.3s ease;
}

.guide-section:hover {
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}

.guide-header {
    background: linear-gradient(135deg, rgba(245, 158, 11, 0.15) 0%, rgba(217, 119, 6, 0.2) 100%);
    color: #2c3e50;
    padding: 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.guide-icon {
    background: rgba(245, 158, 11, 0.2);
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 15px;
}

.guide-icon i {
    font-size: 24px;
    color: #d97706;
}

.guide-title-section {
    flex: 1;
}

.guide-title {
    margin: 0;
    font-size: 1.5rem;
    font-weight: 600;
}

.guide-description {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 0.95rem;
}

.guide-toggle {
    background: rgba(245, 158, 11, 0.2);
    border: 2px solid rgba(245, 158, 11, 0.3);
    color: #d97706;
    padding: 8px 16px;
    border-radius: 25px;
    transition: all 0.3s ease;
    font-weight: 600;
}

.guide-toggle:hover {
    background: rgba(245, 158, 11, 0.3);
    color: #b45309;
    transform: scale(1.05);
}

.guide-content {
    padding: 25px;
    background: #fafafa;
}

.article-guide {
    background: white;
    padding: 20px;
    border-radius: 8px;
    border-left: 4px solid #f59e0b;
    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
}

.article-title {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 15px;
    color: #2c3e50;
    font-weight: 600;
}

.article-title i {
    color: #f59e0b;
}

.views-badge {
    background: #f59e0b;
    color: white;
    padding: 4px 12px;
    border-radius: 15px;
    font-size: 0.8rem;
    font-weight: 500;
    margin-left: auto;
}

.article-description {
    color: #5a6c7d;
    margin-bottom: 20px;
    line-height: 1.6;
}

.steps-section, .tips-section, .important-notes, .methods-section, 
.contact-methods-section, .troubleshooting-section, .solutions-section {
    margin: 20px 0;
    padding: 15px;
    background: #f8f9fa;
    border-radius: 6px;
    border-left: 3px solid #28a745;
}

.steps-section h6, .tips-section h6, .important-notes h6, .methods-section h6,
.contact-methods-section h6, .troubleshooting-section h6, .solutions-section h6 {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 12px;
    color: #2c3e50;
    font-weight: 600;
}

.guide-steps, .guide-tips, .guide-notes, .solution-list {
    margin: 0;
    padding-left: 20px;
}

.guide-steps li, .guide-tips li, .guide-notes li, .solution-list li {
    margin-bottom: 8px;
    line-height: 1.5;
}

.method-card, .contact-method-card, .issue-card {
    background: white;
    padding: 15px;
    margin: 10px 0;
    border-radius: 6px;
    border: 1px solid #e3e6f0;
}

.method-card h7, .contact-method-card h7, .issue-card h7 {
    display: block;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
}

.contact-detail {
    margin: 8px 0;
}

.contact-detail a {
    color: #f59e0b;
    text-decoration: none;
    font-weight: 500;
}

.contact-detail a:hover {
    text-decoration: underline;
}

.when-contact, .what-include {
    margin-top: 15px;
}

.when-contact h7, .what-include h7 {
    display: block;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 10px;
}

.important-notes {
    border-left-color: #ffc107;
    background: #fff8dc;
}

.troubleshooting-section {
    border-left-color: #dc3545;
    background: #fff5f5;
}

.contact-methods-section {
    border-left-color: #17a2b8;
    background: #f0f9ff;
}

.contact-options {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
    margin-top: 30px;
}

.contact-card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    text-align: center;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    transition: all 0.3s ease;
    border: 1px solid #e3e6f0;
}

.contact-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
}

.contact-icon {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 15px;
}

.contact-icon i {
    font-size: 28px;
    color: white;
}

.contact-title {
    margin-bottom: 10px;
    color: #2c3e50;
    font-weight: 600;
}

.contact-description {
    color: #5a6c7d;
    margin-bottom: 15px;
    line-height: 1.5;
}

.contact-info {
    font-weight: 600;
    color: #667eea;
    margin-bottom: 15px;
    font-size: 1.1rem;
}

@media (max-width: 768px) {
    .guide-header {
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }
    
    .guide-toggle {
        align-self: stretch;
    }
    
    .article-title {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }
    
    .views-badge {
        align-self: flex-start;
        margin-left: 0;
    }
}
</style>';
?>

<!-- Help Center Container -->
<div class="container-fluid">
    <!-- Page Header -->
    <div class="page-header">
        <div>
            <h1 class="page-title">Help Center</h1>
            <p class="page-subtitle">Find answers, tutorials, and support resources</p>
        </div>
    </div>
    
    <!-- Help Hero with Search -->
    <div class="help-hero">
        <h2>How can we help you?</h2>
        <p>Search our knowledge base for answers to your questions</p>
        
        <div class="search-container">
            <div class="search-wrapper">
                <i class="bx bx-search search-icon"></i>
                <input type="text" id="help-search-input" class="form-control" placeholder="Search for help topics..." autocomplete="off">
                <button type="button" id="clear-search" class="btn btn-light" style="display: none;">
                    <i class="bx bx-x"></i>
                </button>
                <div class="search-loading" id="search-loading" style="display: none;">
                    <i class="bx bx-loader-alt bx-spin"></i>
                </div>
            </div>
            
            <!-- Live Search Results -->
            <div class="live-search-results" id="live-search-results" style="display: none;"></div>
        </div>
    </div>
    
    <!-- Main Help Content -->
    
    <!-- Help Topics with Detailed Guides -->
    <h3 class="mb-4">Comprehensive Student Guides</h3>
    
    <?php foreach ($topics as $topicIndex => $topic): ?>
    <div class="guide-section mb-5" id="<?= $topic['id'] ?>">
        <div class="guide-header">
            <div class="guide-icon">
                <i class="bx <?= $topic['icon'] ?>"></i>
            </div>
            <div class="guide-title-section">
                <h4 class="guide-title"><?= htmlspecialchars($topic['title']) ?></h4>
                <p class="guide-description"><?= htmlspecialchars($topic['description']) ?></p>
            </div>
            <button class="btn btn-outline-primary guide-toggle" onclick="toggleGuide(<?= $topicIndex ?>)">
                <i class="bx bx-chevron-down"></i> Show Guides
            </button>
        </div>
        
        <div class="guide-content" id="guide-content-<?= $topicIndex ?>" style="display: none;">
            <?php foreach ($topic['articles'] as $articleIndex => $article): ?>
            <div class="article-guide mb-4">
                <h5 class="article-title">
                    <i class="bx bx-file-blank"></i>
                    <?= htmlspecialchars($article['title']) ?>
                    <span class="views-badge"><?= number_format($article['views']) ?> views</span>
                </h5>
                
                <div class="article-content">
                    <p class="article-description"><?= htmlspecialchars($article['content']) ?></p>
                    
                    <?php if (isset($article['steps'])): ?>
                    <div class="steps-section">
                        <h6><i class="bx bx-list-ol"></i> Step-by-Step Guide:</h6>
                        <ol class="guide-steps">
                            <?php foreach ($article['steps'] as $step): ?>
                            <li><?= htmlspecialchars($step) ?></li>
                            <?php endforeach; ?>
                        </ol>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['tips'])): ?>
                    <div class="tips-section">
                        <h6><i class="bx bx-bulb"></i> Pro Tips:</h6>
                        <ul class="guide-tips">
                            <?php foreach ($article['tips'] as $tip): ?>
                            <li><?= htmlspecialchars($tip) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['important_notes'])): ?>
                    <div class="important-notes">
                        <h6><i class="bx bx-info-circle"></i> Important Notes:</h6>
                        <ul class="guide-notes">
                            <?php foreach ($article['important_notes'] as $note): ?>
                            <li><?= htmlspecialchars($note) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['methods'])): ?>
                    <div class="methods-section">
                        <h6><i class="bx bx-network-chart"></i> Communication Methods:</h6>
                        <?php foreach ($article['methods'] as $method): ?>
                        <div class="method-card">
                            <h7><i class="bx bx-message"></i> <?= htmlspecialchars($method['type']) ?></h7>
                            <p><?= htmlspecialchars($method['description']) ?></p>
                            <?php if (isset($method['steps'])): ?>
                            <ol>
                                <?php foreach ($method['steps'] as $step): ?>
                                <li><?= htmlspecialchars($step) ?></li>
                                <?php endforeach; ?>
                            </ol>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['contact_methods'])): ?>
                    <div class="contact-methods-section">
                        <h6><i class="bx bx-support"></i> Support Contact Methods:</h6>
                        <?php foreach ($article['contact_methods'] as $contactMethod): ?>
                        <div class="contact-method-card">
                            <h7><?= htmlspecialchars($contactMethod['method']) ?></h7>
                            <?php if (isset($contactMethod['contact'])): ?>
                            <div class="contact-detail">
                                <strong>Contact:</strong> 
                                <a href="mailto:<?= htmlspecialchars($contactMethod['contact']) ?>"><?= htmlspecialchars($contactMethod['contact']) ?></a>
                            </div>
                            <?php endif; ?>
                            <p><?= htmlspecialchars($contactMethod['description']) ?></p>
                            <?php if (isset($contactMethod['response_time'])): ?>
                            <p><strong>Response Time:</strong> <?= htmlspecialchars($contactMethod['response_time']) ?></p>
                            <?php endif; ?>
                            <?php if (isset($contactMethod['best_for'])): ?>
                            <p><strong>Best for:</strong> <?= htmlspecialchars($contactMethod['best_for']) ?></p>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                        
                        <?php if (isset($article['when_to_contact'])): ?>
                        <div class="when-contact">
                            <h7><i class="bx bx-help-circle"></i> When to Contact Support:</h7>
                            <ul>
                                <?php foreach ($article['when_to_contact'] as $situation): ?>
                                <li><?= htmlspecialchars($situation) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($article['what_to_include'])): ?>
                        <div class="what-include">
                            <h7><i class="bx bx-list-check"></i> What to Include in Your Support Request:</h7>
                            <ul>
                                <?php foreach ($article['what_to_include'] as $item): ?>
                                <li><?= htmlspecialchars($item) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['common_issues'])): ?>
                    <div class="troubleshooting-section">
                        <h6><i class="bx bx-wrench"></i> Common Issues & Solutions:</h6>
                        <?php foreach ($article['common_issues'] as $issue): ?>
                        <div class="issue-card">
                            <h7><i class="bx bx-error"></i> <?= htmlspecialchars($issue['problem']) ?></h7>
                            <div class="solutions">
                                <strong>Solutions:</strong>
                                <ul>
                                    <?php foreach ($issue['solutions'] as $solution): ?>
                                    <li><?= htmlspecialchars($solution) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (isset($article['solutions'])): ?>
                    <div class="solutions-section">
                        <h6><i class="bx bx-check-circle"></i> Solutions:</h6>
                        <ul class="solution-list">
                            <?php foreach ($article['solutions'] as $solution): ?>
                            <li><?= htmlspecialchars($solution) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endforeach; ?>
    
    <!-- Contact Options -->
    <h3 class="mb-4">Still Need Help?</h3>
    <div class="contact-options">
        <div class="contact-card">
            <div class="contact-icon" style="width: 64px; height: 64px; display: flex; align-items: center; justify-content: center; background-color: rgba(245, 158, 11, 0.15); color: #d97706; border-radius: 50%; font-size: 28px; margin: 0 auto 16px auto; border: none;">
                <i class="bx bx-envelope" style="color: #d97706;"></i>
            </div>
            <h4 class="contact-title">Email Support</h4>
            <p class="contact-description">Send us an email for technical support, account issues, or general inquiries. Our support team responds within 24-48 hours.</p>
            <div class="contact-info">admin@panaditeacademy.com</div>
            <a href="mailto:admin@panaditeacademy.com" class="btn btn-primary">Send Email</a>
        </div>
        
        <div class="contact-card">
            <div class="contact-icon" style="width: 64px; height: 64px; display: flex; align-items: center; justify-content: center; background-color: rgba(245, 158, 11, 0.15); color: #d97706; border-radius: 50%; font-size: 28px; margin: 0 auto 16px auto; border: none;">
                <i class="bx bx-help-circle" style="color: #d97706;"></i>
            </div>
            <h4 class="contact-title">Browse Guides</h4>
            <p class="contact-description">Search through our comprehensive guides above for step-by-step instructions on using the platform.</p>
            <a href="#getting-started" class="btn btn-outline-primary">Start Here</a>
        </div>
    </div>
</div>

<!-- CSS for AJAX Search -->
<style>
/* Override Bootstrap CSS variables and contact icon colors with very light amber */
:root {
    --bs-primary-rgb: 245, 158, 11 !important;
    --bs-primary: #f59e0b !important;
}

/* Override contact icon colors with very light amber */
.contact-options {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 24px;
    margin-bottom: 32px;
}

.contact-card {
    background-color: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    padding: 24px;
    text-align: center;
    transition: all 0.3s ease;
}

.contact-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 15px rgba(0, 0, 0, 0.08);
}

.contact-icon,
.contact-card .contact-icon,
div.contact-icon,
.contact-options .contact-icon,
.contact-options .contact-card .contact-icon {
    width: 64px !important;
    height: 64px !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    background: rgba(245, 158, 11, 0.15) !important;
    background-color: rgba(245, 158, 11, 0.15) !important;
    color: #d97706 !important;
    border-radius: 50% !important;
    font-size: 28px !important;
    margin: 0 auto 16px auto !important;
    border: none !important;
}

.contact-icon i,
.contact-card .contact-icon i,
div.contact-icon i {
    color: #d97706 !important;
    background: transparent !important;
}

.contact-title {
    font-size: 18px;
    font-weight: 600;
    margin-bottom: 8px;
    color: #2c3e50;
}

.contact-description {
    font-size: 14px;
    color: #6c757d;
    margin-bottom: 16px;
}

.contact-info {
    font-weight: 500;
    font-size: 16px;
    color: #2c3e50;
    margin-bottom: 16px;
}
.search-wrapper {
    position: relative;
    display: flex;
    align-items: center;
    background: white;
    border-radius: 25px;
    padding: 5px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    max-width: 600px;
    margin: 0 auto;
}

.search-icon {
    position: absolute;
    left: 20px;
    color: #667eea;
    font-size: 20px;
    z-index: 2;
}

#help-search-input {
    flex: 1;
    border: none;
    outline: none;
    padding: 12px 20px 12px 50px;
    font-size: 16px;
    border-radius: 20px;
    background: transparent;
}

#help-search-input:focus {
    box-shadow: none;
    border: none;
}

#clear-search {
    background: none;
    border: none;
    color: #999;
    padding: 8px;
    border-radius: 50%;
    margin-right: 5px;
}

#clear-search:hover {
    background: #f0f0f0;
    color: #333;
}

.search-loading {
    position: absolute;
    right: 15px;
    color: #667eea;
    font-size: 18px;
}

.live-search-results {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border-radius: 10px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
    max-height: 400px;
    overflow-y: auto;
    z-index: 1000;
    margin-top: 10px;
}

.search-result-item {
    display: block;
    padding: 15px 20px;
    border-bottom: 1px solid #f0f0f0;
    text-decoration: none;
    color: inherit;
    transition: background-color 0.2s ease;
}

.search-result-item:hover {
    background-color: #f8f9fa;
    text-decoration: none;
    color: inherit;
}

.search-result-item:last-child {
    border-bottom: none;
}

.search-result-title {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 5px;
    display: flex;
    align-items: center;
    gap: 8px;
}

.search-result-title i {
    color: #667eea;
}

.search-result-content {
    color: #5a6c7d;
    font-size: 14px;
    line-height: 1.4;
    margin-bottom: 8px;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.search-result-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 12px;
}

.search-topic-tag {
    background: #667eea;
    color: white;
    padding: 2px 8px;
    border-radius: 10px;
    font-size: 11px;
}

.search-views {
    color: #999;
    display: flex;
    align-items: center;
    gap: 3px;
}

.search-highlight {
    background: #fff59d;
    padding: 1px 2px;
    border-radius: 2px;
    font-weight: 600;
}

.no-results {
    text-align: center;
    padding: 30px;
    color: #999;
}

.no-results i {
    font-size: 48px;
    margin-bottom: 15px;
    color: #ddd;
}

.search-container {
    position: relative;
}

.search-stats {
    padding: 10px 20px;
    background: #f8f9fa;
    border-bottom: 1px solid #e9ecef;
    font-size: 14px;
    color: #6c757d;
}

@media (max-width: 768px) {
    .search-wrapper {
        margin: 0 15px;
    }
    
    .live-search-results {
        margin: 10px 15px 0;
        left: 15px;
        right: 15px;
    }
}
</style>

<!-- JavaScript for AJAX Search and Guide Toggle Functionality -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
// AJAX Search Functionality
$(document).ready(function() {
    let searchTimeout;
    const $searchInput = $('#help-search-input');
    const $searchResults = $('#live-search-results');
    const $clearButton = $('#clear-search');
    const $searchLoading = $('#search-loading');
    const $guideSections = $('.guide-section');
    
    // Search input event handler
    $searchInput.on('input', function() {
        const query = $(this).val().trim();
        
        // Clear existing timeout
        clearTimeout(searchTimeout);
        
        // Show/hide clear button
        if (query.length > 0) {
            $clearButton.show();
        } else {
            $clearButton.hide();
            $searchResults.hide();
            $guideSections.show();
            return;
        }
        
        // Debounce search (wait 300ms after user stops typing)
        searchTimeout = setTimeout(function() {
            if (query.length >= 2) {
                performSearch(query);
            } else if (query.length === 0) {
                $searchResults.hide();
                $guideSections.show();
            }
        }, 300);
    });
    
    // Clear search button
    $clearButton.on('click', function() {
        $searchInput.val('').focus();
        $clearButton.hide();
        $searchResults.hide();
        $guideSections.show();
    });
    
    // Perform AJAX search
    function performSearch(query) {
        $searchLoading.show();
        $guideSections.hide();
        
        $.ajax({
            url: 'ajax_help_search.php',
            method: 'POST',
            data: {
                query: query
            },
            dataType: 'json',
            success: function(response) {
                $searchLoading.hide();
                
                if (response.success) {
                    displaySearchResults(response.results, response.query, response.count);
                } else {
                    showSearchError(response.message);
                }
            },
            error: function(xhr, status, error) {
                $searchLoading.hide();
                console.error('Search error:', error);
                showSearchError('Search temporarily unavailable. Please try again.');
            }
        });
    }
    
    // Display search results
    function displaySearchResults(results, query, count) {
        let html = '';
        
        if (results.length > 0) {
            html += '<div class="search-stats">Found ' + count + ' result' + (count !== 1 ? 's' : '') + ' for "' + escapeHtml(query) + '"</div>';
            
            results.forEach(function(result) {
                // Highlight search terms in title and content
                const highlightedTitle = highlightSearchTerms(result.title, query);
                const highlightedContent = highlightSearchTerms(result.content, query);
                
                html += '<a href="#' + result.topic_id + '" class="search-result-item" onclick="handleSearchResultClick(\'' + result.topic_id + '\')">'
                + '<div class="search-result-title">'
                + '<i class="bx bx-file-blank"></i>'
                + highlightedTitle
                + '</div>'
                + '<div class="search-result-content">'
                + highlightedContent
                + '</div>'
                + '<div class="search-result-meta">'
                + '<span class="search-topic-tag">' + escapeHtml(result.topic) + '</span>'
                + '<span class="search-views">'
                + '<i class="bx bx-show"></i>'
                + formatNumber(result.views) + ' views'
                + '</span>'
                + '</div>'
                + '</a>';
            });
        } else {
            html = '<div class="no-results">'
            + '<i class="bx bx-search-alt"></i>'
            + '<h4>No Results Found</h4>'
            + '<p>We couldn\'t find any guides matching "' + escapeHtml(query) + '". Try different keywords or browse all topics below.</p>'
            + '</div>';
        }
        
        $searchResults.html(html).show();
    }
    
    // Show search error
    function showSearchError(message) {
        $searchResults.html('<div class="no-results">' +
            '<i class="bx bx-error"></i>' +
            '<h4>Search Error</h4>' +
            '<p>' + escapeHtml(message) + '</p>' +
            '</div>'
        ).show();
    }
    
    // Highlight search terms
    function highlightSearchTerms(text, query) {
        const words = query.split(' ').filter(word => word.length > 1);
        let highlighted = escapeHtml(text);
        
        words.forEach(function(word) {
            const regex = new RegExp('(' + escapeRegExp(word) + ')', 'gi');
            highlighted = highlighted.replace(regex, '<span class="search-highlight">$1</span>');
        });
        
        return highlighted;
    }
    
    // Utility functions
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }
    
    function formatNumber(num) {
        return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    }
    
    // Handle click outside search results to close
    $(document).on('click', function(event) {
        if (!$(event.target).closest('.search-container').length) {
            // $searchResults.hide();
        }
    });
    
    // Handle search result clicks
    window.handleSearchResultClick = function(topicId) {
        $searchInput.val('');
        $clearButton.hide();
        $searchResults.hide();
        $guideSections.show();
        
        // Find and expand the target section
        setTimeout(function() {
            const targetSection = document.getElementById(topicId);
            if (targetSection) {
                const index = Array.from(document.querySelectorAll('.guide-section')).indexOf(targetSection);
                if (index !== -1) {
                    // Open the guide if it's closed
                    const content = document.getElementById('guide-content-' + index);
                    if (content.style.display === 'none') {
                        toggleGuide(index);
                    }
                    // Scroll to the section
                    targetSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            }
        }, 100);
    };
});

// Guide Toggle Functionality
function toggleGuide(index) {
    const content = document.getElementById('guide-content-' + index);
    const button = content.previousElementSibling.querySelector('.guide-toggle');
    const icon = button.querySelector('i');
    
    if (content.style.display === 'none') {
        content.style.display = 'block';
        button.innerHTML = '<i class="bx bx-chevron-up"></i> Hide Guides';
        content.scrollIntoView({ behavior: 'smooth', block: 'start' });
    } else {
        content.style.display = 'none';
        button.innerHTML = '<i class="bx bx-chevron-down"></i> Show Guides';
    }
}

// Auto-expand section if directly linked
window.addEventListener('load', function() {
    const hash = window.location.hash;
    if (hash) {
        const targetElement = document.querySelector(hash);
        if (targetElement && targetElement.classList.contains('guide-section')) {
            const index = Array.from(document.querySelectorAll('.guide-section')).indexOf(targetElement);
            if (index !== -1) {
                toggleGuide(index);
            }
        }
    }
});

// Add smooth scrolling for anchor links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        const href = this.getAttribute('href');
        if (href !== '#') {
            e.preventDefault();
            const target = document.querySelector(href);
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth'
                });
            }
        }
    });
});
</script>
