<?php
/**
 * SITA-Compliant Accredited Certificate System Logic
 * Handles all accreditation workflow stages
 */

// Get current accreditation request status
$accreditation_request = null;
$current_assessment = null;
$assessment_attempt = null;
$submitted_documents = [];
$issued_certificate = null;

if (isset($conn) && isset($user_id) && isset($course_id)) {
    // Check for existing ACTIVE accreditation request (exclude cancelled/rejected)
    $request_sql = "SELECT ar.*, 
                           u.first_name as admin_first_name, u.last_name as admin_last_name
                    FROM accreditation_requests ar
                    LEFT JOIN users u ON ar.approved_by = u.user_id
                    WHERE ar.student_id = ? AND ar.course_id = ? 
                    AND ar.status NOT IN ('cancelled', 'rejected', 'certificate_issued')
                    ORDER BY ar.request_date DESC LIMIT 1";
    
    if ($stmt = $conn->prepare($request_sql)) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $accreditation_request = $result->fetch_assoc();
        $stmt->close();
    }

    // If request exists, get additional data based on status
    if ($accreditation_request) {
        $request_id = $accreditation_request['request_id'];
        
        // Get current assessment if available
        if (in_array($accreditation_request['status'], ['assignment_created', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued'])) {
            $assessment_sql = "SELECT * FROM sita_assessments WHERE request_id = ? AND is_published = 1";
            if ($stmt = $conn->prepare($assessment_sql)) {
                $stmt->bind_param("i", $request_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $current_assessment = $result->fetch_assoc();
                $stmt->close();
            }
        }

        // Get assessment attempt if exists
        if ($current_assessment) {
            $attempt_sql = "SELECT * FROM sita_assessment_attempts 
                           WHERE assessment_id = ? AND student_id = ?
                           ORDER BY attempt_number DESC LIMIT 1";
            if ($stmt = $conn->prepare($attempt_sql)) {
                $stmt->bind_param("ii", $current_assessment['assessment_id'], $user_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $assessment_attempt = $result->fetch_assoc();
                $stmt->close();
            }
        }

        // Get submitted documents
        $docs_sql = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY upload_date DESC";
        if ($stmt = $conn->prepare($docs_sql)) {
            $stmt->bind_param("i", $request_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($doc = $result->fetch_assoc()) {
                $submitted_documents[] = $doc;
            }
            $stmt->close();
        }

        // Get issued certificate if available
        if ($accreditation_request['status'] == 'certificate_issued') {
            $cert_sql = "SELECT * FROM accredited_certificates WHERE request_id = ?";
            if ($stmt = $conn->prepare($cert_sql)) {
                $stmt->bind_param("i", $request_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $issued_certificate = $result->fetch_assoc();
                $stmt->close();
            }
        }
    }
}

// Helper function to get status badge
function getStatusBadge($status) {
    $badges = [
        'pending' => '<span class="badge bg-warning">⏳ Pending Admin Review</span>',
        'approved' => '<span class="badge bg-primary">✅ Approved - Assignment Creating</span>',
        'assignment_created' => '<span class="badge bg-info">📝 SITA Assignment Ready</span>',
        'assessment_created' => '<span class="badge bg-info">📝 SITA Assessment Ready</span>',
        'assessment_completed' => '<span class="badge bg-success">✅ Assessment Completed</span>',
        'documents_pending' => '<span class="badge bg-warning">📄 Documents Required</span>',
        'documents_submitted' => '<span class="badge bg-info">📄 Documents Under Review</span>',
        'under_review' => '<span class="badge bg-primary">👨‍💼 Under Admin Review</span>',
        'approved_final' => '<span class="badge bg-success">🎉 Approved - Certificate Processing</span>',
        'rejected' => '<span class="badge bg-danger">❌ Request Rejected</span>',
        'certificate_issued' => '<span class="badge bg-success">🏆 Certificate Issued</span>'
    ];
    return $badges[$status] ?? '<span class="badge bg-secondary">Unknown Status</span>';
}

// Helper function to get status description
function getStatusDescription($status) {
    $descriptions = [
        'pending' => 'Your accreditation request is awaiting admin approval. You will receive an email notification once reviewed.',
        'approved' => 'Great news! Your request has been approved. Our admin team is preparing your SITA-compliant assignment.',
        'assignment_created' => 'Your SITA assignment is ready! Complete the assignment requirements and upload your documents to proceed.',
        'assessment_created' => 'Your SITA assessment is ready! You must achieve the minimum pass mark to proceed.',
        'assessment_completed' => 'Excellent! You have successfully completed the SITA assessment. Next step: document submission.',
        'documents_pending' => 'Please upload all required documents: CV, qualifications, ID, and your assignment work.',
        'documents_submitted' => 'All documents received. Our admin team is reviewing your submissions.',
        'under_review' => 'Your complete application is under final review by our certification team.',
        'approved_final' => 'Congratulations! Your application has been approved. Your certificate is being generated.',
        'rejected' => 'Unfortunately, your request has been rejected. Please contact support for more information.',
        'certificate_issued' => 'Success! Your SITA-accredited certificate has been issued and is ready for download.'
    ];
    return $descriptions[$status] ?? 'Status information not available.';
}

// Helper function to get next steps
function getNextSteps($status, $assessment_attempt = null) {
    switch ($status) {
        case 'pending':
            return 'Wait for admin approval notification via email.';
        case 'approved':
            return 'Wait for SITA assignment to be prepared by admin team.';
        case 'assignment_created':
            return 'Complete your SITA assignment and upload the required documents. Visit your Assignments page to get started.';
        case 'assessment_created':
            return 'Take the SITA assessment and achieve the minimum pass mark.';
        case 'assessment_completed':
            if ($assessment_attempt && $assessment_attempt['passed']) {
                return 'Upload your required documents using the form below.';
            } else {
                return 'Assessment not passed. Contact admin for retake options.';
            }
        case 'documents_pending':
            return 'Upload all required documents: CV, qualifications, ID document, and assignment work.';
        case 'documents_submitted':
            return 'Wait for admin review of your submitted documents.';
        case 'under_review':
            return 'Your application is under final review. You will be notified of the outcome.';
        case 'approved_final':
            return 'Your certificate is being generated. You will receive it soon.';
        case 'certificate_issued':
            return 'Download your certificate and share your achievement!';
        case 'rejected':
            return 'Contact support to understand the rejection reason and next steps.';
        default:
            return 'Contact support for assistance with your application.';
    }
}
?>
