<?php
/**
 * Get User Profile Data
 * This script retrieves all user profile data from the database
 */

function getUserProfileData($userId) {
    global $conn;
    $userData = [];
    
    if (!$conn) {
        return $userData; // Return empty array if no database connection
    }
    
    try {
        // Fetch basic user data
        $query = "SELECT u.user_id, u.username, u.email, u.first_name, u.last_name, u.user_role, 
                        u.created_at, u.last_login, u.status, u.phone, u.profile_picture,
                        sp.student_id, sp.address, sp.city, sp.country, sp.postal_code, 
                        sp.date_of_birth, sp.gender, sp.bio, sp.major, sp.graduation_year,
                        sp.enrollment_date, sp.gpa
                 FROM users u
                 LEFT JOIN student_profiles sp ON u.user_id = sp.student_id
                 WHERE u.user_id = ?";
        
        $stmt = $conn->prepare($query);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $userData = $result->fetch_assoc();
            
            // Initialize default values for student profile fields if they are null
            $studentFields = ['address', 'city', 'country', 'postal_code', 'date_of_birth', 
                             'gender', 'bio', 'major', 'graduation_year', 'enrollment_date', 'gpa'];
            
            foreach ($studentFields as $field) {
                if (!isset($userData[$field])) {
                    $userData[$field] = null;
                }
            }
            
            // Format data for display
            $userData['fullname'] = $userData['first_name'] . ' ' . $userData['last_name'];
            
            // Get course count
            $countQuery = "SELECT COUNT(*) as course_count FROM enrollments WHERE student_id = ?";
            $countStmt = $conn->prepare($countQuery);
            $countStmt->bind_param("i", $userId);
            $countStmt->execute();
            $countResult = $countStmt->get_result();
            if ($countRow = $countResult->fetch_assoc()) {
                $userData['course_count'] = $countRow['course_count'];
            } else {
                $userData['course_count'] = 0;
            }
            
            // Calculate attendance percentage (example - modify as needed)
            $attendanceQuery = "SELECT AVG(attendance) as avg_attendance 
                               FROM course_attendance 
                               WHERE student_id = ?";
            $attendanceStmt = $conn->prepare($attendanceQuery);
            $attendanceStmt->bind_param("i", $userId);
            $attendanceStmt->execute();
            $attendanceResult = $attendanceStmt->get_result();
            if ($attendanceRow = $attendanceResult->fetch_assoc()) {
                $userData['attendance'] = $attendanceRow['avg_attendance'] ? round($attendanceRow['avg_attendance'], 1) . '%' : '0%';
            } else {
                $userData['attendance'] = '0%';
            }
            
            // Format GPA with 1 decimal place
            $userData['formatted_gpa'] = number_format($userData['gpa'] ?? 0, 1);
            
            // Format dates
            if (!empty($userData['date_of_birth'])) {
                $dob = new DateTime($userData['date_of_birth']);
                $userData['formatted_dob'] = $dob->format('m/d/Y');
            } else {
                $userData['formatted_dob'] = '';
            }
            
            if (!empty($userData['enrollment_date'])) {
                $enrollDate = new DateTime($userData['enrollment_date']);
                $userData['formatted_enrollment_date'] = $enrollDate->format('m/d/Y');
            } else {
                $userData['formatted_enrollment_date'] = '';
            }
            
            // Check for profile picture
            $profilePicPath = "../uploads/profile_pictures/{$userId}.jpg";
            $userData['has_profile_pic'] = file_exists($profilePicPath);
        }
    } catch (Exception $e) {
        error_log("Error fetching user profile: " . $e->getMessage());
    }
    
    return $userData;
}
?>
