<?php
/**
 * AJAX Endpoint: Get Lesson Data
 * Returns lesson data for AJAX video navigation without page reload
 */

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Set JSON response header immediately
header('Content-Type: application/json');

// Add debug logging function
function debug_log($message) {
    error_log("[AJAX DEBUG] " . $message);
}

debug_log("AJAX endpoint called");

try {
    require_once '../../includes/config.php';
    debug_log("Config loaded");
    
    require_once '../../includes/functions.php';
    debug_log("Functions loaded");
    
    require_once '../includes/id_encryption.php';
    debug_log("ID encryption loaded");

    // Security: Require student authentication
    requireRole('student');
    debug_log("Authentication passed");

    // Get JSON input
    $raw_input = file_get_contents('php://input');
    debug_log("Raw input: " . $raw_input);
    
    $input = json_decode($raw_input, true);
    debug_log("Parsed input: " . json_encode($input));
    
    if (!$input || !isset($input['lesson_id']) || !isset($input['course_id'])) {
        throw new Exception('Invalid input data. Raw: ' . $raw_input . ' Parsed: ' . json_encode($input));
    }
    
    $lesson_id = (int)$input['lesson_id'];
    $encrypted_course_id = $input['course_id'];
    $user_id = $_SESSION['user_id'];
    
    debug_log("Lesson ID: $lesson_id, User ID: $user_id, Encrypted Course ID: $encrypted_course_id");
    
    // Decrypt the course ID
    $course_id = null;
    try {
        $course_id = decryptId($encrypted_course_id);
        debug_log("Decrypted Course ID: $course_id");
    } catch (Exception $e) {
        throw new Exception('Invalid course ID format: ' . $e->getMessage());
    }
    
    // Use the global database connection from config
    if (!isset($conn)) {
        throw new Exception('Database connection not available');
    }
    
    // Validate that student is enrolled in this course
    $enrollmentCheck = $pdo->prepare("
        SELECT enrollment_id 
        FROM course_enrollments 
        WHERE student_id = ? AND course_id = ? AND status = 'active'
    ");
    $enrollmentCheck->execute([$user_id, $course_id]);
    
    if (!$enrollmentCheck->fetch()) {
        throw new Exception('Not enrolled in this course');
    }
    
    // Get lesson data with progress information
    $lessonQuery = $pdo->prepare("
        SELECT 
            l.*,
            v.*,
            COALESCE(vp.progress_percentage, 0) as progress_percentage,
            COALESCE(vp.completed, 0) as completed,
            COALESCE(vp.last_position, 0) as last_position
        FROM lessons l
        LEFT JOIN videos v ON l.video_id = v.video_id
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.user_id = ?
        WHERE l.lesson_id = ? AND l.course_id = ?
    ");
    
    $lessonQuery->execute([$user_id, $lesson_id, $course_id]);
    $lesson = $lessonQuery->fetch(PDO::FETCH_ASSOC);
    
    if (!$lesson) {
        throw new Exception('Lesson not found');
    }
    
    // Process video URL for external videos
    if ($lesson['is_external'] && $lesson['video_url']) {
        // Convert YouTube watch URLs to embed URLs
        if (strpos($lesson['video_url'], 'youtube.com/watch') !== false) {
            parse_str(parse_url($lesson['video_url'], PHP_URL_QUERY), $params);
            if (isset($params['v'])) {
                $lesson['video_url'] = 'https://www.youtube.com/embed/' . $params['v'];
            }
        } elseif (strpos($lesson['video_url'], 'youtu.be/') !== false) {
            $video_id = substr(parse_url($lesson['video_url'], PHP_URL_PATH), 1);
            $lesson['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
        }
    } elseif (!$lesson['is_external'] && $lesson['video_file']) {
        // Local video file
        $lesson['video_url'] = '../../uploads/videos/' . $lesson['video_file'];
    }
    
    // Get course progress for context
    $courseProgressQuery = $pdo->prepare("
        SELECT 
            COUNT(*) as total_lessons,
            SUM(CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END) as completed_lessons
        FROM lessons l
        LEFT JOIN videos v ON l.video_id = v.video_id
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.user_id = ?
        WHERE l.course_id = ?
    ");
    
    $courseProgressQuery->execute([$user_id, $course_id]);
    $courseProgress = $courseProgressQuery->fetch(PDO::FETCH_ASSOC);
    
    // Calculate overall progress percentage
    $overallProgress = 0;
    if ($courseProgress['total_lessons'] > 0) {
        $overallProgress = ($courseProgress['completed_lessons'] / $courseProgress['total_lessons']) * 100;
    }
    
    // Get lesson order for auto-advance
    $lessonOrderQuery = $pdo->prepare("
        SELECT lesson_id, title 
        FROM lessons 
        WHERE course_id = ? 
        ORDER BY lesson_order ASC, lesson_id ASC
    ");
    $lessonOrderQuery->execute([$course_id]);
    $allLessons = $lessonOrderQuery->fetchAll(PDO::FETCH_ASSOC);
    
    // Find current and next lesson
    $currentIndex = array_search($lesson_id, array_column($allLessons, 'lesson_id'));
    $nextLesson = null;
    
    if ($currentIndex !== false && isset($allLessons[$currentIndex + 1])) {
        $nextLesson = $allLessons[$currentIndex + 1];
    }
    
    // Return successful response
    echo json_encode([
        'success' => true,
        'lesson' => $lesson,
        'course_progress' => [
            'total_lessons' => (int)$courseProgress['total_lessons'],
            'completed_lessons' => (int)$courseProgress['completed_lessons'],
            'progress_percentage' => round($overallProgress, 1)
        ],
        'next_lesson' => $nextLesson,
        'all_lessons' => array_column($allLessons, 'lesson_id')
    ]);
    
} catch (Exception $e) {
    error_log('Get lesson data error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
