<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

$page_title = 'Digital Library';
$user_id = $_SESSION['user_id'];

// Function to check if table exists
function tableExists($conn, $tableName) {
    if (!$conn) return false;
    $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
    return ($result && $result->num_rows > 0);
}

// Get approved library resources from database
$resources = [];
$resources_by_category = [];

// Get approved resources from library submissions
if (isset($conn) && tableExists($conn, 'library_submissions')) {
    // First, let's debug what's actually in the database
    $debug_stmt = $conn->prepare("SELECT COUNT(*) as total, 
                                         SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                                         SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                                         SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
                                  FROM library_submissions");
    if ($debug_stmt) {
        $debug_stmt->execute();
        $debug_result = $debug_stmt->get_result();
        $debug_data = $debug_result->fetch_assoc();
        // Debug output disabled since library is working
        // echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; margin: 10px; border-radius: 5px; font-family: monospace;'>DEBUG: Total Submissions: {$debug_data['total']}, Approved: {$debug_data['approved']}, Pending: {$debug_data['pending']}, Rejected: {$debug_data['rejected']}</div>";
        $debug_stmt->close();
    }
    
    $stmt = $conn->prepare("
        SELECT ls.submission_id as id, ls.title, ls.description, ls.file_type, 
               ls.file_size, ls.file_path, ls.thumbnail_path, ls.submission_date as publication_date,
               c.category_name, u.username as teacher_name
        FROM library_submissions ls
        LEFT JOIN categories c ON ls.category_id = c.category_id  
        LEFT JOIN users u ON ls.teacher_id = u.user_id
        WHERE ls.status = 'approved'
        ORDER BY c.category_name, ls.submission_date DESC
    ");
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $resources[] = $row;
                $category = $row['category_name'] ?? 'Uncategorized';
                if (!isset($resources_by_category[$category])) {
                    $resources_by_category[$category] = [];
                }
                $resources_by_category[$category][] = $row;
            }
        }
        $stmt->close();
    }
}

// If no approved documents, try public_library_resources as backup
if (empty($resources) && isset($conn) && tableExists($conn, 'public_library_resources')) {
    $stmt = $conn->prepare("
        SELECT plr.*, c.category_name, u.username as teacher_name
        FROM public_library_resources plr
        LEFT JOIN categories c ON plr.category_id = c.category_id  
        LEFT JOIN users u ON plr.teacher_id = u.user_id
        ORDER BY c.category_name, plr.publication_date DESC
    ");
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $resources[] = $row;
                $category = $row['category_name'] ?? 'Uncategorized';
                if (!isset($resources_by_category[$category])) {
                    $resources_by_category[$category] = [];
                }
                $resources_by_category[$category][] = $row;
            }
        }
        $stmt->close();
    }
}
?>

<?php include_once 'components/header.php'; ?>

<style>
    /* Scope styles to library page only */
    .library-page-wrapper {
        background-color: #f5f5f5;
        font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
        min-height: 100vh;
        padding-top: 2rem;
    }
    
    .library-container {
        max-width: 1200px;
        width: 90%;
        margin: 0 auto;
        padding: 2rem;
        box-sizing: border-box;
    }
    
    .library-header {
        text-align: center;
        margin-bottom: 2rem;
        padding: 2rem 0;
    }
    
    .stats-container {
        width: auto;
        margin: 2rem 0;
        display: flex;
        gap: 1rem;
        flex-wrap: wrap;
        justify-content: flex-start;
    }
    
    .stat-card {
        background: #f5f5f5;
        border-radius: 6px;
        padding: 0.75rem;
        text-align: center;
        box-shadow: 0 1px 2px rgba(0,0,0,0.1);
        border: 1px solid #e2e8f0;
        transition: transform 0.2s ease;
        min-width: 120px;
    }
    
    .stat-card:hover {
        transform: translateY(-1px);
    }
    
    .stat-icon {
        width: 32px;
        height: 32px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 0.4rem auto;
        font-size: 1rem;
        color: white;
    }
    
    .stat-icon.red { background: #dc2626; }
    .stat-icon.amber { background: #d97706; }
    .stat-icon.blue { background: #2563eb; }
    
    .stat-number {
        font-size: 1.3rem;
        font-weight: 600;
        color: #1e293b;
        margin-bottom: 0.2rem;
    }
    
    .stat-label {
        font-size: 0.7rem;
        color: #64748b;
        text-transform: uppercase;
        font-weight: 500;
    }
    
    .library-title {
        font-size: 2.5rem;
        font-weight: 700;
        color: #1e293b;
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 1rem;
    }
    
    .library-title i {
        color: #3b82f6;
        font-size: 2.2rem;
    }
    
    .library-subtitle {
        color: #64748b;
        font-size: 1.1rem;
        font-weight: 400;
    }
    
    /* === LIBRARY SEARCH STYLES === */
    .library-search-section {
        background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
        border-radius: 20px;
        border: 1px solid #e2e8f0;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08);
        padding: 3rem 2rem;
        margin: 3rem 0;
        text-align: center;
    }
    
    .search-header {
        margin-bottom: 2rem;
    }
    
    .search-title {
        font-size: 2rem;
        font-weight: 700;
        color: #1e293b;
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.75rem;
    }
    
    .search-title::before {
        content: "🔍";
        font-size: 1.8rem;
    }
    
    .search-subtitle {
        color: #64748b;
        font-size: 1.1rem;
        font-weight: 400;
        margin: 0;
        opacity: 0.8;
    }
    
    .search-container {
        margin-bottom: 1.5rem;
        display: flex;
        justify-content: center;
    }
    
    .search-box {
        position: relative;
        background: #ffffff;
        border-radius: 50px;
        border: 3px solid #e2e8f0;
        transition: all 0.3s ease;
        overflow: hidden;
        width: 100%;
        max-width: 800px;
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
    }
    
    .search-box:focus-within {
        border-color: #3b82f6;
        background: #ffffff;
        box-shadow: 0 0 0 5px rgba(59, 130, 246, 0.15), 0 8px 25px rgba(59, 130, 246, 0.1);
        transform: translateY(-2px);
    }
    
    .search-box i {
        position: absolute;
        left: 25px;
        top: 50%;
        transform: translateY(-50%);
        color: #6b7280;
        font-size: 1.4rem;
        z-index: 2;
        transition: color 0.3s ease;
    }
    
    .search-box:focus-within i {
        color: #3b82f6;
    }
    
    .search-box input {
        width: 100%;
        padding: 20px 70px 20px 65px;
        border: none;
        background: transparent;
        font-size: 1.1rem;
        color: #1f2937;
        outline: none;
        box-sizing: border-box;
        font-weight: 400;
    }
    
    .search-box input::placeholder {
        color: #9ca3af;
        font-weight: 400;
    }
    
    .clear-btn {
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        background: #ef4444;
        color: white;
        border: none;
        border-radius: 50%;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: all 0.3s ease;
        z-index: 2;
        font-size: 16px;
        box-shadow: 0 2px 8px rgba(239, 68, 68, 0.3);
    }
    
    .clear-btn:hover {
        background: #dc2626;
        transform: translateY(-50%) scale(1.1);
        box-shadow: 0 4px 12px rgba(220, 38, 38, 0.4);
    }
    
    .filter-container {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1rem;
        margin-bottom: 1rem;
    }
    
    .filter-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .filter-group label {
        font-size: 0.9rem;
        font-weight: 600;
        color: #374151;
    }
    
    .filter-select {
        padding: 10px 15px;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        background: white;
        color: #1f2937;
        font-size: 0.9rem;
        transition: all 0.3s ease;
        cursor: pointer;
    }
    
    .filter-select:focus {
        outline: none;
        border-color: #3b82f6;
        box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
    }
    
    .filter-select:hover {
        border-color: #cbd5e1;
    }
    
    .search-results-info {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 1.5rem;
        padding: 1.25rem 2rem;
        background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
        border-radius: 50px;
        border: 2px solid #cbd5e1;
        max-width: 500px;
        margin: 0 auto;
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
    }
    
    .results-count {
        font-weight: 600;
        color: #1e293b;
        font-size: 1rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }
    
    .results-count::before {
        content: "📊";
        font-size: 1rem;
    }
    
    .reset-btn {
        background: linear-gradient(135deg, #64748b, #475569);
        color: white;
        border: none;
        padding: 10px 20px;
        border-radius: 25px;
        font-size: 0.9rem;
        font-weight: 500;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        transition: all 0.3s ease;
        box-shadow: 0 2px 8px rgba(100, 116, 139, 0.3);
    }
    
    .reset-btn:hover {
        background: linear-gradient(135deg, #475569, #334155);
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(71, 85, 105, 0.4);
    }
    
    /* Filter Animation */
    .resource-card.filtered-out {
        display: none !important;
    }
    
    .category-section.filtered-out {
        display: none !important;
    }
    
    /* Responsive Library Search */
    @media (max-width: 768px) {
        .library-search-section {
            padding: 1.5rem;
        }
        
        .filter-container {
            grid-template-columns: 1fr;
            gap: 1rem;
        }
        
        .search-results-info {
            flex-direction: column;
            gap: 0.75rem;
            align-items: stretch;
        }
        
        .search-box input {
            padding: 14px 50px 14px 45px;
            font-size: 0.9rem;
        }
    }
    
    .category-section {
        margin-bottom: 3rem;
    }
    
    .category-header {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        margin-bottom: 1.5rem;
        padding-bottom: 0.75rem;
        border-bottom: 2px solid #e2e8f0;
    }
    
    .category-icon {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 1.2rem;
    }
    
    .category-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: #1e293b;
        margin: 0;
    }
    
    .category-count {
        background: #e2e8f0;
        color: #64748b;
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.9rem;
        font-weight: 500;
        margin-left: auto;
    }
    
    .resources-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .resource-card {
        background: white;
        border-radius: 8px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.12);
        transition: all 0.2s ease;
        border: 1px solid #e2e8f0;
        overflow: hidden;
    }
    
    .resource-card:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    
    .resource-header {
        padding: 0.875rem;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 0.75rem;
        transition: background-color 0.2s ease;
    }
    
    .resource-header:hover {
        background-color: #f8fafc;
    }
    
    .resource-thumbnail {
        width: 40px;
        height: 40px;
        border-radius: 6px;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
    }
    
    .file-icon {
        font-size: 1.5rem;
        color: white;
    }
    
    .resource-info {
        flex: 1;
        min-width: 0;
    }
    
    .resource-title {
        font-size: 0.9rem;
        font-weight: 600;
        color: #1e293b;
        margin: 0 0 0.25rem 0;
        line-height: 1.3;
    }
    
    .resource-meta {
        display: flex;
        gap: 0.75rem;
        align-items: center;
        color: #64748b;
        font-size: 0.75rem;
    }
    
    .meta-item {
        display: flex;
        align-items: center;
        gap: 0.2rem;
    }
    
    .resource-details {
        padding: 0 0.875rem 0.875rem 0.875rem;
        border-top: 1px solid #f1f5f9;
        background: #f8fafc;
        display: none;
    }
    
    .resource-description {
        color: #475569;
        font-size: 0.85rem;
        line-height: 1.4;
        margin-bottom: 0.75rem;
        padding-top: 0.75rem;
    }
    
    .resource-actions {
        display: flex;
        gap: 0.75rem;
        align-items: center;
    }
    
    .btn-download {
        background: linear-gradient(135deg, #3b82f6, #1d4ed8);
        color: white;
        border: none;
        padding: 0.5rem 1rem;
        border-radius: 8px;
        font-size: 0.9rem;
        font-weight: 500;
        transition: all 0.2s ease;
        text-decoration: none;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }
    
    .btn-download:hover {
        background: linear-gradient(135deg, #1d4ed8, #1e40af);
        color: white;
        transform: translateY(-1px);
    }
    
    .teacher-info {
        color: #64748b;
        font-size: 0.85rem;
        display: flex;
        align-items: center;
        gap: 0.25rem;
    }
    
    .expand-icon {
        color: #64748b;
        font-size: 1.2rem;
        transition: transform 0.2s ease;
        margin-left: auto;
    }
    
    .resource-card[data-expanded="true"] .expand-icon {
        transform: rotate(180deg);
    }
    
    .resource-card[data-expanded="true"] .resource-details {
        display: block;
    }
    
    /* File type colors - Blue, Orange, Amber */
    .file-pdf { background: #3b82f6; }
    .file-doc { background: #f59e0b; }
    .file-docx { background: #d97706; }
    .file-ppt { background: #f59e0b; }
    .file-pptx { background: #d97706; }
    .file-xls { background: #3b82f6; }
    .file-xlsx { background: #f59e0b; }
    .file-txt { background: #6b7280; }
    .file-default { background: #6b7280; }
    
    .no-resources {
        text-align: center;
        padding: 3rem;
        color: #64748b;
    }
    
    .no-resources i {
        font-size: 4rem;
        color: #cbd5e1;
        margin-bottom: 1rem;
    }
    
    /* Responsive Design */
    @media (max-width: 1400px) {
        .library-container {
            width: 95%;
            padding: 1.5rem;
        }
    }
    
    @media (max-width: 768px) {
        .library-container {
            width: 100%;
            padding: 1rem;
        }
        
        .library-title {
            font-size: 2rem;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .stats-container {
            flex-wrap: wrap;
            gap: 0.75rem;
        }
        
        .stat-card {
            padding: 0.6rem;
            min-width: 100px;
        }
        
        .stat-icon {
            width: 28px;
            height: 28px;
            font-size: 0.9rem;
        }
        
        .stat-number {
            font-size: 1.1rem;
        }
        
        .stat-label {
            font-size: 0.6rem;
        }
        
        .resources-grid {
            grid-template-columns: 1fr;
            gap: 0.75rem;
        }
        
        .resource-actions {
            flex-direction: column;
            align-items: stretch;
            gap: 0.5rem;
        }
        
        .resource-meta {
            flex-wrap: wrap;
            gap: 0.5rem;
        }
        
        .category-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 0.5rem;
        }
        
        .category-count {
            margin-left: 0;
        }
    }
</style>

<div class="library-page-wrapper">
<div class="library-container">
    <!-- Library Header -->
    <div class="library-header">
        <h1 class="library-title">
            <i class="bx bx-library"></i>
            Digital Library
        </h1>
        <p class="library-subtitle">Access educational resources and materials</p>
    </div>
    
    <!-- Stats Cards -->
    <div class="stats-container">
        <div class="stat-card">
            <div class="stat-icon red">
                <i class="bx bx-file"></i>
            </div>
            <div class="stat-number"><?= count($resources) ?></div>
            <div class="stat-label">Resources</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon amber">
                <i class="bx bx-category"></i>
            </div>
            <div class="stat-number"><?= count($resources_by_category) ?></div>
            <div class="stat-label">Categories</div>
        </div>
    </div>
    
    <!-- Search Section -->
    <div class="library-search-section">
        <div class="search-header">
            <h2 class="search-title">Find Your Resources</h2>
            <p class="search-subtitle">Searching through the library collection</p>
        </div>
        
        <div class="search-container">
            <div class="search-box">
                <i class='bx bx-search'></i>
                <input type="text" id="librarySearch" placeholder="Search books, documents, resources, and materials...">
                <button type="button" id="clearLibrarySearch" class="clear-btn" style="display: none;">
                    <i class='bx bx-x'></i>
                </button>
            </div>
        </div>

        <div class="search-results-info" id="searchResults" style="display: none;">
            <span class="results-count">0 resources found</span>
            <button type="button" id="resetFilters" class="reset-btn">
                <i class='bx bx-refresh'></i>
                Clear Search
            </button>
        </div>
    </div>
    
    <!-- Library Content -->
    <?php if (empty($resources_by_category)): ?>
        <div class="no-resources">
            <i class="bx bx-book-open"></i>
            <h3>No Resources Available</h3>
            <p>Resources will appear here once teachers upload and administrators approve them.</p>
        </div>
    <?php else: ?>
        <?php foreach ($resources_by_category as $category => $category_resources): ?>
            <div class="category-section">
                <div class="category-header">
                    <div class="category-icon" style="background-color: <?= $category_config[$category]['color'] ?? '#6b7280' ?>">
                        <i class="bx <?= $category_config[$category]['icon'] ?? 'bx-file' ?>"></i>
                    </div>
                    <h2 class="category-title"><?= htmlspecialchars($category) ?></h2>
                    <span class="category-count"><?= count($category_resources) ?> resources</span>
                </div>
                
                <div class="resources-grid">
                    <?php foreach ($category_resources as $resource): ?>
                        <div class="resource-card" data-expanded="false">
                            <div class="resource-header" onclick="toggleResource(this)">
                                <div class="resource-thumbnail">
                                    <?php if (!empty($resource['thumbnail_path'])): ?>
                                        <img src="../<?= $resource['thumbnail_path'] ?>" 
                                             alt="Thumbnail" 
                                             style="width: 50px; height: 50px; object-fit: cover; border-radius: 8px;">
                                    <?php else: ?>
                                        <div class="file-icon file-<?= strtolower($resource['file_type'] ?? 'default') ?>">
                                            <i class="bx bx-file"></i>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="resource-info">
                                    <h3 class="resource-title"><?= htmlspecialchars($resource['title']) ?></h3>
                                    <div class="resource-meta">
                                        <span class="meta-item">
                                            <i class="bx bx-file"></i>
                                            <?= strtoupper($resource['file_type'] ?? 'FILE') ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <i class="bx bx-chevron-down expand-icon"></i>
                            </div>
                            
                            <div class="resource-details">
                                <p class="resource-description">
                                    <?= htmlspecialchars($resource['description'] ?? 'No description available.') ?>
                                </p>
                                
                                <div class="resource-actions">
                                    <?php if (!empty($resource['file_path'])): ?>
                                        <a href="../<?= $resource['file_path'] ?>" 
                                           class="btn-download" 
                                           target="_blank">
                                            <i class="bx bx-download"></i>
                                            Download
                                        </a>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($resource['teacher_name'])): ?>
                                        <span class="teacher-info">
                                            <i class="bx bx-user"></i>
                                            By <?= htmlspecialchars($resource['teacher_name']) ?>
                                        </span>
                                    <?php endif; ?>
                                    
                                    <span class="teacher-info">
                                        <i class="bx bx-calendar"></i>
                                        <?= isset($resource['publication_date']) ? date('M j, Y', strtotime($resource['publication_date'])) : 'N/A' ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>
</div>

<script>
function toggleResource(header) {
    const card = header.closest('.resource-card');
    const isExpanded = card.getAttribute('data-expanded') === 'true';
    
    // Close all other cards
    document.querySelectorAll('.resource-card').forEach(c => {
        if (c !== card) {
            c.setAttribute('data-expanded', 'false');
        }
    });
    
    // Toggle current card
    card.setAttribute('data-expanded', !isExpanded);
}

// Library Search Functionality
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('librarySearch');
    const clearBtn = document.getElementById('clearLibrarySearch');
    const resetBtn = document.getElementById('resetFilters');
    const searchResults = document.getElementById('searchResults');
    const resultsCount = document.querySelector('.results-count');
    
    let allResources = [];
    let filteredResources = [];
    
    // Initialize resource data
    function initializeResources() {
        allResources = [];
        const resourceCards = document.querySelectorAll('.resource-card');
        
        console.log('Found resource cards:', resourceCards.length); // Debug log
        
        resourceCards.forEach((card, index) => {
            const titleElement = card.querySelector('.resource-title');
            const descriptionElement = card.querySelector('.resource-description');
            const categorySection = card.closest('.category-section');
            const categoryElement = categorySection?.querySelector('.category-title');
            
            const title = titleElement?.textContent?.trim().toLowerCase() || '';
            const description = descriptionElement?.textContent?.trim().toLowerCase() || '';
            const category = categoryElement?.textContent?.trim().toLowerCase() || '';
            const fileType = getFileTypeFromCard(card);
            
            console.log(`Resource ${index + 1}:`, { title, description, category, fileType }); // Debug log
            
            allResources.push({
                element: card,
                categorySection: categorySection,
                title: title,
                description: description,
                category: category,
                fileType: fileType,
                searchText: `${title} ${description} ${category}`.toLowerCase()
            });
        });
        
        filteredResources = [...allResources];
        console.log('Initialized resources:', allResources.length); // Debug log
    }
    
    // Get file type from card
    function getFileTypeFromCard(card) {
        const resourceHeader = card.querySelector('.resource-header');
        if (!resourceHeader) return 'other';
        
        const title = card.querySelector('.resource-title')?.textContent?.toLowerCase() || '';
        
        if (title.includes('.pdf') || resourceHeader.innerHTML.includes('pdf')) return 'pdf';
        if (title.includes('.doc') || title.includes('.docx') || resourceHeader.innerHTML.includes('doc')) return 'doc';
        if (title.includes('.ppt') || title.includes('.pptx') || resourceHeader.innerHTML.includes('ppt')) return 'ppt';
        if (title.includes('.jpg') || title.includes('.png') || title.includes('.gif') || resourceHeader.innerHTML.includes('image')) return 'image';
        if (title.includes('.mp4') || title.includes('.avi') || resourceHeader.innerHTML.includes('video')) return 'video';
        
        return 'other';
    }
    
    // Apply search filter
    function applySearch() {
        const searchTerm = searchInput?.value.toLowerCase().trim() || '';
        
        console.log('Applying search:', { searchTerm }); // Debug log
        
        filteredResources = allResources.filter(resource => {
            const matchesSearch = !searchTerm || resource.searchText.includes(searchTerm);
            
            console.log(`Resource "${resource.title}" - Show: ${matchesSearch}`); // Debug log
            
            return matchesSearch;
        });
        
        console.log('Filtered resources:', filteredResources.length); // Debug log
        
        // Update display
        updateDisplay();
        
        // Update search results info
        updateSearchResults();
    }
    

    
    // Update display
    function updateDisplay() {
        console.log('Updating display...'); // Debug log
        
        // Hide all resources first
        allResources.forEach(resource => {
            resource.element.style.display = 'none';
            resource.element.classList.add('filtered-out');
        });
        
        // Show filtered resources
        const visibleCategories = new Set();
        filteredResources.forEach(resource => {
            resource.element.style.display = 'block';
            resource.element.classList.remove('filtered-out');
            visibleCategories.add(resource.categorySection);
        });
        
        // Show/hide category sections
        const allCategorySections = document.querySelectorAll('.category-section');
        allCategorySections.forEach(section => {
            if (visibleCategories.has(section)) {
                section.style.display = 'block';
                section.classList.remove('filtered-out');
            } else {
                section.style.display = 'none';
                section.classList.add('filtered-out');
            }
        });
        
        console.log(`Display updated - Showing ${filteredResources.length} resources in ${visibleCategories.size} categories`); // Debug log
    }
    
    // Update search results info
    function updateSearchResults() {
        const hasSearch = searchInput?.value.trim();
        
        if (hasSearch) {
            resultsCount.textContent = `${filteredResources.length} resources found`;
            searchResults.style.display = 'flex';
        } else {
            searchResults.style.display = 'none';
        }
        
        // Show/hide clear button
        if (clearBtn) {
            clearBtn.style.display = hasSearch ? 'flex' : 'none';
        }
    }
    
    // Reset search
    function resetSearch() {
        if (searchInput) {
            searchInput.value = '';
        }
        
        filteredResources = [...allResources];
        updateDisplay();
        updateSearchResults();
    }
    
    // Event listeners
    if (searchInput) {
        console.log('Search input found, attaching listeners'); // Debug log
        searchInput.addEventListener('input', function() {
            console.log('Search input changed:', this.value); // Debug log
            applySearch();
        });
        searchInput.addEventListener('input', () => {
            if (clearBtn) {
                clearBtn.style.display = searchInput.value.trim() ? 'flex' : 'none';
            }
        });
    } else {
        console.log('Search input not found!'); // Debug log
    }
    
    if (clearBtn) {
        clearBtn.addEventListener('click', () => {
            searchInput.value = '';
            applySearch();
        });
    }
    
    if (resetBtn) {
        resetBtn.addEventListener('click', resetSearch);
    }
    
    // Initialize immediately and also with delay as fallback
    initializeResources();
    setTimeout(() => {
        console.log('Re-initializing after delay...');
        initializeResources();
    }, 500); // Longer delay to ensure everything is loaded
});
</script>


