/**
 * 🎬 MASTER VIDEO PROGRESS SYSTEM - CLEAN & POWERFUL
 * 
 * Handles ALL video types dynamically with proper progress tracking
 * - YouTube videos (iframe API)
 * - File videos (HTML5 API) 
 * - Dynamic badge updates
 * - Auto-play next video
 * - Database progress sync
 * 
 * @author Cascade AI
 * @version 2.0 - Complete Rewrite
 */

class MasterVideoSystem {
    constructor() {
        // SINGLETON PROTECTION: Prevent multiple initializations
        if (window.masterVideoSystemInitialized) {
            console.log(' MASTER VIDEO SYSTEM: Already initialized, returning existing instance...');
            return window.masterVideoSystemInstance;
        }
        
        console.log(' MASTER VIDEO SYSTEM: Initializing...');
        window.masterVideoSystemInitialized = true;
        window.masterVideoSystemInstance = this;
        
        // Initialize properties
        this.currentVideo = null;
        this.progressTimer = null;
        this.youtubePlayer = null;
        this.completedLessons = new Set();
        this.isYouTubeAPIReady = false;
        
        // Configuration
        this.config = {
            progressInterval: 2000, // Check progress every 2 seconds
            completionThreshold: 90, // 90% watched = completed
            autoplayNext: true,
            debugMode: true
        };
        
        console.log(' MASTER VIDEO SYSTEM: Initializing...');
        // Load YouTube API first
        this.loadYouTubeAPI();
        
        // Load completed lessons from database
        this.loadCompletedLessons();
        
        // Set up event listeners for watch buttons
        this.setupEventListeners();
        
        // Initialize badges
        this.initializeBadges();
        
        // Initialize tab system
        this.initializeTabs();
        
        // Initialize Bootstrap components
        this.initializeBootstrap();
        
        console.log('✅ MASTER VIDEO SYSTEM: Fully initialized with accordion support');
    }
    
    /**
     * 📺 LOAD YOUTUBE IFRAME API
     */
    loadYouTubeAPI() {
        if (window.YT) {
            this.isYouTubeAPIReady = true;
            console.log('✅ YOUTUBE API: Already loaded');
            return;
        }
        
        // Load YouTube iframe API
        const tag = document.createElement('script');
        tag.src = 'https://www.youtube.com/iframe_api';
        const firstScriptTag = document.getElementsByTagName('script')[0];
        firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
        
        // Global callback for when API is ready
        window.onYouTubeIframeAPIReady = () => {
            this.isYouTubeAPIReady = true;
            console.log('✅ YOUTUBE API: Loaded and ready');
        };
    }
    
    /**
     * 💾 LOAD COMPLETED LESSONS FROM DATABASE
     */
    async loadCompletedLessons() {
        try {
            const response = await fetch('get_completed_lessons.php');
            const data = await response.json();
            
            if (data.success) {
                this.completedLessons = new Set(data.completed_lessons.map(id => id.toString()));
                console.log('✅ LOADED COMPLETED LESSONS:', Array.from(this.completedLessons));
            }
        } catch (error) {
            console.error('❌ Failed to load completed lessons:', error);
        }
    }
    
    /**
     * 🎧 SETUP EVENT LISTENERS
     */
    setupEventListeners() {
        // Watch button clicks - Fixed to work with child elements and prevent double handling
        document.addEventListener('click', (e) => {
            // Use closest() instead of matches() to handle clicks on child elements
            const button = e.target.closest('.watch-lesson-btn, .lesson-title');
            if (button) {
                e.preventDefault();
                e.stopPropagation();
                console.log('🖱️ BUTTON CLICK: Detected click on watch button');
                const lessonId = this.extractLessonId(button);
                if (lessonId) {
                    console.log(`🎯 BUTTON CLICK: Loading lesson ${lessonId}`);
                    this.loadLesson(lessonId);
                } else {
                    console.warn('❌ BUTTON CLICK: Could not extract lesson ID from button');
                }
                return; // Exit early to prevent double handling
            }
            
            // Only handle data attributes if it's NOT a watch button (to prevent double clicks)
            if (e.target.hasAttribute('data-lesson-id') || e.target.closest('[data-lesson-id]')) {
                const element = e.target.hasAttribute('data-lesson-id') ? e.target : e.target.closest('[data-lesson-id]');
                const lessonId = element.getAttribute('data-lesson-id');
                
                // Only process if it's NOT a watch button and NOT an accordion button
                if (lessonId && 
                    !element.classList.contains('accordion-button') &&
                    !element.classList.contains('watch-lesson-btn') &&
                    !element.closest('.watch-lesson-btn')) {
                    e.preventDefault();
                    console.log(`🎯 DATA ATTRIBUTE CLICK: Loading lesson ${lessonId} from data attribute`);
                    this.loadLesson(lessonId);
                }
            }
        });
        
        console.log('🎧 EVENT LISTENERS: Set up for watch buttons');
    }
    
    /**
     * 🆔 EXTRACT LESSON ID FROM ELEMENT
     */
    extractLessonId(element) {
        return element.dataset.lessonId || 
               element.closest('[data-lesson-id]')?.dataset.lessonId ||
               element.href?.match(/lesson=(\d+)/)?.[1];
    }
    
    /**
     * 🎬 LOAD LESSON DYNAMICALLY (NO PAGE RELOAD)
     */
    async loadLesson(lessonId) {
        console.log(`🔄 LOADING LESSON: ${lessonId} (AJAX - no reload)`);
        
        try {
            // Show loading state
            this.showStatus(`Loading lesson ${lessonId}...`, 'info');
            this.setBadge(lessonId, 'loading');
            
            // Get lesson data via AJAX
            const courseId = this.getCurrentCourseId();
            console.log('🔍 AJAX REQUEST: course_id =', courseId, 'lesson_id =', lessonId);
            
            const response = await fetch('get_lesson_data_clean.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    lesson_id: lessonId,
                    course_id: courseId
                })
            });
            
            console.log('🔍 AJAX RESPONSE: status =', response.status);
            
            let lessonData;
            try {
                lessonData = await response.json();
            } catch (parseError) {
                const responseText = await response.text();
                console.error('❌ JSON PARSE ERROR:', parseError);
                console.error('❌ RESPONSE TEXT:', responseText.substring(0, 500));
                throw new Error(`JSON Parse Error: ${parseError.message}. Response: ${responseText.substring(0, 200)}`);
            }
            
            console.log('🔍 AJAX RESPONSE DATA:', lessonData);
            
            if (lessonData.success) {
                // Update video player dynamically
                await this.updateVideoPlayer(lessonData.lesson);
                
                // Update URL without reload
                this.updateURL(lessonId);
                
                // Set watching badge
                this.setBadge(lessonId, 'watching');
                
                console.log('✅ LESSON LOADED SUCCESSFULLY:', lessonData.lesson);
            } else {
                // Backend returned detailed error - show it prominently
                console.error('❌ BACKEND ERROR RESPONSE:', lessonData);
                this.displayBackendError(lessonData);
                throw new Error(lessonData.message || 'Failed to load lesson');
            }
            
        } catch (error) {
            console.error('❌ LESSON LOAD ERROR:', error);
            
            // Show detailed error on page for debugging
            let errorMessage = 'Failed to load lesson';
            let debugInfo = '';
            
            if (error.message) {
                errorMessage = error.message;
            }
            
            // Try to get more detailed error from response
            if (typeof error === 'object' && error.response) {
                try {
                    const errorData = typeof error.response === 'string' ? 
                        JSON.parse(error.response) : error.response;
                    
                    if (errorData.message) errorMessage = errorData.message;
                    if (errorData.debug_data) {
                        debugInfo = `
                            <div style="margin-top: 10px; padding: 10px; background: #f8f9fa; border-radius: 5px; font-size: 12px;">
                                <strong>🔍 Debug Info:</strong><br>
                                Course ID (POST): ${errorData.debug_data.post_course_id}<br>
                                Lesson ID (POST): ${errorData.debug_data.post_lesson_id}<br>
                                Database Connected: ${errorData.debug_data.database_connected}<br>
                                Database Error: ${errorData.database_error || 'None'}<br>
                                User ID: ${errorData.debug_data.user_id}
                            </div>
                        `;
                    }
                } catch (parseError) {
                    console.error('Error parsing response:', parseError);
                }
            }
            
            // Display the error to the user
            this.displayBackendError({
                success: false,
                message: errorMessage,
                debug_info: debugInfo
            });
        }
    }
    
    /**
     * 🔗 UPDATE BROWSER URL WITH CURRENT LESSON
     */
    updateURL(lessonId) {
        try {
            // Get current URL parameters
            const url = new URL(window.location);
            url.searchParams.set('lesson', lessonId);
            
            // Update URL without page reload
            window.history.pushState({ lessonId }, '', url);
            
            console.log(`🔗 URL UPDATED: Lesson ${lessonId}`);
        } catch (error) {
            console.error('❌ URL UPDATE ERROR:', error);
        }
    }
    
    /**
     * 📺 UPDATE VIDEO PLAYER WITH NEW LESSON
     */
    async updateVideoPlayer(lesson) {
        const videoContainer = document.getElementById('videoPlayer');
        if (!videoContainer) {
            console.error('❌ Video container not found');
            return;
        }
        
        // Stop current video tracking
        this.stopCurrentVideo();
        
        // Store current lesson data
        this.currentVideo = {
            lessonId: lesson.lesson_id || lesson.video_id,
            courseId: lesson.course_id,
            title: lesson.title,
            videoUrl: lesson.video_url,
            videoType: lesson.video_type,
            platform: this.detectVideoPlatform(lesson)
        };
        
        console.log('🎬 UPDATING VIDEO PLAYER:', this.currentVideo);
        
        // Create appropriate player based on platform
        if (this.currentVideo.platform === 'youtube') {
            await this.createYouTubePlayer(videoContainer);
        } else if (this.currentVideo.platform === 'file') {
            await this.createFilePlayer(videoContainer);
        } else {
            console.error('❌ UNSUPPORTED VIDEO PLATFORM:', this.currentVideo.platform);
        }
    }
    
    /**
     * 🔍 DETECT VIDEO PLATFORM
     */
    detectVideoPlatform(lesson) {
        if (!lesson.video_url) return 'none';
        
        const url = lesson.video_url.toLowerCase();
        
        if (url.includes('youtube.com') || url.includes('youtu.be')) {
            return 'youtube';
        } else if (url.includes('vimeo.com')) {
            return 'vimeo';
        } else if (lesson.video_type === 'file' || url.includes('/uploads/')) {
            return 'file';
        }
        
        return 'external';
    }
    
    /**
     * 📺 CREATE YOUTUBE PLAYER WITH API
     */
    async createYouTubePlayer(container) {
        console.log('🎬 CREATING YOUTUBE PLAYER...');
        
        // Wait for YouTube API to be ready
        if (!this.isYouTubeAPIReady) {
            console.log('⏳ Waiting for YouTube API...');
            await this.waitForYouTubeAPI();
        }
        
        // Extract video ID from URL
        const videoId = this.extractYouTubeVideoId(this.currentVideo.videoUrl);
        if (!videoId) {
            console.error('❌ Could not extract YouTube video ID');
            return;
        }
        
        // 🛑 CLEAR CONTAINER COMPLETELY to prevent duplication
        container.innerHTML = '';
        
        // Create fresh player container with unique ID
        const playerId = `youtube-player-${this.currentVideo.lessonId}-${Date.now()}`;
        container.innerHTML = `<div id="${playerId}"></div>`;
        
        // Initialize YouTube player with API
        this.youtubePlayer = new YT.Player(playerId, {
            width: '100%',
            height: '400',
            videoId: videoId,
            playerVars: {
                autoplay: 1,
                controls: 1,
                rel: 0,
                showinfo: 0,
                modestbranding: 1
            },
            events: {
                onReady: (event) => {
                    console.log('✅ YOUTUBE PLAYER: Ready');
                    this.startProgressTracking();
                },
                onStateChange: (event) => {
                    this.handleYouTubeStateChange(event);
                },
                onError: (event) => {
                    console.error('❌ YOUTUBE PLAYER ERROR:', event.data);
                }
            }
        });
    }
    
    /**
     * 🎵 CREATE FILE VIDEO PLAYER
     */
    async createFilePlayer(container) {
        console.log('🎬 CREATING FILE VIDEO PLAYER...');
        
        // First, check if video file exists to provide better error handling
        const videoUrl = this.currentVideo.videoUrl;
        console.log(`🔍 FILE VIDEO: Checking if video exists: ${videoUrl}`);
        
        container.innerHTML = `
            <div class="video-container position-relative">
                <video id="file-player-${this.currentVideo.lessonId}" 
                       width="100%" 
                       height="400" 
                       controls 
                       autoplay
                       preload="metadata">
                    <source src="${videoUrl}" type="video/mp4">
                    <source src="${videoUrl}" type="video/webm">
                    Your browser does not support the video tag.
                </video>
                <div id="video-error-${this.currentVideo.lessonId}" class="video-error-overlay d-none">
                    <div class="error-content text-center p-4">
                        <i class="bx bx-error-circle fs-1 text-danger"></i>
                        <h5 class="mt-3 mb-2">Video Not Available</h5>
                        <p class="text-muted mb-3">The video file could not be loaded. This may be due to:</p>
                        <ul class="list-unstyled text-start">
                            <li>• File not uploaded yet</li>
                            <li>• File moved or renamed</li>
                            <li>• Server connectivity issues</li>
                        </ul>
                        <small class="text-muted">Please contact your instructor if this problem persists.</small>
                    </div>
                </div>
            </div>
        `;
        
        const videoElement = container.querySelector('video');
        const errorOverlay = container.querySelector(`#video-error-${this.currentVideo.lessonId}`);
        
        // Enhanced error handling for 404 and other video errors
        const handleVideoError = (errorType, errorMessage) => {
            console.error(`❌ FILE VIDEO ${errorType}:`, errorMessage);
            
            // Hide video and show error overlay
            videoElement.style.display = 'none';
            errorOverlay.classList.remove('d-none');
            
            // Update badge to show error
            this.setBadge(this.currentVideo.lessonId, 'error');
        };
        
        // Add comprehensive event listeners for file video
        videoElement.addEventListener('loadedmetadata', () => {
            console.log('✅ FILE VIDEO: Metadata loaded successfully');
            errorOverlay.classList.add('d-none');
            videoElement.style.display = 'block';
            this.startProgressTracking();
        });
        
        videoElement.addEventListener('timeupdate', () => {
            this.handleFileVideoProgress();
        });
        
        videoElement.addEventListener('ended', () => {
            console.log('🏁 FILE VIDEO: Playback ended');
            this.handleVideoCompleted();
        });
        
        videoElement.addEventListener('error', (e) => {
            const error = e.target.error;
            if (error) {
                switch (error.code) {
                    case error.MEDIA_ERR_NETWORK:
                        handleVideoError('NETWORK ERROR', 'Network error while loading video (likely 404)');
                        break;
                    case error.MEDIA_ERR_DECODE:
                        handleVideoError('DECODE ERROR', 'Video format not supported or corrupted');
                        break;
                    case error.MEDIA_ERR_SRC_NOT_SUPPORTED:
                        handleVideoError('FORMAT ERROR', 'Video format not supported by browser');
                        break;
                    default:
                        handleVideoError('UNKNOWN ERROR', 'Unknown video playback error');
                }
            } else {
                handleVideoError('LOADING ERROR', 'Failed to load video file');
            }
        });
        
        // Set a timeout to detect if video never loads (additional 404 detection)
        setTimeout(() => {
            if (videoElement.readyState === 0 && videoElement.networkState === 3) {
                handleVideoError('TIMEOUT ERROR', 'Video failed to load within timeout period (likely 404)');
            }
        }, 5000); // 5 second timeout
    }
    
    /**
     * ⏳ WAIT FOR YOUTUBE API TO BE READY
     */
    waitForYouTubeAPI() {
        return new Promise((resolve) => {
            const checkAPI = () => {
                if (this.isYouTubeAPIReady) {
                    resolve();
                } else {
                    setTimeout(checkAPI, 100);
                }
            };
            checkAPI();
        });
    }
    
    /**
     * 🆔 EXTRACT YOUTUBE VIDEO ID FROM URL
     */
    extractYouTubeVideoId(url) {
        const match = url.match(/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/);
        return match ? match[1] : null;
    }
    
    /**
     * 🎬 HANDLE YOUTUBE STATE CHANGES
     */
    handleYouTubeStateChange(event) {
        const state = event.data;
        
        switch (state) {
            case YT.PlayerState.PLAYING:
                console.log('▶️ YOUTUBE: Playing');
                this.startProgressTracking();
                break;
            case YT.PlayerState.PAUSED:
                console.log('⏸️ YOUTUBE: Paused');
                break;
            case YT.PlayerState.ENDED:
                console.log('🏁 YOUTUBE: Ended');
                this.handleVideoCompleted();
                break;
        }
    }
    
    /**
     * 📊 START PROGRESS TRACKING
     */
    startProgressTracking() {
        if (this.progressTimer) {
            clearInterval(this.progressTimer);
        }
        
        console.log('📊 STARTING PROGRESS TRACKING...');
        
        this.progressTimer = setInterval(() => {
            this.checkProgress();
        }, this.config.progressInterval);
    }
    
    /**
     * 📈 CHECK VIDEO PROGRESS
     */
    checkProgress() {
        if (!this.currentVideo) return;
        
        let currentTime = 0;
        let duration = 0;
        let percentage = 0;
        
        if (this.currentVideo.platform === 'youtube' && this.youtubePlayer) {
            try {
                currentTime = this.youtubePlayer.getCurrentTime();
                duration = this.youtubePlayer.getDuration();
                percentage = duration > 0 ? (currentTime / duration) * 100 : 0;
            } catch (error) {
                console.warn('⚠️ YouTube player not ready for progress check');
                return;
            }
        } else if (this.currentVideo.platform === 'file') {
            const videoElement = document.querySelector(`#file-player-${this.currentVideo.lessonId}`);
            if (videoElement) {
                currentTime = videoElement.currentTime;
                duration = videoElement.duration;
                percentage = duration > 0 ? (currentTime / duration) * 100 : 0;
            }
        }
        
        // Update progress in database every 10 seconds or on completion
        if (percentage > 0) {
            this.updateProgress(percentage, currentTime);
            
            // Check for completion
            if (percentage >= this.config.completionThreshold) {
                this.handleVideoCompleted();
            }
        }
        
        if (this.config.debugMode && percentage > 0) {
            console.log(`📊 PROGRESS: ${percentage.toFixed(1)}% (${currentTime.toFixed(1)}s / ${duration.toFixed(1)}s)`);
        }
    }
    
    /**
     * 📈 HANDLE FILE VIDEO PROGRESS
     */
    handleFileVideoProgress() {
        // This is called frequently by timeupdate event
        // Progress checking is handled by the main checkProgress method
    }
    
    /**
     * 💾 UPDATE PROGRESS IN DATABASE
     */
    async updateProgress(percentage, watchTime) {
        if (!this.currentVideo) return;
        
        try {
            const formData = new FormData();
            formData.append('video_id', this.currentVideo.lessonId);
            formData.append('user_id', this.getCurrentUserId());
            formData.append('completion_percentage', percentage.toFixed(2));
            formData.append('watch_time', watchTime.toFixed(2));
            
            await fetch('update_video_progress.php', {
                method: 'POST',
                body: formData
            });
            
        } catch (error) {
            console.error('❌ PROGRESS UPDATE ERROR:', error);
        }
    }
    
    /**
     * 🏆 HANDLE VIDEO COMPLETED
     */
    async handleVideoCompleted() {
        if (!this.currentVideo || this.completedLessons.has(this.currentVideo.lessonId.toString())) {
            return; // Already completed
        }
        
        console.log('🏆 VIDEO COMPLETED:', this.currentVideo.lessonId);
        
        // Stop progress tracking
        this.stopCurrentVideo();
        
        // Mark as completed in database
        await this.markLessonCompleted();
        
        // Update badge
        this.setBadge(this.currentVideo.lessonId, 'completed');
        
        // Add to completed set
        this.completedLessons.add(this.currentVideo.lessonId.toString());
        
        // Show completion notification
        this.showStatus(`🏆 Lesson ${this.currentVideo.lessonId} completed!`, 'success');
        
        // Auto-play next video if enabled
        if (this.config.autoplayNext) {
            setTimeout(() => {
                this.playNextVideo();
            }, 2000);
        }
    }
    
    /**
     * 💾 MARK LESSON AS COMPLETED IN DATABASE
     */
    async markLessonCompleted() {
        try {
            const formData = new FormData();
            formData.append('video_id', this.currentVideo.lessonId);
            formData.append('user_id', this.getCurrentUserId());
            formData.append('completion_percentage', '100.00');
            formData.append('completed_at', new Date().toISOString().slice(0, 19).replace('T', ' '));
            
            const response = await fetch('mark_video_completed.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            console.log('✅ COMPLETION SAVED:', result);
            
        } catch (error) {
            console.error('❌ COMPLETION SAVE ERROR:', error);
        }
    }
    
    /**
     * ⏭️ PLAY NEXT VIDEO AUTOMATICALLY
     */
    playNextVideo() {
        const currentLessonId = parseInt(this.currentVideo.lessonId);
        const nextLessonId = currentLessonId + 1;
        
        // Check if next lesson exists
        const nextLessonElement = document.querySelector(`[data-lesson-id="${nextLessonId}"]`);
        if (nextLessonElement) {
            console.log(`⏭️ AUTO-PLAYING NEXT VIDEO: ${nextLessonId}`);
            this.showStatus(`Loading next lesson: ${nextLessonId}`, 'info');
            this.loadLesson(nextLessonId);
        } else {
            console.log('🎯 COURSE COMPLETED: No more lessons');
            this.showStatus('🎉 Course completed! Well done!', 'success');
        }
    }
    
    /**
     * 🛑 STOP CURRENT VIDEO TRACKING
     */
    stopCurrentVideo() {
        if (this.progressTimer) {
            clearInterval(this.progressTimer);
            this.progressTimer = null;
        }
        
        if (this.youtubePlayer) {
            try {
                this.youtubePlayer.destroy();
            } catch (error) {
                console.warn('⚠️ Error destroying YouTube player:', error);
            }
            this.youtubePlayer = null;
        }
    }
    
    /**
     * 🎖️ SET LESSON BADGE
     */
    setBadge(lessonId, type) {
        // 🎯 Target the specific badge container directly
        let badgeContainer = document.querySelector(`#lesson-badges-${lessonId}`);
        
        if (!badgeContainer) {
            console.warn(`❌ BADGE CONTAINER: Not found for lesson ${lessonId}`);
            // Fallback: try to find any lesson element
            const lessonElement = document.querySelector(`[data-lesson-id="${lessonId}"]`);
            if (lessonElement) {
                console.log(`🎯 BADGE: Found lesson element, creating badge container`);
                // Try to find a good place to add a badge container
                const parentAccordion = lessonElement.closest('.accordion-item');
                if (parentAccordion) {
                    badgeContainer = parentAccordion.querySelector(`[id*="lesson-badges"]`);
                }
            }
        }
        
        if (!badgeContainer) {
            console.error(`BADGE: Cannot find badge container for lesson ${lessonId}`);
            return;
        }
        
        console.log(`BADGE: Found badge container for lesson ${lessonId}`);
    
    // CRITICAL FIX: If setting a "WATCHING" badge, clear ALL watching badges from other lessons first
    if (type === 'watching') {
        console.log(`GLOBAL CLEAR: Removing all "WATCHING" badges from other lessons`);
        const allWatchingBadges = document.querySelectorAll('.lesson-badge');
        let removedCount = 0;
        allWatchingBadges.forEach(badge => {
            if (badge.textContent.toLowerCase().includes('watching')) {
                const badgeLessonId = badge.getAttribute('data-lesson-id');
                // Only remove if it's NOT the current lesson we're setting
                if (badgeLessonId !== lessonId.toString()) {
                    console.log(`GLOBAL CLEAR: Removing "WATCHING" badge from lesson ${badgeLessonId}`);
                    badge.remove();
                    removedCount++;
                }
            }
        });
        console.log(`GLOBAL CLEAR: Removed ${removedCount} watching badges from other lessons`);
    }
    
    // LOCAL CLEAR: Remove ALL badges from current lesson container first
    const existingLessonBadges = badgeContainer.querySelectorAll('.lesson-badge');
    console.log(`CLEARING: Found ${existingLessonBadges.length} existing badges for lesson ${lessonId}`);
        existingLessonBadges.forEach((badge, index) => {
            console.log(`REMOVING: Badge ${index + 1} - "${badge.textContent.trim()}" for lesson ${lessonId}`);
            badge.remove();
        });
        
        // Check if lesson is completed for rewatch scenario
        const isCompleted = this.completedLessons.has(lessonId.toString()) || 
                           document.querySelector(`#headingLesson${lessonId}`)?.querySelector('.completion-check');
        
        // 🔄 REWATCH: Add completed badge first if lesson is completed and we're showing watching
        if (isCompleted && type === 'watching') {
            const hasCompletedBadge = Array.from(badgeContainer.querySelectorAll('.lesson-badge'))
                .some(badge => badge.textContent.toLowerCase().includes('completed'));
            
            if (!hasCompletedBadge) {
                console.log(`🔄 REWATCH: Adding completed badge for lesson ${lessonId} (rewatching)`);
                // Create completed badge first
                this.createBadgeElement(badgeContainer, lessonId, 'completed');
            }
        }
        
        // Create new badge
        let badgeHtml = '';
        let badgeClass = '';
        
        switch (type) {
            case 'loading':
                badgeHtml = '⏳ Loading...';
                badgeClass = 'badge-secondary';
                break;
            case 'watching':
                badgeHtml = '<i class="bx bx-play-circle me-1"></i>Watching';
                badgeClass = 'badge-light badge-watching-animated';
                break;
            case 'completed':
                badgeHtml = '<i class="bx bx-check-circle me-1"></i>Completed';
                badgeClass = 'badge-success badge-completed-styled';
                break;
            case 'error':
                badgeHtml = '<i class="bx bx-error-circle me-1"></i>Error';
                badgeClass = 'badge-danger';
                break;
        }
        
        if (badgeHtml) {
            const badge = document.createElement('span');
            badge.className = `lesson-badge badge ${badgeClass} me-1`;
            badge.innerHTML = badgeHtml;
            badge.setAttribute('data-lesson-id', lessonId);
            
            // Add specific styling for different badge types
            if (type === 'watching') {
                badge.style.cssText = `
                    font-size: 0.6rem;
                    background-color: #e3f2fd !important;
                    color: #1976d2 !important;
                    border: 1px solid #1976d2 !important;
                    font-weight: 500 !important;
                    animation: bounce 1.5s infinite;
                `;
            } else if (type === 'completed') {
                badge.style.cssText = `
                    font-size: 0.6rem;
                    background-color: #e8f5e8 !important;
                    color: #2e7d32 !important;
                    border: 1px solid #4caf50 !important;
                    font-weight: 600 !important;
                `;
            } else {
                badge.style.cssText = 'font-size: 0.6rem;';
            }
                
            // Add keyframes and styles to document head if not already added
            if (!document.querySelector('#badge-styles')) {
                const style = document.createElement('style');
                style.id = 'badge-styles';
                style.textContent = `
                    @keyframes bounce {
                        0%, 20%, 50%, 80%, 100% {
                            transform: translateY(0);
                        }
                        40% {
                            transform: translateY(-8px);
                        }
                        60% {
                            transform: translateY(-4px);
                        }
                    }
                    .badge-watching-animated {
                        animation: bounce 1.5s infinite;
                        background-color: #e3f2fd !important;
                        color: #1976d2 !important;
                        border: 1px solid #1976d2 !important;
                        font-weight: 500 !important;
                    }
                    .badge-completed-styled {
                        background-color: #e8f5e8 !important;
                        color: #2e7d32 !important;
                        border: 1px solid #4caf50 !important;
                        font-weight: 600 !important;
                    }
                `;
                document.head.appendChild(style);
            }
            
            // Add badge to the container (at the beginning, before other badges)
            badgeContainer.insertBefore(badge, badgeContainer.firstChild);
            console.log(`✅ BADGE: Added ${type} badge to lesson ${lessonId} container`);
        }
        
        console.log(`🎖️ BADGE SET: Lesson ${lessonId} → ${type}`);
    }

    /**
     * 🎯 CREATE BADGE ELEMENT (Helper Method)
     */
    createBadgeElement(badgeContainer, lessonId, type) {
        let badgeHtml = '';
        let badgeClass = '';
        
        switch (type) {
            case 'loading':
                badgeHtml = '⏳ Loading...';
                badgeClass = 'badge-secondary';
                break;
            case 'watching':
                badgeHtml = '<i class="bx bx-play-circle me-1"></i>Watching';
                badgeClass = 'badge-light badge-watching-animated';
                break;
            case 'completed':
                badgeHtml = '<i class="bx bx-check-circle me-1"></i>Completed';
                badgeClass = 'badge-success badge-completed-styled';
                break;
        }
        
        if (badgeHtml) {
            const badge = document.createElement('span');
            badge.className = `lesson-badge badge ${badgeClass} me-1`;
            badge.innerHTML = badgeHtml;
            badge.setAttribute('data-lesson-id', lessonId);
            
            // Add specific styling for different badge types
            if (type === 'watching') {
                badge.style.cssText = `
                    font-size: 0.6rem;
                    background-color: #e3f2fd !important;
                    color: #1976d2 !important;
                    border: 1px solid #1976d2 !important;
                    font-weight: 500 !important;
                    animation: bounce 1.5s infinite;
                `;
            } else if (type === 'completed') {
                badge.style.cssText = `
                    font-size: 0.6rem;
                    background-color: #e8f5e8 !important;
                    color: #2e7d32 !important;
                    border: 1px solid #4caf50 !important;
                    font-weight: 600 !important;
                `;
            } else {
                badge.style.cssText = 'font-size: 0.6rem;';
            }
            
            // Add badge to container (at beginning)
            badgeContainer.insertBefore(badge, badgeContainer.firstChild);
            console.log(`✅ BADGE: Created ${type} badge for lesson ${lessonId}`);
        }
    }
    
    /**
     * 🔄 INITIALIZE ALL BADGES ON LOAD
     */
    initializeBadges() {
        // Mark completed lessons from the completedLessons set
        this.completedLessons.forEach(lessonId => {
            console.log(`🎖️ INITIALIZING: Completed badge for lesson ${lessonId}`);
            this.setBadge(lessonId, 'completed');
        });
        
        // Also check for lessons marked as completed in the HTML (backup method)
        const processedLessons = new Set(); // Track processed lessons to avoid duplicates
        const completedLessonsInHTML = document.querySelectorAll('[data-lesson-id]');
        completedLessonsInHTML.forEach(element => {
            const lessonId = element.getAttribute('data-lesson-id');
            if (lessonId && !processedLessons.has(lessonId)) {
                processedLessons.add(lessonId);
                
                // Check if lesson has completion checkmark in HTML
                const accordionItem = element.closest('.accordion-item');
                if (accordionItem && accordionItem.querySelector('.completion-check')) {
                    console.log(`🎖️ HTML COMPLETION: Found completed lesson ${lessonId} in HTML`);
                    this.completedLessons.add(lessonId.toString());
                    this.setBadge(lessonId, 'completed');
                }
            }
        });
        
        console.log('🎖️ BADGES INITIALIZED for completed lessons:', Array.from(this.completedLessons));
    }

    /**
     * 🏆 HANDLE VIDEO COMPLETED
     */
    async handleVideoCompleted() {
        if (!this.currentVideo || this.completedLessons.has(this.currentVideo.lessonId.toString())) {
            return; // Already completed
        }
        
        console.log('🏆 VIDEO COMPLETED:', this.currentVideo.lessonId);
        
        // Stop progress tracking
        this.stopCurrentVideo();
        
        // Mark as completed in database
        await this.markLessonCompleted();
        
        // Update badge
        this.setBadge(this.currentVideo.lessonId, 'completed');
        
        // Add to completed set
        this.completedLessons.add(this.currentVideo.lessonId.toString());
        
        // Show completion notification
        this.showStatus(`🏆 Lesson ${this.currentVideo.lessonId} completed!`, 'success');
        
        // Auto-play next video if enabled
        if (this.config.autoplayNext) {
            setTimeout(() => {
                this.playNextVideo();
            }, 2000);
        }
    }

    /**
     * 💾 MARK LESSON AS COMPLETED IN DATABASE
     */
    async markLessonCompleted() {
        try {
            const formData = new FormData();
            formData.append('video_id', this.currentVideo.lessonId);
            formData.append('user_id', this.getCurrentUserId());
            formData.append('completion_percentage', '100.00');
            formData.append('completed_at', new Date().toISOString().slice(0, 19).replace('T', ' '));
            
            const response = await fetch('mark_video_completed.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            if (result.success) {
                console.log('✅ LESSON MARKED COMPLETED:', result);
            } else {
                console.error('❌ FAILED TO MARK COMPLETED:', result);
            }
        } catch (error) {
            console.error('❌ MARK COMPLETED ERROR:', error);
        }
    }

    /**
     * 🔴 DISPLAY DETAILED BACKEND ERROR ON PAGE
     */
    displayBackendError(errorData) {
        // Create large, prominent error overlay
        const errorOverlay = document.createElement('div');
        errorOverlay.id = 'backend-error-overlay';
        errorOverlay.style.cssText = `
            position: fixed; top: 0; left: 0; right: 0; bottom: 0; 
            background: rgba(0,0,0,0.8); z-index: 10000;
            display: flex; align-items: center; justify-content: center;
            padding: 20px;
        `;
        
        const errorContent = document.createElement('div');
        errorContent.style.cssText = `
            background: white; max-width: 800px; max-height: 80vh; 
            overflow-y: auto; border-radius: 10px; padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
        `;
        
        errorContent.innerHTML = `
            <div style="text-align: center; margin-bottom: 20px;">
                <h2 style="color: #dc3545; margin: 0;">🚨 BACKEND ERROR DETECTED</h2>
                <p style="margin: 10px 0; color: #666;">Master Video System cannot load lesson data</p>
            </div>
            
            <div style="background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                <h4 style="color: #721c24; margin-top: 0;">Error Message:</h4>
                <p style="margin: 0; color: #721c24; font-weight: 500;">${errorData.message || 'Unknown error'}</p>
            </div>
            
            ${errorData.database_error ? `
                <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                    <h4 style="color: #856404; margin-top: 0;">Database Error:</h4>
                    <p style="margin: 0; color: #856404;">${errorData.database_error}</p>
                    ${errorData.database_errno ? `<p style="margin: 5px 0 0 0; color: #856404;"><strong>Error Code:</strong> ${errorData.database_errno}</p>` : ''}
                </div>
            ` : ''}
            
            ${errorData.debug_data ? `
                <div style="background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                    <h4 style="color: #0c5460; margin-top: 0;">🔍 Debug Information:</h4>
                    <div style="font-family: monospace; font-size: 13px; color: #0c5460;">
                        <p><strong>Course ID (POST):</strong> ${errorData.debug_data.post_course_id}</p>
                        <p><strong>Lesson ID (POST):</strong> ${errorData.debug_data.post_lesson_id}</p>
                        <p><strong>Course ID (GET):</strong> ${errorData.debug_data.get_course_id}</p>
                        <p><strong>Lesson ID (GET):</strong> ${errorData.debug_data.get_lesson_id}</p>
                        <p><strong>User ID:</strong> ${errorData.debug_data.user_id}</p>
                        <p><strong>Database Connected:</strong> ${errorData.debug_data.database_connected}</p>
                        ${errorData.debug_data.session_data ? `
                            <p><strong>Session User ID:</strong> ${errorData.debug_data.session_data.user_id}</p>
                            <p><strong>Session Role:</strong> ${errorData.debug_data.session_data.user_role}</p>
                        ` : ''}
                    </div>
                </div>
            ` : ''}
            
            <div style="background: #e2e3e5; border: 1px solid #d6d8db; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                <h4 style="color: #383d41; margin-top: 0;">💡 What This Means:</h4>
                <ul style="color: #383d41; margin: 0; padding-left: 20px;">
                    <li>The Master Video System tried to load lesson data from the backend</li>
                    <li>The PHP script failed to process the request properly</li>
                    <li>This could be a database connection issue, query problem, or parameter mismatch</li>
                    <li>Check the server error logs for more detailed information</li>
                </ul>
            </div>
            
            <div style="text-align: center;">
                <button onclick="document.getElementById('backend-error-overlay').remove();" 
                        style="background: #007bff; color: white; border: none; padding: 12px 30px; border-radius: 5px; font-size: 16px; cursor: pointer;">
                    Close & Continue Debugging
                </button>
                <button onclick="window.location.reload();" 
                        style="background: #28a745; color: white; border: none; padding: 12px 30px; border-radius: 5px; font-size: 16px; cursor: pointer; margin-left: 10px;">
                    Reload Page
                </button>
            </div>
        `;
        
        errorOverlay.appendChild(errorContent);
        document.body.appendChild(errorOverlay);
        
        console.log('🔴 BACKEND ERROR DISPLAYED ON PAGE:', errorData);
    }

    /**
     * 🆔 GET CURRENT COURSE ID
     */
    getCurrentCourseId() {
        // Try to get from window variables set by PHP first
        if (window.currentCourseId) {
            console.log('✅ COURSE ID: Using window.currentCourseId:', window.currentCourseId);
            return window.currentCourseId;
        }
        
        // Fallback to URL parameter
        const urlParams = new URLSearchParams(window.location.search);
        const urlId = urlParams.get('id') || '';
        console.log('⚠️ COURSE ID: Fallback to URL parameter:', urlId);
        return urlId;
    }
    
    /**
     * 👤 GET CURRENT USER ID
     */
    getCurrentUserId() {
        // Try to get from window variables set by PHP
        return window.currentUserId || window.studentId || 
               document.querySelector('[data-user-id]')?.dataset.userId || '1';
    }

    /**
     * 📢 SHOW STATUS MESSAGE
     */
    showStatus(message, type = 'info', duration = 3000) {
        console.log(`📢 STATUS [${type.toUpperCase()}]: ${message}`);
        
        // Create a simple status notification
        const notification = document.createElement('div');
        notification.className = `alert alert-${type === 'error' ? 'danger' : type === 'success' ? 'success' : 'info'} alert-dismissible fade show`;
        notification.style.cssText = `
            position: fixed; top: 20px; right: 20px; z-index: 9999; 
            min-width: 300px; max-width: 500px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        notification.innerHTML = `
            <div class="d-flex align-items-center">
                <div class="me-2">${type === 'success' ? '✅' : type === 'error' ? '❌' : 'ℹ️'}</div>
                <div class="flex-grow-1">${message}</div>
                <button type="button" class="btn-close ms-2" onclick="this.parentElement.parentElement.remove()"></button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        if (duration > 0) {
            setTimeout(() => {
                if (notification && notification.parentNode) {
                    notification.remove();
                }
            }, duration);
        }
    }

    /**
     * 🚨 SHOW DETAILED ERROR
     */
    showDetailedError(errorMessage, debugInfo = '') {
        console.error('🚨 DETAILED ERROR:', errorMessage, debugInfo);
        
        // Create detailed error overlay
        const errorOverlay = document.createElement('div');
        errorOverlay.style.cssText = `
            position: fixed; top: 0; left: 0; right: 0; bottom: 0;
            background: rgba(0,0,0,0.8); z-index: 10000;
            display: flex; align-items: center; justify-content: center;
            padding: 20px;
        `;
        
        const errorContent = document.createElement('div');
        errorContent.style.cssText = `
            background: white; max-width: 600px; max-height: 80vh;
            overflow-y: auto; border-radius: 10px; padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
        `;
        
        errorContent.innerHTML = `
            <div style="text-align: center; margin-bottom: 20px;">
                <h3 style="color: #dc3545; margin: 0;">🚨 Error Details</h3>
            </div>
            <div style="background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin-bottom: 15px;">
                <strong>Error:</strong> ${errorMessage}
            </div>
            ${debugInfo ? `<div style="background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; margin-bottom: 15px;">
                <strong>Debug Info:</strong><br>${debugInfo}
            </div>` : ''}
            <div style="text-align: center;">
                <button onclick="this.closest('[style*=\"position: fixed\"]').remove();" 
                        style="background: #007bff; color: white; border: none; padding: 10px 20px; border-radius: 5px; cursor: pointer;">
                    Close
                </button>
            </div>
        `;
        
        errorOverlay.appendChild(errorContent);
        document.body.appendChild(errorOverlay);
    }

    /**
     * 🎯 ACCORDION FUNCTIONALITY
     * Enhanced accordion toggle functionality for Bootstrap accordions
     */
    toggleAccordion(accordionId) {
        console.log('🔄 ACCORDION: Toggling:', accordionId);
        
        const accordion = document.getElementById(accordionId);
        if (accordion) {
            // Use Bootstrap's collapse functionality if available
            if (typeof bootstrap !== 'undefined' && bootstrap.Collapse) {
                console.log('✨ ACCORDION: Using Bootstrap 5 collapse');
                const bsCollapse = new bootstrap.Collapse(accordion, {
                    toggle: true
                });
            } else if (typeof $ !== 'undefined' && $.fn.collapse) {
                console.log('✨ ACCORDION: Using Bootstrap 4/jQuery collapse');
                $(accordion).collapse('toggle');
            } else {
                console.log('🔄 ACCORDION: Using manual toggle fallback');
                // Manual fallback toggle
                if (accordion.classList.contains('show')) {
                    accordion.classList.remove('show');
                    accordion.style.display = 'none';
                } else {
                    accordion.classList.add('show');
                    accordion.style.display = 'block';
                }
            }
            console.log(`✅ ACCORDION: Toggled ${accordionId}`);
        } else {
            console.error(`❌ ACCORDION: Could not find accordion with id: ${accordionId}`);
        }
    }

    /**
     * 🎯 TAB SWITCHING FUNCTIONALITY
     * Handles switching between Course Description, Reviews, Notes, Certificate, and Learning Tools tabs
     */
    switchTab(tabName) {
        console.log('🔄 TAB SWITCH: Switching to:', tabName);
        
        // Hide all tab content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.style.display = 'none';
            content.classList.remove('active', 'show');
        });
        
        // Remove active class from all tab buttons
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        
        // Show selected tab content
        const targetContent = document.getElementById(`${tabName}-tab-content`) || 
                             document.querySelector(`[data-tab-content="${tabName}"]`);
        if (targetContent) {
            targetContent.style.display = 'block';
            targetContent.classList.add('active', 'show');
        }
        
        // Set active tab button
        const targetBtn = document.querySelector(`[data-tab="${tabName}"]`);
        if (targetBtn) {
            targetBtn.classList.add('active');
        }
        
        console.log(`✅ TAB SWITCH: Switched to ${tabName}`);
    }

    /**
     * 🎯 BUTTON CLICK HANDLERS
     * Generic button click handler for various interactive elements
     */
    handleButtonClick(action, data) {
        console.log('🖱️ BUTTON CLICK:', { action, data });
        
        switch(action) {
            case 'tab':
                this.switchTab(data.tabName);
                break;
            case 'accordion':
                this.toggleAccordion(data.accordionId);
                break;
            case 'loadLesson':
                this.loadLesson(data.lessonId);
                break;
            default:
                console.log('🔄 BUTTON: Unhandled action:', action);
        }
    }

    /**
     * 🎯 BULLETPROOF ACCORDION SYSTEM
     * Simple, direct accordion that WILL work regardless of Bootstrap
     */
    initializeAccordions() {
        console.log('🔧 ACCORDION: Setting up BULLETPROOF accordion system...');
        
        // Method 1: Handle ALL accordion buttons directly
        const accordionButtons = document.querySelectorAll('.accordion-button');
        console.log(`🔍 ACCORDION: Found ${accordionButtons.length} accordion buttons total`);
        
        // Method 2: Also handle by clicking anywhere in the accordion header
        const accordionHeaders = document.querySelectorAll('.accordion-header');
        console.log(`🔍 ACCORDION: Found ${accordionHeaders.length} accordion headers`);
        
        // FORCE OVERRIDE: Add click handlers to BOTH buttons and headers
        [...accordionButtons, ...accordionHeaders].forEach((element, index) => {
            console.log(`🔧 ACCORDION: Adding FORCE handler ${index + 1}:`, element.tagName, element.className);
            
            element.addEventListener('click', (e) => {
                console.log('🎯🎯🎯 ACCORDION CLICKED!', element);
                console.log('🎯 Event target:', e.target);
                
                // FORCE prevent any default behavior
                e.preventDefault();
                e.stopPropagation();
                e.stopImmediatePropagation();
                
                // Find the button and target
                let button = element;
                if (element.classList.contains('accordion-header')) {
                    button = element.querySelector('.accordion-button');
                }
                
                if (!button) {
                    console.error('❌ No accordion button found');
                    return;
                }
                
                // Get target ID
                const target = button.getAttribute('data-bs-target') || button.getAttribute('data-target');
                console.log('🔍 ACCORDION: Target from button:', target);
                
                if (!target) {
                    console.error('❌ No target found on button');
                    return;
                }
                
                const targetElement = document.querySelector(target);
                console.log('🔍 ACCORDION: Target element:', targetElement);
                
                if (!targetElement) {
                    console.error('❌ Target element not found for:', target);
                    return;
                }
                
                // FIXED LOGIC: Check if accordion should be visible based on Bootstrap collapsed class
                const isCurrentlyCollapsed = button.classList.contains('collapsed');
                const shouldOpen = isCurrentlyCollapsed;
                
                console.log('🔍 ACCORDION: Button collapsed class:', isCurrentlyCollapsed);
                console.log('🔍 ACCORDION: Should open:', shouldOpen);
                console.log('🔍 ACCORDION: Current display:', targetElement.style.display);
                
                if (shouldOpen) {
                    // SHOW IT - accordion should open
                    targetElement.classList.add('show');
                    targetElement.style.display = 'block';
                    button.classList.remove('collapsed');
                    button.setAttribute('aria-expanded', 'true');
                    console.log('✅ ACCORDION: OPENED (was collapsed)');
                } else {
                    // HIDE IT - accordion should close
                    targetElement.classList.remove('show');
                    targetElement.style.display = 'none';
                    button.classList.add('collapsed');
                    button.setAttribute('aria-expanded', 'false');
                    console.log('✅ ACCORDION: CLOSED (was open)');
                }
                
                console.log('🎉 ACCORDION: Toggle completed successfully!');
            }, true); // Use capture phase to ensure we get the event first
        });
        
        // Method 3: FORCE override - document-level click handler as backup
        document.addEventListener('click', (e) => {
            if (e.target.closest('.accordion-button') || e.target.closest('.accordion-header')) {
                console.log('🔥 DOCUMENT LEVEL: Accordion element clicked via document handler');
                
                const accordionButton = e.target.closest('.accordion-button') || 
                                       e.target.closest('.accordion-header')?.querySelector('.accordion-button');
                
                if (accordionButton) {
                    const target = accordionButton.getAttribute('data-bs-target') || 
                                  accordionButton.getAttribute('data-target');
                    
                    if (target) {
                        const targetElement = document.querySelector(target);
                        if (targetElement) {
                            // Force simple toggle
                            const isVisible = targetElement.classList.contains('show');
                            if (isVisible) {
                                targetElement.classList.remove('show');
                                targetElement.style.display = 'none';
                                accordionButton.classList.add('collapsed');
                            } else {
                                targetElement.classList.add('show');
                                targetElement.style.display = 'block';
                                accordionButton.classList.remove('collapsed');
                            }
                            console.log('🔥 DOCUMENT HANDLER: Forced toggle completed');
                        }
                    }
                }
            }
        }, true);
        
        console.log('✅ ACCORDION: BULLETPROOF accordion system initialized with multiple fallbacks!');
    }

    /**
     * 🎯 INITIALIZE TAB SYSTEM
     * Sets up all tab functionality and event listeners
     */
    initializeTabs() {
        console.log('🔧 TAB: Setting up tab switching...');
        
        document.querySelectorAll('.tab-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const tabName = button.getAttribute('data-tab');
                if (tabName) {
                    this.switchTab(tabName);
                }
            });
        });
        
        console.log('✅ TAB: Tab switching initialized');
    }

    /**
     * 🎯 INITIALIZE BOOTSTRAP COMPONENTS
     * Detects and initializes Bootstrap components if available
     */
    initializeBootstrap() {
        console.log('🔍 BOOTSTRAP: Checking for Bootstrap libraries...');
        
        // Check for Bootstrap 5
        if (typeof bootstrap !== 'undefined') {
            console.log('✅ BOOTSTRAP: Bootstrap 5 detected - enhanced functionality available');
            console.log('🔍 BOOTSTRAP: Available Bootstrap 5 components:', Object.keys(bootstrap));
            
            // Verify Collapse component is available
            if (bootstrap.Collapse) {
                console.log('✅ BOOTSTRAP: Collapse component available');
            } else {
                console.log('⚠️ BOOTSTRAP: Collapse component not found');
            }
            
            // Initialize any existing accordions
            document.querySelectorAll('.accordion').forEach((accordion, index) => {
                console.log(`🔧 BOOTSTRAP: Initializing accordion ${index + 1}:`, accordion);
            });
            
        } else if (typeof $ !== 'undefined' && $.fn.collapse) {
            console.log('✅ BOOTSTRAP: Bootstrap 4/jQuery detected - enhanced functionality available');
            console.log('🔍 BOOTSTRAP: jQuery version:', $.fn.jquery);
            
        } else {
            console.log('⚠️ BOOTSTRAP: No Bootstrap detected - using manual fallback functionality');
            console.log('🔍 BOOTSTRAP: Available globals:', {
                bootstrap: typeof bootstrap,
                jQuery: typeof $,
                collapse: typeof $.fn?.collapse
            });
        }
    }
}

// 🚀 AUTO-INITIALIZE WHEN DOM IS READY
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 INITIALIZING MASTER VIDEO SYSTEM...');
    window.masterVideoSystem = new MasterVideoSystem();
});

// Export for global access
window.MasterVideoSystem = MasterVideoSystem;
