<?php
// Start session and include required files
require_once '../includes/session_start.php';
require_once '../config/db_connect.php';

// Authentication check
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || $_SESSION['role'] !== 'student') {
    header("Location: ../auth/login.php");
    exit;
}

$userId = $_SESSION['user_id'];

// Check if this is an AJAX request
$isAjaxRequest = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                 strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

// Function to format message time
function formatMessageTime($datetime) {
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) return 'Just now';
    if ($diff < 3600) return floor($diff / 60) . 'm ago';
    if ($diff < 86400) return floor($diff / 3600) . 'h ago';
    if ($diff < 604800) return floor($diff / 86400) . 'd ago';
    
    return date('M j, Y', $time);
}

// Get contacts (teachers/facilitators) for this student
$contacts = [];
$messages = [];
$selectedContact = null;
$selectedContactId = isset($_GET['contact']) ? intval($_GET['contact']) : null;

try {
    // Get all teachers/facilitators that the student can message
    $contactsQuery = "SELECT DISTINCT 
                        u.user_id, 
                        u.first_name, 
                        u.last_name, 
                        u.email, 
                        u.profile_picture,
                        c.course_id,
                        c.title as course_name
                      FROM users u
                      JOIN teachers t ON u.user_id = t.user_id
                      JOIN courses c ON t.teacher_id = c.teacher_id
                      JOIN enrollments e ON c.course_id = e.course_id
                      WHERE e.student_id = ? AND u.user_role = 'teacher'
                      ORDER BY u.last_name, u.first_name";
    
    $stmt = $conn->prepare($contactsQuery);
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $contactsMap = [];
    while ($row = $result->fetch_assoc()) {
        $contactId = $row['user_id'];
        if (!isset($contactsMap[$contactId])) {
            $contactsMap[$contactId] = [
                'user_id' => $contactId,
                'first_name' => $row['first_name'],
                'last_name' => $row['last_name'],
                'email' => $row['email'],
                'profile_picture' => $row['profile_picture'],
                'courses' => [],
                'unread_count' => 0,
                'last_message_date' => null
            ];
        }
        $contactsMap[$contactId]['courses'][] = [
            'id' => $row['course_id'],
            'name' => $row['course_name']
        ];
    }
    
    $contacts = array_values($contactsMap);
    
    // Get unread message counts and last message dates
    if (!empty($contacts)) {
        // Process each contact individually to avoid complex parameter binding
        foreach ($contacts as &$contact) {
            $contactId = $contact['user_id'];
            
            // Get unread count
            $unreadQuery = "SELECT COUNT(*) as unread_count, MAX(sent_at) as last_message_date
                          FROM messages 
                          WHERE recipient_id = ? AND sender_id = ? AND is_read = 0";
            
            $stmt = $conn->prepare($unreadQuery);
            $stmt->bind_param("ii", $userId, $contactId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($row = $result->fetch_assoc()) {
                $contact['unread_count'] = $row['unread_count'];
                $contact['last_message_date'] = $row['last_message_date'];
            }
        }
        
        // Sort contacts by recent activity
        usort($contacts, function($a, $b) {
            if (isset($a['last_message_date']) && isset($b['last_message_date'])) {
                return strtotime($b['last_message_date']) - strtotime($a['last_message_date']);
            } elseif (isset($a['last_message_date'])) {
                return -1;
            } elseif (isset($b['last_message_date'])) {
                return 1;
            } else {
                return strcmp($a['last_name'], $b['last_name']);
            }
        });
    }
    
    // If a contact is selected, get messages
    if ($selectedContactId && !empty($contacts)) {
        foreach ($contacts as $contact) {
            if ($contact['user_id'] == $selectedContactId) {
                $selectedContact = $contact;
                break;
            }
        }
        
        if ($selectedContact) {
            $messagesQuery = "SELECT 
                                m.message_id, 
                                m.sender_id, 
                                m.recipient_id, 
                                m.message_text, 
                                m.sent_at, 
                                m.is_read,
                                CASE 
                                    WHEN m.sender_id = ? THEN 'sent' 
                                    ELSE 'received' 
                                END as direction,
                                u.first_name as sender_first_name,
                                u.last_name as sender_last_name,
                                u.profile_picture as sender_profile_picture
                              FROM messages m
                              JOIN users u ON m.sender_id = u.user_id
                              WHERE (m.sender_id = ? AND m.recipient_id = ?) 
                                 OR (m.sender_id = ? AND m.recipient_id = ?)
                              ORDER BY m.sent_at ASC";
            
            $stmt = $conn->prepare($messagesQuery);
            $stmt->bind_param("iiiii", $userId, $userId, $selectedContactId, $selectedContactId, $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result) {
                $messages = $result->fetch_all(MYSQLI_ASSOC);
            }
            
            // Mark messages as read
            $markReadQuery = "UPDATE messages SET is_read = 1 
                              WHERE sender_id = ? AND recipient_id = ? AND is_read = 0";
            $stmt = $conn->prepare($markReadQuery);
            $stmt->bind_param("ii", $selectedContactId, $userId);
            $stmt->execute();
        }
    }
    
} catch (Exception $e) {
    error_log("Messages error: " . $e->getMessage());
}

// If this is an AJAX request, return only the conversation content
if ($isAjaxRequest && isset($_GET['ajax'])) {
    if ($selectedContact): ?>
        <div class="message-header">
            <div class="d-flex align-items-center">
                <button class="btn btn-sm btn-outline-secondary d-md-none me-3" onclick="showContactsList()">
                    <i class="fas fa-arrow-left"></i>
                </button>
                <div class="contact-avatar me-3">
                    <?php if ($selectedContact['profile_picture']): ?>
                        <img src="<?= htmlspecialchars($selectedContact['profile_picture']) ?>" 
                             alt="<?= htmlspecialchars($selectedContact['first_name']) ?>" class="rounded-circle">
                    <?php else: ?>
                        <div class="avatar-placeholder">
                            <?= strtoupper(substr($selectedContact['first_name'], 0, 1)) ?>
                        </div>
                    <?php endif; ?>
                </div>
                <div>
                    <h5 class="mb-1"><?= htmlspecialchars($selectedContact['first_name'] . ' ' . $selectedContact['last_name']) ?></h5>
                    <?php if (!empty($selectedContact['courses'])): ?>
                        <small class="text-muted">
                            <i class="bx bx-book-alt me-1"></i>
                            <?= htmlspecialchars($selectedContact['courses'][0]['name']) ?>
                        </small>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="messages-container">
            <div class="messages-list" id="messages-list">
                <?php if (!empty($messages)): ?>
                    <?php
                    $currentDate = '';
                    foreach ($messages as $message):
                        $messageDate = date('Y-m-d', strtotime($message['sent_at']));
                        if ($messageDate !== $currentDate):
                            $currentDate = $messageDate;
                    ?>
                        <div class="date-separator">
                            <span><?= date('F j, Y', strtotime($message['sent_at'])) ?></span>
                        </div>
                    <?php endif; ?>
                    
                    <div class="message-bubble <?= $message['direction'] === 'sent' ? 'message-sent' : 'message-received' ?>">
                        <?= nl2br(htmlspecialchars($message['message_text'])) ?>
                            <div class="message-time">
                                <?= date('g:i A', strtotime($message['sent_at'])) ?>
                                <?php if ($message['direction'] === 'sent'): ?>
                                    <i class="bx bx-check-double <?= $message['is_read'] ? 'text-primary' : 'text-muted' ?>"></i>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="no-messages">
                        <i class="bx bx-message-square-dots"></i>
                        <p>No messages yet. Start a conversation!</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="message-input-area">
            <form id="message-form" class="d-flex w-100">
                <input type="hidden" id="recipient-id" value="<?= $selectedContactId ?>">
                <div class="message-input-wrapper w-100 me-2">
                    <textarea class="form-control" id="message-input" 
                              placeholder="Type your message..." rows="1" required></textarea>
                </div>
                <button type="submit" class="btn btn-primary" id="send-btn">
                    <i class="fas fa-paper-plane"></i>
                </button>
            </form>
        </div>
    <?php else: ?>
        <div class="empty-state">
            <i class="fas fa-comment-slash fa-3x mb-3"></i>
            <h4>Contact not found</h4>
            <p>The selected contact could not be found or you don't have permission to message them.</p>
        </div>
    <?php endif;
    exit;
}

// For regular page requests, include the full page
if (!$isAjaxRequest):
   include_once 'components/header.php';
?>

<!-- Include the enhanced messages CSS -->
<link rel="stylesheet" href="../css/messages_layout_fix.css">

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="page-header mb-4">
                <h2><i class="bx bx-message-square-detail me-2"></i>Messages</h2>
                <p class="text-muted">Communicate with your course facilitators</p>
                <div class="stats-cards d-flex gap-3 mt-3">
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="bx bx-user-voice"></i>
                        </div>
                        <div class="stat-info">
                            <span class="stat-number"><?= count($contacts) ?></span>
                            <span class="stat-label">Facilitators</span>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="bx bx-book-open"></i>
                        </div>
                        <div class="stat-info">
                            <span class="stat-number"><?= array_sum(array_map(function($c) { return count($c['courses']); }, $contacts)) ?></span>
                            <span class="stat-label">Courses</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="messaging-container">
        <div class="row g-0">
            <!-- Contacts Sidebar -->
            <div class="col-lg-4 contacts-column" id="contacts-column">
                <div class="contacts-panel">
                    <div class="contacts-header">
                        <h5><i class="bx bx-user-voice me-2"></i>Course Facilitators</h5>
                        <div class="search-box">
                            <div class="input-group">
                                <span class="input-group-text"><i class="bx bx-search"></i></span>
                                <input type="text" class="form-control" id="contact-search" 
                                       placeholder="Search facilitators...">
                            </div>
                        </div>
                    </div>
                    
                    <div class="contacts-list" id="contacts-list">
                        <?php if (!empty($contacts)): ?>
                            <?php foreach ($contacts as $contact): ?>
                                <div class="contact-item <?= ($selectedContactId == $contact['user_id']) ? 'active' : '' ?>" 
                                     data-contact-id="<?= $contact['user_id'] ?>" 
                                     onclick="loadContact(<?= $contact['user_id'] ?>)">
                                    <div class="contact-avatar">
                                        <?php if ($contact['profile_picture']): ?>
                                            <img src="<?= htmlspecialchars($contact['profile_picture']) ?>" 
                                                alt="<?= htmlspecialchars($contact['first_name']) ?>" class="rounded-circle">
                                        <?php else: ?>
                                            <div class="avatar-initials">
                                                <?= strtoupper(substr($contact['first_name'], 0, 1) . substr($contact['last_name'], 0, 1)) ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-info">
                                        <h6 class="contact-name"><?= htmlspecialchars($contact['first_name'] . ' ' . $contact['last_name']) ?></h6>
                                        <?php if (!empty($contact['courses'])): ?>
                                            <p class="contact-course">
                                                <div class="unread-badge">
                                                    <?= $contact['unread_count'] ?>
                                                </div> 
                                                <?= count($contact['courses']) > 1 ? 
                                                    count($contact['courses']) . ' courses' : 
                                                    htmlspecialchars($contact['courses'][0]['name']) ?>
                                            </p>
                                        <?php endif; ?>
                                        <?php if (isset($contact['last_message_date'])): ?>
                                            <small class="last-message-time text-muted">
                                                <?= formatMessageTime($contact['last_message_date']) ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-actions">
                                        <i class="bx bx-chevron-right"></i>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="no-contacts">
                                <i class="bx bx-user-x"></i>
                                <h5>No Facilitators Found</h5>
                                <p>You don't have any course facilitators to message yet.</p>
                                <a href="courses.php" class="btn btn-primary">
                                    <i class="bx bx-book-open me-2"></i>Browse Courses
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Messages Content -->
            <div class="col-lg-8 messages-column" id="messages-column">
                <div class="messages-panel" id="messages-panel">
                    <?php if ($selectedContact): ?>
                        <!-- This will be loaded via AJAX -->
                        <div class="loading-state">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p>Loading conversation...</p>
                        </div>
                    <?php else: ?>
                        <div class="no-contact-selected">
                            <div class="welcome-message">
                                <i class="bx bx-message-square-dots"></i>
                                <h4>Select a Facilitator</h4>
                                <p>Choose a course facilitator from the list to start messaging</p>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Main JavaScript for messaging functionality -->
<script>
// Wait for DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize variables
    let currentContactId = <?= $selectedContactId ? $selectedContactId : 'null' ?>;
    
    // If a contact is selected, load their conversation
    if (currentContactId) {
        loadConversation(currentContactId);
        markMessagesAsRead(currentContactId);
    }
    
    // Setup message form
    setupMessageForm();
    
    // Setup contact search
    setupContactSearch();
    
    // Helper function to convert newlines to <br> tags like PHP's nl2br
    function nl2br(str) {
        if (typeof str !== 'string') return '';
        return str.replace(/(?:\r\n|\r|\n)/g, '<br>');
    }
    
    // Function to select and load a contact's conversation
    function loadContact(contactId) {
        if (contactId === currentContactId) return;
        
        // Update UI
        document.querySelectorAll('.contact-item').forEach(item => {
            item.classList.remove('active');
        });
        
        const contactItem = document.querySelector(`.contact-item[data-contact-id="${contactId}"]`);
        if (contactItem) {
            contactItem.classList.add('active');
        }
        
        // Update URL without reloading page
        const url = new URL(window.location);
        url.searchParams.set('contact', contactId);
        window.history.pushState({}, '', url);
        
        // Load conversation
        currentContactId = contactId;
        loadConversation(contactId);
        
        // For mobile
        document.querySelector('#contacts-column').classList.remove('d-md-block');
        document.querySelector('#contacts-column').classList.add('d-none');
        document.querySelector('#messages-column').classList.remove('d-none');
        document.querySelector('#messages-column').classList.add('d-block');
    }
    
    // Load conversation via AJAX
    function loadConversation(contactId) {
        const messagesPanel = document.getElementById('messages-panel');
        if (!messagesPanel) return;
        
        messagesPanel.innerHTML = `
            <div class="loading-state">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3">Loading messages...</p>
            </div>
        `;
        
        fetch(`messages.php?contact=${contactId}&ajax=true`)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.text();
            })
            .then(html => {
                messagesPanel.innerHTML = html;
                setupMessageForm();
                setTimeout(() => {
                    scrollToBottom();
                }, 100);
                markMessagesAsRead(contactId);
            })
            .catch(error => {
                console.error('Error loading conversation:', error);
                messagesPanel.innerHTML = `
                    <div class="empty-state">
                        <div class="empty-icon text-danger">
                            <i class="fas fa-exclamation-circle fa-3x mb-3"></i>
                        </div>
                        <h4>Error Loading Messages</h4>
                        <p>There was a problem loading the conversation. Please try again.</p>
                        <button class="btn btn-primary" onclick="loadConversation(${contactId})">Retry</button>
                    </div>
                `;
            });
    }
    
    // Mark messages as read when viewing a conversation
    function markMessagesAsRead(contactId) {
        const formData = new FormData();
        formData.append('contact_id', contactId);
        formData.append('action', 'mark_read');
        
        fetch('ajax_messages.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update unread badge if exists
                const contactItem = document.querySelector(`.contact-item[data-contact-id="${contactId}"]`);
                if (contactItem) {
                    const badge = contactItem.querySelector('.unread-badge');
                    if (badge) {
                        badge.style.display = 'none';
                    }
                }
            }
        })
        .catch(error => {
            console.error('Error marking messages as read:', error);
        });
    }
    
    // Setup message form
    function setupMessageForm() {
        const form = document.getElementById('message-form');
        if (!form) return;
        
        // Remove existing event listeners if any
        const newForm = form.cloneNode(true);
        form.parentNode.replaceChild(newForm, form);
        
        // Add event listener to new form
        newForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const messageInput = document.getElementById('message-input');
            const recipientId = document.getElementById('recipient-id').value;
            const messageText = messageInput.value.trim();
            
            if (!messageText) return;
            
            // Add message locally first for better UX
            const messagesList = document.getElementById('messages-list');
            if (messagesList) {
                const tempMessage = document.createElement('div');
                tempMessage.className = 'message-bubble message-sent';
                tempMessage.innerHTML = `
                    ${nl2br(messageText)}
                    <div class="message-time">
                        Just now
                    </div>
                `;
                messagesList.appendChild(tempMessage);
                scrollToBottom();
            }
            
            // Disable form during sending
            const sendBtn = document.getElementById('send-btn');
            sendBtn.disabled = true;
            messageInput.value = '';
            messageInput.style.height = 'auto';
            
            // Send message via AJAX
            const formData = new FormData();
            formData.append('recipient_id', recipientId);
            formData.append('message', messageText);
            formData.append('action', 'send_message');
            
            fetch('ajax_messages.php', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // We've already cleared the input and shown a temporary message
                    // No need to reload the entire conversation
                } else {
                    alert('Error sending message: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error sending message:', error);
                alert('Error sending message. Please try again.');
            })
            .finally(() => {
                sendBtn.disabled = false;
                messageInput.focus();
            });
        });
        
        // Auto-resize textarea
        const messageInput = document.getElementById('message-input');
        if (messageInput) {
            messageInput.addEventListener('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
            setTimeout(() => {
                messageInput.style.height = 'auto';
                messageInput.style.height = (messageInput.scrollHeight) + 'px';
            }, 100);
        }
    }
    
    // Contact search functionality
    function setupContactSearch() {
        const searchInput = document.getElementById('contact-search');
        if (!searchInput) return;
        
        searchInput.addEventListener('input', function() {
            const query = this.value.toLowerCase();
            document.querySelectorAll('.contact-item').forEach(item => {
                const name = item.querySelector('.contact-name').textContent.toLowerCase();
                if (name.includes(query)) {
                    item.style.display = '';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    }
    
    // Show contacts list on mobile
    function showContactsList() {
        document.querySelector('#contacts-column').classList.remove('d-none');
        document.querySelector('#contacts-column').classList.add('d-md-block');
        document.querySelector('#messages-column').classList.add('d-none');
        document.querySelector('#messages-column').classList.remove('d-block');
    }
    
    // Scroll to bottom of messages
    function scrollToBottom() {
        const messagesList = document.getElementById('messages-list');
        if (messagesList) {
            // Force scroll to bottom with a delay
            setTimeout(() => {
                messagesList.scrollTop = messagesList.scrollHeight + 1000;
            }, 50);
            
            // Do it again after a longer delay to ensure all images and content have loaded
            setTimeout(() => {
                messagesList.scrollTop = messagesList.scrollHeight + 1000;
            }, 500);
        }
    }
    
    // Expose functions to global scope for onclick handlers
    window.loadContact = loadContact;
    window.showContactsList = showContactsList;
});
</script>

<?php endif; // Close the if statement from line 259 ?>

<?php
// Include footer only for regular page requests, not for AJAX calls
if (!$isAjaxRequest) {
    include_once 'components/footer.php';
}
?>
