<?php
// Better payment return handler that doesn't rely on ITN
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// For payment returns, handle authentication more gracefully
// PayFast redirects can sometimes lose session data
$isLoggedIn = isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);

// Debug: Check what's actually in the session
error_log('Payment Return Auth Debug - User ID: ' . ($_SESSION['user_id'] ?? 'NOT SET'));
error_log('Payment Return Auth Debug - Role: ' . ($_SESSION['role'] ?? 'NOT SET'));
error_log('Payment Return Auth Debug - Is Logged In: ' . ($isLoggedIn ? 'YES' : 'NO'));

// Be more lenient with authentication for payment returns
// If user has a user_id, consider them authenticated
if (!$isLoggedIn) {
    $authError = true;
} else {
    // If they have a user_id, assume they're authenticated (payment returns can be tricky)
    $authError = false;
}

// Function to create notifications
function createNotification($user_id, $type, $message) {
    global $conn;
    // Use message as both title and content, set default values for required fields
    $insertNotification = "INSERT INTO notifications (user_id, type, title, content, is_read, created_at) VALUES (?, ?, ?, ?, 0, NOW())";
    $stmt = $conn->prepare($insertNotification);
    $stmt->bind_param("isss", $user_id, $type, $message, $message);
    $stmt->execute();
}

$success = false;
$course = null;
$error = '';
$user_id = $_SESSION['user_id'] ?? 0;

// FETCH USER'S REFERRAL DISCOUNT INFO FOR PAYMENT SUCCESS DISPLAY
$available_discounts = [];
if ($user_id && !$authError) {
    require_once '../config/db_connect.php';
    if (isset($conn)) {
        $discountsQuery = "SELECT * FROM student_referral_discounts 
                          WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                          ORDER BY discount_percentage DESC LIMIT 1";
        $discountStmt = $conn->prepare($discountsQuery);
        if ($discountStmt) {
            $discountStmt->bind_param("i", $user_id);
            $discountStmt->execute();
            $discountResult = $discountStmt->get_result();
            if ($discountResult->num_rows > 0) {
                $available_discounts = $discountResult->fetch_assoc();
            }
        }
    }
}

// Handle authentication error case for payment returns
if ($authError) {
    // Don't process enrollment, but show payment confirmation message
    $success = true; // Payment was successful even if session lost
    $error = '';
    $showAuthMessage = true;
} else {
    $showAuthMessage = false;
}

// Log PayFast parameters for debugging if needed
if (!empty($_GET)) {
    error_log('PayFast Return Parameters: ' . print_r($_GET, true));
}

// Get PayFast return parameters - try different parameter names
$pf_payment_id = $_GET['pf_payment_id'] ?? $_GET['payment_id'] ?? '';
$payment_status = $_GET['payment_status'] ?? $_GET['status'] ?? '';
$m_payment_id = $_GET['m_payment_id'] ?? '';
$amount = $_GET['amount_gross'] ?? $_GET['amount'] ?? 0;

// Handle course_id from URL parameter for sandbox testing
$course_id_from_url = isset($_GET['course_id']) ? intval($_GET['course_id']) : 0;

// PayFast might use different parameter names, let's check common ones
if (empty($pf_payment_id)) {
    foreach (['pf_payment_id', 'payment_id', 'merchant_payment_id', 'transaction_id'] as $key) {
        if (!empty($_GET[$key])) {
            $pf_payment_id = $_GET[$key];
            break;
        }
    }
}

// If we have a course ID in the URL (from a redirect), use that
if (isset($_GET['course_id'])) {
    $_SESSION['payment_course_id'] = intval($_GET['course_id']);
}

if (empty($payment_status)) {
    foreach (['payment_status', 'status', 'transaction_status'] as $key) {
        if (!empty($_GET[$key])) {
            $payment_status = $_GET[$key];
            break;
        }
    }
}

// Check if this is a cart-based payment
$isCartPayment = isset($_GET['type']) && $_GET['type'] === 'cart';

// IMMEDIATE CART CLEARING - As soon as cart payment return is detected
if ($isCartPayment) {
    // Clear session cart immediately
    $_SESSION['cart'] = array();
    
    // Flag for JavaScript to clear localStorage
    $clearCartNow = true;
    
    error_log("Payment Return: Cart payment detected - clearing session cart immediately");
}

// Include email service for purchase confirmation emails - EXACT SAME AS ADMIN COURSE PAGE
require_once '../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but this page uses MySQLi - EXACT SAME AS ADMIN COURSE PAGE
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Include revenue sharing processor for automatic processing
require_once '../includes/purchase_processor.php';

/**
 * Send purchase confirmation notification to student - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
 */
function sendPurchaseConfirmationEmail($user_id, $course_id, $payment_id, $amount) {
    global $conn;
    
    try {
        // Get course and student details - EXACT SAME QUERY PATTERN AS WORKING ADMIN COURSE EMAILS
        $stmt = $conn->prepare("
            SELECT c.title, c.price, 
                   u.first_name, u.last_name, u.email,
                   t.first_name as teacher_first_name, t.last_name as teacher_last_name
            FROM courses c 
            JOIN users u ON u.user_id = ?
            JOIN users t ON c.teacher_id = t.user_id 
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $purchase = $result->fetch_assoc();
        
        if (!$purchase) {
            error_log("Course or user not found for purchase confirmation email: course_id=$course_id, user_id=$user_id");
            return false;
        }
        
        $student_name = $purchase['first_name'] . ' ' . $purchase['last_name'];
        $student_email = $purchase['email'];
        $course_title = $purchase['title'];
        $teacher_name = $purchase['teacher_first_name'] . ' ' . $purchase['teacher_last_name'];
        $purchase_amount = number_format(floatval($amount), 2);
        $purchase_date = date('M j, Y \a\t g:i A');
        
        // Create email content - EXACT SAME STRUCTURE AS WORKING ADMIN COURSE EMAILS
        $subject = "🎉 Purchase Confirmed: \"$course_title\" - Access Now Available!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎉 Purchase Confirmed!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($student_name) . "! 🎓</h2>
                <p>Your purchase of <strong>\"" . htmlspecialchars($course_title) . "\"</strong> has been confirmed and you now have full access!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Purchase Details:</h3>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Instructor:</strong> " . htmlspecialchars($teacher_name) . "</p>
                    <p><strong>Amount Paid:</strong> R" . $purchase_amount . "</p>
                    <p><strong>Purchase Date:</strong> " . $purchase_date . "</p>
                    <p><strong>Payment ID:</strong> " . htmlspecialchars($payment_id) . "</p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>🚀 Ready to Start Learning?</h4>
                    <p>You can now access all course materials, videos, and resources immediately!</p>
                    <p>Start your learning journey right away and unlock your potential.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/my-courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Access Your Course Now
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT SAME PATTERN AS WORKING ADMIN COURSE EMAILS
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending purchase confirmation email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send course sale notification to teacher - EXACT COPY FROM ADMIN COURSES.PHP
 */
function sendTeacherSaleNotificationEmail($course_id) {
    global $conn;
    
    try {
        // Get course and teacher details - EXACT COPY FROM ADMIN COURSES.PHP
        $stmt = $conn->prepare("
            SELECT c.title, c.description, c.price, c.created_at, 
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM courses c 
            JOIN users u ON c.teacher_id = u.user_id 
            LEFT JOIN categories cat ON c.category_id = cat.category_id 
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        
        if (!$course) {
            error_log("Course not found for sale notification email: $course_id");
            return false;
        }
        
        $teacher_name = $course['first_name'] . ' ' . $course['last_name'];
        $teacher_email = $course['email'];
        $course_title = $course['title'];
        $category = $course['category_name'] ?? 'General';
        $price = floatval($course['price']);
        $created_date = date('M j, Y', strtotime($course['created_at']));
        
        // Create email content
        $subject = "🎉 Course Sale: \"$course_title\" - New Student Enrolled!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#f39c12; color:white; padding:20px; text-align:center;'>
                <h1>🎉 Course Sale!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($teacher_name) . "! 💰</h2>
                <p>Great news! A student has just purchased your course <strong>\"" . htmlspecialchars($course_title) . "\"</strong>!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Sale Details:</h3>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>Price:</strong> R" . number_format($price, 2) . "</p>
                    <p><strong>Sale Date:</strong> " . date('M j, Y') . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Dashboard
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT COPY FROM ADMIN COURSES.PHP
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT COPY FROM ADMIN COURSES.PHP
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending course sale notification email: " . $e->getMessage());
        return false;
    }
}

// Process payment return - handle both PayFast parameters and fallback
// Added course_id to the condition to handle sandbox testing where only course_id is passed
// Skip enrollment processing if there's an auth error, but still show success message
if (!$authError && (($pf_payment_id && $payment_status === 'COMPLETE') || isset($_GET['course_id']) || $isCartPayment || empty($_GET))) {
    
    // If no parameters from PayFast except course_id, assume successful payment
    if (empty($_GET) || (count($_GET) === 1 && isset($_GET['course_id']))) {
        // This means user was redirected back from PayFast but no parameters sent
        // This is common in sandbox mode with localhost URLs
        $fallback_mode = true;
        
        // Check for course ID from URL first
        if ($course_id_from_url > 0) {
            $course_id = $course_id_from_url;
            
            // Get course price from database
            $priceQuery = "SELECT price FROM courses WHERE course_id = ?";
            $stmt = $conn->prepare($priceQuery);
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $priceData = $result->fetch_assoc();
                $amount = $priceData['price'];
            } else {
                $amount = 199.00; // Default amount
            }
        }
        // Try to get course ID from session if not in URL
        else if (isset($_SESSION['payment_course_id'])) {
            $course_id = $_SESSION['payment_course_id'];
            
            // Get course price from database
            $priceQuery = "SELECT price FROM courses WHERE course_id = ?";
            $stmt = $conn->prepare($priceQuery);
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $priceData = $result->fetch_assoc();
                $amount = $priceData['price'];
            } else {
                $amount = 199.00; // Default amount
            }
        } else {
            $course_id = 5; // Default to Computer lit course
            $amount = 199.00; // Default amount
        }
    } else {
        $fallback_mode = false;
    }
        
    // Get course and amount info
    $user_id = $_SESSION['user_id']; // Get from session
    
    if (!$fallback_mode) {
        // Try to get course_id from m_payment_id or any PayFast reference data
        // First check if we have it stored in the session
        if (isset($_SESSION['payment_course_id'])) {
            $course_id = $_SESSION['payment_course_id'];
        } 
        // Then check if merchant reference contains course ID
        else if (!empty($m_payment_id) && strpos($m_payment_id, 'course_') !== false) {
            // Extract course_id from m_payment_id (e.g., "course_5" -> 5)
            $course_id = intval(str_replace('course_', '', $m_payment_id));
        } else {
            // Fallback: try to find by amount
            $amount = isset($_GET['amount']) ? floatval($_GET['amount']) : 0;
            
            // Get the latest payment for this user
            $paymentQuery = "SELECT item_id FROM payments WHERE user_id = ? ORDER BY created_at DESC LIMIT 1";
            $stmt = $conn->prepare($paymentQuery);
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $paymentResult = $stmt->get_result();
            
            if ($paymentResult->num_rows > 0) {
                // Use the course ID from the latest payment
                $paymentData = $paymentResult->fetch_assoc();
                $course_id = $paymentData['item_id'];
            } else {
                // Find course by amount (assuming unique pricing)
                $courseQuery = "SELECT course_id FROM courses WHERE price = ?";
                $stmt = $conn->prepare($courseQuery);
                $stmt->bind_param("d", $amount);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result->num_rows > 0) {
                    $courseData = $result->fetch_assoc();
                    $course_id = $courseData['course_id'];
                } else {
                    // If we can't determine the course, check the session for recently viewed course
                    // Default fallback - try different methods to get the right course
                    $course_id = isset($_SESSION['last_purchased_course']) ? $_SESSION['last_purchased_course'] : 
                                (isset($_SESSION['last_viewed_course']) ? $_SESSION['last_viewed_course'] : 5);
                }
            }
        }
    }
    // If fallback_mode, course_id and amount are already set above

// Set success flag - sandbox testing is always considered successful if we have a course_id
if (isset($_GET['course_id'])) {
    $success = true;
    $course_id = intval($_GET['course_id']);
}

// Debug: Add error reporting for this page
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Debug: Log session data for troubleshooting
error_log('Payment Return Debug - Session Data: ' . print_r($_SESSION, true));
error_log('Payment Return Debug - GET params: ' . print_r($_GET, true));
error_log('Payment Return Debug - Is Cart Payment: ' . ($isCartPayment ? 'YES' : 'NO'));

// Handle cart-based payment (multiple courses)
if ($isCartPayment) {
    $success = true; // Always set success for cart payments that reach this page
    
    // Check if we have cart payment data in session
    if (!isset($_SESSION['payment_course_ids']) || !is_array($_SESSION['payment_course_ids'])) {
        // Fallback: Create a simple success message without specific course data
        // This handles cases where session data might be lost
        $_SESSION['enrolled_courses'] = [];
        $_SESSION['cart_payment_total'] = 0;
        $_SESSION['cart_fallback_success'] = true;
        error_log('Payment Return: Using fallback mode for cart payment');
    } else {
    $success = true;
    $enrolledCourses = [];
    $failedCourses = [];
    $totalAmount = $_SESSION['payment_amount'] ?? 0;
    
    foreach ($_SESSION['payment_course_ids'] as $courseId) {
        // Get course info
        $courseQuery = "SELECT c.course_id, c.title, c.price, c.thumbnail, c.description, 
                        u.first_name, u.last_name, cat.category_name 
                        FROM courses c 
                        LEFT JOIN users u ON c.teacher_id = u.user_id 
                        LEFT JOIN categories cat ON c.category_id = cat.category_id 
                        WHERE c.course_id = ?";
        $stmt = $conn->prepare($courseQuery);
        $stmt->bind_param("i", $courseId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $courseData = $result->fetch_assoc();
            
            // Check if already enrolled
            $enrollmentQuery = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $user_id, $courseId);
            $stmt->execute();
            $enrollResult = $stmt->get_result();
            
            if ($enrollResult->num_rows === 0) {
                // Enroll the student with course price for revenue tracking
                $enrollQuery = "INSERT INTO enrollments (student_id, course_id, user_id, course_price, enrolled_date, progress) VALUES (?, ?, ?, ?, NOW(), 0)";
                $stmt = $conn->prepare($enrollQuery);
                $stmt->bind_param("iiid", $user_id, $courseId, $user_id, $courseData['price']);
                
                if ($stmt->execute()) {
                    $enrollment_id = $conn->insert_id;
                    $enrolledCourses[] = $courseData;
                    
                    // Process revenue sharing for this purchase
                    try {
                        $purchase_processor = new PurchaseProcessor($conn);
                        $revenue_result = $purchase_processor->processPurchase(
                            $user_id,
                            $courseId,
                            $courseData['price'],
                            'payfast_cart',
                            $pf_payment_id ?: ('CART_' . time() . '_' . $courseId)
                        );
                        
                        if ($revenue_result['success']) {
                            // Update enrollment with purchase_id
                            $update_enrollment = "UPDATE enrollments SET purchase_id = ? WHERE enrollment_id = ?";
                            $update_stmt = $conn->prepare($update_enrollment);
                            $update_stmt->bind_param("ii", $revenue_result['purchase_id'], $enrollment_id);
                            $update_stmt->execute();
                            
                            // Log successful revenue processing
                            error_log("Revenue sharing processed for enrollment {$enrollment_id}, purchase {$revenue_result['purchase_id']}");
                            
                            // Send purchase confirmation email - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
                            $email_result = sendPurchaseConfirmationEmail($user_id, $courseId, $pf_payment_id ?: ('CART_' . time() . '_' . $courseId), $courseData['price']);
                            error_log("Purchase confirmation email sent: " . ($email_result ? "SUCCESS" : "FAILED") . " for course $courseId to user $user_id");
                            
                            // Send teacher sale notification email - EXACT COPY FROM ADMIN COURSES.PHP
                            $teacher_email_result = sendTeacherSaleNotificationEmail($courseId);
                            error_log("Teacher sale notification email sent: " . ($teacher_email_result ? "SUCCESS" : "FAILED") . " for course $courseId to teacher");
                        } else {
                            // Log revenue processing error but don't fail enrollment
                            error_log("Revenue sharing failed for enrollment {$enrollment_id}: " . $revenue_result['error']);
                        }
                    } catch (Exception $e) {
                        // Log exception but don't fail enrollment
                        error_log("Revenue sharing exception for enrollment {$enrollment_id}: " . $e->getMessage());
                    }
                    
                    // Create notification for enrollment
                    createNotification($user_id, 'enrollment', "You have been successfully enrolled in course: " . $courseData['title']);
                    
                    // Record the successful payment (with error checking)
                    $paymentQuery = "INSERT INTO payments (user_id, course_id, amount, payment_method, payment_status, transaction_id, created_at) 
                                   VALUES (?, ?, ?, 'payfast', 'completed', ?, NOW())";
                    $stmt = $conn->prepare($paymentQuery);
                    if ($stmt) {
                        $transactionId = $pf_payment_id ?: 'cart_' . time();
                        $stmt->bind_param("iids", $user_id, $courseId, $courseData['price'], $transactionId);
                        $stmt->execute();
                    } else {
                        // Log the SQL error for debugging
                        error_log("Payment record SQL error: " . $conn->error);
                        // Continue without recording payment - enrollment is more important
                    }
                } else {
                    $failedCourses[] = $courseData;
                }
            } else {
                // Already enrolled, still count as success
                $enrolledCourses[] = $courseData;
            }
        }
    }
    
    // Store enrollment results for display
    $_SESSION['enrolled_courses'] = $enrolledCourses;
    $_SESSION['failed_courses'] = $failedCourses;
    $_SESSION['cart_payment_total'] = $totalAmount;
    
    // PROCESS REFERRAL REWARDS FOR CART PURCHASES - This was missing!
    if (!empty($enrolledCourses)) {
        try {
            require_once 'payment-success-handler.php';
            $courseIds = array_column($enrolledCourses, 'course_id');
            $referral_result = processPaymentReferralRewards($user_id, 0, $courseIds, $totalAmount);
            
            if ($referral_result['success'] && $referral_result['rewards_processed'] > 0) {
                error_log("Cart referral rewards processed - {$referral_result['rewards_processed']} rewards given");
                createNotification($user_id, 'referral', "Your referral rewards have been processed! Check your referral dashboard for details.");
            }
        } catch (Exception $e) {
            error_log("Cart referral processing error: " . $e->getMessage());
        }
    }
    
    // CLEAR CART AFTER SUCCESSFUL PURCHASE - Both session and localStorage
    $_SESSION['cart'] = array(); // Clear session cart
    
    // Clean up payment session data
    unset($_SESSION['payment_course_ids']);
    unset($_SESSION['payment_course_titles']);
    unset($_SESSION['payment_amount']);
    unset($_SESSION['cart_payment_data']);
    
    // Flag to clear localStorage cart via JavaScript
    $clearLocalStorageCart = true;
    
    } // Close the else block for session data handling
    
} elseif (!$isCartPayment) {
    // Store course_id in session for potential future reference (single course)
    $_SESSION['last_purchased_course'] = $course_id;
    
    // Get course info with more details (single course)
    $courseQuery = "SELECT c.course_id, c.title, c.price, c.thumbnail, c.description, 
                    u.first_name, u.last_name, cat.category_name 
                    FROM courses c 
                    LEFT JOIN users u ON c.teacher_id = u.user_id 
                    LEFT JOIN categories cat ON c.category_id = cat.category_id 
                    WHERE c.course_id = ?";
    $stmt = $conn->prepare($courseQuery);
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $course = $result->fetch_assoc();
        
        // Check if already enrolled
        $enrollmentQuery = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
        $stmt = $conn->prepare($enrollmentQuery);
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $enrollResult = $stmt->get_result();
        
        if ($enrollResult->num_rows === 0) {
            // Enroll the student with course price for revenue tracking
            $enrollQuery = "INSERT INTO enrollments (student_id, course_id, user_id, course_price, enrolled_date, progress) VALUES (?, ?, ?, ?, NOW(), 0)";
            $stmt = $conn->prepare($enrollQuery);
            $stmt->bind_param("iiid", $user_id, $course_id, $user_id, $course['price']);
            
            if ($stmt->execute()) {
                $enrollment_id = $conn->insert_id;
                $success = true;
                
                // Process revenue sharing for this purchase
                try {
                    $purchase_processor = new PurchaseProcessor($conn);
                    $revenue_result = $purchase_processor->processPurchase(
                        $user_id,
                        $course_id,
                        $course['price'],
                        'payfast_single',
                        $pf_payment_id ?: ('SINGLE_' . time() . '_' . $course_id)
                    );
                    
                    if ($revenue_result['success']) {
                        // Update enrollment with purchase_id
                        $update_enrollment = "UPDATE enrollments SET purchase_id = ? WHERE enrollment_id = ?";
                        $update_stmt = $conn->prepare($update_enrollment);
                        $update_stmt->bind_param("ii", $revenue_result['purchase_id'], $enrollment_id);
                        $update_stmt->execute();
                        
                        // Log successful revenue processing
                        error_log("Single course revenue sharing processed for enrollment {$enrollment_id}, purchase {$revenue_result['purchase_id']}");
                        
                        // Send purchase confirmation email - EXACT SAME APPROACH AS CART PURCHASES
                        $email_result = sendPurchaseConfirmationEmail($user_id, $course_id, $pf_payment_id ?: ('SINGLE_' . time() . '_' . $course_id), $course['price']);
                        error_log("Purchase confirmation email sent: " . ($email_result ? "SUCCESS" : "FAILED") . " for course $course_id to user $user_id");
                        
                        // Send teacher sale notification email - EXACT COPY FROM ADMIN COURSES.PHP
                        $teacher_email_result = sendTeacherSaleNotificationEmail($course_id);
                        error_log("Teacher sale notification email sent: " . ($teacher_email_result ? "SUCCESS" : "FAILED") . " for course $course_id to teacher");
                        
                        // PROCESS REFERRAL REWARDS - This was missing!
                        try {
                            require_once 'payment-success-handler.php';
                            $referral_result = processPaymentReferralRewards($user_id, $revenue_result['purchase_id'], [$course_id], $course['price']);
                            
                            if ($referral_result['success'] && $referral_result['rewards_processed'] > 0) {
                                error_log("Referral rewards processed for purchase {$revenue_result['purchase_id']} - {$referral_result['rewards_processed']} rewards given");
                                createNotification($user_id, 'referral', "Your referral rewards have been processed! Check your referral dashboard for details.");
                            }
                        } catch (Exception $e) {
                            error_log("Referral processing error: " . $e->getMessage());
                        }
                        
                        // Enhanced notifications with revenue info
                        createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation. Revenue sharing processed.");
                        createNotification($user_id, 'payment', "Payment of R" . number_format($course['price'], 2) . " confirmed for course: " . $course['title'] . ". Revenue distributed to platform and teacher.");
                    } else {
                        // Log revenue processing error but don't fail enrollment
                        error_log("Single course revenue sharing failed for enrollment {$enrollment_id}: " . $revenue_result['error']);
                        
                        // Standard notifications (enrollment still succeeded)
                        createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                        createNotification($user_id, 'payment', "Payment of R" . number_format($course['price'], 2) . " confirmed for course: " . $course['title']);
                    }
                } catch (Exception $e) {
                    // Log exception but don't fail enrollment
                    error_log("Single course revenue sharing exception for enrollment {$enrollment_id}: " . $e->getMessage());
                    
                    // Standard notifications (enrollment still succeeded)
                    createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                    createNotification($user_id, 'payment', "Payment of R" . number_format($course['price'], 2) . " confirmed for course: " . $course['title']);
                }
            } else {
                $error = "Payment successful but enrollment failed. Please contact support.";
            }
        } else {
            $success = true; // Already enrolled
        }
        
        // Record payment if not already recorded (and if we have a payment ID)
        if (!empty($pf_payment_id)) {
            $paymentCheck = "SELECT id FROM payments WHERE pf_payment_id = ?";
            $stmt = $conn->prepare($paymentCheck);
            if ($stmt) {
                $stmt->bind_param("s", $pf_payment_id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result && $result->num_rows === 0) {
                    $recordPayment = "INSERT INTO payments (payment_id, pf_payment_id, user_id, item_type, item_id, amount, payment_status, created_at) VALUES (?, ?, ?, 'course', ?, ?, 'COMPLETE', NOW())";
                    $stmt = $conn->prepare($recordPayment);
                    if ($stmt) {
                        $stmt->bind_param("ssidd", $pf_payment_id, $pf_payment_id, $user_id, $course_id, $amount);
                        $stmt->execute();
                    } else {
                        error_log("Payment record SQL error: " . $conn->error);
                    }
                }
            } else {
                error_log("Payment check SQL error: " . $conn->error);
            }
        }
    } else {
        $error = "Course not found.";
    }
} // Close the elseif (!$isCartPayment) block

} else {
    $error = "No payment information received or payment not complete.";
}

// Include header for proper display
include_once 'components/header.php';
?>

<style>
/* Modern Payment Success Page */
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    font-family: 'Inter', 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
    min-height: 100vh;
    margin: 0;
    padding: 0;
}

.payment-result {
    max-width: 900px;
    margin: 0 auto;
    padding: 2rem 1rem;
    min-height: calc(100vh - 100px);
    display: flex;
    align-items: center;
    justify-content: center;
}

/* Success Animation */
.success-animation {
    text-align: center;
    margin-bottom: 2rem;
    position: relative;
}

.success-glow {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    background: linear-gradient(135deg, #fef3c7, #fde68a);
    margin: 0 auto 1.5rem;
    position: relative;
    animation: successPulse 2s ease-in-out infinite;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 8px 32px rgba(251, 191, 36, 0.3);
}

.success-icon {
    font-size: 2.5rem;
    color: white;
}

@keyframes successPulse {
    0%, 100% { 
        transform: scale(1); 
        box-shadow: 0 10px 30px rgba(245, 158, 11, 0.3);
    }
    50% { 
        transform: scale(1.05); 
        box-shadow: 0 15px 40px rgba(245, 158, 11, 0.5);
    }
}

/* Success Card */
.success-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(10px);
    border-radius: 24px;
    padding: 3rem;
    box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
    margin-bottom: 2rem;
    position: relative;
    overflow: hidden;
    border: 1px solid rgba(255, 255, 255, 0.2);
    animation: slideInUp 0.6s ease-out;
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Success Header */
.success-header h3 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    background: linear-gradient(135deg, #f59e0b, #d97706);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.success-header p {
    font-size: 1.1rem;
    color: #6b7280;
    font-weight: 400;
    margin: 0;
}

/* Success Body */
.success-body {
    margin-bottom: 2.5rem;
}

/* Course Info Cards */
/* Payment Success Discount Badges */
.payment-success-price-container {
    text-align: right;
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    gap: 4px;
}

.original-price-strike {
    font-size: 14px;
    color: #6b7280;
    text-decoration: line-through;
    font-weight: 500;
}

.discounted-price-main {
    font-size: 20px;
    color: #dc2626;
    font-weight: 700;
}

.savings-badge-success {
    font-size: 11px;
    background: linear-gradient(135deg, #f59e0b, #d97706);
    color: white;
    padding: 3px 8px;
    border-radius: 6px;
    font-weight: 600;
    display: inline-block;
    animation: success-savings-pulse 3s ease-in-out infinite;
    box-shadow: 0 2px 8px rgba(245, 158, 11, 0.3);
}

@keyframes success-savings-pulse {
    0%, 100% { 
        opacity: 1; 
        transform: scale(1);
        box-shadow: 0 2px 8px rgba(245, 158, 11, 0.3);
    }
    50% { 
        opacity: 0.9; 
        transform: scale(1.02);
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.4);
    }
}

/* Alerts */
.alert {
    border-radius: 12px;
    padding: 1rem 1.25rem;
    border: none;
    font-weight: 500;
    margin-bottom: 1.5rem;
}

.alert-success {
    background: linear-gradient(135deg, #fef3c7, #fde68a);
    color: #92400e;
    border-left: 4px solid #f59e0b;
}

.alert-info {
    background: linear-gradient(135deg, #dbeafe, #bfdbfe);
    color: #1e40af;
    border-left: 4px solid #3b82f6;
}

/* Success Actions */
.success-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
    margin-top: 2rem;
}

.btn-success, .btn-outline {
    padding: 0.875rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    font-size: 1rem;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    transition: all 0.3s ease;
    border: 2px solid transparent;
    min-width: 180px;
    justify-content: center;
}

.btn-success {
    background: linear-gradient(135deg, #f59e0b, #d97706);
    color: white;
    box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
}

.btn-success:hover {
    background: linear-gradient(135deg, #d97706, #b45309);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(245, 158, 11, 0.4);
    color: white;
    text-decoration: none;
}

.btn-outline {
    background: rgba(255, 255, 255, 0.9);
    color: #4b5563;
    border-color: #d1d5db;
}

.btn-outline:hover {
    background: white;
    border-color: #f59e0b;
    color: #f59e0b;
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
    text-decoration: none;
}

/* Error Card */
.error-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(10px);
    border-radius: 24px;
    padding: 3rem;
    box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
    margin-bottom: 2rem;
    text-align: center;
    border: 1px solid rgba(255, 255, 255, 0.2);
    animation: slideInUp 0.6s ease-out;
}

.error-header {
    margin-bottom: 2rem;
}

.error-header i {
    color: #ef4444;
}

.error-header h3 {
    font-size: 2rem;
    font-weight: 700;
    color: #dc2626;
    margin-bottom: 0.5rem;
}

.error-header p {
    font-size: 1.1rem;
    color: #6b7280;
    margin: 0;
}

/* Responsive Design */
@media (max-width: 768px) {
    .payment-result {
        padding: 1rem;
        min-height: calc(100vh - 80px);
    }
    
    .success-card, .error-card {
        padding: 2rem 1.5rem;
        border-radius: 20px;
    }
    
    .success-header h3 {
        font-size: 2rem;
    }
    
    .course-info {
        padding: 1rem;
    }
    
    .course-title {
        font-size: 1.1rem;
    }
    
    .course-price {
        font-size: 1.25rem;
    }
    
    .success-actions {
        flex-direction: column;
        align-items: center;
    }
    
    .btn-success, .btn-outline {
        width: 100%;
        max-width: 300px;
    }
    
    .success-glow {
        width: 80px;
        height: 80px;
    }
    
    .success-icon {
        font-size: 2rem;
    }
}

.course-info:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
    border-color: #f59e0b;
}

.course-info::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: linear-gradient(135deg, #f59e0b, #d97706);
}

.course-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
}

.course-instructor {
    font-size: 0.95rem;
    color: #6b7280;
    margin-bottom: 0.5rem;
}

.course-category {
    font-size: 0.9rem;
    color: #d97706;
    font-weight: 500;
}

.course-category i {
    margin-right: 0.25rem;
}

.course-price {
    font-size: 1.5rem;
    font-weight: 700;
    color: #f59e0b;
    text-align: right;
}

/* Payment Summary */
.payment-summary {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border-radius: 16px;
    padding: 1.5rem;
    border: 1px solid #0ea5e9;
    margin-top: 1.5rem;
}

.payment-summary .d-flex {
    margin-bottom: 0.75rem;
}

.payment-summary .d-flex:last-child {
    margin-bottom: 0;
    padding-top: 0.75rem;
    border-top: 1px solid rgba(14, 165, 233, 0.2);
}

.course-info {
    display: flex;
    align-items: flex-start;
    padding: 1.5rem;
    text-align: left;
    background: #f9fafb;
    border-radius: 12px;
    margin-bottom: 1.5rem;
}

.course-thumbnail {
    width: 120px;
    height: 120px;
    object-fit: cover;
    border-radius: 8px;
    margin-right: 1.5rem;
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.course-details h5 {
    font-size: 1.25rem;
    margin-bottom: 0.5rem;
}

.course-meta {
    display: flex;
    align-items: center;
    color: #6c757d;
    margin-bottom: 0.5rem;
    font-size: 0.9rem;
}

.course-meta i {
    margin-right: 0.25rem;
}

.course-meta span {
    margin-right: 1rem;
}

.payment-details {
    border-top: 1px solid #e9ecef;
    padding-top: 1.5rem;
    margin-top: 1.5rem;
}

.payment-amount {
    font-size: 1.5rem;
    font-weight: bold;
    color: #212529;
}

.action-buttons {
    display: flex;
    justify-content: center;
    gap: 1rem;
    margin-top: 2rem;
}

.action-button {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 0.75rem 1.5rem;
    font-size: 1rem;
    font-weight: 600;
    border-radius: 50px;
    transition: all 0.3s ease;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.btn-courses {
    background: #f59e0b;
    border: none;
    color: white;
}

.btn-courses:hover {
    background: #d97706;
    transform: translateY(-2px);
    box-shadow: 0 6px 12px rgba(0,0,0,0.15);
}

.btn-start {
    background: #4e73df;
    border: none;
    color: white;
}

.btn-start:hover {
    background: #3a5fc8;
    transform: translateY(-2px);
    box-shadow: 0 6px 12px rgba(0,0,0,0.15);
}

/* Error styling */
.error-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.08);
    overflow: hidden;
    border: none;
}

.error-header {
    background: linear-gradient(135deg, #dc3545, #f66d7c);
    color: white;
    padding: 1.5rem;
}

.error-body {
    padding: 2rem;
}

.confetti {
    position: absolute;
    width: 10px;
    height: 10px;
    background: #f39c12;
    animation: confetti-fall 3s linear infinite;
}

@keyframes confetti-fall {
    to { transform: translateY(100vh) rotate(360deg); }
}

/* Responsive */
@media (max-width: 767px) {
    .payment-result {
        margin: 1.5rem;
        padding: 0;
    }
    
    .course-info {
        flex-direction: column;
        align-items: center;
        text-align: center;
    }
    
    .course-thumbnail {
        margin-right: 0;
        margin-bottom: 1.5rem;
    }
    
    .action-buttons {
        flex-direction: column;
    }
    
    .success-header h2 {
        font-size: 1.5rem;
    }
}
</style>

<div class="payment-result">

    
    <?php if (!$authError && $isCartPayment && isset($_SESSION['cart_fallback_success'])): ?>
        <!-- Cart Payment Fallback Success Display (User IS authenticated) -->
        <div class="success-card">
            <!-- Success Animation -->
            <div class="success-animation">
                <div class="success-glow">
                    <i class="bx bx-check success-icon"></i>
                </div>
            </div>
            
            <div class="success-header">
                <h3>Payment Successful!</h3>
                <p>Your payment has been processed successfully. You will be enrolled in your courses shortly.</p>
            </div>
            
            <div class="success-body">
                <div class="alert alert-success">
                    <i class="bx bx-check-circle me-2"></i>
                    <strong>Payment Confirmed</strong><br>
                    Your courses are being processed and you will receive enrollment confirmations via email.
                </div>
                
                <div class="payment-summary">
                    <div class="d-flex justify-content-between align-items-center">
                        <span style="font-weight: 600; color: #374151;">Status:</span>
                        <span style="font-weight: 700; color: #10b981;">Payment Successful</span>
                    </div>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="courses.php" class="btn-success">
                    <i class="bx bx-book me-2"></i>
                    View My Courses
                </a>
                <a href="dashboard.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Go to Dashboard
                </a>
            </div>
        </div>
        
        <script>
        // Clear cart from localStorage after successful payment
        localStorage.removeItem('cart');
        localStorage.removeItem('appliedCheckoutPromoCode');
        localStorage.removeItem('checkoutPromoDiscount');
        
        // Update cart count
        const cartBadge = document.querySelector('.cart-count');
        if (cartBadge) {
            cartBadge.textContent = '0';
            cartBadge.style.display = 'none';
        }
        
        const cartIconBadge = document.querySelector('#cart-count');
        if (cartIconBadge) {
            cartIconBadge.textContent = '0';
            cartIconBadge.style.display = 'none';
        }
        </script>
        
    <?php elseif (!$authError && $isCartPayment && isset($_SESSION['enrolled_courses'])): ?>
        <!-- Cart Payment Full Success Display (User IS authenticated, with course details) -->
        <div class="success-card">
            <!-- Success Animation -->
            <div class="success-animation">
                <div class="success-glow">
                    <i class="bx bx-check success-icon"></i>
                </div>
            </div>
            
            <div class="success-header">
                <h3>Payment Successful!</h3>
                <p>You have been enrolled in <?= count($_SESSION['enrolled_courses']) ?> course<?= count($_SESSION['enrolled_courses']) > 1 ? 's' : '' ?>:</p>
            </div>
            
            <div class="success-body">
                <?php foreach ($_SESSION['enrolled_courses'] as $enrolledCourse): ?>
                    <div class="course-info">
                        <img src="<?= htmlspecialchars($enrolledCourse['thumbnail'] ?? '../assets/images/default-thumbnail.jpg') ?>" 
                             alt="<?= htmlspecialchars($enrolledCourse['title']) ?>" 
                             class="course-thumbnail"
                             onerror="this.onerror=null; this.src='../assets/images/default-thumbnail.jpg';">
                        <div class="course-details">
                            <div class="course-title"><?= htmlspecialchars($enrolledCourse['title']) ?></div>
                            <div class="course-instructor">
                                <i class="bx bx-user"></i>
                                by <?= htmlspecialchars(($enrolledCourse['first_name'] ?? '') . ' ' . ($enrolledCourse['last_name'] ?? 'Instructor')) ?>
                            </div>
                            <div class="course-category">
                                <i class="bx bx-category"></i>
                                <?= htmlspecialchars($enrolledCourse['category_name'] ?? 'General') ?>
                            </div>
                        </div>
                        <div class="course-price">
                            <?php 
                            $original_price = $enrolledCourse['price'];
                            $has_discount = false;
                            $discounted_price = $original_price;
                            $savings = 0;
                            
                            // Calculate discount if available
                            if (!empty($available_discounts) && $original_price > 0) {
                                $discount_amount = min(
                                    ($original_price * $available_discounts['discount_percentage'] / 100),
                                    $available_discounts['discount_amount']
                                );
                                $discounted_price = $original_price - $discount_amount;
                                $savings = $discount_amount;
                                $has_discount = true;
                            }
                            ?>
                            
                            <?php if ($has_discount): ?>
                                <div class="payment-success-price-container">
                                    <div class="original-price-strike">R<?= number_format($original_price, 0) ?></div>
                                    <div class="discounted-price-main">R<?= number_format($discounted_price, 0) ?></div>
                                    <div class="savings-badge-success">Saved R<?= number_format($savings, 0) ?>!</div>
                                </div>
                            <?php else: ?>
                                R<?= number_format($original_price, 0) ?>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
                
                <div class="payment-summary">
                    <div class="d-flex justify-content-between align-items-center">
                        <span style="font-weight: 600; color: #374151;">Total Paid:</span>
                        <span style="font-weight: 700; color: #0ea5e9; font-size: 1.25rem;">R<?= number_format($_SESSION['cart_payment_total'] ?? 0, 2) ?></span>
                    </div>
                    <?php if (isset($_SESSION['payment_discount_percent']) && $_SESSION['payment_discount_percent'] > 0): ?>
                        <div class="d-flex justify-content-between align-items-center" style="color: #10b981;">
                            <span>Discount Applied:</span>
                            <span><?= ($_SESSION['payment_discount_percent'] * 100) ?>% off</span>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="courses.php" class="btn-success">
                    <i class="bx bx-book me-2"></i>
                    View My Courses
                </a>
                <a href="dashboard.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Go to Dashboard
                </a>
            </div>
        </div>
        
    <?php elseif (!$authError && $isCartPayment): ?>
        <!-- Cart Payment Simple Success (User IS authenticated, no specific session data) -->
        <div class="success-card">
            <div class="success-header">
                <div class="success-pattern"></div>
                <h3 class="mb-0">🎉 Payment Successful!</h3>
                <p class="mb-0 mt-1">Your cart payment has been processed successfully!</p>
            </div>
            
            <div class="success-body">
                <div class="alert alert-success">
                    <i class="bx bx-check-circle me-2"></i>
                    <strong>Payment Confirmed</strong><br>
                    Your payment was successful! You can now access your purchased courses.
                </div>
                
                <div class="payment-summary mt-3 p-3" style="background: #f8f9fa; border-radius: 8px;">
                    <div class="d-flex justify-content-between align-items-center">
                        <span class="fw-bold">Status:</span>
                        <span class="fw-bold text-success">Payment Successful</span>
                    </div>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="courses.php" class="btn-success">
                    <i class="bx bx-book me-2"></i>
                    View My Courses
                </a>
                <a href="dashboard.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Go to Dashboard
                </a>
            </div>
        </div>
        
        <script>
        // Clear cart from localStorage after successful payment
        localStorage.removeItem('cart');
        localStorage.removeItem('appliedCheckoutPromoCode');
        localStorage.removeItem('checkoutPromoDiscount');
        
        // Update cart count
        const cartBadge = document.querySelector('.cart-count');
        if (cartBadge) {
            cartBadge.textContent = '0';
            cartBadge.style.display = 'none';
        }
        
        const cartIconBadge = document.querySelector('#cart-count');
        if (cartIconBadge) {
            cartIconBadge.textContent = '0';
            cartIconBadge.style.display = 'none';
        }
        </script>
        
    <?php elseif ($authError && ($isCartPayment || isset($_GET['course_id']))): ?>
        <!-- Authentication Issue After Payment Success -->
        <div class="success-card">
            <!-- Success Animation -->
            <div class="success-animation">
                <div class="success-glow">
                    <i class="bx bx-check success-icon"></i>
                </div>
            </div>
            
            <div class="success-header">
                <h3>Payment Successful!</h3>
                <p>Your payment has been processed successfully!</p>
            </div>
            
            <div class="success-body">
                <div class="alert alert-success">
                    <i class="bx bx-check-circle me-2"></i>
                    <strong>Payment Confirmed</strong><br>
                    Your payment was successful and has been recorded. You will receive email confirmation and enrollment details shortly.
                </div>
                
                <div class="alert alert-info">
                    <i class="bx bx-info-circle me-2"></i>
                    <strong>Next Steps:</strong><br>
                    Please log in again to access your purchased courses. Your enrollment will be processed automatically.
                </div>
                
                <div class="payment-summary">
                    <div class="d-flex justify-content-between align-items-center">
                        <span style="font-weight: 600; color: #374151;">Status:</span>
                        <span style="font-weight: 700; color: #10b981;">Payment Successful</span>
                    </div>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="../auth/login.php?redirect=student/courses.php" class="btn-success">
                    <i class="bx bx-log-in me-2"></i>
                    Log In to Access Courses
                </a>
                <a href="../index.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Return to Home
                </a>
            </div>
        </div>
        
        <script>
        // Clear cart from localStorage after successful payment
        localStorage.removeItem('cart');
        localStorage.removeItem('appliedCheckoutPromoCode');
        localStorage.removeItem('checkoutPromoDiscount');
        
        // Update cart count
        const cartBadge = document.querySelector('.cart-count');
        if (cartBadge) {
            cartBadge.textContent = '0';
            cartBadge.style.display = 'none';
        }
        
        const cartIconBadge = document.querySelector('#cart-count');
        if (cartIconBadge) {
            cartIconBadge.textContent = '0';
            cartIconBadge.style.display = 'none';
        }
        </script>
        
    <?php elseif ($success && $isCartPayment && isset($_SESSION['cart_fallback_success'])): ?>
        <!-- Cart Payment Fallback Success Display -->
        <div class="success-card">
            <div class="success-header">
                <div class="success-pattern"></div>
                <h3 class="mb-0">🎉 Payment Successful!</h3>
                <p class="mb-0 mt-1">Your payment has been processed successfully. You will be enrolled in your courses shortly.</p>
            </div>
            
            <div class="success-body">
                <div class="alert alert-info">
                    <i class="bx bx-info-circle me-2"></i>
                    <strong>Payment Confirmed</strong><br>
                    Your courses are being processed and you will receive enrollment confirmations via email.
                </div>
                
                <div class="payment-summary mt-3 p-3" style="background: #f8f9fa; border-radius: 8px;">
                    <div class="d-flex justify-content-between align-items-center">
                        <span class="fw-bold">Status:</span>
                        <span class="fw-bold text-success">Payment Successful</span>
                    </div>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="courses.php" class="btn-success">
                    <i class="bx bx-book me-2"></i>
                    View My Courses
                </a>
                <a href="dashboard.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Go to Dashboard
                </a>
            </div>
        </div>
        
        <script>
        // Clear cart from localStorage after successful payment
        localStorage.removeItem('cart');
        localStorage.removeItem('appliedCheckoutPromoCode');
        localStorage.removeItem('checkoutPromoDiscount');
        
        // Update cart count
        const cartBadge = document.querySelector('.cart-count');
        if (cartBadge) {
            cartBadge.textContent = '0';
            cartBadge.style.display = 'none';
        }
        
        const cartIconBadge = document.querySelector('#cart-count');
        if (cartIconBadge) {
            cartIconBadge.textContent = '0';
            cartIconBadge.style.display = 'none';
        }
        </script>
        
    <?php elseif ($success && $isCartPayment && isset($_SESSION['enrolled_courses'])): ?>
        <!-- Cart Payment Success Display -->
        <div class="success-card">
            <div class="success-header">
                <div class="success-pattern"></div>
                <h3 class="mb-0"> Payment Successful!</h3>
                <p class="mb-0 mt-1">You have been enrolled in <?= count($_SESSION['enrolled_courses']) ?> course<?= count($_SESSION['enrolled_courses']) > 1 ? 's' : '' ?>:</p>
            </div>
            
            <div class="success-body">
                <?php foreach ($_SESSION['enrolled_courses'] as $enrolledCourse): ?>
                    <div class="course-info mb-3" style="border-bottom: 1px solid #e9ecef; padding-bottom: 1rem;">
                        <div class="d-flex align-items-center">
                            <img src="<?= htmlspecialchars($enrolledCourse['thumbnail'] ?? '../assets/images/default-thumbnail.jpg') ?>" 
                                 alt="<?= htmlspecialchars($enrolledCourse['title']) ?>" 
                                 class="course-thumbnail me-3"
                                 style="width: 80px; height: 60px; object-fit: cover; border-radius: 8px;"
                                 onerror="this.onerror=null; this.src='../assets/images/default-thumbnail.jpg';">
                            <div class="flex-grow-1">
                                <h5 class="course-title mb-1"><?= htmlspecialchars($enrolledCourse['title']) ?></h5>
                                <p class="course-instructor mb-1">
                                    by <?= htmlspecialchars(($enrolledCourse['first_name'] ?? '') . ' ' . ($enrolledCourse['last_name'] ?? 'Instructor')) ?>
                                </p>
                                <div class="course-category">
                                    <i class="bx bx-category"></i>
                                    <?= htmlspecialchars($enrolledCourse['category_name'] ?? 'General') ?>
                                </div>
                            </div>
                            <div class="course-price">
                                <strong>R<?= number_format($enrolledCourse['price'], 2) ?></strong>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
                
                <div class="payment-summary mt-3 p-3" style="background: #f8f9fa; border-radius: 8px;">
                    <div class="d-flex justify-content-between align-items-center">
                        <span class="fw-bold">Total Paid:</span>
                        <span class="fw-bold text-primary fs-5">R<?= number_format($_SESSION['cart_payment_total'] ?? 0, 2) ?></span>
                    </div>
                    <?php if (isset($_SESSION['payment_discount_percent']) && $_SESSION['payment_discount_percent'] > 0): ?>
                        <div class="d-flex justify-content-between align-items-center text-success">
                            <span>Discount Applied:</span>
                            <span><?= ($_SESSION['payment_discount_percent'] * 100) ?>% off</span>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="success-actions">
                <a href="courses.php" class="btn-success">
                    <i class="bx bx-book me-2"></i>
                    View My Courses
                </a>
                <a href="dashboard.php" class="btn-outline">
                    <i class="bx bx-home me-2"></i>
                    Go to Dashboard
                </a>
            </div>
        </div>
        
    <?php elseif ($success && $course): ?>
        <!-- Single Course Success Display -->
        <div class="success-card">
            <div class="success-animation">
                <div class="success-glow"></div>
                <div class="success-icon">🎉</div>
            </div>
            <h2 class="mb-4">Payment Successful!</h2>
            
            <div class="success-header">
                <div class="success-pattern"></div>
                <h3 class="mb-0">Enrollment Confirmed</h3>
                <p class="mb-0 mt-1">You have been successfully enrolled in:</p>
            </div>
            
            <div class="success-body">
                <div class="course-info">
                    <?php
// Correctly locate course thumbnail based on how thumbnails are stored in the system
$courseId = $course['course_id'] ?? 5; // Default to course 5 if undefined
$courseName = $course['title'] ?? 'Course'; // Get course name for alt text
$thumbnail = $course['thumbnail'] ?? '';
$thumbnailPath = '';

// Check all possible locations for course thumbnails in order of likelihood
// First check: Course-specific thumbnail pattern in uploads/thumbnails
$courseThumbPattern = "../uploads/thumbnails/course_{$courseId}_*.jpeg";
$matchingFiles = glob($courseThumbPattern);
if (!empty($matchingFiles)) {
    $thumbnailPath = $matchingFiles[0]; // Use the first match
}
// Second check: Direct thumbnail path if specified in the course record
else if (!empty($thumbnail)) {
    if (file_exists("../uploads/thumbnails/{$thumbnail}")) {
        $thumbnailPath = "../uploads/thumbnails/{$thumbnail}";
    } else if (file_exists("../uploads/{$thumbnail}")) {
        $thumbnailPath = "../uploads/{$thumbnail}";
    }
}
// Third check: Default course thumbnail for specific courses
else if ($courseId == 5) { // PowerBI course
    // If there's a specific PowerBI image
    if (file_exists("../assets/images/powerbi-course.jpg")) {
        $thumbnailPath = "../assets/images/powerbi-course.jpg";
    } else {
        // Default thumbnail
        $thumbnailPath = "../assets/images/default-thumbnail.jpg";
    }
}
// Final fallback: use default thumbnail
else {
    $thumbnailPath = "../assets/images/default-thumbnail.jpg";
}
?>
<img src="<?= htmlspecialchars($thumbnailPath) ?>" 
     alt="<?= htmlspecialchars($courseName) ?>" 
     class="course-thumbnail"
     onerror="this.onerror=null; this.src='../assets/images/default-thumbnail.jpg';">

                    
                    <div class="course-details">
                        <h5><?= htmlspecialchars($course['title']) ?></h5>
                        
                        <div class="course-meta">
                            <i class="bx bx-category"></i>
                            <span><?= htmlspecialchars($course['category_name'] ?? 'General') ?></span>
                            
                            <i class="bx bx-user"></i>
                            <span>Instructor: <?= htmlspecialchars(($course['first_name'] ?? 'Instructor') . ' ' . ($course['last_name'] ?? '')) ?></span>
                        </div>
                        
                        <p class="course-description">
                            <?= htmlspecialchars(substr($course['description'] ?? 'Start learning today with this amazing course.', 0, 120)) ?>...
                        </p>
                    </div>
                </div>
                
                <div class="payment-details">
                    <div class="row">
                        <div class="col-md-6 text-md-start">
                            <p class="mb-1"><strong>Payment ID:</strong></p>
                            <p class="text-muted"><?= htmlspecialchars($pf_payment_id ?: 'Local test payment') ?></p>
                        </div>
                        <div class="col-md-6 text-md-end">
                            <p class="mb-1"><strong>Amount Paid:</strong></p>
                            <p class="payment-amount">R<?= number_format($course['price'], 2) ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="action-buttons">
            <a href="courses.php" class="btn action-button btn-courses">
                <i class="bx bx-book me-1"></i>
                View My Courses
            </a>
            <a href="course-detail.php?id=<?= $course['course_id'] ?>" class="btn action-button btn-start">
                <i class="bx bx-play-circle me-1"></i>
                Start Learning
            </a>
        </div>
        
        <!-- Confetti Animation -->
        <script>
            // IMMEDIATE CART CLEARING ON PAGE LOAD - MUST RUN FIRST
            <?php if (isset($clearCartNow) && $clearCartNow): ?>
            console.log('🛒 PAYMENT SUCCESS: Clearing cart immediately');
            localStorage.removeItem('cart');  // FIXED: Key is 'cart' not 'cartItems'
            localStorage.removeItem('appliedPromoCode');
            localStorage.removeItem('promoDiscount');
            console.log('✅ Cart cleared from localStorage after successful payment');
            <?php endif; ?>
            
            function createConfetti() {
                // More confetti for a better effect
                for(let i = 0; i < 100; i++) {
                    let confetti = document.createElement('div');
                    confetti.className = 'confetti';
                    confetti.style.left = Math.random() * 100 + 'vw';
                    confetti.style.animationDelay = Math.random() * 3 + 's';
                    confetti.style.width = Math.random() * 15 + 5 + 'px';
                    confetti.style.height = Math.random() * 15 + 5 + 'px';
                    confetti.style.opacity = Math.random() * 0.8 + 0.2;
                    // More vibrant colors
                    confetti.style.backgroundColor = [
                        '#f39c12', '#e74c3c', '#3498db', '#2ecc71', '#9b59b6', 
                        '#1abc9c', '#f1c40f', '#e67e22', '#c0392b', '#8e44ad'
                    ][Math.floor(Math.random() * 10)];
                    document.body.appendChild(confetti);
                }
                
                // Store a session flag to not show again on refresh
                sessionStorage.setItem('payment_success_shown', 'true');
                
                // Remove confetti after animation
                setTimeout(() => {
                    document.querySelectorAll('.confetti').forEach(el => {
                        el.style.opacity = '0';
                        setTimeout(() => el.remove(), 500);
                    });
                }, 4000);
            }
            
            // Only show confetti if not already shown in this session
            if (sessionStorage.getItem('payment_success_shown') !== 'true') {
                createConfetti();
            }
            
            // Store the course ID in local storage for future reference
            localStorage.setItem('last_purchased_course', '<?= $course["course_id"] ?>');
            
            // CLEAR CART AFTER SUCCESSFUL PURCHASE
            <?php if (isset($clearLocalStorageCart) && $clearLocalStorageCart): ?>
            console.log('Payment successful - clearing cart from localStorage');
            localStorage.removeItem('cartItems');
            localStorage.removeItem('appliedPromoCode');
            localStorage.removeItem('promoDiscount');
            console.log('Cart cleared from localStorage after successful purchase');
            <?php endif; ?>
        </script>
        
    <?php elseif ($error): ?>
        <div class="error-card">
            <div class="error-header text-center">
                <div class="mb-3">
                    <i class="bx bx-error-circle" style="font-size: 4rem;"></i>
                </div>
                <h3 class="mb-0">Payment Issue</h3>
            </div>
            
            <div class="error-body">
                <div class="alert alert-warning mb-4">
                    <i class="bx bx-warning me-2"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
                
                <div class="alert alert-info mb-4">
                    <h5><i class="bx bx-help-circle me-2"></i> Need Help?</h5>
                    <p class="mb-0">If you believe you have completed the payment successfully, please contact our support team with your payment reference number.</p>
                </div>
                
                <div class="d-flex justify-content-center gap-3 mt-4">
                    <a href="courses.php" class="btn btn-outline-primary px-4 py-2">
                        <i class="bx bx-arrow-back me-1"></i> Back to Courses
                    </a>
                    <button class="btn btn-outline-secondary px-4 py-2" onclick="window.location.reload()">
                        <i class="bx bx-refresh me-1"></i> Refresh Page
                    </button>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php include_once 'components/footer.php'; ?>
