<?php
/**
 * Process Checkout Payment
 * Handles PayFast integration for multiple course purchases
 */

// Authentication and session
require_once '../auth/functions.php';
requireRole('student');

// Include PayFast functions
require_once '../payments/payfast.php';

// Initialize variables
$error = '';
$success = '';
$user = null;
$cartItems = [];
$paymentFormData = [];
$debugInfo = []; // Store debug information for display

// Check if database connection is available
require_once '../config/db_connect.php';
require_once '../includes/id_encryption.php';

// Get the current user ID from session
$user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

if ($user_id && isset($conn)) {
    // Get user details
    $userQuery = "SELECT user_id, first_name, last_name, email FROM users WHERE user_id = ?";
    $stmt = $conn->prepare($userQuery);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $userResult = $stmt->get_result();
    
    if ($userResult->num_rows > 0) {
        $user = $userResult->fetch_assoc();
    }
    
    // Get cart items from POST data
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cart_items'])) {
        $cartData = json_decode($_POST['cart_items'], true);
        
        // DEBUG: Store cart data for display
        $debugInfo['cart_data_received'] = $cartData;
        $debugInfo['user_id'] = $user_id;
        
        if (is_array($cartData) && !empty($cartData)) {
            // Validate each course in cart and get fresh data from database
            $courseIds = [];
            foreach ($cartData as $item) {
                if (isset($item['id'])) {
                    // Handle both numeric IDs and encrypted IDs
                    if (is_numeric($item['id'])) {
                        $courseIds[] = intval($item['id']);
                    } else if (is_string($item['id'])) {
                        // Try to decrypt encrypted course ID using IdEncryption class
                        $decryptedId = IdEncryption::decrypt($item['id']);
                        if ($decryptedId !== false && is_numeric($decryptedId)) {
                            $courseIds[] = intval($decryptedId);
                            $debugInfo['id_decrypt_attempts'][] = [
                                'original' => $item['id'],
                                'decrypted' => $decryptedId,
                                'is_numeric' => is_numeric($decryptedId),
                                'result' => 'SUCCESS'
                            ];
                        } else {
                            // If decryption fails, log for debugging
                            $debugInfo['id_decrypt_attempts'][] = [
                                'original' => $item['id'],
                                'decrypted' => $decryptedId,
                                'is_numeric' => is_numeric($decryptedId),
                                'result' => 'FAILED'
                            ];
                        }
                    }
                }
            }
            
            $debugInfo['course_ids_to_validate'] = $courseIds;
            
            if (!empty($courseIds)) {
                // Get all courses from database
                $placeholders = str_repeat('?,', count($courseIds) - 1) . '?';
                $courseQuery = "
                    SELECT c.course_id, c.title, c.description, c.price, c.thumbnail,
                           u.first_name as teacher_first_name, u.last_name as teacher_last_name,
                           cat.category_name
                    FROM courses c 
                    JOIN users u ON c.teacher_id = u.user_id
                    JOIN categories cat ON c.category_id = cat.category_id
                    WHERE c.course_id IN ($placeholders) AND c.is_published = 1
                ";
                
                $stmt = $conn->prepare($courseQuery);
                $types = str_repeat('i', count($courseIds));
                $stmt->bind_param($types, ...$courseIds);
                $stmt->execute();
                $courseResult = $stmt->get_result();
                
                $coursesFound = $courseResult->num_rows;
                $debugInfo['courses_found_in_db'] = $coursesFound;
                $debugInfo['course_validation'] = [];
                
                while ($course = $courseResult->fetch_assoc()) {
                    $courseDebug = [
                        'course_id' => $course['course_id'],
                        'title' => $course['title'],
                        'price' => $course['price']
                    ];
                    
                    // Check if user is already enrolled in this course
                    $enrollmentQuery = "
                        SELECT enrollment_id 
                        FROM enrollments 
                        WHERE student_id = ? AND course_id = ?
                    ";
                    
                    $enrollStmt = $conn->prepare($enrollmentQuery);
                    $enrollStmt->bind_param("ii", $user_id, $course['course_id']);
                    $enrollStmt->execute();
                    $enrollmentResult = $enrollStmt->get_result();
                    
                    $isEnrolled = $enrollmentResult->num_rows > 0;
                    $courseDebug['is_enrolled'] = $isEnrolled;
                    
                    if ($enrollmentResult->num_rows == 0) {
                        // Check if user has already paid for this course
                        $hasPurchased = hasUserPurchasedCourse($user_id, $course['course_id']);
                        $courseDebug['has_purchased'] = $hasPurchased;
                        
                        if (!$hasPurchased) {
                            $cartItems[] = $course;
                            $courseDebug['result'] = 'ADDED TO CART';
                        } else {
                            $courseDebug['result'] = 'REJECTED - Already purchased';
                        }
                    } else {
                        $courseDebug['has_purchased'] = 'N/A (already enrolled)';
                        $courseDebug['result'] = 'REJECTED - Already enrolled';
                    }
                    
                    $debugInfo['course_validation'][] = $courseDebug;
                }
                
                $debugInfo['final_valid_cart_items'] = count($cartItems);
            }
        } else {
            $debugInfo['error'] = 'Invalid or empty cart data';
        }
    }
    
    // Process payment if we have valid cart items
    if (!empty($cartItems) && $user && !$error) {
        // Calculate total amount
        $totalAmount = 0;
        $courseNames = [];
        $courseIds = [];
        
        foreach ($cartItems as $course) {
            $totalAmount += floatval($course['price']);
            $courseNames[] = $course['title'];
            $courseIds[] = $course['course_id'];
        }
        
        // Apply promo code discount if provided
        $discountPercent = 0;
        if (isset($_POST['promo_code']) && !empty($_POST['promo_code'])) {
            $promoCode = strtoupper(trim($_POST['promo_code']));
            $validCodes = [
                'WELCOME10' => 0.10,
                'STUDENT20' => 0.20,
                'PANADITE15' => 0.15
            ];
            
            if (isset($validCodes[$promoCode])) {
                $discountPercent = $validCodes[$promoCode];
                $totalAmount = $totalAmount * (1 - $discountPercent);
            }
        }
        
        // Check for available referral discounts (CONSISTENT WITH FRONTEND)
        $referral_discount = 0;
        $discountsQuery = "SELECT * FROM student_referral_discounts 
                          WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                          ORDER BY discount_percentage DESC LIMIT 1";
        $discountStmt = $conn->prepare($discountsQuery);
        if ($discountStmt) {
            $discountStmt->bind_param("i", $user_id);
            $discountStmt->execute();
            $discountResult = $discountStmt->get_result();
            if ($discountResult->num_rows > 0) {
                $discount_data = $discountResult->fetch_assoc();
                
                // Calculate discount using SAME LOGIC as frontend
                $discount_amount = min(
                    ($totalAmount * $discount_data['discount_percentage'] / 100),
                    $discount_data['discount_amount']
                );
                $referral_discount = $discount_amount;
                $totalAmount -= $referral_discount;
                
                // Store discount info for later use
                $_SESSION['applied_referral_discount'] = [
                    'discount_id' => $discount_data['id'],
                    'amount' => $referral_discount
                ];
                
                $debugInfo['referral_discount_applied'] = $referral_discount;
            }
        }

        if ($totalAmount > 0) {
            // Create item for PayFast
            $item = [
                'title' => count($cartItems) . ' Course' . (count($cartItems) > 1 ? 's' : '') . ' - ' . implode(', ', array_slice($courseNames, 0, 2)) . (count($courseNames) > 2 ? '...' : ''),
                'amount' => $totalAmount,
                'item_type' => 'courses',
                'item_id' => 'cart_' . time()
            ];
            
            // Store payment info in session for return page
            $_SESSION['payment_amount'] = $totalAmount;
            $_SESSION['payment_course_titles'] = $courseNames;
            $_SESSION['payment_course_ids'] = $courseIds;
            $_SESSION['payment_discount_percent'] = $discountPercent;
            $_SESSION['payment_promo_code'] = isset($_POST['promo_code']) ? $_POST['promo_code'] : null;
            $_SESSION['cart_payment_data'] = $cartItems;
            
            // Generate return and cancel URLs
            global $site_config;
            $returnUrl = $site_config['site_url'] . '/student/payment-return.php?type=cart';
            $cancelUrl = $site_config['site_url'] . '/student/checkout.php';
            $notifyUrl = $site_config['site_url'] . '/payments/notify.php';
            
            try {
                $paymentFormData = generatePayfastFormData($item, $user, $returnUrl, $cancelUrl, $notifyUrl);
                
                // CLEAR CART IMMEDIATELY AFTER PAYFAST PAYMENT IS INITIATED
                // This ensures cart is cleared in the current valid session
                $_SESSION['cart'] = array(); // Clear session cart
                error_log("CART CLEARING: Cleared session cart during checkout process for user " . $user_id);
                
            } catch (Exception $e) {
                $error = "Payment system error: " . $e->getMessage();
            }
        } else {
            $error = "Invalid total amount.";
        }
    } else if (empty($cartItems)) {
        $error = "No valid courses found in cart or all courses are already purchased.";
    }
} else {
    $error = "Unable to process your request. Please try again.";
}

// Include layout components
include_once 'components/header.php';
?>

<style>
/* Payment Processing Page Styling */
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
}

.payment-container {
    max-width: 800px;
    margin: 0 auto;
    padding: 2rem;
    min-height: calc(100vh - 100px);
    display: flex;
    align-items: center;
    justify-content: center;
}

.payment-card {
    background: white;
    border-radius: 20px;
    padding: 3rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    text-align: center;
    width: 100%;
}

.payment-header h1 {
    color: #2c3e50;
    font-size: 2.5rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
}

.payment-header p {
    color: #6c757d;
    font-size: 1.1rem;
    margin-bottom: 2rem;
}

.course-summary {
    background: #f8f9fa;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 2rem;
    text-align: left;
}

.course-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #e9ecef;
}

.course-item:last-child {
    border-bottom: none;
}

.course-name {
    font-weight: 600;
    color: #495057;
}

.course-price {
    font-weight: 600;
    color: #f59e0b;
}

.total-row {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 2px solid #dee2e6;
    font-size: 1.2rem;
    font-weight: bold;
}

.payment-button {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border: none;
    border-radius: 12px;
    color: white;
    padding: 1rem 2rem;
    font-size: 1.2rem;
    font-weight: 600;
    width: 100%;
    transition: all 0.3s ease;
    cursor: pointer;
}

.payment-button:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
}

.error-message {
    background: #fee;
    border: 1px solid #f5c6cb;
    border-radius: 8px;
    padding: 1rem;
    color: #721c24;
    margin-bottom: 1rem;
}

.back-link {
    display: inline-block;
    margin-top: 1rem;
    color: #6c757d;
    text-decoration: none;
    transition: color 0.3s ease;
}

.back-link:hover {
    color: #495057;
}

.payfast-logo {
    margin-top: 1rem;
    opacity: 0.7;
}
</style>

<div class="payment-container">
    <div class="payment-card">
        <div class="payment-header">
            <h1><i class="bx bx-credit-card me-3"></i>Complete Purchase</h1>
            <p>Secure payment for your selected courses</p>
        </div>

        <?php if ($error): ?>
            <div class="error-message">
                <i class="bx bx-error-circle me-2"></i>
                <?= htmlspecialchars($error) ?>
            </div>
            <a href="checkout.php" class="back-link">
                <i class="bx bx-arrow-back me-1"></i>Back to Checkout
            </a>
        <?php elseif (!empty($cartItems) && !empty($paymentFormData)): ?>
            <div class="course-summary">
                <h5><i class="bx bx-list-ul me-2"></i>Order Summary</h5>
                <?php 
                $subtotal = 0;
                foreach ($cartItems as $course): 
                    $subtotal += floatval($course['price']);
                ?>
                    <div class="course-item">
                        <span class="course-name"><?= htmlspecialchars($course['title']) ?></span>
                        <span class="course-price">R<?= number_format($course['price'], 2) ?></span>
                    </div>
                <?php endforeach; ?>
                
                <?php if ($discountPercent > 0): ?>
                    <div class="course-item">
                        <span class="course-name text-success">
                            <i class="bx bx-tag me-1"></i>
                            Discount (<?= ($discountPercent * 100) ?>%)
                        </span>
                        <span class="course-price text-success">-R<?= number_format($subtotal * $discountPercent, 2) ?></span>
                    </div>
                <?php endif; ?>
                
                <div class="course-item total-row">
                    <span class="course-name">Total:</span>
                    <span class="course-price text-primary">R<?= number_format($totalAmount, 2) ?></span>
                </div>
            </div>

            <form action="<?= $paymentFormData['url'] ?>" method="post" id="payfast-form">
                <?php foreach ($paymentFormData['data'] as $key => $value): ?>
                    <input type="hidden" name="<?= htmlspecialchars($key) ?>" value="<?= htmlspecialchars($value) ?>">
                <?php endforeach; ?>
                
                <p class="small text-muted mb-3">
                    By clicking "Pay Now", you agree to our Terms of Service and confirm that you want to purchase these courses.
                </p>
                
                <button type="submit" class="payment-button">
                    <i class="bx bx-credit-card me-2"></i>
                    Pay Now - R<?= number_format($totalAmount, 2) ?>
                </button>
            </form>
            
            <div class="payfast-logo">
                <img src="../assets/images/payfast-logo.png" alt="Secured by PayFast" height="30" 
                     onerror="this.onerror=null; this.src='https://www.payfast.co.za/assets/images/logos/payfast-badge.svg';">
            </div>
            
            <a href="checkout.php" class="back-link">
                <i class="bx bx-arrow-back me-1"></i>Back to Checkout
            </a>
        <?php else: ?>
            <div class="error-message">
                <i class="bx bx-warning me-2"></i>
                No courses found for payment or payment system unavailable.
            </div>
            <a href="checkout.php" class="back-link">
                <i class="bx bx-arrow-back me-1"></i>Back to Checkout
            </a>
        <?php endif; ?>
    </div>
</div>

<script>
// Store cart information before payment
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('payfast-form');
    if (form) {
        form.addEventListener('submit', function() {
            // CLEAR CART IMMEDIATELY WHEN PAYFAST FORM IS SUBMITTED
            // This ensures cart is cleared in the current session before any redirects
            console.log('🛒 CLEARING CART: PayFast payment initiated - clearing localStorage');
            localStorage.removeItem('cart');  // Clear cart items
            localStorage.removeItem('appliedPromoCode');  // Clear promo codes
            localStorage.removeItem('promoDiscount');  // Clear discounts
            console.log('✅ CART CLEARED: localStorage cleared during checkout process');
            
            // Store payment initiation timestamp
            localStorage.setItem('payment_initiated', Date.now());
            
            // Store course info for success page
            const courseData = <?= json_encode($cartItems) ?>;
            localStorage.setItem('payment_courses', JSON.stringify(courseData));
        });
    }
});
</script>

<?php include_once 'components/footer.php'; ?>
