<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

$page_title = 'Profile';
$user_id = $_SESSION['user_id'];

// Get user information
$user_info = [];
if (isset($conn)) {
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_info = $result->fetch_assoc();
        $stmt->close();
    }
}

// Get academic stats
$enrolled_count = 0;
$completed_count = 0;

if (isset($conn)) {
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $enrolled_count = $row['count'];
        }
        $stmt->close();
    }
    
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ? AND status = 'completed'");
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $completed_count = $row['count'];
        }
        $stmt->close();
    }
}

// Handle profile picture upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_picture']) && isset($_FILES['profile_picture'])) {
    $upload_success = false;
    $upload_error = '';
    
    $file = $_FILES['profile_picture'];
    
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $upload_error = 'File upload error occurred.';
    } else {
        // Validate file type
        $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $file_type = mime_content_type($file['tmp_name']);
        
        if (!in_array($file_type, $allowed_types)) {
            $upload_error = 'Only JPG, PNG, and GIF images are allowed.';
        } elseif ($file['size'] > 5 * 1024 * 1024) { // 5MB limit
            $upload_error = 'File size must be less than 5MB.';
        } else {
            // Create uploads directory if it doesn't exist
            $upload_dir = '../uploads/profile_pictures/';
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            // Generate unique filename
            $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'profile_' . $user_id . '_' . time() . '.' . $file_extension;
            $filepath = $upload_dir . $filename;
            
            // Delete old profile picture if exists
            if (!empty($user_info['profile_picture'])) {
                $old_file = '../uploads/profile_pictures/' . $user_info['profile_picture'];
                if (file_exists($old_file)) {
                    unlink($old_file);
                }
            }
            
            // Move uploaded file
            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                // Update database
                $stmt = $conn->prepare("UPDATE users SET profile_picture = ? WHERE user_id = ?");
                if ($stmt) {
                    $stmt->bind_param("si", $filename, $user_id);
                    if ($stmt->execute()) {
                        $user_info['profile_picture'] = $filename;
                        $success_message = 'Profile picture updated successfully!';
                        $upload_success = true;
                    } else {
                        $upload_error = 'Database update failed.';
                        unlink($filepath); // Remove uploaded file if DB update fails
                    }
                    $stmt->close();
                } else {
                    $upload_error = 'Database prepare error.';
                    unlink($filepath);
                }
            } else {
                $upload_error = 'Failed to save uploaded file.';
            }
        }
    }
    
    if (!$upload_success && !empty($upload_error)) {
        $error_message = $upload_error;
    }
}

// Handle profile picture removal
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['remove_picture'])) {
    if (!empty($user_info['profile_picture'])) {
        // Delete file from server
        $file_path = '../uploads/profile_pictures/' . $user_info['profile_picture'];
        if (file_exists($file_path)) {
            unlink($file_path);
        }
        
        // Update database
        $stmt = $conn->prepare("UPDATE users SET profile_picture = NULL WHERE user_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            if ($stmt->execute()) {
                $user_info['profile_picture'] = null;
                $success_message = 'Profile picture removed successfully!';
            } else {
                $error_message = 'Failed to remove profile picture from database.';
            }
            $stmt->close();
        }
    }
}

// Handle updates
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    $firstName = trim($_POST['first_name']);
    $lastName = trim($_POST['last_name']);
    $email = trim($_POST['email']);
    $bio = trim($_POST['bio']);
    $cellNumber = trim($_POST['cell_number']);
    $suburb = trim($_POST['suburb']);
    $province = trim($_POST['province']);
    $communicationMethod = trim($_POST['communication_method']);
    $trainingPreference = trim($_POST['training_preference']);
    $jobFieldInterest = trim($_POST['job_field_interest']);
    $educationStatus = trim($_POST['education_status']);
    
    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = "Please enter a valid email address.";
    } else {
        // Check if email is already taken by another user
        $checkStmt = $conn->prepare("SELECT user_id FROM users WHERE email = ? AND user_id != ?");
        if ($checkStmt) {
            $checkStmt->bind_param("si", $email, $user_id);
            $checkStmt->execute();
            $result = $checkStmt->get_result();
            
            if ($result->num_rows > 0) {
                $error_message = "This email address is already registered to another account.";
            } else {
                // Update user information
                $sql = "UPDATE users SET first_name = ?, last_name = ?, email = ?, bio = ?, cell_number = ?, suburb = ?, province = ?, communication_method = ?, training_preference = ?, job_field_interest = ?, education_status = ? WHERE user_id = ?";
                
                $stmt = $conn->prepare($sql);
                if ($stmt) {
                    $stmt->bind_param("sssssssssssi", $firstName, $lastName, $email, $bio, $cellNumber, $suburb, $province, $communicationMethod, $trainingPreference, $jobFieldInterest, $educationStatus, $user_id);
                    if ($stmt->execute()) {
                        $success_message = "Profile updated successfully!";
                        // Update the user_info array with new values
                        $user_info['first_name'] = $firstName;
                        $user_info['last_name'] = $lastName;
                        $user_info['email'] = $email;
                        $user_info['bio'] = $bio;
                        $user_info['cell_number'] = $cellNumber;
                        $user_info['suburb'] = $suburb;
                        $user_info['province'] = $province;
                        $user_info['communication_method'] = $communicationMethod;
                        $user_info['training_preference'] = $trainingPreference;
                        $user_info['job_field_interest'] = $jobFieldInterest;
                        $user_info['education_status'] = $educationStatus;
                    } else {
                        $error_message = "Failed to update profile. Please try again.";
                    }
                    $stmt->close();
                } else {
                    $error_message = "Database prepare error: " . $conn->error;
                }
            }
            $checkStmt->close();
        }
    }
}

include_once 'components/header.php';
?>

<style>
/* Modern Profile Styling */
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
}

.profile-wrapper {
    padding: 2rem 0;
    min-height: 100vh;
}

.profile-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0 2rem;
}

/* Clean Header */
.profile-header {
    background: whitesmoke;
    padding: 1.5rem 0;
    margin-bottom: 3rem;
    border-bottom: 4px solid #dc3545;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}

.profile-header h1 {
    font-size: 1.5rem;
    font-weight: 600;
    margin: 0;
    color: #000000;
    text-align: center;
}

/* Main Grid */
.profile-grid {
    display: grid;
    grid-template-columns: 350px 1fr;
    gap: 2rem;
}

/* Profile Card */
.profile-card {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    height: fit-content;
}

.avatar-section {
    text-align: center;
    margin-bottom: 2rem;
}

.profile-user-avatar {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 3rem;
    color: white;
    font-weight: 700;
    margin: 0 auto 1rem;
    box-shadow: 0 15px 35px rgba(78,115,223,0.3);
    overflow: hidden;
    position: relative;
}

.profile-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: 50%;
}

.profile-picture-controls {
    margin: 1rem 0;
    text-align: center;
}

.picture-form {
    margin-bottom: 0.5rem;
}

.file-upload-wrapper {
    margin-bottom: 0.5rem;
}

.file-input {
    display: none;
}

.file-label {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border: 2px dashed #6c757d;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.9rem;
    color: #495057;
}

.file-label:hover {
    border-color: #4e73df;
    color: #4e73df;
    background: linear-gradient(135deg, #f8f9ff 0%, #e6f0ff 100%);
}

.btn-upload {
    background: linear-gradient(135deg, #1cc88a 0%, #13855c 100%);
    color: white;
    font-size: 0.85rem;
    padding: 0.5rem 1rem;
}

.btn-upload:hover {
    background: linear-gradient(135deg, #13855c 0%, #0f6848 100%);
}

.btn-remove {
    background: linear-gradient(135deg, #e74a3b 0%, #c0392b 100%);
    color: white;
    font-size: 0.85rem;
    padding: 0.5rem 1rem;
}

.btn-remove:hover {
    background: linear-gradient(135deg, #c0392b 0%, #a93226 100%);
}

.user-name {
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.user-email {
    color: #64748b;
    font-size: 1rem;
    margin-bottom: 1.5rem;
}

.stats-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.stat-card {
    background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
    border-radius: 16px;
    padding: 1.5rem;
    text-align: center;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
}

.stat-card:hover {
    border-color: #4e73df;
    transform: translateY(-2px);
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #4e73df;
    margin-bottom: 0.25rem;
}

.stat-label {
    color: #64748b;
    font-size: 0.9rem;
    font-weight: 500;
}

/* Content Card */
.content-card {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.section-title {
    font-size: 1.4rem;
    font-weight: 600;
    color: #2d3748;
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.section-subtitle {
    font-size: 1.2rem;
    font-weight: 600;
    color: #4a5568;
    margin: 2rem 0 1rem 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding-bottom: 0.5rem;
    border-bottom: 2px solid #e2e8f0;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.5rem;
    font-size: 0.95rem;
}

.form-input {
    padding: 1rem 1.25rem;
    border: 2px solid #e2e8f0;
    border-radius: 16px;
    font-size: 1rem;
    transition: all 0.3s ease;
    background: #f8fafc;
}

.form-input:focus {
    outline: none;
    border-color: #4e73df;
    box-shadow: 0 0 0 3px rgba(78, 115, 223, 0.1);
}

.form-input:disabled {
    background: #f8f9fa;
    color: #6c757d;
    cursor: not-allowed;
}

.form-note {
    font-size: 0.85rem;
    color: #6c757d;
    margin-top: 0.25rem;
    font-style: italic;
}

textarea.form-input {
    resize: vertical;
    min-height: 120px;
    font-family: inherit;
}

/* Submit Button */
.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 2rem;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    text-decoration: none;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
    margin-top: 2rem;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #224abe 0%, #1e3f99 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(78, 115, 223, 0.4);
}
    font-size: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(78,115,223,0.3);
}

.btn:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(78,115,223,0.4);
}

.success-alert {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
    padding: 1rem 1.5rem;
    border-radius: 12px;
    margin-bottom: 2rem;
    border: 1px solid #c3e6cb;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    box-shadow: 0 4px 12px rgba(21, 87, 36, 0.1);
}

.error-alert {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
    padding: 1rem 1.5rem;
    border-radius: 12px;
    margin-bottom: 2rem;
    border: 1px solid #f5c6cb;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    box-shadow: 0 4px 12px rgba(114, 28, 36, 0.1);
}

/* Responsive Design */
@media (max-width: 1024px) {
    .profile-grid {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .profile-container {
        padding: 0 1rem;
    }
    
    .profile-header h1 {
        font-size: 2.5rem;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .stats-grid {
        grid-template-columns: repeat(4, 1fr);
    }
}

@media (max-width: 480px) {
    .profile-header h1 {
        font-size: 2rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr 1fr;
    }
    
    .user-avatar {
        width: 100px;
        height: 100px;
        font-size: 2.5rem;
    }
}
</style>

<div class="profile-wrapper">
    <div class="profile-container">
        <!-- Header -->
        <div class="profile-header">
            <h1>My Profile</h1>
        </div>

        <!-- Success Message -->
        <?php if ($success_message): ?>
            <div class="success-alert">
                <i class="bx bx-check-circle"></i>
                <?= $success_message ?>
            </div>
        <?php endif; ?>
        
        <!-- Error Message -->
        <?php if ($error_message): ?>
            <div class="error-alert">
                <i class="bx bx-error-circle"></i>
                <?= $error_message ?>
            </div>
        <?php endif; ?>
        


        <!-- Main Grid -->
        <div class="profile-grid">
            <!-- Profile Card -->
            <div class="profile-card">
                <div class="avatar-section">
                    <div class="profile-user-avatar">
                        <?php if (!empty($user_info['profile_picture'])): ?>
                            <img src="../uploads/profile_pictures/<?= htmlspecialchars($user_info['profile_picture']) ?>" 
                                 alt="Profile Picture" class="profile-image">
                        <?php else: ?>
                            <?= strtoupper(substr($user_info['first_name'] ?? 'S', 0, 1) . substr($user_info['last_name'] ?? 'T', 0, 1)) ?>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Profile Picture Upload Form -->
                    <div class="profile-picture-controls">
                        <?php if (!empty($user_info['profile_picture'])): ?>
                            <form method="POST" class="picture-form">
                                <button type="submit" name="remove_picture" class="btn btn-remove" 
                                        onclick="return confirm('Are you sure you want to remove your profile picture?')">
                                    <i class="bx bx-trash"></i> Remove Picture
                                </button>
                            </form>
                        <?php endif; ?>
                        
                        <form method="POST" enctype="multipart/form-data" class="picture-form">
                            <div class="file-upload-wrapper">
                                <input type="file" id="profile_picture" name="profile_picture" 
                                       accept="image/jpeg,image/jpg,image/png,image/gif" 
                                       class="file-input" required>
                                <label for="profile_picture" class="file-label">
                                    <i class="bx bx-camera"></i>
                                    <?= !empty($user_info['profile_picture']) ? 'Change Picture' : 'Upload Picture' ?>
                                </label>
                            </div>
                            <button type="submit" name="upload_picture" class="btn btn-upload">
                                <i class="bx bx-upload"></i> Save Picture
                            </button>
                        </form>
                    </div>
                    
                    <div class="user-name">
                        <?= htmlspecialchars(($user_info['first_name'] ?? '') . ' ' . ($user_info['last_name'] ?? '')) ?>
                    </div>
                    <div class="user-email">
                        <?= htmlspecialchars($user_info['email'] ?? '') ?>
                    </div>
                </div>
                
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-number"><?= $enrolled_count ?></div>
                        <div class="stat-label">Enrolled</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?= $completed_count ?></div>
                        <div class="stat-label">Completed</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?= $enrolled_count > 0 ? round(($completed_count / $enrolled_count) * 100) : 0 ?>%</div>
                        <div class="stat-label">Progress</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?= date('M Y', strtotime($user_info['created_at'] ?? 'now')) ?></div>
                        <div class="stat-label">Joined</div>
                    </div>
                </div>
            </div>

            <!-- Content Card -->
            <div class="content-card">
                <h2 class="section-title">
                    <i class="bx bx-edit"></i>Personal Information
                </h2>
                
                <form method="POST">
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">First Name</label>
                            <input type="text" name="first_name" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['first_name'] ?? '') ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Last Name</label>
                            <input type="text" name="last_name" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['last_name'] ?? '') ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Email Address</label>
                            <input type="email" name="email" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['email'] ?? '') ?>" required>
                            <div class="form-note">Make sure this email is valid and accessible</div>
                        </div>
                    </div>

                    <!-- Student Information Section -->
                    <h3 class="section-subtitle">
                        <i class="bx bx-user-detail"></i>Student Details
                    </h3>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">Title</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['title'] ?? '') ?>" readonly>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Cell Number</label>
                            <input type="tel" name="cell_number" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['cell_number'] ?? '') ?>"
                                   placeholder="e.g., 0823456789">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">ID Number</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['id_number'] ?? '') ?>" readonly>
                            <div class="form-note">ID number cannot be changed for security reasons</div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Suburb</label>
                            <input type="text" name="suburb" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['suburb'] ?? '') ?>"
                                   placeholder="e.g., Pretoria Central">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Province</label>
                            <select name="province" class="form-input">
                                <option value="">Select Province</option>
                                <option value="Eastern Cape" <?= ($user_info['province'] ?? '') === 'Eastern Cape' ? 'selected' : '' ?>>Eastern Cape</option>
                                <option value="Free State" <?= ($user_info['province'] ?? '') === 'Free State' ? 'selected' : '' ?>>Free State</option>
                                <option value="Gauteng" <?= ($user_info['province'] ?? '') === 'Gauteng' ? 'selected' : '' ?>>Gauteng</option>
                                <option value="KwaZulu-Natal" <?= ($user_info['province'] ?? '') === 'KwaZulu-Natal' ? 'selected' : '' ?>>KwaZulu-Natal</option>
                                <option value="Limpopo" <?= ($user_info['province'] ?? '') === 'Limpopo' ? 'selected' : '' ?>>Limpopo</option>
                                <option value="Mpumalanga" <?= ($user_info['province'] ?? '') === 'Mpumalanga' ? 'selected' : '' ?>>Mpumalanga</option>
                                <option value="Northern Cape" <?= ($user_info['province'] ?? '') === 'Northern Cape' ? 'selected' : '' ?>>Northern Cape</option>
                                <option value="North West" <?= ($user_info['province'] ?? '') === 'North West' ? 'selected' : '' ?>>North West</option>
                                <option value="Western Cape" <?= ($user_info['province'] ?? '') === 'Western Cape' ? 'selected' : '' ?>>Western Cape</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Communication Method</label>
                            <select name="communication_method" class="form-input">
                                <option value="">Select Method</option>
                                <option value="Email" <?= ($user_info['communication_method'] ?? '') === 'Email' ? 'selected' : '' ?>>Email</option>
                                <option value="WhatsApp" <?= ($user_info['communication_method'] ?? '') === 'WhatsApp' ? 'selected' : '' ?>>WhatsApp</option>
                                <option value="SMS" <?= ($user_info['communication_method'] ?? '') === 'SMS' ? 'selected' : '' ?>>SMS</option>
                                <option value="Phone Call" <?= ($user_info['communication_method'] ?? '') === 'Phone Call' ? 'selected' : '' ?>>Phone Call</option>
                            </select>
                        </div>
                    </div>

                    <!-- Learning Preferences Section -->
                    <h3 class="section-subtitle">
                        <i class="bx bx-laptop"></i>Learning Preferences
                    </h3>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">Owns Laptop/PC</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['owns_laptop'] ?? '') ?>" readonly>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Training Preference</label>
                            <select name="training_preference" class="form-input">
                                <option value="">Select Preference</option>
                                <option value="Online" <?= ($user_info['training_preference'] ?? '') === 'Online' ? 'selected' : '' ?>>Online</option>
                                <option value="Onsite" <?= ($user_info['training_preference'] ?? '') === 'Onsite' ? 'selected' : '' ?>>Onsite</option>
                                <option value="Hybrid" <?= ($user_info['training_preference'] ?? '') === 'Hybrid' ? 'selected' : '' ?>>Hybrid</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Job Field Interest</label>
                            <select name="job_field_interest" class="form-input">
                                <option value="">Select Job Field</option>
                                <option value="Data Capturing" <?= ($user_info['job_field_interest'] ?? '') === 'Data Capturing' ? 'selected' : '' ?>>Data Capturing</option>
                                <option value="Administrative Assistant" <?= ($user_info['job_field_interest'] ?? '') === 'Administrative Assistant' ? 'selected' : '' ?>>Administrative Assistant</option>
                                <option value="Customer Service" <?= ($user_info['job_field_interest'] ?? '') === 'Customer Service' ? 'selected' : '' ?>>Customer Service</option>
                                <option value="IT Support" <?= ($user_info['job_field_interest'] ?? '') === 'IT Support' ? 'selected' : '' ?>>IT Support</option>
                                <option value="Digital Marketing" <?= ($user_info['job_field_interest'] ?? '') === 'Digital Marketing' ? 'selected' : '' ?>>Digital Marketing</option>
                                <option value="Web Development" <?= ($user_info['job_field_interest'] ?? '') === 'Web Development' ? 'selected' : '' ?>>Web Development</option>
                                <option value="Graphic Design" <?= ($user_info['job_field_interest'] ?? '') === 'Graphic Design' ? 'selected' : '' ?>>Graphic Design</option>
                                <option value="Project Management" <?= ($user_info['job_field_interest'] ?? '') === 'Project Management' ? 'selected' : '' ?>>Project Management</option>
                                <option value="Other" <?= ($user_info['job_field_interest'] ?? '') === 'Other' ? 'selected' : '' ?>>Other</option>
                            </select>
                        </div>
                        
                        <?php if (!empty($user_info['job_field_other'])): ?>
                        <div class="form-group">
                            <label class="form-label">Other Job Field</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['job_field_other'] ?? '') ?>" readonly>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Background Information Section -->
                    <h3 class="section-subtitle">
                        <i class="bx bx-briefcase"></i>Background Information
                    </h3>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">Education Status</label>
                            <select name="education_status" class="form-input">
                                <option value="">Select Education Level</option>
                                <option value="High School" <?= ($user_info['education_status'] ?? '') === 'High School' ? 'selected' : '' ?>>High School</option>
                                <option value="University Degree/Diploma" <?= ($user_info['education_status'] ?? '') === 'University Degree/Diploma' ? 'selected' : '' ?>>University Degree/Diploma</option>
                                <option value="Postgraduate" <?= ($user_info['education_status'] ?? '') === 'Postgraduate' ? 'selected' : '' ?>>Postgraduate</option>
                                <option value="Professional Certification" <?= ($user_info['education_status'] ?? '') === 'Professional Certification' ? 'selected' : '' ?>>Professional Certification</option>
                                <option value="Other" <?= ($user_info['education_status'] ?? '') === 'Other' ? 'selected' : '' ?>>Other</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Employment Status</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['employment_status'] ?? '') ?>" readonly>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">How You Heard About Us</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['heard_about_us'] ?? '') ?>" readonly>
                        </div>
                        
                        <?php if (!empty($user_info['referral_person_name'])): ?>
                        <div class="form-group">
                            <label class="form-label">Referred By</label>
                            <input type="text" class="form-input" 
                                   value="<?= htmlspecialchars($user_info['referral_person_name'] ?? '') ?>" readonly>
                        </div>
                        <?php endif; ?>
                        
                        <div class="form-group full-width">
                            <label class="form-label">Bio</label>
                            <textarea name="bio" class="form-input" 
                                      placeholder="Tell us about yourself, your interests, and your learning goals..."><?= htmlspecialchars($user_info['bio'] ?? '') ?></textarea>
                        </div>
                    </div>
                    
                    <button type="submit" name="update_profile" class="btn btn-primary">
                        <i class="bx bx-save"></i>
                        Update Profile
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include_once 'components/footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Profile picture file input handling
    const fileInput = document.getElementById('profile_picture');
    const fileLabel = document.querySelector('.file-label');
    const uploadBtn = document.querySelector('.btn-upload');
    
    if (fileInput) {
        fileInput.addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                // Validate file size (5MB limit)
                if (file.size > 5 * 1024 * 1024) {
                    alert('File size must be less than 5MB');
                    this.value = '';
                    return;
                }
                
                // Validate file type
                const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                if (!allowedTypes.includes(file.type)) {
                    alert('Only JPG, PNG, and GIF images are allowed');
                    this.value = '';
                    return;
                }
                
                // Update label text
                fileLabel.innerHTML = `<i class="bx bx-camera"></i> ${file.name}`;
                uploadBtn.style.display = 'inline-flex';
                
                // Preview image
                const reader = new FileReader();
                reader.onload = function(e) {
                    const avatar = document.querySelector('.profile-user-avatar');
                    const existingImg = avatar.querySelector('.profile-image');
                    
                    if (existingImg) {
                        existingImg.src = e.target.result;
                    } else {
                        avatar.innerHTML = `<img src="${e.target.result}" alt="Profile Picture" class="profile-image">`;
                    }
                };
                reader.readAsDataURL(file);
            }
        });
    }
    
    // Form validation
    const forms = document.querySelectorAll('.picture-form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const uploadForm = form.querySelector('input[name="upload_picture"]');
            if (uploadForm && !fileInput.files[0]) {
                e.preventDefault();
                alert('Please select an image file to upload');
            }
        });
    });
});
</script>
