<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$quiz_id = intval($_GET['id'] ?? 0);

if ($quiz_id <= 0) {
    header('Location: assignments.php');
    exit;
}

// Get quiz details and result info
$query = "
    SELECT 
        q.quiz_id,
        q.title,
        q.description,
        q.instructions,
        q.time_limit,
        q.questions_count,
        q.max_attempts,
        q.passing_score,
        q.created_at,
        c.course_id,
        c.title as course_title,
        c.thumbnail as course_thumbnail,
        u.first_name,
        u.last_name,
        u.email as teacher_email,
        qr.quiz_result_id,
        qr.score,
        qr.completed_at,
        qr.attempt_number,
        qr.time_spent,
        CASE 
            WHEN qr.quiz_result_id IS NOT NULL THEN 'completed'
            ELSE 'available'
        END as status
    FROM quizzes q
    JOIN courses c ON q.course_id = c.course_id
    JOIN users u ON c.teacher_id = u.user_id
    JOIN enrollments e ON c.course_id = e.course_id
    LEFT JOIN quiz_results qr ON q.quiz_id = qr.quiz_id AND qr.student_id = ?
    WHERE q.quiz_id = ? AND e.student_id = ?
";

$stmt = $pdo->prepare($query);
$stmt->execute([$student_id, $quiz_id, $student_id]);
$quiz = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$quiz) {
    header('Location: assignments.php');
    exit;
}

// Get quiz questions (Q&A)
$qa_query = "
    SELECT qq.question, qq.created_at, qq.answer, qq.answered_at
    FROM quiz_questions qq
    WHERE qq.quiz_id = ? AND qq.student_id = ?
    ORDER BY qq.created_at DESC
";

$stmt = $pdo->prepare($qa_query);
$stmt->execute([$quiz_id, $student_id]);
$questions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = $quiz['title'] . " - Quiz Details";
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?> - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --error-color: #ef4444;
            --warning-color: #f59e0b;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1000px;
            margin: 0 auto;
            padding: 0 1rem;
            text-align: center;
        }

        .page-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }

        .page-subtitle {
            font-size: 1rem;
            opacity: 0.9;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--panadite-primary);
            text-decoration: none;
            font-weight: 500;
            margin-bottom: 1.5rem;
            transition: all 0.3s ease;
        }

        .back-link:hover {
            text-decoration: underline;
            transform: translateX(-2px);
        }

        .quiz-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
            overflow: hidden;
        }

        .quiz-header {
            padding: 2rem;
            background: linear-gradient(135deg, #e0f2fe 0%, #b3e5fc 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .quiz-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 1rem;
        }

        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
            margin-bottom: 1rem;
        }

        .status-available {
            background: #e0f2fe;
            color: #0369a1;
        }

        .status-completed {
            background: #d1fae5;
            color: #059669;
        }

        .quiz-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            padding: 1rem;
            background: white;
            border-radius: 12px;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .meta-icon {
            color: var(--panadite-primary);
            font-size: 1.1rem;
        }

        .quiz-content {
            padding: 2rem;
        }

        .content-section {
            margin-bottom: 2rem;
        }

        .section-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 1rem;
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .description, .instructions {
            color: var(--text-secondary);
            line-height: 1.6;
            background: var(--bg-light);
            padding: 1rem;
            border-radius: 12px;
        }

        .instructions {
            background: #fef3c7;
            border-left: 4px solid var(--panadite-accent);
        }

        .result-section {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
        }

        .score-display {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);
            color: white;
            padding: 2rem;
            border-radius: 12px;
            text-align: center;
            margin-bottom: 1rem;
        }

        .score-number {
            font-size: 3rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }

        .score-percentage {
            font-size: 1.2rem;
            opacity: 0.9;
            margin-bottom: 0.5rem;
        }

        .passing-status {
            font-size: 1rem;
            font-weight: 600;
        }

        .action-buttons {
            display: flex;
            gap: 1rem;
            margin-top: 1.5rem;
            flex-wrap: wrap;
        }

        .action-btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.9rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(37, 99, 235, 0.3);
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--panadite-accent) 0%, #f97316 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 2px solid var(--panadite-primary);
        }

        .action-btn:hover {
            transform: translateY(-2px);
        }

        .qa-section {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }

        .qa-item {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1rem;
            border: 2px solid var(--border-color);
        }

        .question {
            background: var(--bg-light);
            margin-bottom: 0.5rem;
            padding: 1rem;
            border-radius: 8px;
        }

        .answer {
            background: #d1fae5;
            padding: 1rem;
            border-radius: 8px;
            border-left: 4px solid var(--success-color);
        }

        .qa-meta {
            font-size: 0.8rem;
            color: var(--text-secondary);
            margin-top: 0.5rem;
        }

        .attempts-info {
            background: #dbeafe;
            border-left: 4px solid var(--panadite-primary);
            padding: 1rem;
            border-radius: 8px;
            margin-top: 1rem;
        }

        @media (max-width: 768px) {
            .page-title {
                font-size: 1.5rem;
            }
            
            .quiz-header, .quiz-content, .result-section, .qa-section {
                padding: 1.5rem;
            }
            
            .quiz-meta {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .score-number {
                font-size: 2rem;
            }
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">Quiz Details</h1>
            <p class="page-subtitle"><?= htmlspecialchars($quiz['course_title']) ?></p>
        </div>
    </div>

    <div class="container">
        <a href="assignments.php" class="back-link">
            <i class="bx bx-arrow-back"></i>
            Back to Assignments & Quizzes
        </a>

        <div class="quiz-card">
            <div class="quiz-header">
                <h2 class="quiz-title">
                    <i class="bx bx-brain"></i>
                    <?= htmlspecialchars($quiz['title']) ?>
                </h2>
                
                <div class="status-badge status-<?= $quiz['status'] ?>">
                    <?php
                    $status_icons = [
                        'available' => 'bx-play-circle',
                        'completed' => 'bx-check-double'
                    ];
                    ?>
                    <i class="bx <?= $status_icons[$quiz['status']] ?>"></i>
                    <?= ucfirst($quiz['status']) ?>
                </div>
                
                <div class="quiz-meta">
                    <div class="meta-item">
                        <i class="meta-icon bx bx-time"></i>
                        <span><?= $quiz['time_limit'] ?> minutes</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-help-circle"></i>
                        <span><?= $quiz['questions_count'] ?> Questions</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-refresh"></i>
                        <span><?= $quiz['max_attempts'] ?> Max Attempts</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-target-lock"></i>
                        <span><?= $quiz['passing_score'] ?>% to Pass</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-user"></i>
                        <span><?= htmlspecialchars($quiz['first_name'] . ' ' . $quiz['last_name']) ?></span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-calendar"></i>
                        <span>Created: <?= date('M d, Y', strtotime($quiz['created_at'])) ?></span>
                    </div>
                    <?php if ($quiz['completed_at']): ?>
                        <div class="meta-item">
                            <i class="meta-icon bx bx-check"></i>
                            <span>Completed: <?= date('M d, Y g:i A', strtotime($quiz['completed_at'])) ?></span>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="quiz-content">
                <div class="content-section">
                    <h3 class="section-title">
                        <i class="bx bx-info-circle"></i>
                        Description
                    </h3>
                    <div class="description">
                        <?= nl2br(htmlspecialchars($quiz['description'])) ?>
                    </div>
                </div>
                
                <?php if ($quiz['instructions']): ?>
                    <div class="content-section">
                        <h3 class="section-title">
                            <i class="bx bx-list-ul"></i>
                            Instructions
                        </h3>
                        <div class="instructions">
                            <?= nl2br(htmlspecialchars($quiz['instructions'])) ?>
                        </div>
                    </div>
                <?php endif; ?>
                
                <?php if ($quiz['max_attempts'] > ($quiz['attempt_number'] ?? 0)): ?>
                    <div class="attempts-info">
                        <strong>Attempts Remaining:</strong> 
                        <?= $quiz['max_attempts'] - ($quiz['attempt_number'] ?? 0) ?> of <?= $quiz['max_attempts'] ?>
                    </div>
                <?php endif; ?>
                
                <div class="action-buttons">
                    <?php if ($quiz['status'] === 'available'): ?>
                        <a href="take-quiz.php?id=<?= $quiz['quiz_id'] ?>" class="action-btn btn-primary">
                            <i class="bx bx-play"></i>
                            Start Quiz
                        </a>
                    <?php else: ?>
                        <a href="quiz-results.php?id=<?= $quiz['quiz_id'] ?>" class="action-btn btn-secondary">
                            <i class="bx bx-bar-chart"></i>
                            View Results
                        </a>
                        
                        <?php if ($quiz['max_attempts'] > ($quiz['attempt_number'] ?? 0)): ?>
                            <a href="retake-quiz.php?id=<?= $quiz['quiz_id'] ?>" class="action-btn btn-outline">
                                <i class="bx bx-refresh"></i>
                                Retake Quiz
                            </a>
                        <?php endif; ?>
                    <?php endif; ?>
                    
                    <a href="mailto:<?= htmlspecialchars($quiz['teacher_email']) ?>" class="action-btn btn-outline">
                        <i class="bx bx-envelope"></i>
                        Contact Instructor
                    </a>
                </div>
            </div>
        </div>

        <!-- Quiz Results -->
        <?php if ($quiz['quiz_result_id']): ?>
            <div class="result-section">
                <h3 class="section-title">
                    <i class="bx bx-trophy"></i>
                    Your Results
                </h3>
                
                <div class="score-display">
                    <div class="score-number"><?= $quiz['score'] ?>%</div>
                    <div class="score-percentage">
                        <?= $quiz['score'] >= $quiz['passing_score'] ? 'PASSED' : 'FAILED' ?>
                    </div>
                    <div class="passing-status">
                        Attempt <?= $quiz['attempt_number'] ?> of <?= $quiz['max_attempts'] ?>
                    </div>
                </div>
                
                <div class="quiz-meta">
                    <div class="meta-item">
                        <i class="meta-icon bx bx-time-five"></i>
                        <span>Time Spent: <?= $quiz['time_spent'] ?> minutes</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-target-lock"></i>
                        <span>Passing Score: <?= $quiz['passing_score'] ?>%</span>
                    </div>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-calendar"></i>
                        <span>Completed: <?= date('M d, Y g:i A', strtotime($quiz['completed_at'])) ?></span>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Q&A Section -->
        <div class="qa-section">
            <h3 class="section-title">
                <i class="bx bx-help-circle"></i>
                Questions & Answers
            </h3>
            
            <?php if (!empty($questions)): ?>
                <?php foreach ($questions as $qa): ?>
                    <div class="qa-item">
                        <div class="question">
                            <strong>Your Question:</strong><br>
                            <?= nl2br(htmlspecialchars($qa['question'])) ?>
                            <div class="qa-meta">
                                Asked on <?= date('M d, Y g:i A', strtotime($qa['created_at'])) ?>
                            </div>
                        </div>
                        
                        <?php if ($qa['answer']): ?>
                            <div class="answer">
                                <strong>Instructor's Answer:</strong><br>
                                <?= nl2br(htmlspecialchars($qa['answer'])) ?>
                                <div class="qa-meta">
                                    Answered on <?= date('M d, Y g:i A', strtotime($qa['answered_at'])) ?>
                                </div>
                            </div>
                        <?php else: ?>
                            <div style="color: var(--text-secondary); font-style: italic; margin-top: 0.5rem;">
                                Waiting for instructor's response...
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div style="text-align: center; color: var(--text-secondary); padding: 2rem;">
                    <i class="bx bx-message-square-dots" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                    <p>No questions asked yet.</p>
                    <p>Go back to the assignments page to ask a question about this quiz.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
