<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Include encryption for quiz IDs
require_once '../includes/id_encryption.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Check if attempt ID is provided
if (!isset($_GET['attempt_id']) || !is_numeric($_GET['attempt_id'])) {
    $_SESSION['error'] = "Invalid attempt ID";
    header('Location: all_quizzes.php');
    exit();
}

$attempt_id = mysqli_real_escape_string($conn, $_GET['attempt_id']);

// Get quiz attempt details and quiz info
$attemptQuery = "
    SELECT qa.*, q.*, c.course_id, c.title as course_title
    FROM quiz_attempts qa
    JOIN quizzes q ON qa.quiz_id = q.quiz_id
    JOIN courses c ON q.course_id = c.course_id
    WHERE qa.attempt_id = ? AND qa.student_id = ?
";

$attemptStmt = $conn->prepare($attemptQuery);
$attemptStmt->bind_param("ii", $attempt_id, $student_id);
$attemptStmt->execute();
$attemptResult = $attemptStmt->get_result();

if ($attemptResult->num_rows == 0) {
    $_SESSION['error'] = "Quiz attempt not found or access denied";
    header('Location: all_quizzes.php');
    exit();
}

$attempt_data = $attemptResult->fetch_assoc();
$quiz = $attempt_data; // Contains both quiz and attempt data
$course_id = $quiz['course_id'];
$quiz_id = $quiz['quiz_id'];

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

if (!$is_enrolled) {
    $_SESSION['error'] = "You are not enrolled in this course";
    header('Location: courses.php');
    exit();
}

// We already have the attempt data from the earlier query, no need for another query
// The $quiz variable already contains both quiz and attempt data from the JOIN query above
$attempt = $quiz; // This contains the attempt data

// The score in database is already stored as percentage
$percentageScore = $attempt['score'] ?? 0;

// Calculate actual earned points and total possible points
$max_score_query = "SELECT SUM(points) as total_points FROM quiz_questions WHERE quiz_id = $quiz_id";
$max_score_result = $conn->query($max_score_query);
$maxScore = $max_score_result->fetch_assoc()['total_points'] ?? 0;

// Calculate earned points from percentage
$earnedPoints = $maxScore > 0 ? round(($percentageScore / 100) * $maxScore) : 0;

// Determine color based on score
$scoreColor = '#dc3545'; // Red (default for low scores)
if ($percentageScore >= 90) {
    $scoreColor = '#198754'; // Green for excellent
} elseif ($percentageScore >= 75) {
    $scoreColor = '#0d6efd'; // Blue for good
} elseif ($percentageScore >= 60) {
    $scoreColor = '#fd7e14'; // Orange for passing
}

$page_title = "Quiz Results: " . $quiz['title'];

// Include header
include __DIR__ . '/components/header.php';
?>

<!-- Quiz CSS -->
<link rel="stylesheet" href="assets/css/pages/quiz.css?v=<?= time(); ?>">

<div class="page-content">
    <div class="container py-4">
        <div class="row">
            <div class="col-12">
                <!-- Quiz header -->
                <div class="quiz-header mb-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="quiz-title">Results: <?= htmlspecialchars($quiz['title']) ?></h2>
                            <div class="quiz-meta">
                                <span class="course-name">
                                    <i class='bx bx-book'></i> <?= htmlspecialchars($quiz['course_title']) ?>
                                </span>
                                <span class="quiz-attempt">
                                    <i class='bx bx-calendar'></i> Attempted on <?= date('M d, Y - h:i A', strtotime($attempt['start_time'])) ?>
                                </span>
                            </div>
                        </div>
                        <?php
                        // Encrypt quiz ID for the back button
                        $encrypted_quiz_id = '';
                        try {
                            if (class_exists('IdEncryption') && method_exists('IdEncryption', 'encrypt')) {
                                $encrypted_quiz_id = IdEncryption::encrypt($quiz_id);
                                if (empty($encrypted_quiz_id) || $encrypted_quiz_id == $quiz_id) {
                                    $encrypted_quiz_id = $quiz_id;
                                }
                            } else {
                                $encrypted_quiz_id = $quiz_id;
                            }
                        } catch (Exception $e) {
                            $encrypted_quiz_id = $quiz_id;
                        }
                        ?>
                        <a href="take_quiz.php?id=<?= htmlspecialchars($encrypted_quiz_id) ?>" class="btn btn-outline-primary">
                            <i class='bx bx-arrow-back'></i> Back to Quiz
                        </a>
                    </div>
                </div>

                <!-- Quiz results -->
                <div class="quiz-results">
                    <div class="quiz-score-summary">
                        <div class="quiz-score-circle" style="background-color: <?= $scoreColor ?>20; color: <?= $scoreColor ?>;">
                            <div class="quiz-score-percent"><?= $percentageScore ?>%</div>
                            <div class="quiz-score-text">Score</div>
                        </div>
                        
                        <div class="quiz-score-details">
                            <div class="quiz-score-detail">
                                <div class="detail-value"><?= $earnedPoints ?> / <?= $maxScore ?></div>
                                <div class="detail-label">Points</div>
                            </div>
                            
                            <div class="quiz-score-detail">
                                <div class="detail-value">
                                    <?php
                                    if ($percentageScore >= 90) echo "Excellent";
                                    elseif ($percentageScore >= 75) echo "Good";
                                    elseif ($percentageScore >= 60) echo "Satisfactory";
                                    else echo "Needs Improvement";
                                    ?>
                                </div>
                                <div class="detail-label">Performance</div>
                            </div>
                            
                            <div class="quiz-score-detail">
                                <div class="detail-value"><?= $attempt['attempt_number'] ?? 1 ?></div>
                                <div class="detail-label">Attempt</div>
                            </div>
                        </div>
                    </div>

                    <?php
                    // Get the student's answers with option text and correct answers
                    $answersQuery = "
                        SELECT 
                            qr.*,
                            qq.question_text, 
                            qq.question_type, 
                            qq.points as question_points,
                            student_option.option_text as given_answer,
                            correct_option.option_text as correct_answer
                        FROM quiz_responses qr
                        JOIN quiz_questions qq ON qr.question_id = qq.question_id
                        LEFT JOIN quiz_question_options student_option ON qr.answer_id = student_option.option_id
                        LEFT JOIN quiz_question_options correct_option ON qq.question_id = correct_option.question_id AND correct_option.is_correct = 1
                        WHERE qr.attempt_id = ?
                        ORDER BY qq.question_id
                    ";
                    
                    $answersStmt = $conn->prepare($answersQuery);
                    $answersStmt->bind_param("i", $attempt['attempt_id']);
                    $answersStmt->execute();
                    $answersResult = $answersStmt->get_result();
                    
                    if ($answersResult->num_rows > 0) {
                    ?>
                    
                    <div class="question-review">
                        <h4 class="mb-4">Question Review</h4>
                        
                        <?php
                        $questionNumber = 1;
                        while ($answer = $answersResult->fetch_assoc()) {
                            $questionId = $answer['question_id'];
                            $isCorrect = $answer['is_correct'] ?? 0;
                            $givenAnswer = $answer['given_answer'] ?? '';
                            $correctAnswer = $answer['correct_answer'] ?? '';
                            $pointsEarned = $answer['points_earned'] ?? 0;
                            $maxPoints = $answer['question_points'] ?? 1;
                            
                            // Determine question status
                            $questionStatus = "incorrect";
                            if ($isCorrect == 1) {
                                $questionStatus = "correct";
                            } else if ($pointsEarned > 0 && $pointsEarned < $maxPoints) {
                                $questionStatus = "partial";
                            }
                        ?>
                        
                        <div class="question-review-item">
                            <div class="question-review-header">
                                <div class="question-number">Question <?= $questionNumber ?></div>
                                <div class="d-flex align-items-center">
                                    <div class="question-points me-3"><?= $pointsEarned ?> / <?= $maxPoints ?> points</div>
                                    <div class="question-status <?= $questionStatus ?>">
                                        <?php 
                                        if ($questionStatus == "correct") echo "Correct";
                                        else if ($questionStatus == "partial") echo "Partially Correct";
                                        else echo "Incorrect";
                                        ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="question-content">
                                <?= htmlspecialchars($answer['question_text']) ?>
                            </div>
                            
                            <div class="answer-review mt-3">
                                <div class="answer-given">
                                    <strong>Your Answer:</strong> <?= htmlspecialchars($givenAnswer) ?>
                                </div>
                                
                                <?php if ($questionStatus != "correct") { ?>
                                <div class="answer-correct mt-2">
                                    <strong>Correct Answer:</strong> <?= htmlspecialchars($correctAnswer) ?>
                                </div>
                                <?php } ?>
                                
                                <?php if (!empty($answer['feedback'])) { ?>
                                <div class="explanation mt-3">
                                    <strong>Explanation:</strong><br>
                                    <?= htmlspecialchars($answer['feedback']) ?>
                                </div>
                                <?php } ?>
                            </div>
                        </div>
                        
                        <?php
                            $questionNumber++;
                        }
                        ?>
                    </div>
                    
                    <?php } else { ?>
                    <div class="text-center my-5">
                        <div class="alert alert-info">
                            <i class='bx bx-info-circle'></i> Detailed question review is not available for this quiz attempt.
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Action buttons -->
                    <div class="d-flex justify-content-between mt-5">
                        <a href="course-quizzes.php?id=<?= $course_id ?>" class="btn btn-outline-secondary">
                            <i class='bx bx-arrow-back'></i> Back to Quizzes
                        </a>
                        
                        <?php
                        // Check if student can attempt the quiz again
                        $maxAttempts = isset($quiz['max_attempts']) ? $quiz['max_attempts'] : PHP_INT_MAX;
                        $attemptCountQuery = "SELECT COUNT(*) as attempt_count FROM quiz_attempts WHERE student_id = ? AND quiz_id = ?";
                        $attemptCountStmt = $conn->prepare($attemptCountQuery);
                        $attemptCountStmt->bind_param("ii", $student_id, $quiz_id);
                        $attemptCountStmt->execute();
                        $attemptCountResult = $attemptCountStmt->get_result();
                        $attemptCount = $attemptCountResult->fetch_assoc()['attempt_count'] ?? 0;
                        
                        $now = new DateTime();
                        $startTime = !empty($quiz['start_time']) ? new DateTime($quiz['start_time']) : new DateTime();
                        $endTime = !empty($quiz['end_time']) ? new DateTime($quiz['end_time']) : new DateTime('+1 day');
                        $isAvailable = ($now >= $startTime && $now <= $endTime);
                        
                        if ($isAvailable && $attemptCount < $maxAttempts) {
                        ?>
                        <a href="quiz_start.php?id=<?= $quiz_id ?>" class="btn btn-primary">
                            <i class='bx bx-refresh'></i> Attempt Again
                        </a>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include __DIR__ . '/components/footer.php'; ?>
