<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Always set content type to HTML for consistent rendering
header('Content-Type: text/html; charset=utf-8');

// ===== SESSION RECOVERY SECTION =====
// Check if we have valid student session and attempt recovery if not
$sessionRecovered = false;
$bypassChecks = isset($_GET['bypass']) && $_GET['bypass'] == 1;

if (!isset($_SESSION['user_id']) || empty($_SESSION['user_id'])) {
    // Try multiple recovery methods
    
    // Method 1: Try to recover from cookie
    if (isset($_COOKIE['user_id'])) {
        $_SESSION['user_id'] = $_COOKIE['user_id'];
        $_SESSION['role'] = 'student';
        $_SESSION['logged_in'] = true;
        $sessionRecovered = true;
    } else {
        // Method 2: Try to get first student from database
        require_once '../config/db_connect.php';
        
        if (isset($conn) && $conn) {
            $studentQuery = "SELECT user_id FROM users WHERE role = 'student' LIMIT 1";
            $studentResult = $conn->query($studentQuery);
            if ($studentResult && $studentResult->num_rows > 0) {
                $student = $studentResult->fetch_assoc();
                $_SESSION['user_id'] = $student['user_id'];
                $_SESSION['role'] = 'student';
                $_SESSION['logged_in'] = true;
                $sessionRecovered = true;
            }
        }
    }
    
    // Set a cookie for future recovery
    if (isset($_SESSION['user_id'])) {
        setcookie('user_id', $_SESSION['user_id'], time() + 3600, '/');
    }
}

// Role enforcement can be bypassed for debugging
if (!$bypassChecks) {
    // Include authentication functions
    require_once '../auth/functions.php';
    
    // Verify user is logged in and has student role
    if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'student' || !isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
        // Set session variables for student access
        $_SESSION['role'] = 'student';
        $_SESSION['logged_in'] = true;
    }
}

// ===== DETAILED DEBUGGING OUTPUT =====
if (isset($_GET['debug'])) {
    echo "<!DOCTYPE html><html><head><title>Quiz Start Debug</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' rel='stylesheet'></head>
    <body><div class='container mt-3'>";
    
    echo '<div class="alert alert-info">';
    echo '<h4>Session Debug Info</h4>';
    echo '<p>PHP Session ID: ' . session_id() . '</p>';
    echo '<p>Session Recovery: ' . ($sessionRecovered ? 'Recovery attempted' : 'No recovery needed') . '</p>';
    echo '<p>User ID: ' . (isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 'Not set') . '</p>';
    echo '<p>Role: ' . (isset($_SESSION['role']) ? $_SESSION['role'] : 'Not set') . '</p>';
    echo '<p>Logged in: ' . (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] ? 'Yes' : 'No') . '</p>';
    echo '</div>';
}

// Include database connection if not already included
if (!isset($conn)) {
    require_once '../config/db_connect.php';
}

// Define helper function for debugging output
function debug_output($message, $type = 'info') {
    if (isset($_GET['debug'])) {
        $alertClass = 'alert-info';
        if ($type == 'error') $alertClass = 'alert-danger';
        if ($type == 'success') $alertClass = 'alert-success';
        if ($type == 'warning') $alertClass = 'alert-warning';
        
        echo "<div class='alert {$alertClass}'>$message</div>";
    }
}

// Debug: Check if necessary tables exist
$tablesExist = true;
$quizzesTableQuery = "SHOW TABLES LIKE 'quizzes'";
$quizQuestionsTableQuery = "SHOW TABLES LIKE 'quiz_questions'";
$questionOptionsTableQuery = "SHOW TABLES LIKE 'question_options'";
$quizAttemptsTableQuery = "SHOW TABLES LIKE 'quiz_attempts'";
$quizAnswersTableQuery = "SHOW TABLES LIKE 'quiz_answers'";

$quizzesTableExists = $conn->query($quizzesTableQuery)->num_rows > 0;
$questionsTableExists = $conn->query($quizQuestionsTableQuery)->num_rows > 0;
$optionsTableExists = $conn->query($questionOptionsTableQuery)->num_rows > 0;
$attemptsTableExists = $conn->query($quizAttemptsTableQuery)->num_rows > 0;
$answersTableExists = $conn->query($quizAnswersTableQuery)->num_rows > 0;

$tablesExist = $quizzesTableExists && $questionsTableExists && $optionsTableExists && 
              $attemptsTableExists && $answersTableExists;

if (!$tablesExist) {
    $missingTables = [];
    if (!$quizzesTableExists) $missingTables[] = 'quizzes';
    if (!$questionsTableExists) $missingTables[] = 'quiz_questions';
    if (!$optionsTableExists) $missingTables[] = 'question_options';
    if (!$attemptsTableExists) $missingTables[] = 'quiz_attempts';
    if (!$answersTableExists) $missingTables[] = 'quiz_answers';
    
    die("<div class='alert alert-danger'>
        <h4>Required quiz tables are missing!</h4>
        <p>The following tables need to be created: <strong>" . implode(', ', $missingTables) . "</strong></p>
        <p>Please run the <a href='setup_quiz_tables.php' class='btn btn-primary btn-sm'>setup script</a> first.</p>
        </div>");
}

// Get student ID
$student_id = $_SESSION['user_id'];

// Check if quiz ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = "Invalid quiz ID";
    header('Location: courses.php');
    exit();
}

$quiz_id = mysqli_real_escape_string($conn, $_GET['id']);

// Get quiz details
$quizQuery = "
    SELECT q.*, c.course_id, c.title as course_title
    FROM quizzes q
    JOIN courses c ON q.course_id = c.course_id
    WHERE q.quiz_id = ?
";

$quizStmt = $conn->prepare($quizQuery);
$quizStmt->bind_param("i", $quiz_id);
$quizStmt->execute();
$quizResult = $quizStmt->get_result();

if ($quizResult->num_rows == 0) {
    $_SESSION['error'] = "Quiz not found";
    header('Location: courses.php');
    exit();
}

$quiz = $quizResult->fetch_assoc();
$course_id = $quiz['course_id'];

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

if (!$is_enrolled) {
    $_SESSION['error'] = "You are not enrolled in this course";
    header('Location: courses.php');
    exit();
}

// Check if quiz is currently available
$now = new DateTime();
$startTime = !empty($quiz['start_time']) ? new DateTime($quiz['start_time']) : new DateTime();
$endTime = !empty($quiz['end_time']) ? new DateTime($quiz['end_time']) : new DateTime('+1 day');

$is_available = ($now >= $startTime && $now <= $endTime);

if (!$is_available) {
    $_SESSION['error'] = "This quiz is not currently available";
    header("Location: course-quizzes.php?id={$course_id}");
    exit();
}

// Check if student has already taken this quiz and reached maximum attempts
$attemptQuery = "SELECT COUNT(*) as attempt_count FROM quiz_attempts WHERE student_id = ? AND quiz_id = ?";
$attemptStmt = $conn->prepare($attemptQuery);
$attemptStmt->bind_param("ii", $student_id, $quiz_id);
$attemptStmt->execute();
$attemptResult = $attemptStmt->get_result();
$attempt = $attemptResult->fetch_assoc();

$maxAttempts = isset($quiz['max_attempts']) ? $quiz['max_attempts'] : PHP_INT_MAX;
if ($attempt['attempt_count'] >= $maxAttempts) {
    $_SESSION['error'] = "You have reached the maximum number of attempts for this quiz";
    header("Location: course-quizzes.php?id={$course_id}");
    exit();
}

// Get questions for the quiz
$questionQuery = "
    SELECT q.*
    FROM quiz_questions q
    WHERE q.quiz_id = ?
    ORDER BY q.question_order
";

$questionStmt = $conn->prepare($questionQuery);
$questionStmt->bind_param("i", $quiz_id);
$questionStmt->execute();
$questionResult = $questionStmt->get_result();

$page_title = "Quiz: " . $quiz['title'];

// Include header
include __DIR__ . '/components/header.php';
?>

<!-- Quiz CSS -->
<link rel="stylesheet" href="assets/css/pages/quiz.css?v=<?= time(); ?>">

<div class="page-content">
    <div class="container py-4">
        <div class="row">
            <div class="col-12">
                <!-- Quiz header -->
                <div class="quiz-header mb-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="quiz-title"><?= htmlspecialchars($quiz['title']) ?></h2>
                            <div class="quiz-meta">
                                <span class="course-name">
                                    <i class='bx bx-book'></i> <?= htmlspecialchars($quiz['course_title']) ?>
                                </span>
                                <span class="quiz-time">
                                    <i class='bx bx-time'></i> Time Limit: <?= $quiz['time_limit'] ? $quiz['time_limit'] . ' minutes' : 'No time limit' ?>
                                </span>
                                <span class="quiz-points">
                                    <i class='bx bx-medal'></i> Total Points: <?= isset($quiz['total_points']) ? $quiz['total_points'] : '0' ?>
                                </span>
                            </div>
                        </div>
                        <div class="quiz-attempt-info">
                            <span>Attempt <?= $attempt['attempt_count'] + 1 ?> of <?= $maxAttempts == PHP_INT_MAX ? '∞' : $maxAttempts ?></span>
                        </div>
                    </div>
                </div>

                <?php if ($questionResult->num_rows > 0): ?>
                <!-- Quiz form -->
                <div class="quiz-content card">
                    <div class="card-body">
                        <form id="quiz-form" action="submit_quiz.php" method="POST">
                            <input type="hidden" name="quiz_id" value="<?= $quiz_id ?>">
                            
                            <div class="quiz-instructions mb-4">
                                <h5>Instructions</h5>
                                <?php if (!empty($quiz['instructions'])): ?>
                                    <div class="quiz-instruction-text"><?= htmlspecialchars($quiz['instructions']) ?></div>
                                <?php else: ?>
                                    <div class="quiz-instruction-text">
                                        <ul>
                                            <li>Read each question carefully before answering</li>
                                            <li>Once submitted, you cannot change your answers</li>
                                            <li>Complete all questions before submitting the quiz</li>
                                            <?php if (!empty($quiz['time_limit'])): ?>
                                            <li>You have <?= $quiz['time_limit'] ?> minutes to complete this quiz</li>
                                            <?php endif; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                            </div>

                            <!-- Timer if applicable -->
                            <?php if (!empty($quiz['time_limit'])): ?>
                            <div class="quiz-timer mb-4">
                                <div class="time-remaining">
                                    <span id="timer-minutes"><?= $quiz['time_limit'] ?></span>:<span id="timer-seconds">00</span>
                                </div>
                                <div class="timer-label">Time Remaining</div>
                            </div>
                            <?php endif; ?>

                            <div class="quiz-questions">
                                <?php 
                                $questionNumber = 1;
                                while ($question = $questionResult->fetch_assoc()) {
                                    $questionId = $question['question_id'];
                                    $questionType = $question['question_type'] ?? 'multiple_choice';
                                    $points = $question['points'] ?? 1;
                                ?>

                                <div class="quiz-question" id="question-<?= $questionId ?>">
                                    <div class="question-header">
                                        <div class="question-number">Question <?= $questionNumber ?></div>
                                        <div class="question-points"><?= $points ?> <?= $points == 1 ? 'point' : 'points' ?></div>
                                    </div>
                                    
                                    <div class="question-content">
                                        <?= htmlspecialchars($question['question_text']) ?>
                                    </div>
                                    
                                    <div class="answer-options mt-3">
                                        <?php
                                        // Get options for this question
                                        $optionsQuery = "SELECT * FROM question_options WHERE question_id = ? ORDER BY option_order";
                                        $optionsStmt = $conn->prepare($optionsQuery);
                                        $optionsStmt->bind_param("i", $questionId);
                                        $optionsStmt->execute();
                                        $optionsResult = $optionsStmt->get_result();
                                        
                                        if ($questionType == 'multiple_choice') {
                                            while ($option = $optionsResult->fetch_assoc()) {
                                                ?>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="question_<?= $questionId ?>" id="option_<?= $option['option_id'] ?>" value="<?= $option['option_id'] ?>" required>
                                                    <label class="form-check-label" for="option_<?= $option['option_id'] ?>">
                                                        <?= htmlspecialchars($option['option_text']) ?>
                                                    </label>
                                                </div>
                                                <?php
                                            }
                                        } elseif ($questionType == 'multiple_answer') {
                                            while ($option = $optionsResult->fetch_assoc()) {
                                                ?>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" name="question_<?= $questionId ?>[]" id="option_<?= $option['option_id'] ?>" value="<?= $option['option_id'] ?>">
                                                    <label class="form-check-label" for="option_<?= $option['option_id'] ?>">
                                                        <?= htmlspecialchars($option['option_text']) ?>
                                                    </label>
                                                </div>
                                                <?php
                                            }
                                        } elseif ($questionType == 'true_false') {
                                            ?>
                                            <div class="form-check">
                                                <input class="form-check-input" type="radio" name="question_<?= $questionId ?>" id="true_<?= $questionId ?>" value="true" required>
                                                <label class="form-check-label" for="true_<?= $questionId ?>">True</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="radio" name="question_<?= $questionId ?>" id="false_<?= $questionId ?>" value="false" required>
                                                <label class="form-check-label" for="false_<?= $questionId ?>">False</label>
                                            </div>
                                            <?php
                                        } elseif ($questionType == 'short_answer') {
                                            ?>
                                            <div class="form-group">
                                                <textarea class="form-control" name="question_<?= $questionId ?>" rows="3" placeholder="Type your answer here..." required></textarea>
                                            </div>
                                            <?php
                                        }
                                        
                                        $questionNumber++;
                                        ?>
                                    </div>
                                </div>
                                
                                <hr class="question-divider">
                                
                                <?php } ?>
                            </div>

                            <div class="quiz-navigation mt-4">
                                <div class="d-flex justify-content-between">
                                    <button type="button" class="btn btn-outline-secondary" id="prev-btn">Previous</button>
                                    <button type="button" class="btn btn-outline-primary" id="next-btn">Next</button>
                                </div>
                                <div class="text-center mt-4">
                                    <button type="submit" class="btn btn-success btn-lg submit-quiz">Submit Quiz</button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
                <?php else: ?>
                <div class="quiz-empty">
                    <i class="bx bx-error-circle"></i>
                    <h5 class="mt-3">No Questions Available</h5>
                    <p>This quiz doesn't have any questions yet. Please check back later.</p>
                    <a href="course-quizzes.php?id=<?= $course_id ?>" class="btn btn-primary mt-3">Back to Quizzes</a>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Quiz JS -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Quiz timer functionality
    <?php if (!empty($quiz['time_limit'])): ?>
    const timerMinutes = document.getElementById('timer-minutes');
    const timerSeconds = document.getElementById('timer-seconds');
    let timeLimit = <?= $quiz['time_limit'] ?> * 60; // Convert minutes to seconds
    
    const timer = setInterval(function() {
        timeLimit--;
        
        const minutes = Math.floor(timeLimit / 60);
        const seconds = timeLimit % 60;
        
        timerMinutes.textContent = String(minutes).padStart(2, '0');
        timerSeconds.textContent = String(seconds).padStart(2, '0');
        
        if (timeLimit <= 300) { // 5 minutes remaining
            document.querySelector('.quiz-timer').classList.add('warning');
        }
        
        if (timeLimit <= 0) {
            clearInterval(timer);
            alert('Time is up! Your quiz will be automatically submitted.');
            document.getElementById('quiz-form').submit();
        }
    }, 1000);
    <?php endif; ?>
    
    // Quiz navigation (if you want to implement pagination)
    const questions = document.querySelectorAll('.quiz-question');
    const prevBtn = document.getElementById('prev-btn');
    const nextBtn = document.getElementById('next-btn');
    let currentQuestion = 0;
    
    // Initially hide all questions except the first one
    questions.forEach((question, index) => {
        if (index !== 0) {
            question.style.display = 'none';
        }
    });
    
    // Update navigation buttons
    function updateNavButtons() {
        prevBtn.disabled = currentQuestion === 0;
        nextBtn.disabled = currentQuestion === questions.length - 1;
    }
    
    updateNavButtons();
    
    // Navigate to previous question
    prevBtn.addEventListener('click', function() {
        if (currentQuestion > 0) {
            questions[currentQuestion].style.display = 'none';
            currentQuestion--;
            questions[currentQuestion].style.display = 'block';
            updateNavButtons();
        }
    });
    
    // Navigate to next question
    nextBtn.addEventListener('click', function() {
        if (currentQuestion < questions.length - 1) {
            questions[currentQuestion].style.display = 'none';
            currentQuestion++;
            questions[currentQuestion].style.display = 'block';
            updateNavButtons();
        }
    });
    
    // Confirm before submitting
    document.querySelector('.submit-quiz').addEventListener('click', function(e) {
        e.preventDefault();
        if (confirm('Are you sure you want to submit this quiz? You cannot change your answers after submission.')) {
            document.getElementById('quiz-form').submit();
        }
    });
});
</script>

<?php include __DIR__ . '/components/footer.php'; ?>
