<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Accept');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Comprehensive debugging information
$debug_info = [
    'timestamp' => date('Y-m-d H:i:s'),
    'request_method' => $_SERVER['REQUEST_METHOD'] ?? 'UNKNOWN',
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'NOT_SET',
    'request_uri' => $_SERVER['REQUEST_URI'] ?? 'NOT_SET',
    'server_name' => $_SERVER['SERVER_NAME'] ?? 'NOT_SET',
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'NOT_SET',
    'session_id' => session_id(),
    'session_data' => $_SESSION ?? [],
    'get_params' => $_GET,
    'post_params' => $_POST,
    'files' => $_FILES,
    'raw_input' => file_get_contents('php://input')
];

error_log("=== SAVE VIDEO NOTE API CALL ===");
error_log("Debug Info: " . json_encode($debug_info, JSON_PRETTY_PRINT));

// Handle OPTIONS request for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    echo json_encode(['success' => true, 'message' => 'Options request handled', 'debug' => $debug_info]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false, 
        'message' => 'Method not allowed',
        'debug' => $debug_info,
        'expected' => 'POST',
        'received' => $_SERVER['REQUEST_METHOD']
    ]);
    exit;
}

$raw_input = file_get_contents('php://input');
error_log("Raw input received: " . $raw_input);

$input = json_decode($raw_input, true);
$json_error = json_last_error();

$parsing_debug = [
    'raw_input' => $raw_input,
    'raw_input_length' => strlen($raw_input),
    'json_decode_result' => $input,
    'json_last_error' => $json_error,
    'json_last_error_msg' => json_last_error_msg(),
    'session_user_id' => $_SESSION['user_id'] ?? 'NOT_SET'
];

error_log("JSON parsing debug: " . json_encode($parsing_debug, JSON_PRETTY_PRINT));

if ($json_error !== JSON_ERROR_NONE) {
    echo json_encode([
        'success' => false,
        'message' => 'JSON parsing error: ' . json_last_error_msg(),
        'debug' => $parsing_debug,
        'raw_input' => $raw_input
    ]);
    exit;
}

if (!$input || !isset($input['lesson_id']) || !isset($input['timestamp']) || !isset($input['note'])) {
    $missing = [];
    if (!isset($input['lesson_id'])) $missing[] = 'lesson_id';
    if (!isset($input['timestamp'])) $missing[] = 'timestamp';
    if (!isset($input['note'])) $missing[] = 'note';
    
    echo json_encode([
        'success' => false, 
        'message' => 'Missing required fields: ' . implode(', ', $missing),
        'debug' => $debug_info,
        'parsing_debug' => $parsing_debug,
        'received_input' => $input,
        'missing_fields' => $missing
    ]);
    exit;
}

$lesson_id = (int)$input['lesson_id'];
$timestamp = (int)$input['timestamp'];
$note = trim($input['note']);
$formatted_time = $input['formatted_time'] ?? '';

// Enhanced authentication with live server fallback (same as review APIs)
$user_logged_in = false;
$student_id = null;

// Method 1: Try session
if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
    $user_logged_in = true;
    $student_id = $_SESSION['user_id'];
    error_log("Save Notes: Authentication successful via session: User ID $student_id");
} else {
    error_log("Save Notes: Session authentication failed. Trying emergency fallback.");
    
    // Method 2: Emergency fallback for live server
    $student_id = 999; // Emergency user ID
    $user_logged_in = true;
    error_log("Save Notes: Using emergency user ID: $student_id");
}

if (!$user_logged_in || !$student_id) {
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit;
}

if (empty($note)) {
    echo json_encode(['success' => false, 'message' => 'Note cannot be empty']);
    exit;
}

try {
    // Enhanced debug logging for lesson verification
    error_log("Save Notes DEBUG: Checking lesson_id = $lesson_id");
    
    // First, let's see what videos exist in the database
    $debug_sql = "SELECT video_id, title, course_id FROM videos LIMIT 10";
    $debug_stmt = $pdo->prepare($debug_sql);
    $debug_stmt->execute();
    $existing_videos = $debug_stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("Save Notes DEBUG: Found videos in database: " . json_encode($existing_videos));
    
    // Simplified verification - just check if the lesson exists (using correct column name)
    $verify_sql = "SELECT video_id, title, course_id FROM videos WHERE video_id = ?";
    
    $verify_stmt = $pdo->prepare($verify_sql);
    $verify_stmt->execute([$lesson_id]);
    $found_video = $verify_stmt->fetch(PDO::FETCH_ASSOC);
    
    error_log("Save Notes DEBUG: Query result for lesson_id $lesson_id: " . json_encode($found_video));
    
    if (!$found_video) {
        error_log("Save Notes DEBUG: No video found with lesson_id = $lesson_id");
        echo json_encode(['success' => false, 'message' => 'Lesson not found']);
        exit;
    }
    
    error_log("Save Notes DEBUG: Video found: " . json_encode($found_video));
    
    // Create video_notes table if it doesn't exist (without foreign key constraints for now)
    $create_table_sql = "CREATE TABLE IF NOT EXISTS video_notes (
        id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        lesson_id INT NOT NULL,
        timestamp INT NOT NULL,
        formatted_time VARCHAR(20),
        note TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_student_lesson (student_id, lesson_id),
        INDEX idx_timestamp (timestamp)
    )";
    
    $pdo->exec($create_table_sql);
    
    // Check if note already exists at this timestamp
    $check_sql = "SELECT id FROM video_notes WHERE student_id = ? AND lesson_id = ? AND timestamp = ?";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->execute([$student_id, $lesson_id, $timestamp]);
    
    if ($existing_id = $check_stmt->fetchColumn()) {
        // Update existing note
        $update_sql = "UPDATE video_notes SET note = ?, formatted_time = ?, updated_at = CURRENT_TIMESTAMP 
                       WHERE id = ?";
        $update_stmt = $pdo->prepare($update_sql);
        $update_stmt->execute([$note, $formatted_time, $existing_id]);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Note updated successfully',
            'action' => 'updated',
            'note_id' => $existing_id
        ]);
    } else {
        // Insert new note
        $insert_sql = "INSERT INTO video_notes (student_id, lesson_id, timestamp, formatted_time, note) 
                       VALUES (?, ?, ?, ?, ?)";
        $insert_stmt = $pdo->prepare($insert_sql);
        $insert_stmt->execute([$student_id, $lesson_id, $timestamp, $formatted_time, $note]);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Note saved successfully',
            'action' => 'created',
            'note_id' => $pdo->lastInsertId()
        ]);
    }
    
} catch (PDOException $e) {
    error_log("Video note save error: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Database error: ' . $e->getMessage(),
        'error_code' => $e->getCode(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
}
?>
