<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Initialize page title for header
$page_title = 'Account Settings';

// Add page-specific CSS
$page_specific_css = '<link rel="stylesheet" href="assets/css/pages/settings.css">';

// Include database connection
require_once '../config/db_connect.php';

// Get user ID from session - should be set if user is logged in
$userId = $_SESSION['user_id'];

// Fetch user data from database
$user_data = [
    'first_name' => '',
    'last_name' => '',
    'email' => '',
    'username' => '',
    'profile_picture' => null
];

// Use prepared statement to fetch user data
$query = "SELECT user_id, first_name, last_name, email, username, profile_picture, user_role FROM users WHERE user_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$userResult = $stmt->get_result();

if ($userResult && $userResult->num_rows > 0) {
    $userData = $userResult->fetch_assoc();
    $user_data = $userData;
    
    // Update session variables for consistency
    $_SESSION['user_id'] = $userData['user_id'];
    $_SESSION['username'] = $userData['username'];
    $_SESSION['user_first_name'] = $userData['first_name'];
    $_SESSION['user_last_name'] = $userData['last_name'];
    $_SESSION['first_name'] = $userData['first_name'];
    $_SESSION['last_name'] = $userData['last_name'];
    $_SESSION['email'] = $userData['email'];
    $_SESSION['user_role'] = $userData['user_role'];
    
    // Check if student profile exists
    $checkProfileQuery = "SELECT student_id FROM student_profiles WHERE student_id = ?";
    $checkStmt = $conn->prepare($checkProfileQuery);
    $checkStmt->bind_param("i", $userId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult && $checkResult->num_rows === 0) {
        // Create empty student profile
        $createProfileQuery = "INSERT INTO student_profiles (student_id) VALUES (?)";
        $createStmt = $conn->prepare($createProfileQuery);
        $createStmt->bind_param("i", $userId);
        $createStmt->execute();
    }
}
?>

<?php
// Set current page for navigation
$currentPage = 'settings';

// Include header at the end of PHP logic, following the pattern of other working pages
include_once 'components/header.php';
?>

<style>
/* Modern Settings Page Styling */
.settings-wrapper {
    background: #f8fafc;
    min-height: 100vh;
    padding: 2rem 0;
}

.settings-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem;
}

.settings-header {
    text-align: center;
    margin-bottom: 3rem;
    background: rgba(245, 158, 11, 0.08);
    border-radius: 20px;
    padding: 2.5rem;
    border: 1px solid rgba(245, 158, 11, 0.15);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.settings-header h1 {
    color: #1f2937;
    font-size: 2.5rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
}

.settings-header p {
    color: #6b7280;
    font-size: 1.2rem;
    margin: 0;
}

.settings-layout {
    display: grid;
    grid-template-columns: 250px 1fr;
    gap: 2rem;
    align-items: start;
}

.settings-nav {
    background: #ffffff;
    border-radius: 15px;
    padding: 1.5rem;
    border: 1px solid #e2e8f0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    position: sticky;
    top: 2rem;
}

.nav-pills .nav-link {
    background: none;
    border: none;
    color: #64748b;
    padding: 1rem 1.5rem;
    margin-bottom: 0.5rem;
    border-radius: 12px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    transition: all 0.3s ease;
    text-align: left;
    width: 100%;
}

.nav-pills .nav-link:hover {
    background: rgba(79, 115, 223, 0.1);
    color: #4e73df;
    transform: translateX(5px);
}

.nav-pills .nav-link.active {
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    color: white;
    box-shadow: 0 5px 15px rgba(78, 115, 223, 0.3);
}

.nav-pills .nav-link i {
    font-size: 1.2rem;
}

.settings-content {
    background: #ffffff;
    border-radius: 20px;
    padding: 0;
    border: 1px solid #e2e8f0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    overflow: hidden;
}

.settings-card {
    padding: 2.5rem;
    border: none;
    background: none;
}

.settings-title {
    color: #2c3e50;
    font-size: 1.8rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.settings-subtitle {
    color: #64748b;
    font-size: 1.1rem;
    margin-bottom: 2rem;
}

.form-label {
    color: #374151;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.form-control, .form-select {
    border: 2px solid #e2e8f0;
    border-radius: 10px;
    padding: 0.75rem 1rem;
    font-size: 1rem;
    transition: all 0.3s ease;
    background: #ffffff;
}

.form-control:focus, .form-select:focus {
    border-color: #4e73df;
    box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.25);
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 10px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(78, 115, 223, 0.4);
}

.password-strength {
    margin-top: 0.5rem;
}

.progress {
    height: 6px;
    border-radius: 3px;
    background: #e2e8f0;
}

.progress-bar {
    border-radius: 3px;
    transition: all 0.3s ease;
}

.strength-text {
    font-size: 0.9rem;
    font-weight: 600;
    margin-top: 0.5rem;
}

@media (max-width: 768px) {
    .settings-layout {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .settings-nav {
        position: static;
    }
    
    .settings-container {
        padding: 0 1rem;
    }
    
    .settings-header {
        padding: 2rem 1.5rem;
    }
    
    .settings-header h1 {
        font-size: 2rem;
    }
}
</style>

<!-- Modern Settings Container -->
<div class="settings-wrapper">
    <div class="settings-container">
        <!-- Modern Header -->
        <div class="settings-header">
            <h1><i class="bx bx-shield-check"></i> Security Settings</h1>
            <p>Manage your password and account security</p>
        </div>
        
        <div class="settings-layout">
            <!-- Clean Navigation -->
            <div class="settings-nav">
                <div class="nav flex-column nav-pills" id="settings-tab" role="tablist" aria-orientation="vertical">
                    <button class="nav-link active" id="security-tab" data-bs-toggle="pill" data-bs-target="#security" type="button" role="tab" aria-controls="security" aria-selected="true">
                        <i class="bx bx-shield-check"></i> Security
                    </button>
                </div>
            </div>
            <!-- Modern Settings Content -->
            <div class="settings-content">
                <div class="tab-content" id="settings-tabContent">
                    <!-- Security Settings -->
                    <div class="tab-pane fade show active" id="security" role="tabpanel" aria-labelledby="security-tab">
                        <div class="settings-card">
                            <h3 class="settings-title"><i class="bx bx-shield-check"></i> Password Security</h3>
                            <p class="settings-subtitle">Update your password to keep your account secure</p>
                            
                            <form id="passwordForm" class="settings-form">
                                <div class="mb-3">
                                    <label for="currentPassword" class="form-label">Current Password</label>
                                    <input type="password" class="form-control" id="currentPassword">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="newPassword" class="form-label">New Password</label>
                                    <input type="password" class="form-control" id="newPassword">
                                    <div class="password-strength mt-2">
                                        <div class="progress">
                                            <div class="progress-bar" role="progressbar" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                                        </div>
                                        <small class="text-muted">Password strength: <span id="passwordStrength">Very Weak</span></small>
                                    </div>
                                </div>
                                
                                <div class="mb-4">
                                    <label for="confirmPassword" class="form-label">Confirm New Password</label>
                                    <input type="password" class="form-control" id="confirmPassword">
                                </div>
                                
                                <div class="d-flex justify-content-end">
                                    <button type="button" class="btn btn-primary" id="savePasswordBtn">Update Password</button>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Delete Account Section -->
                        <div class="settings-card mt-4 border-danger">
                            <h3 class="settings-title text-danger"><i class="bx bx-trash"></i> Delete Account</h3>
                            <p class="settings-subtitle">Permanently delete your account and all associated data</p>
                            
                            <div class="alert alert-danger" role="alert">
                                <strong>Warning!</strong> This action cannot be undone. All your courses, progress, and account data will be permanently deleted.
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="confirmDelete">
                                    <label class="form-check-label" for="confirmDelete">
                                        I understand that this action is permanent and cannot be reversed
                                    </label>
                                </div>
                            </div>
                            
                            <div class="d-flex justify-content-end">
                                <button type="button" class="btn btn-danger" id="deleteAccountBtn" disabled>Delete My Account</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Close home-content div -->
        </div>
    </section>
</body>
</html>

<!-- JavaScript for Settings Page -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Password strength meter
    const passwordInput = document.getElementById('newPassword');
    const progressBar = document.querySelector('.password-strength .progress-bar');
    const strengthText = document.getElementById('passwordStrength');
    
    if (passwordInput) {
        passwordInput.addEventListener('input', function() {
            const password = this.value;
            let strength = 0;
            let status = '';
            
            if (password.length > 0) {
                // Length check
                if (password.length >= 8) strength += 25;
                
                // Lowercase letters check
                if (password.match(/[a-z]+/)) strength += 25;
                
                // Uppercase letters check
                if (password.match(/[A-Z]+/)) strength += 25;
                
                // Numbers and special characters check
                if (password.match(/[0-9]+/) || password.match(/[$@#&!]+/)) strength += 25;
                
                // Set the status text
                if (strength <= 25) {
                    status = 'Very Weak';
                    progressBar.className = 'progress-bar bg-danger';
                } else if (strength <= 50) {
                    status = 'Weak';
                    progressBar.className = 'progress-bar bg-warning';
                } else if (strength <= 75) {
                    status = 'Medium';
                    progressBar.className = 'progress-bar bg-info';
                } else {
                    status = 'Strong';
                    progressBar.className = 'progress-bar bg-success';
                }
            } else {
                status = 'Very Weak';
                strength = 0;
                progressBar.className = 'progress-bar';
            }
            
            progressBar.style.width = strength + '%';
            progressBar.setAttribute('aria-valuenow', strength);
            strengthText.textContent = status;
        });
    }
    
    // Toggle delete account button based on confirmation checkbox
    const confirmDelete = document.getElementById('confirmDelete');
    const deleteAccountBtn = document.getElementById('deleteAccountBtn');
    
    if (confirmDelete && deleteAccountBtn) {
        confirmDelete.addEventListener('change', function() {
            deleteAccountBtn.disabled = !this.checked;
        });
        
        // Delete account functionality
        deleteAccountBtn.addEventListener('click', function() {
            if (confirmDelete.checked) {
                if (confirm('Are you absolutely sure you want to delete your account? This action cannot be undone.')) {
                    // Here you would typically make an AJAX request to delete the account
                    alert('Account deletion functionality would be implemented here.');
                }
            }
        });
    }
    
    // Save buttons functionality
    const saveButtons = document.querySelectorAll('#savePasswordBtn');
    
    saveButtons.forEach(button => {
        if (button) {
            button.addEventListener('click', function() {
                // Here you would typically make an AJAX request to save settings
                // For now, just show a success message
                
                // Show success notification
                const toastHTML = `
                    <div class="toast align-items-center text-white bg-success border-0 position-fixed bottom-0 end-0 m-3" role="alert" aria-live="assertive" aria-atomic="true">
                        <div class="d-flex">
                            <div class="toast-body">
                                <i class="bx bx-check-circle me-2"></i> Settings saved successfully!
                            </div>
                            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                        </div>
                    </div>
                `;
                
                document.body.insertAdjacentHTML('beforeend', toastHTML);
                const toast = new bootstrap.Toast(document.querySelector('.toast'));
                toast.show();
            });
        }
    });
});
</script>

<?php
// Include footer
include_once 'components/footer.php';
?>
