<?php
/**
 * Simple & Reliable Certificate Download - Panadite Academy
 * Uses step-by-step queries to avoid SQL join issues
 */

// Start session and include database
require_once '../includes/session_start.php';
require_once '../config/db_connect.php';

// Get parameters
$course_id = intval($_GET['course_id'] ?? 0);
$user_id = $_SESSION['user_id'] ?? null;

// Validate user session
if (!$user_id || !$course_id) {
    http_response_code(400);
    die('Invalid request. Please login and specify a course.');
}

// Simple step-by-step validation and data collection
try {
    
    // Step 1: Get student data
    $student_data = null;
    if ($conn) {
        $user_query = "SELECT first_name, last_name, email FROM users WHERE user_id = $user_id";
        $user_result = $conn->query($user_query);
        if ($user_result && $user_result->num_rows > 0) {
            $student_data = $user_result->fetch_assoc();
        }
    }
    
    if (!$student_data) {
        throw new Exception("Student data not found for user ID: $user_id");
    }
    
    // Step 2: Get course data
    $course_data = null;
    if ($conn) {
        $course_query = "SELECT title, description FROM courses WHERE course_id = $course_id";
        $course_result = $conn->query($course_query);
        if ($course_result && $course_result->num_rows > 0) {
            $course_data = $course_result->fetch_assoc();
        }
    }
    
    if (!$course_data) {
        throw new Exception("Course data not found for course ID: $course_id");
    }
    
    // Step 3: Check enrollment (FIXED: enrolled_date not enrollment_date)
    $enrollment_data = null;
    if ($conn) {
        $enrollment_query = "SELECT enrolled_date FROM enrollments WHERE user_id = $user_id AND course_id = $course_id";
        $enrollment_result = $conn->query($enrollment_query);
        if ($enrollment_result && $enrollment_result->num_rows > 0) {
            $enrollment_data = $enrollment_result->fetch_assoc();
        }
    }
    
    if (!$enrollment_data) {
        throw new Exception("Student not enrolled in this course. User ID: $user_id, Course ID: $course_id");
    }
    
    // Generate and download certificate
    generateAndDownloadCertificate($student_data, $course_data, $enrollment_data, $course_id);
    
} catch (Exception $e) {
    http_response_code(500);
    echo "Error generating certificate: " . $e->getMessage();
    echo "<br><br><a href='debug_certificate.php?course_id=$course_id'>🔍 Run Debug Script</a>";
}

function generateAndDownloadCertificate($student_data, $course_data, $enrollment_data, $course_id) {
    
    $student_name = trim($student_data['first_name'] . ' ' . $student_data['last_name']);
    $course_title = $course_data['title'];
    $completion_date = date('F j, Y', strtotime($enrollment_data['enrolled_date']));
    $certificate_id = 'CERT-' . strtoupper(substr(md5($student_name . $course_title . time()), 0, 8));
    
    // Generate filename
    $safe_course_title = preg_replace('/[^A-Za-z0-9\-_]/', '_', $course_title);
    $safe_student_name = preg_replace('/[^A-Za-z0-9\-_]/', '_', $student_name);
    $filename = "Certificate_{$safe_course_title}_{$safe_student_name}_" . date('Y-m-d') . ".pdf";
    
    // Create beautiful HTML certificate that downloads as PDF-like
    $html = "
<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Certificate - $student_name</title>
    <style>
        @page { 
            size: A4 landscape; 
            margin: 0; 
        }
        body { 
            font-family: 'Times New Roman', serif; 
            margin: 0; 
            padding: 0;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
        }
        .certificate {
            width: 1000px;
            height: 700px;
            background: #ffffff;
            margin: 0 auto;
            padding: 60px;
            border: 15px solid #2c3e50;
            border-radius: 20px;
            box-shadow: 0 30px 60px rgba(0,0,0,0.2);
            position: relative;
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
        }
        .decorative-border {
            position: absolute;
            top: 25px;
            left: 25px;
            right: 25px;
            bottom: 25px;
            border: 3px solid #3498db;
            border-radius: 15px;
        }
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        .academy-name {
            font-size: 48px;
            font-weight: bold;
            color: #2c3e50;
            margin: 0;
            text-transform: uppercase;
            letter-spacing: 8px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
        }
        .tagline {
            font-size: 20px;
            color: #7f8c8d;
            margin: 15px 0 0 0;
            font-style: italic;
        }
        .certificate-title {
            font-size: 42px;
            font-weight: bold;
            text-align: center;
            color: #e74c3c;
            margin: 50px 0;
            text-transform: uppercase;
            letter-spacing: 4px;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
        }
        .content {
            text-align: center;
            line-height: 2;
        }
        .certify-text {
            font-size: 24px;
            color: #34495e;
            margin-bottom: 30px;
        }
        .student-name {
            font-size: 56px;
            font-weight: bold;
            color: #2c3e50;
            margin: 30px 0;
            border-bottom: 4px solid #3498db;
            display: inline-block;
            padding-bottom: 15px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
        }
        .completed-text {
            font-size: 24px;
            color: #34495e;
            margin: 30px 0;
        }
        .course-title {
            font-size: 36px;
            font-weight: bold;
            color: #3498db;
            margin: 30px 0;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
        }
        .date {
            font-size: 20px;
            color: #7f8c8d;
            margin: 40px 0;
        }
        .footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 60px;
            padding-top: 40px;
            border-top: 3px solid #ecf0f1;
        }
        .signature {
            text-align: center;
        }
        .signature-line {
            width: 250px;
            height: 3px;
            background: #2c3e50;
            margin: 30px auto 15px auto;
        }
        .signature-text {
            font-size: 18px;
            font-weight: bold;
            color: #2c3e50;
        }
        .cert-id {
            position: absolute;
            bottom: 30px;
            left: 30px;
            font-size: 14px;
            color: #bdc3c7;
        }
        .website {
            position: absolute;
            bottom: 30px;
            right: 30px;
            font-size: 14px;
            color: #bdc3c7;
        }
        .seal {
            position: absolute;
            bottom: 150px;
            left: 80px;
            width: 120px;
            height: 120px;
            border: 6px solid #e74c3c;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: rgba(231, 76, 60, 0.1);
            color: #e74c3c;
            font-weight: bold;
            font-size: 16px;
            text-align: center;
        }
        @media print {
            body { background: white; }
            .certificate { box-shadow: none; }
        }
    </style>
</head>
<body>
    <div class='certificate'>
        <div class='decorative-border'></div>
        
        <div class='header'>
            <h1 class='academy-name'>Panadite Academy</h1>
            <p class='tagline'>Excellence in Digital Education</p>
        </div>
        
        <h2 class='certificate-title'>Certificate of Completion</h2>
        
        <div class='content'>
            <p class='certify-text'>This is to certify that</p>
            
            <div class='student-name'>$student_name</div>
            
            <p class='completed-text'>has successfully completed the course</p>
            
            <div class='course-title'>$course_title</div>
            
            <p class='date'>Date of Completion: $completion_date</p>
        </div>
        
        <div class='footer'>
            <div class='signature'>
                <div class='signature-line'></div>
                <p class='signature-text'>Panadite Academy</p>
            </div>
            <div class='signature'>
                <div class='signature-line'></div>
                <p class='signature-text'>Date Issued: " . date('F j, Y') . "</p>
            </div>
        </div>
        
        <div class='seal'>
            OFFICIAL<br>SEAL
        </div>
        
        <div class='cert-id'>Certificate ID: $certificate_id</div>
        <div class='website'>Verify at: panaditeacademy.com</div>
    </div>
    
    <script>
        // Auto-print when page loads (creates PDF-like experience)
        window.onload = function() {
            setTimeout(function() {
                window.print();
            }, 1000);
        };
    </script>
</body>
</html>
    ";
    
    // Set headers for download
    header('Content-Type: text/html; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '.html"');
    header('Cache-Control: private, max-age=0, must-revalidate');
    
    // Output the certificate
    echo $html;
}

?>
