<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
require_once '../auth/functions.php';
requireRole('teacher');
require_once '../config/db_connect.php';

// Get current teacher ID
$teacher_id = $_SESSION['user_id'];

// Get attempt ID from URL
$attempt_id = isset($_GET['attempt_id']) ? intval($_GET['attempt_id']) : 0;

if (!$attempt_id) {
    header('Location: quiz_submissions.php');
    exit;
}

// Get quiz attempt details with verification that it belongs to teacher's quiz
$attempt_query = "
    SELECT 
        qa.*,
        q.title as quiz_title,
        q.description as quiz_description,
        q.passing_score,
        q.teacher_id,
        c.title as course_title,
        u.first_name,
        u.last_name,
        u.email,
        COUNT(DISTINCT qq.question_id) as total_questions,
        SUM(qq.points) as total_possible_points
    FROM quiz_attempts qa
    JOIN quizzes q ON qa.quiz_id = q.quiz_id
    JOIN courses c ON q.course_id = c.course_id
    JOIN users u ON qa.student_id = u.user_id
    LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
    WHERE qa.attempt_id = ? AND q.teacher_id = ?
    GROUP BY qa.attempt_id
";

$attempt_stmt = $pdo->prepare($attempt_query);
$attempt_stmt->execute([$attempt_id, $teacher_id]);
$attempt = $attempt_stmt->fetch(PDO::FETCH_ASSOC);

if (!$attempt) {
    header('Location: quiz_submissions.php');
    exit;
}

// Get quiz questions and student responses
$questions_query = "
    SELECT 
        qq.*,
        qr.response_id,
        qr.answer_id,
        qr.text_response,
        qr.is_correct,
        qr.points_earned,
        qo_student.option_text as student_answer,
        qo_correct.option_text as correct_answer
    FROM quiz_questions qq
    LEFT JOIN quiz_responses qr ON qq.question_id = qr.question_id AND qr.attempt_id = ?
    LEFT JOIN quiz_question_options qo_student ON qr.answer_id = qo_student.option_id
    LEFT JOIN quiz_question_options qo_correct ON qq.question_id = qo_correct.question_id AND qo_correct.is_correct = 1
    WHERE qq.quiz_id = ?
    ORDER BY qq.question_id
";

$questions_stmt = $pdo->prepare($questions_query);
$questions_stmt->execute([$attempt_id, $attempt['quiz_id']]);
$questions = $questions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all options for multiple choice questions
$options_query = "
    SELECT qo.*, qq.question_id
    FROM quiz_question_options qo
    JOIN quiz_questions qq ON qo.question_id = qq.question_id
    WHERE qq.quiz_id = ?
    ORDER BY qq.question_id, qo.option_id
";

$options_stmt = $pdo->prepare($options_query);
$options_stmt->execute([$attempt['quiz_id']]);
$all_options = $options_stmt->fetchAll(PDO::FETCH_ASSOC);

// Organize options by question
$question_options = [];
foreach ($all_options as $option) {
    $question_options[$option['question_id']][] = $option;
}

$page_title = "Quiz Results - " . $attempt['quiz_title'];
?>

<style>
    .results-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 2rem 0;
        margin-bottom: 2rem;
        border-radius: 15px;
    }

    .student-info {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .score-summary {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .score-circle {
        width: 120px;
        height: 120px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.5rem;
        font-weight: bold;
        color: white;
        margin: 0 auto;
    }

    .score-passed { background: linear-gradient(135deg, #10b981 0%, #059669 100%); }
    .score-failed { background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); }

    .question-card {
        background: white;
        border-radius: 15px;
        padding: 2rem;
        margin-bottom: 1.5rem;
        box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        border-left: 4px solid #e5e7eb;
    }

    .question-correct {
        border-left-color: #10b981;
        background: rgba(16, 185, 129, 0.03);
    }

    .question-incorrect {
        border-left-color: #ef4444;
        background: rgba(239, 68, 68, 0.03);
    }

    .question-header {
        display: flex;
        justify-content: between;
        align-items: flex-start;
        margin-bottom: 1.5rem;
    }

    .question-number {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 600;
        color: white;
        margin-right: 1rem;
    }

    .question-number.correct { background: #10b981; }
    .question-number.incorrect { background: #ef4444; }

    .question-text {
        flex: 1;
        font-size: 1.1rem;
        font-weight: 500;
        line-height: 1.6;
    }

    .answer-status {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        margin-left: auto;
    }

    .status-badge {
        padding: 0.25rem 0.75rem;
        border-radius: 50px;
        font-weight: 600;
        font-size: 0.875rem;
    }

    .status-correct { background: #d1fae5; color: #065f46; }
    .status-incorrect { background: #fee2e2; color: #991b1b; }

    .option-item {
        background: #f8fafc;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        padding: 1rem;
        margin-bottom: 0.75rem;
        transition: all 0.2s ease;
    }

    .option-correct {
        background: rgba(16, 185, 129, 0.1);
        border-color: #10b981;
        color: #065f46;
    }

    .option-student {
        background: rgba(59, 130, 246, 0.1);
        border-color: #3b82f6;
        border-width: 3px;
    }

    .option-student-wrong {
        background: rgba(239, 68, 68, 0.1);
        border-color: #ef4444;
        border-width: 3px;
    }

    .text-answer {
        background: #f8fafc;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        padding: 1rem;
        min-height: 100px;
        font-family: inherit;
    }

    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        background: #6b7280;
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 500;
        transition: all 0.2s ease;
    }

    .back-button:hover {
        background: #4b5563;
        color: white;
        transform: translateY(-1px);
    }
</style>

<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="results-header">
        <div class="container">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="display-6 fw-bold mb-0">
                        <i class="bx bx-bar-chart-alt-2"></i>
                        Quiz Results
                    </h1>
                    <p class="mb-0 opacity-75">Detailed student performance analysis</p>
                </div>
                <a href="quiz_submissions.php" class="back-button">
                    <i class="bx bx-arrow-back"></i>
                    Back to Submissions
                </a>
            </div>
        </div>
    </div>

    <div class="container">
        <!-- Student Information -->
        <div class="student-info">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h4 class="mb-3">
                        <i class="bx bx-user"></i>
                        Student Information
                    </h4>
                    <div class="mb-2">
                        <strong>Name:</strong> <?= htmlspecialchars($attempt['first_name'] . ' ' . $attempt['last_name']) ?>
                    </div>
                    <div class="mb-2">
                        <strong>Email:</strong> <?= htmlspecialchars($attempt['email']) ?>
                    </div>
                    <div class="mb-2">
                        <strong>Quiz:</strong> <?= htmlspecialchars($attempt['quiz_title']) ?>
                    </div>
                    <div class="mb-0">
                        <strong>Course:</strong> <?= htmlspecialchars($attempt['course_title']) ?>
                    </div>
                </div>
                <div class="col-md-6">
                    <h4 class="mb-3">
                        <i class="bx bx-time"></i>
                        Submission Details
                    </h4>
                    <div class="mb-2">
                        <strong>Started:</strong> <?= date('M d, Y g:i A', strtotime($attempt['start_time'])) ?>
                    </div>
                    <div class="mb-2">
                        <strong>Completed:</strong> <?= date('M d, Y g:i A', strtotime($attempt['end_time'])) ?>
                    </div>
                    <div class="mb-0">
                        <strong>Duration:</strong> 
                        <?php
                        $duration = strtotime($attempt['end_time']) - strtotime($attempt['start_time']);
                        $minutes = floor($duration / 60);
                        $seconds = $duration % 60;
                        echo "{$minutes} minutes {$seconds} seconds";
                        ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Score Summary -->
        <div class="score-summary">
            <div class="row align-items-center">
                <div class="col-md-3 text-center">
                    <div class="score-circle <?= $attempt['pass_status'] ? 'score-passed' : 'score-failed' ?>">
                        <?= number_format($attempt['score'], 1) ?>%
                    </div>
                    <div class="mt-2">
                        <?php if ($attempt['pass_status']): ?>
                            <span class="status-badge status-correct">
                                <i class="bx bx-check"></i> PASSED
                            </span>
                        <?php else: ?>
                            <span class="status-badge status-incorrect">
                                <i class="bx bx-x"></i> FAILED
                            </span>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="col-md-9">
                    <div class="row">
                        <div class="col-md-3">
                            <div class="text-center">
                                <div class="fs-4 fw-bold text-primary"><?= $attempt['total_questions'] ?></div>
                                <div class="text-muted">Total Questions</div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="text-center">
                                <div class="fs-4 fw-bold text-success">
                                    <?= count(array_filter($questions, function($q) { return $q['is_correct']; })) ?>
                                </div>
                                <div class="text-muted">Correct Answers</div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="text-center">
                                <div class="fs-4 fw-bold text-primary"><?= $attempt['total_possible_points'] ?></div>
                                <div class="text-muted">Total Points</div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="text-center">
                                <div class="fs-4 fw-bold text-warning"><?= $attempt['passing_score'] ?>%</div>
                                <div class="text-muted">Passing Score</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Question Results -->
        <div class="mb-4">
            <h4 class="mb-3">
                <i class="bx bx-list-check"></i>
                Question-by-Question Analysis
            </h4>
            
            <?php $question_num = 1; ?>
            <?php foreach ($questions as $question): ?>
                <div class="question-card <?= $question['is_correct'] ? 'question-correct' : 'question-incorrect' ?>">
                    <div class="question-header">
                        <div class="question-number <?= $question['is_correct'] ? 'correct' : 'incorrect' ?>">
                            <?= $question_num ?>
                        </div>
                        <div class="question-text">
                            <?= nl2br(htmlspecialchars($question['question_text'])) ?>
                        </div>
                        <div class="answer-status">
                            <?php if ($question['is_correct']): ?>
                                <span class="status-badge status-correct">
                                    <i class="bx bx-check"></i> Correct
                                </span>
                            <?php else: ?>
                                <span class="status-badge status-incorrect">
                                    <i class="bx bx-x"></i> Incorrect
                                </span>
                            <?php endif; ?>
                            <span class="badge bg-primary">
                                <?= $question['points_earned'] ?: 0 ?>/<?= $question['points'] ?> points
                            </span>
                        </div>
                    </div>

                    <?php if ($question['question_type'] === 'multiple_choice' || $question['question_type'] === 'true_false'): ?>
                        <?php if (isset($question_options[$question['question_id']])): ?>
                            <?php foreach ($question_options[$question['question_id']] as $option): ?>
                                <?php
                                $is_student_answer = $question['answer_id'] == $option['option_id'];
                                $is_correct_option = $option['is_correct'];
                                $option_class = 'option-item';
                                
                                if ($is_correct_option) {
                                    $option_class .= ' option-correct';
                                }
                                if ($is_student_answer && !$is_correct_option) {
                                    $option_class .= ' option-student-wrong';
                                } elseif ($is_student_answer) {
                                    $option_class .= ' option-student';
                                }
                                ?>
                                
                                <div class="<?= $option_class ?>">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span><?= htmlspecialchars($option['option_text']) ?></span>
                                        <div class="d-flex gap-2">
                                            <?php if ($is_student_answer): ?>
                                                <span class="badge bg-secondary">Student's Answer</span>
                                            <?php endif; ?>
                                            <?php if ($is_correct_option): ?>
                                                <span class="badge bg-success">
                                                    <i class="bx bx-check"></i> Correct
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>

                    <?php elseif ($question['question_type'] === 'short_answer'): ?>
                        <div class="mb-3">
                            <strong>Student's Answer:</strong>
                            <div class="text-answer">
                                <?= htmlspecialchars($question['text_response'] ?: 'No answer provided') ?>
                            </div>
                            <small class="text-muted mt-2 d-block">
                                Note: Short answer questions require manual grading by the instructor.
                            </small>
                        </div>
                    <?php endif; ?>
                </div>
                <?php $question_num++; ?>
            <?php endforeach; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Animate elements on load
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.question-card, .student-info, .score-summary');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
    </script>
</body>
</html>
