<?php
// Use session_start.php for consistent session handling
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Initialize variables
$error = null;
$video_id = null;
$redirect_url = 'videos.php';

// Check if a video ID was provided
if (isset($_POST['video_id']) && !empty($_POST['video_id'])) {
    $video_id = intval($_POST['video_id']);
} else {
    $_SESSION['video_error'] = "No video was specified for purchase.";
    header("Location: $redirect_url");
    exit;
}

// Get the logged-in user ID
$user_id = $_SESSION['user_id'] ?? 0;

if (!$user_id) {
    $_SESSION['video_error'] = "You must be logged in to purchase a video.";
    header("Location: $redirect_url");
    exit;
}

try {
    // Start transaction
    $conn->begin_transaction();

    // Check if video exists and is published
    $videoQuery = "SELECT * FROM videos WHERE video_id = ? AND is_published = 1";
    $stmt = $conn->prepare($videoQuery);
    $stmt->bind_param("i", $video_id);
    $stmt->execute();
    $videoResult = $stmt->get_result();

    if ($videoResult->num_rows === 0) {
        throw new Exception("The requested video does not exist or is not available for purchase.");
    }

    $video = $videoResult->fetch_assoc();
    
    // Check if the video is free (no purchase needed)
    if ($video['is_free']) {
        // For free videos, just record a purchase with 0.00 amount
        $checkPurchaseQuery = "SELECT * FROM video_purchases WHERE user_id = ? AND video_id = ?";
        $stmt = $conn->prepare($checkPurchaseQuery);
        $stmt->bind_param("ii", $user_id, $video_id);
        $stmt->execute();
        $purchaseResult = $stmt->get_result();

        if ($purchaseResult->num_rows === 0) {
            // Insert purchase record with 0.00 amount and completed status
            $insertQuery = "INSERT INTO video_purchases (user_id, video_id, amount, payment_id, payment_status) 
                            VALUES (?, ?, 0.00, 'free-video', 'completed')";
            $stmt = $conn->prepare($insertQuery);
            $stmt->bind_param("ii", $user_id, $video_id);
            $stmt->execute();
            $_SESSION['video_success'] = "You now have access to this free video.";
        } else {
            $_SESSION['video_info'] = "You already have access to this video.";
        }
    } else {
        // Check if already purchased
        $checkPurchaseQuery = "SELECT * FROM video_purchases WHERE user_id = ? AND video_id = ?";
        $stmt = $conn->prepare($checkPurchaseQuery);
        $stmt->bind_param("ii", $user_id, $video_id);
        $stmt->execute();
        $purchaseResult = $stmt->get_result();

        if ($purchaseResult->num_rows > 0) {
            // Already purchased
            $purchase = $purchaseResult->fetch_assoc();
            if ($purchase['payment_status'] === 'completed') {
                $_SESSION['video_info'] = "You already own this video.";
            } else {
                $_SESSION['video_info'] = "You have a pending purchase for this video.";
            }
        } else {
            // In a real application, here we would:
            // 1. Integrate with a payment gateway
            // 2. Redirect to payment page 
            // 3. Process payment
            // 4. Update purchase record on successful payment
            
            // For demo purposes, we'll simulate a successful payment
            $paymentId = 'demo-' . time() . '-' . rand(1000, 9999);
            $amount = $video['price'] ?? 9.99;
            
            // Insert purchase record
            $insertQuery = "INSERT INTO video_purchases (user_id, video_id, amount, payment_id, payment_status) 
                          VALUES (?, ?, ?, ?, 'completed')";
            $stmt = $conn->prepare($insertQuery);
            $stmt->bind_param("iids", $user_id, $video_id, $amount, $paymentId);
            $stmt->execute();
            
            // Increment video views
            $updateViewsQuery = "UPDATE videos SET views = views + 1 WHERE video_id = ?";
            $stmt = $conn->prepare($updateViewsQuery);
            $stmt->bind_param("i", $video_id);
            $stmt->execute();
            
            $_SESSION['video_success'] = "Video purchased successfully! You now have access to watch it.";
        }
    }

    // Commit transaction
    $conn->commit();
    
    // If the request came from the video detail page, redirect back there
    if (isset($_POST['redirect']) && !empty($_POST['redirect'])) {
        $redirect_url = $_POST['redirect'];
    }
    
    header("Location: $redirect_url");
    exit;
    
} catch (Exception $e) {
    // Rollback transaction on error
    $conn->rollback();
    
    $_SESSION['video_error'] = "Error during purchase: " . $e->getMessage();
    error_log("Video purchase error: " . $e->getMessage());
    header("Location: $redirect_url");
    exit;
}
?>
