<?php
// Use session_start.php for consistent session handling
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Add custom CSS for the videos page - add cache-busting parameter to force reload
$cacheBuster = time(); // Use current timestamp to prevent caching
$customCss = '<link rel="stylesheet" href="../assets/css/pages/videos.css?v=' . $cacheBuster . '">';

// Add inline critical CSS to ensure styling is applied immediately
$customCss .= '<style>
    :root {
        --primary-color: #3a7bd5;
        --primary-gradient: linear-gradient(135deg, #3a7bd5 0%, #2d5ba0 100%);
        --card-shadow: 0 8px 20px rgba(0, 0, 0, 0.08);
        --hover-shadow: 0 15px 30px rgba(0, 0, 0, 0.12);
        --border-radius: 16px;
        --card-border-radius: 12px;
        --transition: all 0.3s ease;
    }
    
    .video-card {
        border-radius: var(--card-border-radius);
        overflow: hidden;
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        background-color: #ffffff;
        height: 100%;
        border: 1px solid rgba(232, 232, 232, 0.7);
        display: flex;
        flex-direction: column;
    }
    
    .video-card:hover {
        transform: translateY(-8px);
        box-shadow: var(--hover-shadow);
    }
    
    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        background-color: #fff;
        border-radius: var(--card-border-radius);
        box-shadow: var(--card-shadow);
        margin-top: 1rem;
        position: relative;
        overflow: hidden;
    }
</style>';

// Initialize variables
$videos = [];
$categories = [];
$featuredVideo = null;
$error = null;
$videoProgress = [];

// Get user ID from session
$userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

if ($userId) {
    try {
        // Extract unique category names from videos table for filter
        $categoryQuery = "SELECT DISTINCT category_name FROM videos WHERE is_published = 1 ORDER BY category_name";
        $categoryResult = $conn->query($categoryQuery);
        
        if ($categoryResult) {
            while ($row = $categoryResult->fetch_assoc()) {
                if (!empty($row['category_name'])) {
                    $categories[] = ['name' => $row['category_name']];
                }
            }
        }
        
        // Get purchased videos for the student
        $videoQuery = "
            SELECT 
                v.video_id as id,
                v.title,
                v.description,
                v.thumbnail,
                v.video_url,
                v.duration,
                CONCAT(FLOOR(LENGTH(v.video_url)/1024/1024), ' MB') as file_size,
                v.created_at,
                v.is_free,
                v.category_name,
                v.views,
                u.first_name,
                u.last_name,
                vp.payment_status,
                vp.purchase_date as last_watched
            FROM videos v
            JOIN video_purchases vp ON v.video_id = vp.video_id 
            JOIN users u ON v.teacher_id = u.user_id
            WHERE vp.user_id = ? 
            AND vp.payment_status = 'completed'
            AND v.is_published = 1
            ORDER BY vp.purchase_date DESC
        ";
        
        $stmt = $conn->prepare($videoQuery);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result) {
            $videos = $result->fetch_all(MYSQLI_ASSOC);
            
            // Set the first video as featured if available
            if (!empty($videos)) {
                $featuredVideo = $videos[0];
            }
        }
        
        // Create a temporary video progress tracking table if it doesn't exist
        // This is for tracking video progress since there's no dedicated table in schema
        $progressTableCheck = "SHOW TABLES LIKE 'video_progress'";
        $progressResult = $conn->query($progressTableCheck);
        
        if ($progressResult->num_rows == 0) {
            // Table doesn't exist, create it
            $createProgressTable = "
                CREATE TABLE IF NOT EXISTS video_progress (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    student_id INT NOT NULL,
                    video_id INT NOT NULL,
                    last_position INT DEFAULT 0,
                    completed BOOLEAN DEFAULT FALSE,
                    last_watched TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
                    FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE CASCADE,
                    UNIQUE KEY (student_id, video_id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ";
            $conn->query($createProgressTable);
        }
        
        // Get video progress for all videos
        if (!empty($videos)) {
            $progressQuery = "
                SELECT 
                    video_id,
                    last_position,
                    completed
                FROM video_progress 
                WHERE student_id = ?
            ";
            
            $stmt = $conn->prepare($progressQuery);
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $progressResult = $stmt->get_result();
            
            if ($progressResult) {
                while ($row = $progressResult->fetch_assoc()) {
                    $videoProgress[$row['video_id']] = [
                        'last_position' => $row['last_position'],
                        'completed' => $row['completed']
                    ];
                }
            }
        }
    } catch (Exception $e) {
        $error = "Error fetching videos: " . $e->getMessage();
        error_log($error);
    }

// No demo data - we'll only use real data from the database
}

// No default categories - we only use data from the database

// Set featured video if not already set but we have videos
if (!isset($featuredVideo) && !empty($videos)) {
    // Find the most recently watched incomplete video
    foreach ($videos as $video) {
        $videoId = $video['id'];
        $isCompleted = isset($videoProgress[$videoId]) && $videoProgress[$videoId]['completed'];
        
        if (!$isCompleted) {
            $featuredVideo = $video;
            break;
        }
    }
    
    // If all videos are completed or no progress info, use the first one
    if (!isset($featuredVideo)) {
        $featuredVideo = $videos[0];
    }
}

// Get search query
$searchQuery = isset($_GET['search']) ? $_GET['search'] : '';
$categoryFilter = isset($_GET['category']) ? $_GET['category'] : '';

// Filter videos based on search query and category
if (!empty($searchQuery) || !empty($categoryFilter)) {
    $filteredVideos = [];
    
    foreach ($videos as $video) {
        $titleMatch = empty($searchQuery) || stripos($video['title'], $searchQuery) !== false;
        $categoryMatch = empty($categoryFilter) || $video['category_name'] === $categoryFilter;
        
        if ($titleMatch && $categoryMatch) {
            $filteredVideos[] = $video;
        }
    }
    
    $videos = $filteredVideos;
}

// Page title
$pageTitle = "My Videos";

// Include header
include_once 'components/header.php';
?>

<!-- Dynamic Video Banner Section -->
<?php if (isset($featuredVideo) && !empty($featuredVideo)): ?>
<div class="container-fluid py-4">
    <div class="video-banner">
        <div class="video-banner-bg" style="background-image: url('<?= $featuredVideo['thumbnail'] ?>'); background-position: center;"></div>
        <div class="video-banner-content">
            <h1>Continue Your Learning Journey</h1>
            <p><?= $featuredVideo['title'] ?></p>
            <a href="video-player.php?id=<?= $featuredVideo['id'] ?>" class="btn btn-featured">
                <i class="bx bx-play-circle"></i>
                <?php 
                $videoId = $featuredVideo['id'];
                $completed = isset($videoProgress[$videoId]) && $videoProgress[$videoId]['completed'];
                $hasProgress = isset($videoProgress[$videoId]) && $videoProgress[$videoId]['last_position'] > 0;
                
                if ($completed): ?>
                    Rewatch Now
                <?php elseif ($hasProgress): ?>
                    Continue Watching
                <?php else: ?>
                    Start Learning
                <?php endif; ?>
            </a>
        </div>
    </div>
</div>
<?php endif; ?>

<div class="container-fluid">
    <div class="page-header mb-4 d-flex justify-content-between align-items-center">
        <div class="page-title">
            <h1>My Video Library</h1>
            <p>Access your purchased video content and track your learning progress</p>
        </div>
        <div class="page-actions">
            <a href="browse-videos.php" class="btn btn-primary">
                <i class="bx bx-plus-circle"></i> Browse Available Videos
            </a>
        </div>
    </div>
    
    <?php if (isset($_SESSION['video_success'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $_SESSION['video_success'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php unset($_SESSION['video_success']); endif; ?>

    <?php if (isset($_SESSION['video_error'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $_SESSION['video_error'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php unset($_SESSION['video_error']); endif; ?>

    <!-- Search and Filter -->
    <div class="search-filter-area">
        <form action="" method="GET" class="row g-3">
            <div class="col-md-6">
                <div class="input-group">
                    <span class="input-group-text"><i class="bx bx-search"></i></span>
                    <input type="text" class="form-control" placeholder="Search videos by title or description" name="search" value="<?= htmlspecialchars($searchQuery) ?>">
                </div>
            </div>
            <div class="col-md-4">
                <select class="form-select" name="category">
                    <option value="">All Categories</option>
                    <?php foreach($categories as $category): ?>
                    <option value="<?= $category['name'] ?>" <?= $categoryFilter === $category['name'] ? 'selected' : '' ?>>
                        <?= $category['name'] ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-primary w-100">Apply Filter</button>
            </div>
        </form>
    </div>

    <!-- Video Grid -->
    <div class="row g-3">
        <?php if (count($videos) > 0): ?>
            <?php foreach ($videos as $index => $video): ?>
                <div class="col-md-6 col-lg-4 mb-4" style="animation-delay: <?= $index * 0.05 ?>s;">
                    <div class="video-card">
                        <div class="video-thumbnail">
                            <?php if (isset($video['thumbnail']) && (file_exists($video['thumbnail']) || strpos($video['thumbnail'], 'http') === 0)): ?>
                            <img src="<?= $video['thumbnail'] ?>" alt="<?= $video['title'] ?>" loading="lazy">
                        <?php else: ?>
                        <div class="placeholder-thumbnail">
                            <i class="bx bx-video"></i>
                        </div>
                        <?php endif; ?>
                        <div class="video-duration">
                            <?= $video['duration'] ?>
                        </div>
                        <?php if ($video['completed']): ?>
                        <div class="video-completed">
                            <i class="bx bx-check-circle"></i>
                        </div>
                        <?php endif; ?>
                        <div class="video-play-button">
                            <a href="video-player.php?id=<?= $video['id'] ?>" class="play-icon">
                                <i class="bx bx-play"></i>
                            </a>
                        </div>
                        <?php if ($video['last_position'] > 0 && !$video['completed']): ?>
                        <?php
                            // Calculate progress percentage
                            $durationSeconds = 0;
                            list($minutes, $seconds) = explode(':', $video['duration']);
                            $durationSeconds = ($minutes * 60) + $seconds;
                            $progressPercent = $durationSeconds > 0 ? min(100, ($video['last_position'] / $durationSeconds) * 100) : 0;
                        ?>
                        <div class="video-progress">
                            <div class="progress">
                                <div class="progress-bar" role="progressbar" style="width: <?= $progressPercent ?>%"></div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="card-body">
                        <h5 class="video-title"><?= $video['title'] ?></h5>
                        <p class="video-category">
                            <span class="badge bg-primary-light text-primary"><?= $video['category_name'] ?></span>
                        </p>
                        <div class="video-meta">
                            <span><i class="bx bx-time"></i> <?= $video['duration'] ?></span>
                            <span><i class="bx bx-file"></i> <?= $video['file_size'] ?></span>
                        </div>
                        <p class="video-description"><?= substr($video['description'], 0, 80) ?>...</p>
                    </div>
                    <div class="card-footer">
                        <a href="video-player.php?id=<?= $video['id'] ?>" class="btn btn-primary btn-sm">
                            <?php 
                            $videoId = $video['id'];
                            $completed = isset($videoProgress[$videoId]) && $videoProgress[$videoId]['completed'];
                            $hasProgress = isset($videoProgress[$videoId]) && $videoProgress[$videoId]['last_position'] > 0;
                            
                            if ($completed): ?>
                                <i class="bx bx-refresh"></i> Rewatch
                            <?php elseif ($hasProgress): ?>
                                <i class="bx bx-play"></i> Continue
                            <?php else: ?>
                                <i class="bx bx-play"></i> Watch
                            <?php endif; ?>
                        </a>
                        <?php if ($video['is_free']): ?>
                        <a href="download-video.php?id=<?= $video['id'] ?>" class="btn btn-outline-primary btn-sm ms-1">
                            <i class="bx bx-download"></i> Download
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    <?php else: ?>
        <div class="col-12">
            <div class="empty-state">
                <div class="empty-state-icon">
                    <i class="bx bx-video-off"></i>
                </div>
                <h4>No Videos Found</h4>
                <p>You haven't purchased any videos yet or no videos match your search criteria. Explore our video library to find content that matches your learning interests.</p>
                <a href="browse-videos.php" class="btn btn-primary"><i class="bx bx-search"></i> Browse Video Library</a>
            </div>
        </div>
    <?php endif; ?>
</div>
