<?php
// Set current page for active menu highlighting
$current_page = 'assignments.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Get assignment ID from URL
$assignment_id = intval($_GET['id'] ?? 0);
$userId = $_SESSION['user_id'];

if ($assignment_id <= 0) {
    header('Location: assignments.php');
    exit;
}

// Get assignment details and submissions
try {
    echo "<!-- Debug: Starting database queries -->\n";
    
    // Get assignment information (removed score reference to fix column error)
    $assignment_query = "
        SELECT a.*, c.title as course_title, c.course_id,
               COUNT(DISTINCT s.student_id) as total_submissions,
               0 as avg_score
        FROM assignments a 
        JOIN courses c ON a.course_id = c.course_id 
        LEFT JOIN assignment_submissions s ON a.assignment_id = s.assignment_id 
        WHERE a.assignment_id = ? AND a.teacher_id = ?
        GROUP BY a.assignment_id
    ";
    
    echo "<!-- Debug: Preparing assignment query -->\n";
    $stmt = $conn->prepare($assignment_query);
    if (!$stmt) {
        die("Assignment query prepare failed: " . $conn->error);
    }
    
    echo "<!-- Debug: Executing assignment query for ID: $assignment_id, User: $userId -->\n";
    $stmt->bind_param("ii", $assignment_id, $userId);
    $stmt->execute();
    $assignment_result = $stmt->get_result();
    $assignment = $assignment_result->fetch_assoc();
    
    echo "<!-- Debug: Assignment found: " . ($assignment ? "YES" : "NO") . " -->\n";
    
    if (!$assignment) {
        echo "<!-- Debug: Assignment not found, redirecting -->\n";
        die("Assignment not found or you don't have permission to view it. <a href='assignments.php'>← Back to Assignments</a>");
    }
    
    // Get all submissions for this assignment
    echo "<!-- Debug: Getting submissions for assignment $assignment_id -->\n";
    $submissions_query = "
        SELECT s.submission_id, s.assignment_id, s.student_id, 
               s.submission_text, s.file_url, s.submission_date,
               u.first_name, u.last_name, u.email
        FROM assignment_submissions s
        JOIN users u ON s.student_id = u.user_id
        WHERE s.assignment_id = ?
        ORDER BY s.submission_date DESC
    ";
    
    $stmt = $conn->prepare($submissions_query);
    if (!$stmt) {
        die("Submissions query prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $assignment_id);
    $stmt->execute();
    $submissions_result = $stmt->get_result();
    $submissions = [];
    while ($row = $submissions_result->fetch_assoc()) {
        $submissions[] = $row;
    }
    
    echo "<!-- Debug: Found " . count($submissions) . " submissions -->\n";
    
    // Parse assignment data
    $submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
    $allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];
    
    // Fix double-encoded formats if needed
    if (!empty($allowed_formats) && is_array($allowed_formats) && is_string($allowed_formats[0])) {
        $allowed_formats = json_decode($allowed_formats[0], true) ?: [];
    }
    
} catch (Exception $e) {
    error_log("Error in view-submissions.php: " . $e->getMessage());
    header('Location: assignments.php');
    exit;
}

// Handle score update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_score') {
    // Debug: Log all POST data
    error_log("POST data received: " . print_r($_POST, true));
    
    $student_id = intval($_POST['student_id'] ?? 0);
    $score = floatval($_POST['score'] ?? 0);
    $feedback = trim($_POST['feedback'] ?? '');
    
    error_log("Processing grade update - Student ID: $student_id, Score: $score, Assignment ID: $assignment_id");
    
    if ($student_id > 0) {
        try {
            $update_query = "UPDATE assignment_submissions SET grade = ?, feedback = ? WHERE assignment_id = ? AND student_id = ?";
            $stmt = $conn->prepare($update_query);
            
            if (!$stmt) {
                throw new Exception("Prepare failed: " . $conn->error);
            }
            
            $stmt->bind_param("dsii", $score, $feedback, $assignment_id, $student_id);
            
            if ($stmt->execute()) {
                $affected_rows = $stmt->affected_rows;
                error_log("Grade update executed. Affected rows: $affected_rows");
                
                if ($affected_rows > 0) {
                    $success_message = "Grade updated successfully!";
                    // Add a small delay to ensure the update is committed
                    sleep(1);
                    header("Location: view-submissions.php?id=" . $assignment_id . "&success=1");
                    exit;
                } else {
                    $error_message = "No rows were updated. Check if submission exists.";
                    error_log("No rows affected in grade update");
                }
            } else {
                throw new Exception("Execute failed: " . $stmt->error);
            }
        } catch (Exception $e) {
            $error_message = "Error updating grade: " . $e->getMessage();
            error_log("Grade update error: " . $e->getMessage());
        }
    } else {
        $error_message = "Invalid student ID: $student_id";
        error_log("Invalid student ID received: $student_id");
    }
}

$page_title = "View Submissions - " . $assignment['title'];
?>

<!-- Page Content Starts Here -->
<style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }

        .header {
            background: linear-gradient(135deg, var(--panadite-primary), var(--panadite-secondary));
            color: white;
            padding: 2rem 0;
            text-align: center;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: white;
            text-decoration: none;
            margin-bottom: 1rem;
            padding: 0.5rem 1rem;
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.1);
            transition: background 0.3s;
        }

        .back-link:hover {
            background: rgba(255, 255, 255, 0.2);
            color: white;
            text-decoration: none;
        }

        .assignment-info {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            margin: 2rem 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .assignment-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem;
            background: var(--bg-light);
            border-radius: 8px;
        }

        .submissions-section {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            margin: 2rem 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .submissions-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--border-color);
        }

        .submission-card {
            border: 1px solid var(--border-color);
            border-radius: 12px;
            margin-bottom: 1.5rem;
            overflow: hidden;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .submission-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .submission-header {
            background: var(--bg-light);
            padding: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .student-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .submission-content {
            padding: 1.5rem;
        }

        .files-list {
            margin: 1rem 0;
        }

        .file-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            background: var(--bg-light);
            border-radius: 8px;
            margin-bottom: 0.5rem;
        }

        .grading-section {
            background: #f8f9fa;
            padding: 1.5rem;
            border-top: 1px solid var(--border-color);
        }

        .grading-form {
            display: grid;
            grid-template-columns: 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .form-control {
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-family: inherit;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-primary {
            background: var(--panadite-primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--panadite-secondary);
        }

        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }

        .badge-secondary {
            background: #e2e3e5;
            color: #383d41;
        }

        .no-submissions {
            text-align: center;
            padding: 3rem;
            color: var(--text-secondary);
        }

        .download-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: var(--panadite-primary);
            color: white;
            text-decoration: none;
            border-radius: 6px;
            font-size: 0.9rem;
            transition: background 0.3s;
        }

        .download-btn:hover {
            background: var(--panadite-secondary);
            color: white;
            text-decoration: none;
        }

        @media (max-width: 768px) {
            .grading-form {
                grid-template-columns: 1fr;
            }
            
            .assignment-meta {
                grid-template-columns: 1fr;
            }
        }
    </style>

<!-- Main Content Container -->
<div class="container-fluid mt-4">
    <div class="row">
        <div class="col-12">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2 class="mb-1">View Submissions</h2>
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="assignments.php">Assignments</a></li>
                            <li class="breadcrumb-item active" aria-current="page"><?= htmlspecialchars($assignment['title']) ?></li>
                        </ol>
                    </nav>
                </div>
                <a href="assignments.php" class="btn btn-outline-primary">
                    <i class="bx bx-arrow-back"></i> Back to Assignments
                </a>
            </div>

    <div class="container">
        <!-- Success/Error Messages -->
        <?php if (isset($success_message)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bx bx-check-circle"></i> <?= htmlspecialchars($success_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bx bx-error-circle"></i> <?= htmlspecialchars($error_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['success']) && $_GET['success'] == '1'): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bx bx-check-circle"></i> Grade updated successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Assignment Information -->
        <div class="assignment-info">
            <h2><i class="bx bx-info-circle"></i> <?= htmlspecialchars($assignment['title']) ?> Assignment Details</h2>
            
            <div class="assignment-meta">
                <div class="meta-item">
                    <i class="bx bx-calendar" style="color: var(--panadite-primary);"></i>
                    <div>
                        <strong>Due Date:</strong><br>
                        <?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?>
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-star" style="color: var(--panadite-accent);"></i>
                    <div>
                        <strong>Points:</strong><br>
                        <?= $assignment['points'] ?> points
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-user-check" style="color: var(--success-color);"></i>
                    <div>
                        <strong>Submissions:</strong><br>
                        <?= $assignment['total_submissions'] ?> received
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-trending-up" style="color: var(--panadite-primary);"></i>
                    <div>
                        <strong>Average Score:</strong><br>
                        <?= $assignment['avg_score'] ? $assignment['avg_score'] . '%' : 'Not graded' ?>
                    </div>
                </div>
            </div>

            <?php if (!empty($submission_types)): ?>
            <div style="margin-top: 1.5rem;">
                <strong>Required Submission Types:</strong>
                <div style="margin-top: 0.5rem;">
                    <?php foreach ($submission_types as $type): ?>
                        <span class="badge badge-secondary" style="margin-right: 0.5rem;">
                            <?= ucwords(str_replace('_', ' ', $type)) ?>
                        </span>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($allowed_formats)): ?>
            <div style="margin-top: 1rem;">
                <strong>Allowed File Formats:</strong>
                <span style="color: var(--text-secondary);"><?= strtoupper(implode(', ', $allowed_formats)) ?></span>
            </div>
            <?php endif; ?>
        </div>

        <!-- Submissions Section -->
        <div class="submissions-section">
            <div class="submissions-header">
                <h2><i class="bx bx-file-blank"></i> Student Submissions</h2>
                <?php if (count($submissions) > 0): ?>
                    <div>
                        <span class="badge badge-success"><?= count($submissions) ?> submissions</span>
                    </div>
                <?php endif; ?>
            </div>

            <?php if (empty($submissions)): ?>
                <div class="no-submissions">
                    <i class="bx bx-inbox" style="font-size: 4rem; color: var(--text-secondary); margin-bottom: 1rem;"></i>
                    <h3>No submissions yet</h3>
                    <p>Students haven't submitted any work for this assignment.</p>
                </div>
            <?php else: ?>
                <?php foreach ($submissions as $submission): ?>
                    <div class="submission-card">
                        <div class="submission-header">
                            <div class="student-info">
                                <div>
                                    <h4><?= htmlspecialchars($submission['first_name'] . ' ' . $submission['last_name']) ?></h4>
                                    <p style="color: var(--text-secondary); margin: 0;"><?= htmlspecialchars($submission['email']) ?></p>
                                </div>
                                <div style="text-align: right;">
                                    <div style="color: var(--text-secondary); font-size: 0.9rem;">
                                        Submitted: <?= date('M d, Y g:i A', strtotime($submission['submission_date'])) ?>
                                    </div>
                                    <?php if ($submission['grade'] !== null && $submission['grade'] != ''): ?>
                                        <div style="font-weight: 600; color: var(--panadite-primary); margin-top: 0.25rem;">
                                            Score: <?= $submission['grade'] ?>%
                                        </div>
                                    <?php else: ?>
                                        <span class="badge badge-warning">Not Graded</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="submission-content">
                            <?php if (!empty($submission['submission_text'])): ?>
                                <div style="margin-bottom: 1.5rem;">
                                    <h5><i class="bx bx-text"></i> Text Submission:</h5>
                                    <div style="background: var(--bg-light); padding: 1rem; border-radius: 8px; margin-top: 0.5rem; white-space: pre-wrap;">
                                        <?= htmlspecialchars($submission['submission_text']) ?>
                                    </div>
                                </div>
                            <?php endif; ?>

                            <?php if (!empty($submission['file_url'])): ?>
                                <?php
                                $file_url_raw = $submission['file_url'];
                                $files = null;
                                $display_files = [];
                                
                                // Method 1: Try JSON format (new format)
                                $files = json_decode($file_url_raw, true);
                                
                                // Method 2: If failed, try unescaping JSON
                                if (!is_array($files)) {
                                    $files = json_decode(stripslashes($file_url_raw), true);
                                }
                                
                                // Method 3: Try fixing common JSON issues
                                if (!is_array($files)) {
                                    $cleaned = str_replace(['\\"', "\\'"], ['"', "'"], $file_url_raw);
                                    $files = json_decode($cleaned, true);
                                }
                                
                                // Method 4: Handle truncated JSON (fix incomplete data)
                                if (!is_array($files) && substr($file_url_raw, 0, 1) === '[') {
                                    $last_brace = strrpos($file_url_raw, '}');
                                    if ($last_brace !== false) {
                                        $fixed_json = substr($file_url_raw, 0, $last_brace + 1) . ']';
                                        $files = json_decode($fixed_json, true);
                                    }
                                }
                                
                                // Method 5: Handle LEGACY format (simple string filename)
                                if (!is_array($files)) {
                                    // Legacy format: simple filename string
                                    $filename = trim($file_url_raw);
                                    if (!empty($filename)) {
                                        // Extract original name from legacy filename
                                        $original_name = $filename;
                                        // Try to get a more readable name
                                        if (preg_match('/assignment_\d+_student_\d+_\d+\.(\w+)$/', $filename, $matches)) {
                                            $extension = $matches[1];
                                            $original_name = "Submitted File." . $extension;
                                        }
                                        
                                        $display_files[] = [
                                            'original_name' => $original_name,
                                            'file_path' => $filename,
                                            'type' => 'legacy'
                                        ];
                                    }
                                } else {
                                    // JSON format worked - use parsed files
                                    $display_files = $files;
                                }
                                
                                if (!empty($display_files)):
                                ?>
                                    <div class="files-list">
                                        <h6><i class="bx bx-file"></i> Uploaded Files:</h6>
                                        <?php foreach ($display_files as $file): ?>
                                            <div class="file-item mb-2">
                                                <i class="bx bx-file-blank me-2"></i>
                                                <span class="file-name"><?= htmlspecialchars($file['original_name'] ?? 'Unknown file') ?></span>
                                                <a href="../uploads/assignments/<?= htmlspecialchars($file['file_path']) ?>" 
                                                   class="btn btn-sm btn-outline-primary ms-2" 
                                                   download="<?= htmlspecialchars($file['original_name'] ?? 'file') ?>">
                                                    <i class="bx bx-download"></i> Download
                                                </a>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php else: ?>
                                    <p class="text-muted">No files attached.</p>
                                <?php endif; ?>
                            <?php endif; ?>

                            <?php if (!empty($submission['feedback'])): ?>
                                <div style="margin-top: 1.5rem;">
                                    <h5><i class="bx bx-message-detail"></i> Instructor Feedback:</h5>
                                    <div style="background: #fff3cd; padding: 1rem; border-radius: 8px; margin-top: 0.5rem; border-left: 4px solid var(--panadite-accent);">
                                        <?= nl2br(htmlspecialchars($submission['feedback'])) ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Grading Section -->
                        <div class="grading-section">
                            <h5><i class="bx bx-edit"></i> Grade Submission</h5>
                            <form method="POST" class="grading-form">
                                <input type="hidden" name="action" value="update_score">
                                <input type="hidden" name="student_id" value="<?= $submission['student_id'] ?>">
                                
                                <div class="form-group">
                                    <label for="score_<?= $submission['student_id'] ?>">Score (%):</label>
                                    <input type="number" 
                                           id="score_<?= $submission['student_id'] ?>" 
                                           name="score" 
                                           min="0" 
                                           max="100" 
                                           step="0.1"
                                           value="<?= $submission['grade'] ?? '' ?>" 
                                           class="form-control" 
                                           placeholder="Enter score">
                                </div>
                                
                                <div class="form-group">
                                    <label for="feedback_<?= $submission['student_id'] ?>">Feedback:</label>
                                    <textarea id="feedback_<?= $submission['student_id'] ?>" 
                                              name="feedback" 
                                              class="form-control" 
                                              rows="3" 
                                              placeholder="Enter feedback for student"><?= htmlspecialchars($submission['feedback'] ?? '') ?></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bx bx-save"></i>
                                        Update Grade
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</div>
</div>
